package com.gtis.archive.core.support.hibernate.envers;

import org.hibernate.Query;
import org.hibernate.envers.RevisionType;
import org.hibernate.envers.configuration.AuditConfiguration;
import org.hibernate.envers.configuration.AuditEntitiesConfiguration;
import org.hibernate.envers.entities.mapper.relation.MiddleIdData;
import org.hibernate.envers.query.criteria.AuditCriterion;
import org.hibernate.envers.query.impl.AbstractAuditQuery;
import org.hibernate.envers.reader.AuditReaderImplementor;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * 与EntitiesAtRevisionQuery区别是添加了classloader, 并且在list里使用fixedEntityInstantiator替代entityInstantiator
 *
 * @author linlong
 * @since 2019.04.09
 */
public class FixedEntityAtRevisionQuery extends AbstractAuditQuery {

    private final ClassLoader classLoader;

    private final Number revision;

    /**
     * 用于初始化，生成对象
     */
    private FixedEntityInstantiator fixedEntityInstantiator;

    public FixedEntityAtRevisionQuery(AuditConfiguration verCfg,
                                   AuditReaderImplementor versionsReader, Class<?> cls,
                                   Number revision, ClassLoader classLoader) {
        super(verCfg, versionsReader, cls);
        this.revision = revision;
        this.classLoader = classLoader;
        this.fixedEntityInstantiator = new FixedEntityInstantiator(verCfg, versionsReader, classLoader);
    }

    public FixedEntityAtRevisionQuery(AuditConfiguration verCfg,
                                   AuditReaderImplementor versionsReader, Class<?> cls,
                                      String entityName, Number revision, ClassLoader classLoader) {
        super(verCfg, versionsReader, cls, entityName);
        this.revision = revision;
        this.classLoader = classLoader;
        this.fixedEntityInstantiator = new FixedEntityInstantiator(verCfg, versionsReader, classLoader);
    }

    @SuppressWarnings({"unchecked"})
    @Override
    public List list() {
        /*
         * The query that we need to create:
         *   SELECT new list(e) FROM versionsReferencedEntity e
         *   WHERE
         * (all specified conditions, transformed, on the "e" entity) AND
         * (selecting e entities at revision :revision)
         *   --> for DefaultAuditStrategy:
         *     e.revision = (SELECT max(e2.revision) FROM versionsReferencedEntity e2
         *       WHERE e2.revision <= :revision AND e2.id = e.id)
         *
         *   --> for ValidityAuditStrategy:
         *     e.revision <= :revision and (e.endRevision > :revision or e.endRevision is null)
         *
         *     AND
         * (only non-deleted entities)
         *     e.revision_type != DEL
         */
        AuditEntitiesConfiguration verEntCfg = verCfg.getAuditEntCfg();
        String revisionPropertyPath = verEntCfg.getRevisionNumberPath();
        String originalIdPropertyName = verEntCfg.getOriginalIdPropName();

        MiddleIdData referencedIdData = new MiddleIdData(verEntCfg, verCfg.getEntCfg().get(entityName).getIdMappingData(),
                null, entityName, verCfg.getEntCfg().isVersioned(entityName));

        // (selecting e entities at revision :revision)
        // --> based on auditStrategy (see above)
        verCfg.getAuditStrategy().addEntityAtRevisionRestriction(verCfg.getGlobalCfg(), qb, revisionPropertyPath,
                verEntCfg.getRevisionEndFieldName(), true, referencedIdData,
                revisionPropertyPath, originalIdPropertyName, "e", "e2");

        // e.revision_type != DEL
        qb.getRootParameters().addWhereWithParam(verEntCfg.getRevisionTypePropName(), "<>", RevisionType.DEL);

        // all specified conditions
        for (AuditCriterion criterion : criterions) {
            criterion.addToQuery(verCfg, entityName, qb, qb.getRootParameters());
        }

        Query query = buildQuery();
        // add named parameter (only used for ValidAuditTimeStrategy)
        List<String> params = Arrays.asList(query.getNamedParameters());
        if (params.contains("revision")) {
            query.setParameter("revision", revision);
        }
        List queryResult = query.list();

        if (hasProjection) {
            return queryResult;
        } else {
            List result = new ArrayList();
            fixedEntityInstantiator.addInstancesFromVersionsEntities(entityName, result, queryResult, revision);
            return result;
        }
    }
}
