package com.gtis.archive.core.support.hibernate.envers;

import org.hibernate.HibernateException;
import org.hibernate.NonUniqueResultException;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.engine.SessionImplementor;
import org.hibernate.envers.configuration.AuditConfiguration;
import org.hibernate.envers.exception.AuditException;
import org.hibernate.envers.exception.NotAuditedException;
import org.hibernate.envers.exception.RevisionDoesNotExistException;
import org.hibernate.envers.query.AuditEntity;
import org.hibernate.envers.query.AuditQueryCreator;
import org.hibernate.envers.reader.AuditReaderImplementor;
import org.hibernate.envers.reader.FirstLevelCache;
import org.hibernate.envers.synchronization.AuditProcess;
import org.hibernate.event.EventSource;
import org.hibernate.proxy.HibernateProxy;

import javax.persistence.NoResultException;
import java.util.*;

import static org.hibernate.envers.tools.ArgumentsTools.checkNotNull;
import static org.hibernate.envers.tools.ArgumentsTools.checkPositive;
import static org.hibernate.envers.tools.Tools.getTargetClassIfProxied;

/**
 * @author linlong
 * @since 2019.04.09
 */
public class FixedAuditReaderImpl implements AuditReaderImplementor {


    private final AuditConfiguration verCfg;
    private final SessionImplementor sessionImplementor;
    private final Session session;
    private final FirstLevelCache firstLevelCache;
    private final ClassLoader classLoader;

    private String entityNameParam = "Entity name";
    private String primaryKeyParam = "Primary key";
    private String entityRevisionParam = "Entity revision";

    public FixedAuditReaderImpl(AuditConfiguration verCfg, Session session,
                                SessionImplementor sessionImplementor, ClassLoader classLoader) {
        firstLevelCache = new FirstLevelCache();
        this.session = session;
        this.sessionImplementor = sessionImplementor;
        this.verCfg = verCfg;
        this.classLoader = classLoader;
    }

    private void checkSession() {
        if (!session.isOpen()) {
            throw new IllegalStateException("The associated entity manager is closed!");
        }
    }

    @Override
    public SessionImplementor getSessionImplementor() {
        return sessionImplementor;
    }

    @Override
    public Session getSession() {
        return session;
    }

    @Override
    public FirstLevelCache getFirstLevelCache() {
        return firstLevelCache;
    }

    @Override
    public <T> T find(Class<T> cls, Object primaryKey, Number revision) throws NotAuditedException {
        cls = getTargetClassIfProxied(cls);
        return this.find(cls, cls.getName(), primaryKey, revision);
    }

    @Override
    @SuppressWarnings({"unchecked"})
    public <T> T find(Class<T> cls, String entityName, Object primaryKey, Number revision) throws NotAuditedException{
        cls = getTargetClassIfProxied(cls);
        checkNotNull(cls, "Entity class");
        checkNotNull(entityName, entityNameParam);
        checkNotNull(primaryKey, primaryKeyParam);
        checkNotNull(revision, entityRevisionParam);
        checkPositive(revision, entityRevisionParam);
        checkSession();

        if (!verCfg.getEntCfg().isVersioned(entityName)) {
            throw new NotAuditedException(entityName, entityName + " is not versioned!");
        }

        if (firstLevelCache.contains(entityName, revision, primaryKey)) {
            return (T) firstLevelCache.get(entityName, revision, primaryKey);
        }

        Object result;
        try {
            // The result is put into the cache by the entity instantiator called from the query
            result = createQuery().forEntitiesAtRevision(cls, entityName, revision)
                    .add(AuditEntity.id().eq(primaryKey)).getSingleResult();
        } catch (NoResultException e) {
            result = null;
        } catch (NonUniqueResultException e) {
            throw new AuditException(e);
        }

        return (T) result;
    }

    @Override
    public List<Number> getRevisions(Class<?> cls, Object primaryKey) throws NotAuditedException {
        cls = getTargetClassIfProxied(cls);
        return this.getRevisions(cls, cls.getName(), primaryKey);
    }

    @Override
    @SuppressWarnings({"unchecked"})
    public List<Number> getRevisions(Class<?> cls, String entityName, Object primaryKey) throws NotAuditedException {
        // if a class is not versioned from the beginning, there's a missing ADD rev - what then?
        cls = getTargetClassIfProxied(cls);
        checkNotNull(cls, "Entity class");
        checkNotNull(entityName, entityNameParam);
        checkNotNull(primaryKey, primaryKeyParam);
        checkSession();

        if (!verCfg.getEntCfg().isVersioned(entityName)) {
            throw new NotAuditedException(entityName, entityName + " is not versioned!");
        }

        return createQuery().forRevisionsOfEntity(cls, entityName, false, true)
                .addProjection(AuditEntity.revisionNumber())
                .add(AuditEntity.id().eq(primaryKey))
                .getResultList();
    }

    @Override
    public Date getRevisionDate(Number revision) throws RevisionDoesNotExistException {
        checkNotNull(revision, entityRevisionParam);
        checkPositive(revision, entityRevisionParam);
        checkSession();

        Query query = verCfg.getRevisionInfoQueryCreator().getRevisionDateQuery(session, revision);

        try {
            Object timestampObject = query.uniqueResult();
            if (timestampObject == null) {
                throw new RevisionDoesNotExistException(revision);
            }

            // The timestamp object is either a date or a long
            return timestampObject instanceof Date ? (Date) timestampObject : new Date((Long) timestampObject);
        } catch (NonUniqueResultException e) {
            throw new AuditException(e);
        }
    }

    @Override
    public Number getRevisionNumberForDate(Date date) {
        checkNotNull(date, "Date of revision");
        checkSession();

        Query query = verCfg.getRevisionInfoQueryCreator().getRevisionNumberForDateQuery(session, date);

        try {
            Number res = (Number) query.uniqueResult();
            if (res == null) {
                throw new RevisionDoesNotExistException(date);
            }

            return res;
        } catch (NonUniqueResultException e) {
            throw new AuditException(e);
        }
    }

    @Override
    @SuppressWarnings({"unchecked"})
    public <T> T findRevision(Class<T> revisionEntityClass, Number revision) throws RevisionDoesNotExistException {
        checkNotNull(revision, entityRevisionParam);
        checkPositive(revision, entityRevisionParam);
        checkSession();

        Set<Number> revisions = new HashSet<Number>(1);
        revisions.add(revision);
        Query query = verCfg.getRevisionInfoQueryCreator().getRevisionsQuery(session, revisions);

        try {
            T revisionData = (T) query.uniqueResult();

            if (revisionData == null) {
                throw new RevisionDoesNotExistException(revision);
            }

            return revisionData;
        } catch (NonUniqueResultException e) {
            throw new AuditException(e);
        }
    }

    @Override
    @SuppressWarnings({"unchecked"})
    public <T> Map<Number, T> findRevisions(Class<T> revisionEntityClass, Set<Number> revisions) throws RuntimeException {
        Map<Number, T> result = new HashMap<Number, T>(revisions.size());

        for (Number revision : revisions) {
            checkNotNull(revision, entityRevisionParam);
            checkPositive(revision, entityRevisionParam);
        }
        checkSession();

        Query query = verCfg.getRevisionInfoQueryCreator().getRevisionsQuery(session, revisions);

        try {
            List<T> revisionList = query.list();
            for (T revision : revisionList) {
                Number revNo = verCfg.getRevisionInfoNumberReader().getRevisionNumber(revision);
                result.put(revNo, revision);
            }

            return result;
        } catch (HibernateException e) {
            throw new AuditException(e);
        }
    }

    @Override
    @SuppressWarnings({"unchecked"})
    public <T> T getCurrentRevision(Class<T> revisionEntityClass, boolean persist) {
        if (!(session instanceof EventSource)) {
            throw new IllegalArgumentException("The provided session is not an EventSource!");
        }

        // Obtaining the current audit sync
        AuditProcess auditProcess = verCfg.getSyncManager().get((EventSource) session);

        // And getting the current revision data
        return (T) auditProcess.getCurrentRevisionData(session, persist);
    }

    @Override
    public AuditQueryCreator createQuery() {
        return new FixedAuditQueryCreator(verCfg, this, classLoader);
    }

    @Override
    public boolean isEntityClassAudited(Class<?> entityClass) {
        entityClass = getTargetClassIfProxied(entityClass);
        return this.isEntityNameAudited(entityClass.getName());
    }


    @Override
    public boolean isEntityNameAudited(String entityName) {
        checkNotNull(entityName, entityNameParam);
        checkSession();
        return (verCfg.getEntCfg().isVersioned(entityName));
    }


    @Override
    public String getEntityName(Object primaryKey, Number revision, Object entity) throws HibernateException {
        checkNotNull(primaryKey, primaryKeyParam);
        checkNotNull(revision, entityRevisionParam);
        checkPositive(revision, entityRevisionParam);
        checkNotNull(entity, "Entity");
        checkSession();

        // Unwrap if necessary
        if (entity instanceof HibernateProxy) {
            entity = ((HibernateProxy) entity).getHibernateLazyInitializer().getImplementation();
        }
        if (firstLevelCache.containsEntityName(primaryKey, revision, entity)) {
            // it's on envers FLC!
            return firstLevelCache.getFromEntityNameCache(primaryKey, revision, entity);
        } else {
            throw new HibernateException(
                    "Envers can't resolve entityName for historic entity. The id, revision and entity is not on envers first level cache.");
        }
    }
}
