package com.gtis.archive.core.support.hibernate.envers;

import org.hibernate.Session;
import org.hibernate.engine.SessionImplementor;
import org.hibernate.envers.AuditReader;
import org.hibernate.envers.event.AuditEventListener;
import org.hibernate.envers.exception.AuditException;
import org.hibernate.event.EventListeners;
import org.hibernate.event.PostInsertEventListener;

import javax.persistence.EntityManager;

import static org.hibernate.envers.tools.ArraysTools.arrayIncludesInstanceOf;

/**
 * @author linlong
 * @since 2019.04.09
 */
public class FixedAuditReaderFactory {

    /**
     * Create an audit reader associated with an open session.
     * @param session An open session.
     * @return An audit reader associated with the given sesison. It shouldn't be used
     * after the session is closed.
     * @throws AuditException When the given required listeners aren't installed.
     */
    public static AuditReader get(Session session, ClassLoader classLoader) throws AuditException {
        SessionImplementor sessionImpl;
        if (!(session instanceof SessionImplementor)) {
            sessionImpl = (SessionImplementor) session.getSessionFactory().getCurrentSession();
        } else {
            sessionImpl = (SessionImplementor) session;
        }

        EventListeners listeners = sessionImpl.getListeners();

        for (PostInsertEventListener listener : listeners.getPostInsertEventListeners()) {
            if (listener instanceof AuditEventListener) {
                if (arrayIncludesInstanceOf(listeners.getPostUpdateEventListeners(), AuditEventListener.class) &&
                        arrayIncludesInstanceOf(listeners.getPostDeleteEventListeners(), AuditEventListener.class)) {
                    return new FixedAuditReaderImpl(((AuditEventListener) listener).getVerCfg(), session,
                            sessionImpl, classLoader);
                }
            }
        }

        throw new AuditException("You need to install the org.hibernate.envers.event.AuditEventListener " +
                "class as post insert, update and delete event listener.");
    }

    /**
     * Create an audit reader associated with an open entity manager.
     * @param entityManager An open entity manager.
     * @return An audit reader associated with the given entity manager. It shouldn't be used
     * after the entity manager is closed.
     * @throws AuditException When the given entity manager is not based on Hibernate, or if the required
     * listeners aren't installed.
     */
    public static AuditReader get(EntityManager entityManager, ClassLoader classLoader) throws AuditException {
        if (entityManager.getDelegate() instanceof Session) {
            return get((Session) entityManager.getDelegate(), classLoader);
        }

        if (entityManager.getDelegate() instanceof EntityManager) {
            return get((EntityManager) entityManager.getDelegate(), classLoader);
        }

        throw new AuditException("Hibernate EntityManager not present!");
    }
}
