package com.gtis.archive.core.support.hibernate.envers;

import org.hibernate.envers.configuration.AuditConfiguration;
import org.hibernate.envers.query.AuditQuery;
import org.hibernate.envers.query.AuditQueryCreator;
import org.hibernate.envers.reader.AuditReaderImplementor;

import static org.hibernate.envers.tools.ArgumentsTools.checkNotNull;
import static org.hibernate.envers.tools.ArgumentsTools.checkPositive;
import static org.hibernate.envers.tools.Tools.getTargetClassIfProxied;

/**
 * @author linlong
 * @since 2019.04.09
 */
public class FixedAuditQueryCreator extends AuditQueryCreator {

    private final AuditConfiguration auditCfg;
    private final AuditReaderImplementor auditReaderImplementor;

    private final ClassLoader classLoader;

    String entityRevision = "Entity revision";

    public FixedAuditQueryCreator(AuditConfiguration auditCfg, AuditReaderImplementor auditReaderImplementor,
                                  ClassLoader classLoader) {
        super(auditCfg, auditReaderImplementor);
        this.classLoader = classLoader;
        this.auditCfg = auditCfg;
        this.auditReaderImplementor = auditReaderImplementor;
    }

    /**
     * Creates a query, which will return entities satisfying some conditions (specified later),
     * at a given revision.
     * @param c Class of the entities for which to query.
     * @param revision Revision number at which to execute the query.
     * @return A query for entities at a given revision, to which conditions can be added and which
     * can then be executed. The result of the query will be a list of entities (beans), unless a
     * projection is added.
     */
    @Override
    public AuditQuery forEntitiesAtRevision(Class<?> c, Number revision) {
        checkNotNull(revision, entityRevision);
        checkPositive(revision, entityRevision);
        c = getTargetClassIfProxied(c);
        return new FixedEntityAtRevisionQuery(auditCfg, auditReaderImplementor, c, revision, classLoader);
    }

    /**
     * Creates a query, which will return entities satisfying some conditions (specified later),
     * at a given revision and a given entityName.
     * @param c Class of the entities for which to query.
     * @param entityName Name of the entity (if can't be guessed basing on the {@code c}).
     * @param revision Revision number at which to execute the query.
     * @return A query for entities at a given revision, to which conditions can be added and which
     * can then be executed. The result of the query will be a list of entities (beans), unless a
     * projection is added.
     */
    public AuditQuery forEntitiesAtRevision(Class<?> c, String entityName, Number revision) {
        checkNotNull(revision, entityRevision);
        checkPositive(revision, entityRevision);
        c = getTargetClassIfProxied(c);
        return new FixedEntityAtRevisionQuery(auditCfg, auditReaderImplementor, c, entityName, revision, classLoader);
    }

    /**
     * Creates a query, which selects the revisions, at which the given entity was modified.
     * Unless an explicit projection is set, the result will be a list of three-element arrays, containing:
     * <ol>
     * <li>the entity instance</li>
     * <li>revision entity, corresponding to the revision at which the entity was modified. If no custom
     * revision entity is used, this will be an instance of {@link org.hibernate.envers.DefaultRevisionEntity}</li>
     * <li>type of the revision (an enum instance of class {@link org.hibernate.envers.RevisionType})</li>.
     * </ol>
     * Additional conditions that the results must satisfy may be specified.
     * @param c Class of the entities for which to query.
     * @param selectEntitiesOnly If true, instead of a list of three-element arrays, a list of entites will be
     * returned as a result of executing this query.
     * @param selectDeletedEntities If true, also revisions where entities were deleted will be returned. The additional
     * entities will have revision type "delete", and contain no data (all fields null), except for the id field.
     * @return A query for revisions at which instances of the given entity were modified, to which
     * conditions can be added (for example - a specific id of an entity of class <code>c</code>), and which
     * can then be executed. The results of the query will be sorted in ascending order by the revision number,
     * unless an order or projection is added.
     */
    public AuditQuery forRevisionsOfEntity(Class<?> c, boolean selectEntitiesOnly, boolean selectDeletedEntities) {
        c = getTargetClassIfProxied(c);
        return new FixedRevisionOfEntityQuery(auditCfg, auditReaderImplementor, c, selectEntitiesOnly,selectDeletedEntities, classLoader);
    }

    /**
     * Creates a query, which selects the revisions, at which the given entity was modified and with a given entityName.
     * Unless an explicit projection is set, the result will be a list of three-element arrays, containing:
     * <ol>
     * <li>the entity instance</li>
     * <li>revision entity, corresponding to the revision at which the entity was modified. If no custom
     * revision entity is used, this will be an instance of {@link org.hibernate.envers.DefaultRevisionEntity}</li>
     * <li>type of the revision (an enum instance of class {@link org.hibernate.envers.RevisionType})</li>.
     * </ol>
     * Additional conditions that the results must satisfy may be specified.
     * @param c Class of the entities for which to query.
     * @param entityName Name of the entity (if can't be guessed basing on the {@code c}).
     * @param selectEntitiesOnly If true, instead of a list of three-element arrays, a list of entites will be
     * returned as a result of executing this query.
     * @param selectDeletedEntities If true, also revisions where entities were deleted will be returned. The additional
     * entities will have revision type "delete", and contain no data (all fields null), except for the id field.
     * @return A query for revisions at which instances of the given entity were modified, to which
     * conditions can be added (for example - a specific id of an entity of class <code>c</code>), and which
     * can then be executed. The results of the query will be sorted in ascending order by the revision number,
     * unless an order or projection is added.
     */
    public AuditQuery forRevisionsOfEntity(Class<?> c, String entityName, boolean selectEntitiesOnly, boolean selectDeletedEntities) {
        c = getTargetClassIfProxied(c);
        return new FixedRevisionOfEntityQuery(auditCfg, auditReaderImplementor, c, entityName, selectEntitiesOnly,selectDeletedEntities, classLoader);
    }
}
