package com.gtis.archive.core.impl;

import com.gtis.archive.core.Model;
import com.gtis.archive.core.ModelService;
import com.gtis.archive.core.ModelManager;
import com.gtis.archive.core.generater.ModelCodeGenerater;
import groovy.lang.GroovyClassLoader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;

import java.io.*;
import java.util.Collection;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2010-8-9
 */
public class ModelServiceImpl implements ModelService {
    private ModelManager modelManager;
    private ClassLoader classLoader;
    private Map<String, Class> loadedclasses = new ConcurrentHashMap<String, Class>();
    private Map<Class, String> namesMap = new ConcurrentHashMap<Class, String>();
    private Resource tempLocation;
    private final Logger logger = LoggerFactory.getLogger(getClass());

    @Override
    public ClassLoader getClassLoader() {
        return classLoader;
    }

    public void setModelManager(ModelManager modelManager) {
        this.modelManager = modelManager;
    }

    public void setTempLocation(Resource tempLocation) {
        this.tempLocation = tempLocation;
    }

    @Override
    public Class getClass(String modelName) throws ClassNotFoundException {
        Class clazz = loadedclasses.get(modelName);
        if (clazz == null) {
            clazz = classLoader.loadClass(Model.PACKAGE_NAME + "." + modelName);
            loadedclasses.put(modelName, clazz);
        }
        return clazz;
    }

    @Override
    public Collection<Class> getClasses() {
        return loadedclasses.values();
    }

    @Override
    public Model getModel(String modelName) {
        return modelManager.getModel(modelName);
    }

    @Override
    public Model getModel(Object entity) {
        return getModel(getModelName(entity));
    }

    @Override
    public String getModelName(Object obj) {
        Class clazz = obj.getClass();
        String modelName = namesMap.get(clazz);
        if (modelName == null) {
            for (String name : loadedclasses.keySet()) {
                if (loadedclasses.get(name) == clazz) {
                    namesMap.put(clazz, name);
                    modelName = name;
                    break;
                }
            }
        }
        return modelName;
    }

    @Override
    public Collection<Model> getModels() {
        return modelManager.getModels();
    }

    @Override
    public void refresh() {
        init();
    }

    @Override
    public void reload() {
        loadedclasses.clear();
        namesMap.clear();
        refresh();
    }

    public void init() {
        GroovyClassLoader gcl = new GroovyClassLoader();
        try {
            File path = tempLocation.getFile();
            if (!path.exists() && !path.mkdirs()) {
                logger.error("mkdir [{}] error",path.getAbsolutePath());
            }
            gcl.addURL(path.toURI().toURL());
            gcl.setShouldRecompile(true);
            this.classLoader = gcl;
        } catch (Exception e) {
            logger.error("error to init GroovyClassLoader,tempLocation [" + tempLocation + "] not exist", e);
        }
        Collection<Model> models = modelManager.getModels();
        for (Model model : models) {
            if (!model.isConstant()) {
                generateModelCode(model);
            }
            try {
                Class clazz = gcl.loadClass(toClassName(model), true, false);
                loadedclasses.put(model.getName(), clazz);
            } catch (ClassNotFoundException e) {
                logger.error("load model[" + model + "],error:[{}]", e);
            }
        }
    }

    private void generateModelCode(Model model) {
        Writer wf = null;
        try {
            File file = new File(tempLocation.getFile(), toClassName(model).replace(".", "/") + ".groovy");
            File dir = file.getParentFile();
            if (!dir.exists() && !dir.mkdirs()) {
                logger.error("mkdir[{}]error",dir.getAbsolutePath());
            }
            if (file.exists() && !file.delete()) {
                logger.error("删除文件[{}]出错！",file.getPath());
            }
            wf = new OutputStreamWriter(new FileOutputStream(file));
            ModelCodeGenerater.GROOVY.generate(model, wf);
        } catch (IOException e) {
            logger.error("generate model [" + model + "] error", e);
        } finally {
            try {
                if (wf != null) {
                    wf.close();
                }
            } catch (IOException ignored) {
                logger.error(ignored.getMessage());
            }
        }
    }

    private String toClassName(Model model) {
        return Model.PACKAGE_NAME + "." + model.getName();
    }
}
