package cn.gtmap.helium.client.util;

import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.ResourceUtils;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;

/**
 * gtmap egov conf helper
 * Author: <a href="mailto:yingxiufeng@gtmap.cn">yingxiufeng</a>
 * Date:  2016/6/18 15:04
 */
public final class EgovConfigHelper {

    public static final Logger logger = LoggerFactory.getLogger(EgovConfigHelper.class);

    public static final String EGOV_HOME_FOLDER = "egov-home";
    public static final String DEFAULT_ACTIVE_CONF = "default";
    public static final String PROPERTY_FILE_NAME = "egov.properties";


    /**
     * 获取egov的conf文件夹
     *
     * @param paths
     * @return
     */
    public static File getEgovConfHome(final String... paths) {
        File root = determineRootDir(paths);
        File home;
        if (root != null) {
            logger.info("Use egov-home dir:[" + root + "]");
            File activeFile = new File(root, "active.conf");
            String active = DEFAULT_ACTIVE_CONF;
            try {
                active = FileUtils.readFileToString(activeFile);
            } catch (IOException e) {
                logger.info("Read active profile from [" + activeFile.getAbsolutePath() + "] fail,use default");
            }
            home = new File(root, active);
            if (!home.exists()) {
                home = new File(root, DEFAULT_ACTIVE_CONF);
                if (!home.exists()) {
                    logger.error("Egov active conf [" + home.getAbsolutePath() + "] not exsit");
                }
            }
            logger.info("Use Custom config home dir:[" + home + "]");
        } else {
            home = new File("/opt/gtis/config/egov/default");
            if (home.exists()) {
                logger.warn("Use compact config home dir:[" + home.getAbsolutePath() + "],please convert to new format");
                return home;
            }
            URL url = null;
            try {
                url = ResourceUtils.getURL("/META-INF/conf/" + PROPERTY_FILE_NAME);
                if (url != null) {
                    try {
                        home = new File(url.toURI()).getParentFile().getParentFile();
                        logger.info("Custom config home not found,Use classpath config home dir [" + home.getAbsolutePath() + "]");
                    } catch (URISyntaxException ignored) {
                        return null;
                    }
                } else {
                    logger.error("Load config error,config not found");
                    return null;
                }
            } catch (FileNotFoundException e) {
                logger.error("Load config error,config not found");
                return null;
            }

        }
        return null;
    }

    /**
     * @param paths
     * @return
     */
    public static File determineRootDir(final String... paths) {
        File root;
        if (paths != null) {
            for (String path : paths) {
                root = new File(path);
                if (root.exists())
                    return root;
            }
        }
        for (String path : new String[]{
                System.getProperty("EGOV_HOME"),
                System.getenv("EGOV_HOME")
        }) {
            if (path != null) {
                root = new File(path);
                if (root.exists())
                    return root;
            }
        }
        for (String path : new String[]{
                System.getProperty("catalina.base"),
                System.getProperty("catalina.home"),
                System.getProperty("user.home")
        }) {
            if (path != null) {
                root = new File(path, EGOV_HOME_FOLDER);
                if (root.exists())
                    return root;
            }
        }
        return null;
    }
}
