/**
 * 地图及一些调用接口
 * @author by <a href="mailto:xxx@gmail.com">cyj</a>
 * @version v1.0 2019  Copyright gtmap Corp
 */
define([
    'leaflet',
    'esri-leaflet',
    'proj4-leaflet',
    'leaflet-pip',
    // 'webjars/lp-webjars/js/map/layersConfig',
    'webjars/lp-webjars/js/map/ServiceLayer',
    'webjars/lp-webjars/js/map/TimeLine',
    'webjars/lp-webjars/js/map/Legend'
],function (
    L,
    LEsri,
    LProj,
    LeafletPip,
    // Config,
    ServiceLayer,
    TimeLine,
    Legend
) {
    var $map,Config,that,layer = [];

    // 地图坐标系及层级参数的获取
    // todo 获取系统坐标系
    var crs = 4490;
    // 自定义比例尺
    var CSYS = {
        4490: {
            resolution: [
                1.40625,
                0.703125,
                0.3515625,
                0.17578125,
                0.087890625,
                0.0439453125,
                0.02197265625,
                0.010986328125,
                0.0054931640625,
                0.00274658203125,
                0.001373291015625,
                6.866455078125E-4,
                3.4332275390625E-4,
                1.71661376953125E-4,
                8.58306884765625E-5,
                4.291534423828125E-5,
                2.1457672119140625E-5,
                1.0728836059570312E-5,
                5.364418029785156E-6,
                2.682209064925356E-6,
                1.3411045324626732E-6
            ],
            proj4: '+proj=longlat +ellps=GRS80 +no_defs'
        },
        4528: {
            resolution: [],
            proj4: '+proj=tmerc +lat_0=0 +lon_0=120 +k=1 +x_0=40500000 +y_0=0 +ellps=GRS80 +units=m +no_defs'
        }
    };
    var origin = [-180, 90]; // 定义中心原点

    function InitMap() {
        that = this;
        // 模拟发布订阅模式进行监听更改
        // (图层切换)
        ServiceLayer.addEventListener(changeLayerControl);
        // (时间轴切换)
        $(document).on("timeLine-change",changeTimelineLayer);
    }

    /**
     * 调用接口
     * @type {{init: init, getMap: getMap, changeLayer: changeLayer, addLayer: addLayer, createMapLayer: createMapLayer, setOption: setOption, setLyr: setLyr}}
     */
    InitMap.prototype = {
        /**
         * 实例化一个地图
         * @param mapId
         * @param baseMap
         */
        init: function (mapId,baseLayer,baseMap) {
            that._id = mapId;
            Config = baseLayer;
            // 若不传底图字段时默认为天地图
            if(!baseMap){
                baseMap = "TDT";
            }
            try {
                render(mapId, baseLayer, baseMap)
            }catch (e){
                alert('地图初始化失败');
                console.log(e);
            }
        },

        /**
         * 获取实例的地图对象
         * @returns {*}
         */
        getMap: function () {
            return $map;
        },

        /**
         * 移除所有存储的图层并清空
         */
        changeLayer: function () {
            if(layer.length > 0){
                $.each(layer,function (i,e) {
                    $map.removeLayer(e);
                });
                layer = [];
            }
        },

        /**
         * 加载动态图层
         * @param url
         */
        addLayer: function (url) {
            //加油站
            var lyr = LEsri.dynamicMapLayer({url:url});
            layer.push(lyr);
            lyr.addTo($map);
        },

        /**
         * 根据类型判断创建服务图层
         * @param options
         */
        createMapLayer: function (options) {
            createLayerByOptions(options);
        },

        /**
         * 根据参数展示地图要素
         * @param option
         */
        setOption: function (option) {
            // 默认值继承传值
            var config = {
                title: true, // 标题
                layers: "FWSS", // 图层配置id
                legend: true, // 图例
                menu: true, // 是否展示菜单
                region: false // 是否展示行政区定位
            };
            $.extend(config,option);
            // 执行函数
            changeOption(config);
        },

        /**
         * push图层
         * @param lyr
         */
        setLyr: function (lyr) {
            layer.push(lyr);
        }
    };


    /***********************************************************逻辑处理层***/

    /********操作地图*******/

    /**
     * 渲染地图
     * @param mapId
     */
    function render(mapId,Config,baseMap) {
        // 获取底图配置
        var baseConfig = Config.baseLayer[baseMap];
        // 定义坐标系
        var CRS_4490 = new LProj.CRS("EPSG:4490", CSYS[crs].proj4,{
            // 这里可以有transformation、scales、resulutions、bounds几个参数提供
            // 选择，其中scales与resolutions不能同时配置
            resolutions: CSYS[crs].resolution,
            origin: origin,
            bounds: L.bounds(
                [114.32398008091434,30.60241588421052],
                [124.87085527765123,35.41992085688668]
            )
        });
        // 实例化地图去掉右下角水印
        var map = L.map(mapId,
            {   crs:CRS_4490,
                attributionControl: false,
                zoomControl: false
            });
        // 定位范围
        if(baseConfig.extent){
            var extent = baseConfig.extent;
            map.fitBounds([
                [extent.ymin,extent.xmin],
                [extent.ymax,extent.xmax]
            ]);
        }
        // 创建pane
        map.createPane("basePane");
        map.createPane("basePane2");
        map.getPane("basePane").style.zIndex = 100;
        map.getPane("basePane2").style.zIndex = 120;
        //创建底图图层组
        var baseGroup = [];
        $.each(baseConfig.layers,function (i,lyr) {
            baseGroup.push(createLayerByOptions(lyr))
        });

        L.layerGroup(baseGroup).addTo(map);
        // 传递地图对象
        $map = map;
    }

    /**
     * 根据类型返回对应服务图层
     * @param options
     * @returns {*}
     */
    function createLayerByOptions(options) {
        var _layer,type = options.type.toLocaleLowerCase();
        switch (type) {
            case "ags_tile":
                _layer = new LEsri.TiledMapLayer(options);
                break;
            case "ags_rest":
                _layer = new LEsri.DynamicMapLayer(options);
                break;
            default:
                console.log("未包含服务类型");
        }
        return _layer;
    }

    /********操作地图要素*******/

    /**
     * 根据参数添加地图状态
     * @param option
     */
    function changeOption(option) {
        var titleDom = $("#monitorMapTitle");
        removeRegionLayer();
        // 添加标题
        if(option.title){
            titleDom.show();
        }else{
            titleDom.hide();
        }
        // 控制图例
        if(option.legend){
            Legend.showLegend(true)
        }else{
            Legend.showLegend(false)
        }
        // 控制图层控制显示
        if(option.menu){
            ServiceLayer.showService(true);
        }else {
            ServiceLayer.showService(false);
        }

        addLayerByConfig(option.layers);

        // 控制行政区定位
        if(option.region){
            addBackButton();
            loadRegionData();
            openClickEvent($map);
        }else {
            closeClickEvent($map);
        }
    }

    /**
     * 根据参数加载图层
     * @param layerName
     */
    var showNode = false; // 标识
    function addLayerByConfig(layerName) {
        var timeLineDom = $("#monitorTimeLine");
        showNode = false;
        // todo 读取接口或者配置里的配置
        var lyrContent = Config.content;
        if(!lyrContent[layerName]){
            // alert("未找到该图层配置");
            return;
        }

        // 配置图层服务属性时
        if(lyrContent[layerName].layers){
            var allLayers = lyrContent[layerName].layers;
            // 隐藏时间轴
            timeLineDom.hide("slow");
            // 判断配置单图层且为动态服务时
            showNode = (allLayers.length === 1
                && allLayers[0].type === "ags_rest" )?true: false
            ServiceLayer.layers(allLayers);
            changeLayerControl(allLayers[0]);
        }

        // 配置多年度属性时
        // 进行渲染时间轴
        else if(lyrContent[layerName].years){
            // 显示时间轴
            timeLineDom.show("slow");
            var allYears = lyrContent[layerName].years;
            allYears.id = layerName;
            TimeLine.init("#time_select_left", allYears);
        }
    }

    /**
     * 图层控制监听事件
     * @param btn
     */
    function changeLayerControl(btn) {
        try {
            $(":radio[name='rdoLayerName'][value='"+btn.alias+"']").prop("checked", "checked");
            $("#monitorMapTitle").text(btn.alias);
            // 判断动态服务多图层
            if (btn.isNode) {
                // 刷新图层
                layer[layer.length - 1].options.layers = [btn.id];
                layer[layer.length - 1].redraw();
            } else {
                that.changeLayer();
                var lyr = createLayerByOptions(btn);
                that.setLyr(lyr);
                lyr.addTo($map);
            }
            // 请求服务参数并定位
            $.support.cors = true;
            $.ajax({
                url: btn.url,
                type: 'get',
                dataType: 'jsonp',
                data: {f: 'pjson'},
                success: function (res) {
                    // 处理多图层时动态服务的图层切换
                    if (showNode && res.layers.length > 0) {
                        var newLayer = [btn];
                        $.each(res.layers, function (i, e) {
                            $.extend(e, {
                                alias: e.name,
                                url: btn.url,
                                isNode: true
                            });
                            newLayer.push(e);
                            // 更新viewModel
                            ServiceLayer.layers(newLayer);
                            showNode = false;
                        });
                    }
                    // fit bounds
                    var extent = res.fullExtent;
                    $map.fitBounds([
                        [extent.ymin,extent.xmin],
                        [extent.ymax,extent.xmax]
                    ]);
                }
            });
            if(Legend.showLegend()){
                Legend.init(btn);
            }
        }catch (e){console.log(e)}
    }


    /**
     * 时间轴改变的回调事件
     * @param e
     * @param data
     */
    function changeTimelineLayer(e, data) {
        // 联动标题和图层切换
        $("#monitorMapTitle").text(data.label);
        // 图例标题
        Legend.legendTitle(data.label);
        ServiceLayer.layers(data.layers);
        // 添加默认最后一个年份里的第一个图层
        changeLayerControl(data.layers[0]);
    }


    /********开启地图进行行政区点击定位*******/

    function openClickEvent(map) {
        map.on('click', clickEvent);
    }

    function closeClickEvent(map) {
        map.off('click');
    }

    // 加载行政区数据
    //存储范围结果数据
    var regionJson = [],judgeGeoJson = [],fullMap = [];
    function loadRegionData(){
        var getAllRegionsUrl = Config.region.getAllRegionsUrl;
        var queryPre = Config.region.queryUrlPre;
        var regionCode = Config.region.regionCode,provincialCode = Config.region.provincialCode;
        var regionIndex = Config.region.regionIndex;
        var queryUrl = queryPre + "?index=" + regionIndex;
        var cityUrl = queryPre + "?index=" + regionIndex.split(',')[0];
        // 若缓存数组为空
        if(regionJson.length === 0){
            $.ajax({
                url: getAllRegionsUrl + provincialCode,
                type: 'get',
                success: function (data) {
                    $.each(data, function (i, e) {
                        if (e.value === regionCode.substr(0, 4)) {
                            // 市级面数据
                            $.ajax({
                                url: cityUrl + "&XZQDM=" + e.value,
                                type: "get",
                                success: function (data) {
                                    // todo 传递数据
                                    // 定位
                                    fullMap = data.content;
                                }
                            });
                            // 区级面数据
                            $.each(e.children, function (index, value) {
                                $.ajax({
                                    url: queryUrl + "&XZQDM=" + value.value,
                                    type: "get",
                                    success: function (data) {
                                        // todo 传递数据
                                        // 定位
                                        judgeGeoJson.push(data.content[0]);
                                    }
                                })
                            });
                        }
                    });
                }
            })
        }
    }

    /**
     * 地图点击事件
     * @param e
     */
    var regionLyr = [];
    function clickEvent(e) {
        if(judgeGeoJson.length === 0){
            alert('行政区范围尚未请求完成');
        }else {
            // 判断点所在的位置
            var point = [e.latlng.lng, e.latlng.lat];
            // 清除上次点击图层
            removeRegionLayer();

            $.each(judgeGeoJson, function (i, item) {
                var gjLayer = L.geoJson(item);
                var pipResult = LeafletPip.pointInLayer(point, gjLayer, false);
                if (pipResult.length !== 0) {

                    // 定位
                    var bounds = pipResult[0].getBounds();
                    var zoom = $map.getBoundsZoom(bounds);
                    pipResult[0].addTo($map);
                    regionLyr.push(pipResult[0]);

                    //地图放大
                    //$map.flyTo(bounds.getCenter(), zoom);

                    //地图点击事件
                    $(document).trigger("region-change",{regionCode: pipResult[0].feature.properties.REGION_CODE});

                    // 绑定图层事件，todo后期优化，不需要再次点击
                    pipResult[0].on('mouseover',function (e) {
                        var toolTip = this.bindTooltip(e.target.feature.properties.TITLE,{opacity:'1.0'});
                        toolTip.openTooltip();
                        setTimeout(function () {
                            toolTip.closeTooltip();
                        }, 2000);
                    }).on('click',function (e) {
                        $(document).trigger("region-change",{regionCode: e.target.feature.properties.REGION_CODE});
                    });
                }
            });
        }
    }

    /**
     * 清除定位图层
     */
    function removeRegionLayer() {
        if (regionLyr.length > 0) {
            $.each(regionLyr,function (i,e) {
                $map.removeLayer(e);
            });
            regionLyr = [];
        }
    }


    /**
     * 增加视图按钮回退
     */
    function addBackButton() {
        var base = base +'/webjars/lp-webjars/image/ghfl/fullMap.png' ;
        var html = "<div id='fullMap' " +
            "style='position: absolute;bottom:30px;right: 10px;z-index:999;" +
            "width:30px;height:30px;border-radius:3px;background: #fff;" +
            "border: 1px solid #c3b4b4;text-align: center;line-height:30px;cursor: pointer'>" +
            "<img src="+base+">"+
            "</div>";
        $("#" + that._id).append(html);
        var el = document.getElementById("fullMap");

        L.DomEvent.addListener(el, 'click', function (e) {

            $(document).trigger("region-change",{regionCode: "320400"});
            L.DomEvent.stopPropagation(e);
            L.DomEvent.preventDefault(e);
            removeRegionLayer();
            $map.setZoom(8)
        });
        L.DomEvent.addListener(el, 'mousedown dblclick drag', function (e) {
            L.DomEvent.stopPropagation(e);
            L.DomEvent.preventDefault(e);
        });
    }

    return new InitMap;

});