/**
 *  地图打印
 *  1/支持勾选资源目录树加载图层的打印(多图层以叠加次序显示)
 *  2/支持导入文件的地块图形打印
 *  3/支持多边形手绘图形打印
 *  4/打印纸张格式暂配A4 landscape模板
 *
 *   var web_map_as_json = {
        //mapOptions为必须的，定义地图显示属性
        // mapOptions:{<extent>(必须,定义映射范围),<scale>(可选),<spatialReference>(可选,地图空间参考)}
        mapOptions: {},
        //operationalLayers包含在地图中显示的所有操作层。
        // ！数组顺序决定了图层顺序
        operationalLayers: [],
        //该映射包含一个底图和底图属性
        baseMap: {
            title: "display_base_map",
            baseMapLayers: []
        },
        //输出映射的配置
        //exportOptions:{ <dpi>,outputSize:{<width>,<height>}
        exportOptions: {},
        //定义了不同页面的布局设置(标题/版权文本/比例尺/作者姓名/自定义文本等)
        layoutOptions: {}
    };
 * @author by <a href="mailto:chaomashi@gmail.com">Du</a>
 * @version v1.0 2019/6/12/14:10  Copyright gtmap Corp
 */

define([
    'easyMap/main',
    'leaflet',
    'esri-leaflet',
    'easyMap/tp/esri-leaflet-gp/esri-leaflet-gp',
    'popup',
    'knockout',
    'jquery',
    'jQueryXDomainRequest'
], function (
    MainMap,
    L,
    LEsri,
    LEsriGP,
    Popup,
    ko,
    $
) {
    var instance,
        crs,
        printTempData,
        me = {
            // init fun
            init: function () {
                $map = MainMap.map.get$Map();
                crs = MainMap.map.CrsFactory;
                wkid = parseInt(crs.getWkid());
                printUrl = MainMap.appConfig.interface.exportMapService;
                printConfig = this.config;
                paperType = printConfig.defaultPaper;
                exportType = printConfig.defaultExport

            },
            // open event
            onOpen: function () {

            },
            // close event
            onClose: function () {
                clearArea();
            },
            // pause event
            onPause: function () {
                clearArea();
            }
        };
    // map wkid 打印地址 layui layer
    var $map, wkid, printUrl, layuiLayer, printConfig, rectangle, tmpRect, paperType, exportType;
    var lgLyr = [];

    // operationalLayers
    var operationalLayers = [],
        layoutOptions = {
            titleText: '标题',
            authorText: '作者：',
            copyrightText: '版权：',
            scaleBarOptions: {},
            legendOptions: {"operationalLayers":[]}
            // legendOptions: {}
        },
        exportInputs = [
            {
                "label": "标题",
                "name": "title",
                "value": ""
            },
            {
                "label": "作者",
                "name": "author",
                "value": ""
            },
            {
                "label": "版权",
                "name": "copyright",
                "value": ""
            }
        ];


    // WidgetUtils.loadScrollBar2('#layPrintForm', 185);

    /**
     * init执行前渲染页面
     */
    function renderPage(){
        ko.applyBindings(new PrintViewModel(), document.getElementById("exportMap"));
        layui.use(['form', 'layer'], function (form, layer) {
            form = layui.form;
            layuiLayer = layui.layer;
            //点击打印按钮使用模板进行解析打印
            form.on('submit(printBtn)', function (data) {
                paperType = printConfig.defaultPaper;
                exportType = printConfig.defaultExport;
                hasAdobeReader(exportType);
                onPrintMap(exportType, paperType);
                return false;
            });

            form.on('select(temp)', function (data) {
                paperType = data.value;
            });

            form.on('select(paper)', function (data) {
                exportType = data.value;
            });

            //点击框选按钮，绘制打印范围
            form.on('submit(drawBtn)', function (data) {
                //清除框选的矩形
                clearArea();
                drawRectangle();
                return false;
            });

            //点击清除按钮，清除框选的矩形
            form.on('submit(clearBtn)', function (data) {
                clearArea();
                return false;
            });

        });
    }

    printTempData = getPrintTypes(renderPage);



    function getPrintTypes(callBack) {
        var printUrl = MainMap.appConfig.interface.exportMapService;
        if (!printUrl) {
            return Popup.msg("请配置打印服务！");
        }

        LEsri.Support.cors = true;
        LEsri.get(printUrl, {f: 'pjson'}, function (error, response) {
            var tempTypes = [], paperTypes = [];
            if (error) {
                console.log('打印请求错误!请检查服务地址')
            }

            // 拼装渲染数据
            response.parameters.forEach(function (value) {
                if (value.hasOwnProperty('choiceList')) {
                    var choiceList = value.choiceList;
                    if (value.name === 'Layout_Template') {
                        tempTypes = choiceList;
                    } else if (value.name === 'Format') {
                        paperTypes = choiceList;
                    }
                }
            });
            printTempData = {temp: tempTypes, paper: paperTypes};
            if(callBack){
                callBack(printTempData);
            }
        })

    }
    
    function PrintViewModel() {
        var self = this,
            types = printTempData;

        self.tempTypes = ko.observableArray(types.temp);

        self.paperTypes = ko.observableArray(types.paper);

        self.exportInputs = ko.observableArray(exportInputs);

        self.selectedChoice = ko.observable("JPG");

        self.defaultTemp = ko.observable("A4纵向土地利用总图规划（局部图）");

        // 监听input数据赋值
        self.inputChange = function (type) {
            layoutOptions[type.name + 'Text'] = type.value;
        }
    }

    /**
     * 拼接手绘面的点数组
     * @param lPoint
     * @returns {Array}
     */
    function getPoint(lPoint) {
        var arr = [];
        arr.push(lPoint.lng);
        arr.push(lPoint.lat);
        return arr;
    }


    /**
     * assemble Data
     */
    function assembleData() {

        var zoom = $map.getZoom(),
            layerBounds = $map.getBounds();

        var web_map_as_json = {

            baseMap: {
                title: "display_base_map",
                baseMapLayers: []
            },

            exportOptions: {
                "dpi": 96,
                "outputSize": [800, 1100]
            }

        };

        //定义存放polygon图层的json
        var features = {
            geometry: {
                rings: [],
                spatialReference: {wkid: wkid, latestWkid: wkid}
            },
            symbol: {
                color: [252, 179, 179, 51],
                outline: {
                    color: [252, 179, 179, 255],
                    width: 1.5,
                    type: "esriSLS",
                    style: "esriSLSSolid"
                },
                type: "esriSFS",
                style: "esriSFSSolid"
            }
        };

        var map_graphics_layer_all_points = {
            id: "map_graphics_layer_all_points",
            featureCollection: {
                layers: [
                    {
                        layerDefinition: {
                            name: "pointLayer",
                            geometryType: "esriGeometryPoint"
                        },

                        featureSet: {
                            geometryType: "esriGeometryPoint",
                            features: []
                        }
                    }
                ]
            }
        };

        // map_graphics_layer_all_ploylines 存放当前map的graphics_layer的ploylines
        var map_graphics_layer_all_ploylines = {
            id: "map_graphics_layer_all_ploylines",
            featureCollection: {
                layers: [
                    {
                        layerDefinition: {
                            name: "polylineLayer",
                            geometryType: "esriGeometryPolyline"
                        },
                        featureSet: {
                            geometryType: "esriGeometryPolyline",
                            features: []
                        }
                    }
                ]
            }
        };

        // map_graphics_layer_all_ploygons 存放当前map的graphics_layer的ploygons
        var map_graphics_layer_all_ploygons = {
            id: "map_graphics_layer_all_ploygons",
            featureCollection: {
                layers: [
                    {
                        layerDefinition: {
                            name: "polygonLayer",
                            geometryType: "esriGeometryPolygon"
                        },
                        featureSet: {
                            geometryType: "esriGeometryPolygon",
                            features: []
                        }
                    }
                ]
            }
        };

        // 拼装map ch
        if (rectangle !== undefined) {
            try {
                var mapOptions = {
                    extent: LEsri.Util.boundsToExtent(rectangle.getBounds()),
                    spatialReference: {wkid: wkid, latestWkid: wkid}
                };
                rectangle.remove();
            } catch (e) {
                console.log(e.toString());
            }
        } else {
            var mapOptions = {
                extent: LEsri.Util.boundsToExtent(layerBounds), //必须,定义映射范围
                scale: crs.getLods(zoom),
                spatialReference: {wkid: wkid, latestWkid: wkid}
            };
        }

        $.each(MainMap.map.getAllLayers(), function (index, layer) {
            //图例
            lgLyr.push({id: layer.options.id});

            operationalLayers.push({
                id: layer.options.id,
                title: "",
                opacity: layer.options.opacity,
                minScale: 0,
                maxScale: 0,
                url: layer.options.url
            });

            //筛选layer中的polygon层信息，截取构成polygon的坐标点存储进数组中
            if (layer.hasOwnProperty("_rings")) {
                var latArr = layer._latlngs[0];
                var ringPoint = [];
                for (var a = 0; a < latArr.length; a++) {
                    ringPoint.push(getPoint(latArr[a]));
                }
                features.geometry.rings.push(ringPoint);
            }
        });

        //将底图上绘制的图层json拼接到对应的web_map_json中
        map_graphics_layer_all_ploygons.featureCollection.layers[0].featureSet.features.push(features);
        operationalLayers.push(map_graphics_layer_all_ploygons);
        operationalLayers.push(map_graphics_layer_all_ploylines);
        operationalLayers.push(map_graphics_layer_all_points);

        layoutOptions.legendOptions.operationalLayers = lgLyr;

        return JSON.stringify($.extend(web_map_as_json, {
            layoutOptions: layoutOptions,
            mapOptions: mapOptions,
            operationalLayers: operationalLayers
        }));
    }

    /**
     * 打印地图
     * @param format
     * @param template
     * @constructor
     */
    function onPrintMap(format, template) {
        var waiting = layuiLayer.msg("加载中..."),
            web_map_as_json = assembleData();

        var service_GP = LEsriGP.service({url: printUrl}),
            exportMapTask = service_GP.createTask();

        // 设置GP服务参数
        exportMapTask.setParam('Web_Map_as_JSON', web_map_as_json);
        exportMapTask.setParam('Format', format);
        exportMapTask.setParam('Layout_Template', template);
        exportMapTask.setParam('f', 'pjson');

        LEsri.Support.cors = true;
        exportMapTask.run(function (error, data, response) {
            if (error) {
                Popup.msg("加载访问异常，请重试！");
            }
            if (data) {
                window.open(data.Output_File.url);
            }
            layer.close(waiting);
        });

    }

    /**
     * hasAdobeReader
     * @param Format
     */
    function hasAdobeReader(Format) {
        if (Format === "PDF" && !isAcrobatPluginInstall) {
            Popup.msg({
                type: 'warning',
                message: '检测到您的浏览器没有Adobe Reader插件'
            });
        }

        return false;
    }

    /**
     * 检测浏览器 adboe pdf 插件是否安装
     * @returns {boolean}
     */
    function isAcrobatPluginInstall() {
        var flag = false;

        var useAgent = navigator.userAgent;
        //chrome 支持pdf预览
        if (useAgent.match(/Chrome/i) === 'Chrome') {
            flag = true;
        }
        // 下面代码都是处理IE浏览器的情况
        else if (window.ActiveXObject) {
            for (var x = 2; x < 10; x++) {
                try {
                    var oAcro = eval("new ActiveXObject('PDF.PdfCtrl." + x + "');");
                    if (oAcro) {
                        flag = true;
                    }
                } catch (e) {
                    flag = false;
                }
            }
            try {
                var oAcro4 = new ActiveXObject('PDF.PdfCtrl.1');
                if (oAcro4)
                    flag = true;
            } catch (e) {
                flag = false;
            }
            try {
                var oAcro7 = new ActiveXObject('AcroPDF.PDF.1');
                if (oAcro7)
                    flag = true;
            } catch (e) {
                flag = false;
            }
        }
        return flag;
    }

    /**
     * 绘制矩形
     */
    function drawRectangle() {
        var latlngs = [];
        //开始绘制
        $map.on('mousedown', function (e) {
            $map.dragging.disable();
            if (typeof tmpRect != 'undefined') {
                tmpRect.remove();
            }
            //左上角坐标
            latlngs[0] = [e.latlng.lat, e.latlng.lng];
            //绘制时，监听鼠标移动事件
            $map.on('mousemove', function (e) {
                //禁止地图拖拽事件
                $map.dragging.disable();
                latlngs[1] = [e.latlng.lat, e.latlng.lng];
                //删除临时矩形
                if (typeof tmpRect != 'undefined') {
                    tmpRect.remove();
                }
                //添加临时矩形
                tmpRect = L.rectangle(latlngs, {color: '#2A8AFF', fillOpacity: 0.1, weight: 2}).addTo($map);
            });
        });
        //结束绘制
        $map.on('mouseup', function (e) {
            //激活地图拖拽事件
            $map.dragging.enable();
            //矩形绘制完成，移除临时矩形，并停止监听鼠标移动事件
            tmpRect.remove();
            $map.off('mousemove')
                .off('mousedown')
                .off('mouseup');
            //右下角坐标
            latlngs[1] = [e.latlng.lat, e.latlng.lng];
            rectangle = L.rectangle(latlngs, {color: '#2A8AFF', fillOpacity: 0.1, weight: 2});
            rectangle.addTo($map);
            $map.fitBounds(latlngs);
        });
    }

    /**
     * 清除框选范围矩形
     */
    function clearArea() {
        if (rectangle !== undefined) {
            rectangle.remove();
            rectangle = undefined;
        }
    }


    /**
     * get instance
     *
     * @returns {*}
     */
    me.getInstance = function () {
        if (instance === undefined) {
            instance = me;
        }
        return instance;
    };
    return me;
});