package cn.gtmap.realestate.common.util;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;

import java.lang.reflect.Field;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
 * @version 1.0, 2018/11/14
 * @description Spring Bean解析处理工具类
 */
@Component
public class BeansResolveUtils implements ApplicationContextAware {
    /**
     * 日期格式化
     */
    private static SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    /**
     * 日志操作
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(BeansResolveUtils.class);

    /**
     * BeansResolveUtils限定类名
     */
    private static final String CLASS_NAME = BeansResolveUtils.class.getName();

    /**
     * Spring应用上下文
     */
    private ApplicationContext applicationContext;

    @Override
    public void setApplicationContext(ApplicationContext applicationContext){
        if(null == this.applicationContext){
            this.applicationContext = applicationContext;
        }
    }

    /**
     * @author  <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @param   name bean名称
     * @return  bean对象
     * @description 根据bean名称获取对应bean
     */
    public <T> T getBeanByName(String name){
        if(null == this.applicationContext || StringUtils.isBlank(name)){
            return null;
        }

        return (T) this.applicationContext.getBean(name);
    }

    /**
     * @author  <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @param   beansNameMap
     * @param   key 目标bean对应键值
     * @return  bean对象
     * @description 获取bean名称键值对Map中指定key对应的bean
     */
    public <T> T getBeanOfMapItem(Map<Object, String> beansNameMap, Object key){
        if(MapUtils.isEmpty(beansNameMap) || StringUtils.isBlank(String.valueOf(key))){
            LOGGER.warn("{}：获取配置项Bean集合失败，原因可能：入参Bean名称集合为空、未指定要获取bean对应的键值！", CLASS_NAME);
            return null;
        }

        String beanName = MapUtils.getString(beansNameMap, key);
        return this.getBeanByName(beanName);
    }

    /**
     * @author  <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @param   beansNameList bean名称集合
     * @return  {List} bean集合
     * @description  获取bean名称对应的bean集合
     */
    public <T> List<T> listBeans(List<String> beansNameList){
        if(CollectionUtils.isEmpty(beansNameList)){
            LOGGER.warn("{}：获取配置项Bean集合失败，原因：入参Bean名称集合为空！", CLASS_NAME);
            return Collections.emptyList();
        }

        //存储bean结果集
        List<T> beansList = new ArrayList<>(beansNameList.size());
        for (String beanName : beansNameList){
            if(StringUtils.isNotBlank(beanName)){
                //获取配置项对应bean
                T bean = this.getBeanByName(beanName);
                if(null == bean){
                    LOGGER.warn("{}：获取配置项Bean集合中止，原因：未找到{}对应bean！", CLASS_NAME, beanName);
                    continue;
                }

                beansList.add(bean);
            }
        }

        return beansList;
    }

    /**
     * @author  <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @param   beansNameMap bean名称键值对集合
     * @return  {Map} bean键值对集合
     * @description
     *     <p>
     *        1、获取键值对形式bean名称对应的bean集合
     *        2、key-value中的key为索引，value为bean名称
     *     </p>
     */
    public <T> Map<Object, T> listBeansOfMapType(Map<Object, String> beansNameMap){
        if(MapUtils.isEmpty(beansNameMap)){
            LOGGER.warn("{}：获取配置项Bean集合失败，原因：入参Bean名称集合为空！", CLASS_NAME);
            return Collections.emptyMap();
        }

        //存储bean结果集
        Map<Object, T> beansMap = new HashMap<>(beansNameMap.size());
        for(Map.Entry<Object, String> entry : beansNameMap.entrySet()){
            if(StringUtils.isNotBlank(entry.getValue())){
                //获取配置项对应bean
                T bean = this.getBeanByName(entry.getValue());
                if(null == bean){
                    LOGGER.warn("{}：获取配置项Bean集合中止，原因：未找到{}对应bean！", CLASS_NAME, entry.getValue());
                    continue;
                }

                beansMap.put(entry.getKey(), bean);
            }
        }

        return beansMap;
    }

    /**
     * @author  <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @param  originObject 源对象
     * @param  targetObject 目标对象
     * @description
     *      将源对象属性值赋值给目标对象
     *    1、相当于属性值对应复制
     *    2、值非空才赋值
     *    3、对应属性名称要一致，有get方法、set方法
     */
    public static void clonePropertiesValue(Object originObject, Object targetObject){
        // 获取源对象所有属性
        Field[] originFields = originObject.getClass().getDeclaredFields();
        List<String> originFieldsNameList = BeansResolveUtils.getClassFieldsName(originFields);

        // 循环设置属性值
        Field[] targetFields = targetObject.getClass().getDeclaredFields();
        try {
            for(Field targetField : targetFields){
                if(originFieldsNameList.contains(targetField.getName())){
                    // 获取源对象属性值
                    Field originField = originObject.getClass().getDeclaredField(targetField.getName());
                    originField.setAccessible(true);
                    Object value = originField.get(originObject);

                    if(StringUtils.isNotBlank(String.valueOf(value))){
                        // 赋值给目标属性
                        targetField.setAccessible(true);
                        targetField.set(targetObject, value);
                    }
                }
            }
        } catch (NoSuchFieldException | IllegalAccessException e){
            LOGGER.error(e.getMessage(), e);
        }
    }

    /**
     * @author  <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @param  originObject 源对象
     * @param  targetObject 目标对象
     * @description
     *      将源对象属性值赋值给目标对象(包括目标对象父类属性)
     */
    public static void clonePropsValueWithParentProps(Object originObject, Object targetObject){
        // 获取源对象所有属性
        Field[] originFields = originObject.getClass().getDeclaredFields();
        List<String> originFieldsNameList = BeansResolveUtils.getClassFieldsName(originFields);

        // 循环设置属性值
        Field[] targetFields = targetObject.getClass().getDeclaredFields();
        Field[] parentFields = targetObject.getClass().getSuperclass().getDeclaredFields();
        targetFields = ArrayUtils.addAll(targetFields, parentFields);

        try {
            for(Field targetField : targetFields){
                if(originFieldsNameList.contains(targetField.getName())){
                    // 获取源对象属性值
                    Field originField = originObject.getClass().getDeclaredField(targetField.getName());
                    originField.setAccessible(true);
                    Object value = originField.get(originObject);

                    if(StringUtils.isNotBlank(String.valueOf(value))){
                        // 赋值给目标属性
                        targetField.setAccessible(true);
                        targetField.set(targetObject, value);
                    }
                }
            }
        } catch (NoSuchFieldException | IllegalAccessException e){
            LOGGER.error(e.getMessage(), e);
        }
    }

    /**
     * @author  <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @param   fields class对象成员变量数组
     * @return  {List} 成员变量名称集合
     * @description  获取class对象成员变量对应的名称集合
     */
    public static List<String> getClassFieldsName(Field[] fields){
        if(null == fields || 0 == fields.length){
            return Collections.emptyList();
        }

        List<String> fieldsNameList = new ArrayList<>(fields.length);
        for(Field field : fields){
            if(null != field){
                fieldsNameList.add(field.getName());
            }
        }
        return fieldsNameList;
    }

    /**
     * @author  <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @param   list 目标对象集合
     * @return  {String} 非空字段属性JSON字符串
     * @description  获取集合中的对象非空字段内容，序列化字符串
     */
    public static <T> String toNotNullPropertyListJSON(List<T> list) throws IllegalAccessException {
        if(CollectionUtils.isEmpty(list)){
            return "";
        }

        JSONArray jsonArray = new JSONArray(list.size());
        for(T object : list){
            JSONObject jsonObject = new JSONObject();

            Field[] fields = object.getClass().getDeclaredFields();
            for(Field field : fields){
                field.setAccessible(true);
                Object obj = field.get(object);

                if(null != obj && StringUtils.isNotBlank(String.valueOf(obj))){
                    if (obj instanceof  Object[]){
                        jsonObject.put(field.getName(), StringUtils.join((Object[])obj, ","));
                    } else if (obj instanceof Date){
                        jsonObject.put(field.getName(), formatter.format((Date) obj));
                    } else {
                        jsonObject.put(field.getName(), String.valueOf(obj));
                    }
                }
            }

            jsonArray.add(jsonObject);
        }
        return JSON.toJSONString(jsonArray);
    }

    /**
     * @author  <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @param   object 目标对象
     * @return  {String} 非空字段属性JSON字符串
     * @description  获取对象非空字段内容，序列化字符串
     */
    public static <T extends Object> String toNotNullPropertyJSON(T object) throws IllegalAccessException {
        if(null == object){
            return "";
        }

        JSONObject jsonObject = new JSONObject();

        Field[] fields = object.getClass().getDeclaredFields();
        for(Field field : fields){
            field.setAccessible(true);
            Object obj = field.get(object);

            if(null != obj && StringUtils.isNotBlank(String.valueOf(obj))){
                if (obj instanceof  Object[]){
                    jsonObject.put(field.getName(), StringUtils.join((Object[])obj, ","));
                } else if (obj instanceof Date){
                    jsonObject.put(field.getName(), formatter.format((Date) obj));
                } else {
                    jsonObject.put(field.getName(), String.valueOf(obj));
                }
            }
        }
        return JSON.toJSONString(jsonObject);
    }
}
