package cn.gtmap.realestate.common.core.support.pdf.service.impl.thread;

import cn.gtmap.realestate.common.core.support.pdf.service.OfficeDocService;
import cn.gtmap.realestate.common.util.UUIDGenerator;
import com.deepoove.poi.NiceXWPFDocument;
import org.docx4j.Docx4J;
import org.docx4j.convert.out.FOSettings;
import org.docx4j.fonts.IdentityPlusMapper;
import org.docx4j.fonts.Mapper;
import org.docx4j.fonts.PhysicalFont;
import org.docx4j.fonts.PhysicalFonts;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.packages.WordprocessingMLPackage;
import org.docx4j.wml.RFonts;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.Map;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.CountDownLatch;

/**
 * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
 * @version 1.0, 2020/01/07
 * @description 单个PDF文档处理线程任务
 */
public class PdfTask implements Runnable{
    /**
     * 日志操作
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(PdfTask.class);
    /**
     * 文档处理逻辑
     */
    private OfficeDocService officeDocService;
    /**
     * 数据内容
     */
    private Map<String, Object> pdfData;
    /**
     * 文档顺序号
     */
    private int index;
    /**
     * 文件临时操作父目录，同打印模板同一目录
     */
    private String printPath;
    /**
     * 模板名称
     */
    private String modelName;
    /**
     * 临时PDF文件缓存集合
     */
    private ConcurrentSkipListMap<String, String> pdfMap;
    /**
     * 计数器
     */
    private CountDownLatch countDownLatch;


    public PdfTask(OfficeDocService officeDocService,
                   Map<String, Object> pdfData,
                   int index,
                   String modelName,
                   String printPath,
                   ConcurrentSkipListMap<String, String> pdfMap,
                   CountDownLatch countDownLatch){

        this.officeDocService = officeDocService;
        this.printPath = printPath;
        this.pdfData = pdfData;
        this.index = index;
        this.modelName = modelName;
        this.pdfMap = pdfMap;
        this.countDownLatch = countDownLatch;
    }

    @Override
    public void run() {
        LOGGER.debug("PDF导出正在处理第{}份文件！", countDownLatch.getCount());

        // 生成临时word
        String tempWordFileName = this.generateWordFileName();
        // 生成临时pdf
        String tempPdfFileName = this.generatePdfFileName();

        try{
            // 生成word
            FileOutputStream wordFileOutStream = new FileOutputStream(new File(tempWordFileName));
            NiceXWPFDocument document = officeDocService.getNiceXWPFDocument(modelName, pdfData);
            document.write(wordFileOutStream);

            // 转换为pdf
            this.convertDocxToPDF(tempWordFileName, tempPdfFileName);
            pdfMap.put(String.valueOf(index), tempPdfFileName);
            LOGGER.debug("当前处理第{}页文档完成", index + 1);
        } catch (Exception e){
            LOGGER.error("处理WORD转换为PDF出错：{}", e.getMessage());
        } finally {
            countDownLatch.countDown();

            File wordFile = new File(tempWordFileName);
            if(wordFile.exists()){
                wordFile.delete();
            }
        }
    }

    /**
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @param docxPath word文件位置
     * @param pdfPath  pdf文件位置
     * @description 将word文档转为pdf
     */
    private void convertDocxToPDF(String docxPath, String pdfPath) {
        try(OutputStream os = new FileOutputStream(pdfPath)) {
            WordprocessingMLPackage mlPackage = WordprocessingMLPackage.load(new FileInputStream(new File(docxPath)));
            // 设置字体
            this.configSimSunFont(mlPackage);
            FOSettings foSettings = Docx4J.createFOSettings();
            foSettings.setWmlPackage(mlPackage);
            Docx4J.toFO(foSettings, os, Docx4J.FLAG_EXPORT_PREFER_XSL);
        }catch(Exception ex){
            LOGGER.error("word转为pdf出错，{}", ex.getMessage());
        }
    }

    /**
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @description  设置文件字体
     */
    private void configSimSunFont(WordprocessingMLPackage wordMLPackage) throws Exception {
        // 默认文档为宋体
        String fontFamily = "SimSun";
        // 加载字体文件
        URL simsunUrl = new URL("file:" + printPath + "simsun.ttc");
        PhysicalFonts.addPhysicalFonts(fontFamily, simsunUrl);
        PhysicalFont simsunFont = PhysicalFonts.get(fontFamily);
        Mapper fontMapper = new IdentityPlusMapper();
        fontMapper.put(fontFamily, simsunFont);
        wordMLPackage.setFontMapper(fontMapper);

        RFonts rfonts = Context.getWmlObjectFactory().createRFonts();
        rfonts.setAsciiTheme(null);
        rfonts.setAscii(fontFamily);
        wordMLPackage.getMainDocumentPart().getPropertyResolver().getDocumentDefaultRPr().setRFonts(rfonts);
    }

    /**
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @description  生成临时word文件名称
     */
    private String generateWordFileName(){
        return printPath + "temp/" + UUIDGenerator.generate16() + ".docx";
    }

    /**
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @description  生成临时pdf文件名称
     */
    private String generatePdfFileName(){
        return printPath + "temp/" + UUIDGenerator.generate16() + ".pdf";
    }
}
