package cn.gtmap.realestate.common.core.support.pdf.service.impl;

import cn.gtmap.realestate.common.core.dto.OfficeExportDTO;
import cn.gtmap.realestate.common.core.ex.AppException;
import cn.gtmap.realestate.common.core.support.pdf.service.OfficeDataService;
import cn.gtmap.realestate.common.core.support.pdf.service.OfficeDocService;
import cn.gtmap.realestate.common.core.support.pdf.service.OfficePdfService;
import cn.gtmap.realestate.common.core.support.pdf.service.impl.thread.PdfTask;
import cn.gtmap.realestate.common.util.UUIDGenerator;
import com.itextpdf.text.pdf.PdfCopy;
import com.itextpdf.text.pdf.PdfReader;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.dom4j.DocumentException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.*;

/**
 * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
 * @version 1.0, 2020/01/06
 * @description PDF相关数据处理逻辑
 */
@Service
public class OfficePdfServiceImpl implements OfficePdfService {
    /**
     * 日志操作
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(OfficePdfServiceImpl.class);
    /**
     * 字体存放目录，同打印模板同一目录
     */
    @Value("${print.path:/usr/local/bdc3/print/}")
    private String printPath;
    /**
     * 文档对象操作
     */
    @Autowired
    private OfficeDocService officeDocService;
    /**
     * 数据处理
     */
    @Autowired
    private OfficeDataService officeDataService;

    /**
     * PDF任务处理共享线程池定义
     */
    private ExecutorService executor = new ThreadPoolExecutor(
            // 核心线程数量
            4,
            // 最大线程数
            6,
            // 超时30秒
            30, TimeUnit.SECONDS,
            // 最大允许等待200个任务
            new ArrayBlockingQueue<>(200),

            // 自定义线程工厂，处理线程异常捕获
            new ThreadFactory() {
                @Override
                public Thread newThread(Runnable r) {
                    Thread thread = new Thread(r);
                    thread.setUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler() {
                        @Override
                        public void uncaughtException(Thread t, Throwable e) {
                            LOGGER.error("PDF处理线程发生异常：{}", e.toString());
                        }
                    });
                    return thread;
                }
            },

            // 拒绝策略
            (task, executor) -> LOGGER.error("生成PDF任务被拒绝，请检查导出数量！")
    );


    /**
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @description 构造文件操作临时目录
     */
    @PostConstruct
    public void init(){
        File file = new File(printPath + "temp/");
        if(!file.exists()){
            file.mkdirs();
        }
    }

    /**
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @param officeExportDTO 导出所需参数信息
     * @description 生成PDF文档（多页合并成一个）
     *
     *  注意：调用端需要考虑是否删除生成的临时PDF文件
     */
    @Override
    public String generatePdfFile(OfficeExportDTO officeExportDTO) {
        // 缓存临时生成的一个个单个PDF文件，集合需要保证安全且有序
        ConcurrentSkipListMap<String, String> pdfMap = new ConcurrentSkipListMap();

        try{
            // 解析打印数据（数据量过大，XML解析时候可能会大量占用内容，需要相关模块优化）
            List<Map<String, Object>> dataList = officeDataService.getValDataList(officeExportDTO.getXmlData());
            if(CollectionUtils.isEmpty(dataList)){
                return null;
            }

            int dataSize = dataList.size();
            LOGGER.debug("导出PDF文档：{}，共{}页", officeExportDTO.getFileName() ,dataSize);

            CountDownLatch countDownLatch = new CountDownLatch(dataSize);
            BlockingQueue queue = ((ThreadPoolExecutor)executor).getQueue();

            for(int index = 0; index < dataSize; index++){
                int freeCount = queue.remainingCapacity();

                // 数量控制，避免大批量OOM
                if(freeCount <= 10){
                    LOGGER.debug("导出PDF线程池队列剩余空间较少，开始等待提交任务，文件名：{}", officeExportDTO.getFileName());
                    long start = System.currentTimeMillis();
                    do{
                        Thread.sleep(500);
                    }while (queue.remainingCapacity() <= 10 || System.currentTimeMillis() - start < 5000);
                }

                // 每个线程任务生成一个word，再转pdf
                PdfTask pdfTask = new PdfTask(officeDocService,
                        dataList.get(index),
                        index,
                        officeExportDTO.getModelName(),
                        printPath,
                        pdfMap,
                        countDownLatch
                );
                LOGGER.debug("导出PDF处理第{}个子任务提交", index + 1);
                executor.submit(pdfTask);
            }

            // 整体等待5分钟
            boolean waitRes = countDownLatch.await(300, TimeUnit.SECONDS);
            if(!waitRes){
                // 超时
                LOGGER.error("PDF导出超时5分钟，处理终止，目标文件：{}", officeExportDTO.getFileName());
                throw new AppException("PDF导出超时5分钟，处理终止");
            }

            // 合并PDF文件
            LOGGER.debug("PDF导出中间临时文件处理完毕，开始合并文件:{}！", officeExportDTO.getFileName());
            return this.mergePdfFiles(pdfMap);
        } catch (Exception e) {
            throw new AppException("生成的临时PDF文件报错，处理终止, 异常原因：" + e.toString());
        } finally {
            // 删除临时pdf
            if(MapUtils.isNotEmpty(pdfMap)){
                for(Map.Entry<String, String> entry : pdfMap.entrySet()){
                    File pdfFile = new File(entry.getValue());
                    if(pdfFile.exists()){
                        pdfFile.delete();
                    }
                }
            }
        }
    }

    /**
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @param files 要合并的PDF文件
     * @return 临时文件地址
     * @description 合并多个PDF文件为一个
     */
    @Override
    public String mergePdfFiles(Map<String, String> files) {
        if(MapUtils.isEmpty(files)){
            LOGGER.error("合并PDF失败，原因：未指定要合并的PDF文件！");
            throw new AppException("合并PDF失败，原因：未指定要合并的PDF文件！");
        }

        String pdfFileName = this.generatePdfFileName();
        com.itextpdf.text.Document document = null;
        FileOutputStream fileOutputStream = null;

        try {
            // 新建Document，设置纸张大小，根据随便一个PDF文档大小设置
            document = new com.itextpdf.text.Document(new PdfReader(files.get("0")).getPageSize(1));
            fileOutputStream = new FileOutputStream(pdfFileName);
            PdfCopy copy = new PdfCopy(document, fileOutputStream);
            document.open();

            // 遍历需要保证取出按照文档顺序
            for(int index = 0; index < files.size(); index++){
                // 每一份PDF
                PdfReader reader = new PdfReader(files.get(String.valueOf(index)));
                // 处理每一页（需要注意索引从1开始）
                for (int i = 1; i <= reader.getNumberOfPages(); i++) {
                    document.newPage();
                    copy.addPage(copy.getImportedPage(reader, i));
                }
                reader.close();
            }
            copy.close();
            LOGGER.debug("PDF导出合并文件处理完毕！");

            return pdfFileName;
        } catch (Exception e) {
            throw new AppException("合并PDF失败，原因：" + e.toString());
        } finally {
            if(null != document){
                document.close();
            }

            if(null != fileOutputStream){
                try {
                    fileOutputStream.close();
                } catch (IOException e) {
                    LOGGER.error("系统导出PDF关闭资源报错：{}", e.getMessage());
                }
            }
        }
    }

    /**
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @param files 要合并的PDF文件
     * @return 临时文件地址
     * @description 合并多个PDF文件为一个
     */
    @Override
    public String mergePdfFiles(Collection<String> files) {
        if(CollectionUtils.isEmpty(files)){
            LOGGER.error("合并PDF失败，原因：未指定要合并的PDF文件！");
            throw new AppException("合并PDF失败，原因：未指定要合并的PDF文件！");
        }

        String pdfFileName = this.generatePdfFileName();
        com.itextpdf.text.Document document = null;
        FileOutputStream fileOutputStream = null;

        try {
            // 新建Document，设置纸张大小，直接根据第一个PDF文档大小设置
            document = new com.itextpdf.text.Document(new PdfReader(files.iterator().next()).getPageSize(1));
            fileOutputStream = new FileOutputStream(pdfFileName);
            PdfCopy copy = new PdfCopy(document, fileOutputStream);
            document.open();

            Iterator<String> iterator = files.iterator();
            while (iterator.hasNext()){
                // 每一份PDF
                PdfReader reader = new PdfReader(iterator.next());
                // 处理每一页（需要注意索引从1开始）
                for (int i = 1; i <= reader.getNumberOfPages(); i++) {
                    document.newPage();
                    copy.addPage(copy.getImportedPage(reader, i));
                }
                reader.close();
            }
            copy.close();
            LOGGER.debug("PDF导出合并文件处理完毕！");

            return pdfFileName;
        } catch (Exception e) {
            throw new AppException("合并PDF失败，原因：" + e.toString());
        } finally {
            if(null != document){
                document.close();
            }

            if(null != fileOutputStream){
                try {
                    fileOutputStream.close();
                } catch (IOException e) {
                    LOGGER.error("系统导出PDF关闭资源报错：{}", e.getMessage());
                }
            }
        }
    }

    /**
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @description  生成临时pdf文件名称
     */
    private String generatePdfFileName(){
        return printPath + "temp/" + UUIDGenerator.generate16() + ".pdf";
    }
}
