package cn.gtmap.realestate.common.config;

import cn.gtmap.realestate.common.util.CommonConstantUtils;
import cn.gtmap.realestate.common.util.RedisUtils;
import com.alibaba.fastjson.JSON;
import com.ctrip.framework.apollo.Config;
import com.ctrip.framework.apollo.ConfigService;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.env.*;

import javax.annotation.PostConstruct;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 从数据库加载特殊业务配置
 * CertificateApp未配置
 */
@Configuration
public class AutomaticLoadingConfig {

    @Autowired
    private ConfigurableEnvironment configurableEnvironment;

    @Autowired
    private RedisUtils redisUtils;

    private static final Logger LOGGER = LoggerFactory.getLogger(AutomaticLoadingConfig.class);

    @PostConstruct
    public void initSystemConfig(){

        String loadConfigFromDataBaseFlag = "false";
        String name = "DataBasePropertySources";

        // 获取所有的系统配置 MutablePropertySources 是PropertySources的一个实现类,它将所有的PropertySource都放置在一个名叫propertySourceList集合
        MutablePropertySources propertySources = configurableEnvironment.getPropertySources();

        // 获取apollo所有的namespaces,以及yml里的配置
        Map<Object, Object> ymlConfig = new ConcurrentHashMap<>();
        for (PropertySource<?> source : propertySources) {
            ymlConfig.putAll((Map)JSON.parse(JSON.toJSON(source.getSource()).toString()));
        }

        if(ymlConfig.containsKey("config.loadingConfigFlag") && Objects.nonNull(ymlConfig.get("config.loadingConfigFlag"))){
            loadConfigFromDataBaseFlag = ymlConfig.get("config.loadingConfigFlag").toString();
            LOGGER.info("loadConfigFromDataBaseFlag from yml:{}", loadConfigFromDataBaseFlag);
        }

        if(ymlConfig.containsKey("apollo.bootstrap.namespaces") && Objects.nonNull(ymlConfig.get("apollo.bootstrap.namespaces"))){
            String[] namespaces = ymlConfig.get("apollo.bootstrap.namespaces").toString().split(",");
            for(String namespace : namespaces){
                Config appConfig = ConfigService.getConfig(namespace);
                Set<String> propertyNames = appConfig.getPropertyNames();
                if(propertyNames.contains("config.loadingConfigFlag")){
                    loadConfigFromDataBaseFlag = appConfig.getProperty("config.loadingConfigFlag","false");
                    LOGGER.info("loadConfigFromDataBaseFlag from apollo:{}", loadConfigFromDataBaseFlag);
                }
            }
        }

        LOGGER.info("is loading config from dataBase:{}", loadConfigFromDataBaseFlag);
        // yml配置项 配置了从数据库读取数据
        if (StringUtils.equals("true",loadConfigFromDataBaseFlag)) {
            /*
             * 从redis获取自定义变量列表,将转换后的列表加入属性中,将属性转换为属性集合,并指定名称为当前sourceName,替换配置项的值
             */
            Map<Object, Object> configFromDataBase = redisUtils.getHash(CommonConstantUtils.REDIS_TSYW_PZ);
            LOGGER.info("config date size from redis:{}", configFromDataBase.size());
            LOGGER.info("config date from redis:{}", JSON.toJSONString(configFromDataBase));
            Properties properties = new Properties();
            properties.putAll(configFromDataBase);
            PropertiesPropertySource dataBaseConstants = new PropertiesPropertySource(name, properties);
            propertySources.addFirst(dataBaseConstants);
        }

    }
}
