package cn.gtmap.realestate.common.util;

import cn.gtmap.gtc.clients.UserManagerClient;
import cn.gtmap.gtc.sso.domain.dto.OrganizationDto;
import cn.gtmap.gtc.sso.domain.dto.RoleDto;
import cn.gtmap.gtc.sso.domain.dto.UserDto;
import cn.gtmap.realestate.common.core.ex.MissingArgumentException;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;

import java.util.Collections;
import java.util.List;

/**
 * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
 * @version 1.0  2018/11/5 19:34
 * @description 用户操作工具类
 */
@Component
public class UserManagerUtils {
    /**
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @description 引入基础平台用户操作FeignClient
     */
    @Autowired
    private UserManagerClient userManagerClient;


    /**
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @param userName 用户名称
     * @return {UserDto} 用户信息实体
     * @description 根据用户名称获取用户信息实体
     */
    public UserDto getUserByName(String userName) {
        if (StringUtils.isBlank(userName)) {
            return null;
        }
        return this.userManagerClient.getUserDetailByUsername(userName);
    }

    /**
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @return {UserDto} 用户信息实体
     * @description 获取当前系统登录用户
     */
    public UserDto getCurrentUser() {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        if (null == authentication || StringUtils.isBlank(authentication.getName())) {
            return null;
        }
        return this.getUserByName(authentication.getName());
    }

    /**
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @param userName 用户名
     * @return {List} 用户所在机构（可能对应多个）
     * @description 根据用户名称获取所在机构列表
     */
    public List<OrganizationDto> getOrgListByUserName(String userName) {
        if (StringUtils.isBlank(userName)) {
            return Collections.emptyList();
        }

        //根据用户名获取用户
        UserDto userDto = this.getUserByName(userName);
        if (null == userDto || StringUtils.isBlank(userDto.getUsername())) {
            return Collections.emptyList();
        }

        //获取用户所在机构
        return userDto.getOrgRecordList();
    }

    /**
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @return {String} 机构代码
     * @description 获取当前登录用户所在组织机构代码
     */
    public String getCurrentUserOrgCode() {
        UserDto userDto = this.getCurrentUser();
        if (null == userDto || StringUtils.isBlank(userDto.getUsername())) {
            return StringUtils.EMPTY;
        }

        return this.getOrgCodeByUserName(userDto.getUsername());
    }

    /**
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @param userName 用户名
     * @return {String} 机构代码
     * @description 根据用户名称（登录账号，数字/字母）获取用户所在机构编码
     */
    public String getOrgCodeByUserName(String userName) {
        if (StringUtils.isBlank(userName)) {
            return null;
        }

        //获取机构
        ///可能用户对应多个机构，目前因为基础平台支持不足先只考虑一个情况
        List<OrganizationDto> organizationDtoList = this.getOrgListByUserName(userName);
        if (CollectionUtils.isEmpty(organizationDtoList)) {
            return StringUtils.EMPTY;
        }

        //获取编码
        OrganizationDto organizationDto =
                (OrganizationDto) CollectionUtils.get(organizationDtoList, 0);
        return organizationDto.getCode();
    }

    /**
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @param userName 用户名
     * @return {String} 用户所在机构行政区划编码
     * @description 根据用户名获取机构所在行政区划编码
     */
    public String getRegionCodeByUserName(String userName) {
        if (StringUtils.isBlank(userName)) {
            return null;
        }

        //获取机构
        List<OrganizationDto> organizationDtoList = this.getOrgListByUserName(userName);
        if (CollectionUtils.isEmpty(organizationDtoList)) {
            return StringUtils.EMPTY;
        }

        //获取机构所在区划编码
        OrganizationDto organizationDto =
                (OrganizationDto) CollectionUtils.get(organizationDtoList, 0);
        return organizationDto.getRegionCode();
    }

    /**
     * @return String userName 用户姓名
     * @author <a href="mailto:zhangwentao@gtmap.cn">zhangwentao</a>
     * @description 只获取当前用户姓名
     */
    public String getCurrentUserName() {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        if (authentication != null && StringUtils.isNotBlank(authentication.getName())) {
            return authentication.getName();
        }
        return "";
    }

    /**
     * @param userName 用户姓名
     * @return String organizatioName 机构名称
     * @author <a href="mailto:zhangwentao@gtmap.cn">zhangwentao</a>
     * @description 获取第一个机构名称
     */
    public String getOrganizationByUserName(String userName) {
        if (StringUtils.isNotBlank(userName)) {
            List<OrganizationDto> organizationDtos = getOrgListByUserName(userName);
            if (CollectionUtils.isNotEmpty(organizationDtos)) {
                return organizationDtos.get(0).getName();
            }
        }
        return "";
    }

    /**
     * @param userName
     * @return signid
     * @author <a href ="mailto:songhaowen@gtmap.cn">songhaowen</a>
     * @description 根据用户名获取签名id
     */
    public String getUserSignByUserName(String userName) {
        if (StringUtils.isEmpty(userName)) {
            throw new MissingArgumentException("userName");
        }
        return userManagerClient.getUserSign(userName);
    }

    /**
     * @param userId 用户ID
     * @return UserDto
     * @author <a href="mailto:zhangwentao@gtmap.cn">zhangwentao</a>
     * @description 根据userId获取用户所有的信息
     */
    public UserDto getUserByUserid(String userId) {
        return userManagerClient.getUserDetail(userId);
    }

    /**
     * 根据userID获取用户角色  ccx
     * @param userId
     * @return
     */
    public List<RoleDto> getRolesByUserId(String userId) {
        if (StringUtils.isBlank(userId)) {
            return null;
        }
        return this.userManagerClient.findRoles(userId);
    }
    /**
     * 判断当前用户是否有管理员角色  ccx
     * @param userId
     * @return
     */
    public boolean isAdminByUserId(String userId) {
        if (StringUtils.isBlank(userId)) {
            return false;
        }
        // 查询当前用户是否管理员角色  管理员角色不加区县代码过滤条件
        List<RoleDto> roleList = getRolesByUserId(userId);
        if(CollectionUtils.isNotEmpty(roleList)){
            for(RoleDto role:roleList){
                if(role != null && StringUtils.equals( role.getName(),"admin")){
                   return true;
                }
            }
        }
        return false;
    }
}
