package cn.gtmap.realestate.common.util;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;

/**
 * @author <a href="mailto:liyinqiao@gtmap.cn">liyinqiao</a>
 * @version 1.0  2019-05-03
 * @description
 */
public class DateUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(DateUtils.class);

    public static final String sdf_China = "yyyy年MM月dd日";
    public static final String sdf = "yyyy-MM-dd";
    public static final String sdf_ymd = "yyyyMMdd";
    public static final String sdf_ymdWithSpilt = "yyyy/MM/dd";
    public static final String sdf_ymdhms = "yyyy-MM-dd hh:mm:ss";
    private static final DateTimeFormatter DATE_FORMATYMDHMS = DateTimeFormatter.ofPattern("yyyyMMddHHmmss", Locale.CHINA);
    private static final DateTimeFormatter DATE_FORMATYMD = DateTimeFormatter.ofPattern(sdf_China, Locale.CHINA);
    private static final DateTimeFormatter DATE_FORMATYMDHMS_SPLIT = DateTimeFormatter.ofPattern(sdf_ymdhms, Locale.CHINA);

    /**
     * 给Date类型的日期时间 后面加上 time
     * @param date
     * @return
     */
    public static Date dealDate(Date date,String time){
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
        String dateStr = dateFormat.format(date) + " " + time;
        SimpleDateFormat dateTimeFormat = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss");
        Date resultDate = null;
        try {
            resultDate = dateTimeFormat.parse(dateStr);
        } catch (ParseException e) {
            LOGGER.error(null, e);
        }
        return resultDate;
    }


    /**
     * @param date 日期
     * @return
     * @author <a href="mailto:lisongtao@gtmap.cn">lisongtao</a>
     * @description 格式化日期 返回某格式
     */
    public static String formateTime(Date date,DateTimeFormatter dateTimeFormatter) {
        LocalDateTime localDateTime = date2LocalDateTime(date);
        return localDateTime.format(dateTimeFormatter);
    }

    /**
     * @param date 日期
     * @return
     * @author <a href="mailto:lisongtao@gtmap.cn">lisongtao</a>
     * @description 格式化日期 返回yyyyMMddHHmmss 格式
     */
    public static String formateTimeYmdhms(Date date) {
        LocalDateTime localDateTime = date2LocalDateTime(date);
        return localDateTime.format(DATE_FORMATYMDHMS);
    }

    /**
     * @param date 日期
     * @return
     * @author <a href="mailto:lisongtao@gtmap.cn">lisongtao</a>
     * @description 格式化日期 返回yyyy年MM月dd日 格式
     */
    public static String formateYmdZw(Date date) {
        LocalDateTime localDateTime = date2LocalDateTime(date);
        return localDateTime.format(DATE_FORMATYMD);
    }

    /**
     * 格式化日期 返回 yyyy-MM-dd hh:mm:ss 格式
     *
     * @param date 日期
     * @return {String} yyyy-MM-dd hh:mm:ss
     * @author <a href="mailto:lixin1@gtmap.cn">lixin</a>
     */
    public static String formateYmdhms(Date date) {
        LocalDateTime localDateTime = date2LocalDateTime(date);
        return localDateTime.format(DATE_FORMATYMDHMS_SPLIT);
    }

    /**
     * 将日期进行格式化（参数类型java.util.Date），格式：yyyy-MM-dd
     *
     * @param str
     * @return
     */
    public static Date formatDate(String str) {
        Date date = null;
        try {
            if (StringUtils.isNotBlank(str)) {
                date = new SimpleDateFormat(sdf).parse(str);
            }

        } catch (Exception e) {
            LOGGER.info("日期转换失败：{}",str);
            try {
                date = new SimpleDateFormat(sdf_ymd).parse(str);
            } catch (ParseException e1) {
                LOGGER.info("日期转换失败：{}",str);
                try {
                    date = new SimpleDateFormat(sdf_China).parse(str);
                } catch (ParseException e2) {
                    LOGGER.info("日期转换失败：{}",str);
                    try {
                        date = new SimpleDateFormat(sdf_ymdWithSpilt).parse(str);
                    } catch (ParseException e3) {
                        LOGGER.info("日期转换失败：{}",str);
                        return null;
                    }
                }
            }
        }
        return date;
    }

    /**
     * @author <a href="mailto:liyinqiao@gtmap.cn">liyinqiao</a>
     * @param str
     * @param pattern 模板
     * @return java.util.Date
     * @description 字符串转换DATE类型
     */
    public static Date formatDate(String str,String pattern){
        Date date = null;
        try {
        if (StringUtils.isNotBlank(str)) {
            date = new SimpleDateFormat(pattern).parse(str);
            }
        } catch (ParseException e) {
            LOGGER.info("", e);
        }
        return date;
    }


    /**
     * @param date 日期
     * @return
     * @author <a href="mailto:yinyao@gtmap.cn">yinyao</a>
     * @description Date转换为LocalDateTime
     */
    public static LocalDateTime date2LocalDateTime(Date date) {
        Instant instant = date.toInstant();//An instantaneous point on the time-line.(时间线上的一个瞬时点。)
        ZoneId zoneId = ZoneId.systemDefault();//A time-zone ID, such as {@code Europe/Paris}.(时区)
        return instant.atZone(zoneId).toLocalDateTime();
    }

    /**
     * 获取季度
     * @param date 日期
     * @return 1 第一季度 2 第二季度 3 第三季度 4 第四季度
     */
    public static int getSeason(Date date) {
        int season = 0;
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        int month = c.get(Calendar.MONTH);
        switch (month) {
            case Calendar.JANUARY:
            case Calendar.FEBRUARY:
            case Calendar.MARCH:
                season = 1;
                break;
            case Calendar.APRIL:
            case Calendar.MAY:
            case Calendar.JUNE:
                season = 2;
                break;
            case Calendar.JULY:
            case Calendar.AUGUST:
            case Calendar.SEPTEMBER:
                season = 3;
                break;
            case Calendar.OCTOBER:
            case Calendar.NOVEMBER:
            case Calendar.DECEMBER:
                season = 4;
                break;
            default:
                break;
        }
        return season;
    }

    /**
     * 获取指定日期零时零分零秒时间戳
     * @param date 日期
     * @return
     */
    public static long getDayTimeOfZeroHMS(Date date){
        if(null == date){
            return -1;
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        return calendar.getTime().getTime();
    }

    /**
     * 获取指定日期23点时59分59秒时间戳
     * @param date 日期
     * @return
     */
    public static long getDayTimeOfLastHMS(Date date){
        if(null == date){
            return -1;
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);
        return calendar.getTime().getTime();
    }
}
