package cn.gtmap.realestate.common.core.support.excel;

import cn.gtmap.realestate.common.core.dto.ExcelExportDTO;
import cn.gtmap.realestate.common.util.CommonConstantUtils;
import cn.gtmap.realestate.common.util.StringToolUtils;
import com.alibaba.fastjson.JSON;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.poi.hssf.usermodel.*;
import org.apache.poi.ss.usermodel.HorizontalAlignment;
import org.apache.poi.ss.util.CellRangeAddress;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URLEncoder;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
 * @version 1.0, 2019/5/22
 * @description Excel处理控制器
 */
@RestController
@RequestMapping(value = "/excel")
public class ExcelController {
    /**
     * 日志操作
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(ExcelController.class);
    /**
     * 限定类名
     */
    private static final String CLASS_NAME = ExcelController.class.getName();

    /**
     * @param response       返回响应
     * @param excelExportDTO Excel信息实体
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @description 导出Excel（前台直接封装定义数据，不走后台再次查询处理逻辑）
     */
    @PostMapping(value = "/export")
    public void exportExcel(HttpServletResponse response, @ModelAttribute ExcelExportDTO excelExportDTO) {
        if (null == excelExportDTO || isExistEmpty(excelExportDTO)) {
            LOGGER.error("{}：导出Excel中止，原因：提交的数据存在问题！", CLASS_NAME);
            return;
        }
        if (null == excelExportDTO.getRownum()) {
            excelExportDTO.setRownum(0);
        }
        try (HSSFWorkbook workbook = new HSSFWorkbook();
             OutputStream outputStream = response.getOutputStream()) {
            // 默认工作簿名称
            String sheetName = excelExportDTO.getSheetName();
            if (StringUtils.isBlank(sheetName)) {
                sheetName = "工作簿";
            }

            // 创建工作簿
            HSSFSheet sheet = workbook.createSheet(sheetName);
            this.setCellTitle(workbook, sheet, excelExportDTO);
            this.setCellWidth(sheet, excelExportDTO.getCellWidth());
            // 判断是否需要执行 行合并操作
            if(excelExportDTO.getMergeSameCell() && StringUtils.isNotBlank(excelExportDTO.getMergeCellKey())){
                this.setCellDataAndMergeSameCell(workbook, sheet, excelExportDTO);
            }else{
                this.setCellData(workbook, sheet, excelExportDTO);
            }
            //浏览器下载
            String fileName = URLEncoder.encode(excelExportDTO.getFileName() + ".xls", "utf-8");
            response.setContentType("application/octet-stream");
            response.setHeader("Content-Disposition", "attachment;filename=" + fileName);
            outputStream.flush();
            workbook.write(outputStream);
        } catch (IOException e) {
            LOGGER.error("{}系统导出Excel报错：{}", CLASS_NAME, e.getMessage());
        }
    }

    /**
     * @return {Boolean}   false: 都不为空； true: 只要有一个为空
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @description 判断实体属性是否存在空
     */
    private boolean isExistEmpty(ExcelExportDTO excelExportDTO) {
        return StringToolUtils.existItemNullOrEmpty(excelExportDTO.getFileName(),
                excelExportDTO.getCellTitle(), excelExportDTO.getCellKey(), excelExportDTO.getData());
    }

    /**
     * @param
     * @param workbook Excel模板
     * @param sheet    Excel工作簿
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @description 设置列标题
     */
    private void setCellTitle(HSSFWorkbook workbook, HSSFSheet sheet, ExcelExportDTO excelExportDTO) {
        int rowNum = excelExportDTO.getRownum();
        // 标题内容
        String title = excelExportDTO.getCellTitle();
        String[] cellTitleArr = title.split(CommonConstantUtils.ZF_YW_DH);

        // 标题样式
        HSSFCellStyle style = workbook.createCellStyle();
        HSSFFont font = workbook.createFont();
        font.setBold(true);
        style.setAlignment(HorizontalAlignment.CENTER);
        style.setFont(font);

        // 创建表名
        HSSFRow rowTableName = sheet.createRow(rowNum);
        HSSFCell cellTableName = rowTableName.createCell(0);
        cellTableName.setCellValue(sheet.getSheetName());
        cellTableName.setCellStyle(style);
        sheet.addMergedRegion(new CellRangeAddress(
                rowNum, rowNum, 0, cellTitleArr.length - 1
        ));
        rowNum++;

        // 创建表头汇总内容
        if (StringUtils.isNotBlank(excelExportDTO.getSummaryContent())) {
            String[] summaryArr = excelExportDTO.getSummaryContent().split(CommonConstantUtils.ZF_YW_DH);
            Integer cellNum = 0;
            if (summaryArr.length > 1 && NumberUtils.isNumber(summaryArr[1])){
                cellNum = Integer.parseInt(summaryArr[1]);
            }

            rowTableName = sheet.createRow(rowNum);
            cellTableName = rowTableName.createCell(cellNum);
            cellTableName.setCellValue(summaryArr[0]);
            sheet.addMergedRegion(new CellRangeAddress(
                    rowNum, rowNum, cellNum, cellTitleArr.length - 1
            ));
            rowNum++;
        }

        // 创建标题行
        HSSFRow rowTitle = sheet.createRow(rowNum);
        for (int i = 0; i < cellTitleArr.length; i++) {
            HSSFCell cell = rowTitle.createCell(i);
            cell.setCellValue(cellTitleArr[i]);
            cell.setCellStyle(style);
        }
        rowNum++;

        excelExportDTO.setRownum(rowNum);
    }

    /**
     * @param sheet Excel工作簿
     * @param width 多列宽
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @description 设置列宽
     */
    private void setCellWidth(HSSFSheet sheet, String width) {
        if (StringUtils.isBlank(width)) {
            return;
        }

        String[] cellWidthArr = width.split(",");
        for (int i = 0; i < cellWidthArr.length; i++) {
            sheet.setColumnWidth(i, Integer.valueOf(cellWidthArr[i]) * 256);
        }
    }

    /**
     * @param workbook       Excel模板
     * @param sheet          Excel工作簿
     * @param excelExportDTO Excel信息实体
     * @author <a href="mailto:zhuyong@gtmap.cn">zhuyong</a>
     * @description 设置列数据
     */
    private void setCellData(HSSFWorkbook workbook, HSSFSheet sheet, ExcelExportDTO excelExportDTO) {
        // 解析封装的JSON数据
        List<Map<String, Object>> dataList = JSON.parseObject(excelExportDTO.getData(), List.class);
        // 每列数据对应的key
        String[] cellKeyArr = excelExportDTO.getCellKey().split(",");
        // 列样式：默认居中
        HSSFCellStyle dataCellStyle = workbook.createCellStyle();
        dataCellStyle.setAlignment(HorizontalAlignment.CENTER);

        int rowNum = excelExportDTO.getRownum();
        for (int j = 0; j < dataList.size(); j++) {
            HSSFRow dataRow = sheet.createRow(j + rowNum);
            Map<String, Object> map = dataList.get(j);

            int cellNum = 0;
            for (String cellKey : cellKeyArr) {
                String value = "";
                if (map.containsKey(cellKey) && null != map.get(cellKey)) {
                    value = String.valueOf(map.get(cellKey));
                }
                HSSFCell dataCell = dataRow.createCell(cellNum++);
                dataCell.setCellValue(value);
                dataCell.setCellStyle(dataCellStyle);
            }
        }
    }

    /**
     *
     * @param workbook       Excel模板
     * @param sheet          Excel工作簿
     * @param excelExportDTO Excel信息实体
     * @author <a href="mailto:yaoyi@gtmap.cn">yaoyi</a>
     * @description 设置列数据，合并指定列中相同值。
     * <p>例如：excel中A列 A1：测试 A2：测试 A3:结果  会将A1与A2进行合并</p>
     */
    private void setCellDataAndMergeSameCell(HSSFWorkbook workbook, HSSFSheet sheet, ExcelExportDTO excelExportDTO){
        // 解析封装的JSON数据
        List<Map<String, Object>> dataList = JSON.parseObject(excelExportDTO.getData(), List.class);
        // 每列数据对应的key
        String[] cellKeyArr = excelExportDTO.getCellKey().split(",");

        // 列样式：默认居中、垂直居中
        HSSFCellStyle dataCellStyle = workbook.createCellStyle();
        dataCellStyle.setAlignment(HorizontalAlignment.CENTER);
        dataCellStyle.setVerticalAlignment(dataCellStyle.getVerticalAlignmentEnum().CENTER);

        // 数据起始行
        int startRowNum = excelExportDTO.getRownum();
        // 存储记录合并列的数量
        Map<String,Integer> tempMap = new HashMap<>(1);
        // 自定义需要合并的cellKey
        List<String> mergeCellArr = Arrays.asList(
                excelExportDTO.getMergeCellKey().split(CommonConstantUtils.ZF_YW_DH));
        // 当前列与下一列进行数据对比与合并
        for (int rowNum = 0, next = 1; rowNum < dataList.size(); rowNum++, next++) {
            int currentRowNum = rowNum + startRowNum; // 当前行数
            HSSFRow dataRow = sheet.createRow(currentRowNum);
            // 当前行的数据集合
            Map<String, Object> map = dataList.get(rowNum);
            // 下一行的数据集合,最后一条数据和第一条数据进行比对
            Map<String, Object> nextDataMap = next==dataList.size() ? dataList.get(0):dataList.get(next);

            for(int colNum = 0; colNum<cellKeyArr.length; colNum++){
                String cellKey = cellKeyArr[colNum];
                String value = "";
                if (map.containsKey(cellKey) && null != map.get(cellKey)) {
                    value = String.valueOf(map.get(cellKey));
                }
                HSSFCell dataCell = dataRow.createCell(colNum);
                dataCell.setCellValue(value);
                dataCell.setCellStyle(dataCellStyle);
                // 判断当前列数据是否要合并
                if(mergeCellArr.contains(cellKey) && nextDataMap.containsKey(cellKey)){
                    String nextData = String.valueOf(nextDataMap.get(cellKey)); // 下一行当前列的数据
                    this.mergeCell(sheet, currentRowNum, colNum, value,
                            cellKey, nextData, mergeCellArr, tempMap);
                }

            }
        }
    }


    /**
     * 合并单元格逻辑处理
     * @param sheet  Excel工作簿
     * @param currentRowNum 当前行数
     * @param currentColNum 当前列数
     * @param cellValue 单元格的值
     * @param cellKey 单元格所对应的KEY
     * @param nextData 下一行当前列的数据
     * @param mergeCellArr 自定义合并列的cellKey
     * @param tempMap 临时
     */
    private void mergeCell(HSSFSheet sheet, int currentRowNum, int currentColNum,
                           String cellValue, String cellKey, String nextData,
                           List<String> mergeCellArr,Map<String, Integer> tempMap){
        // 比较当前行的列与下一行的列值是否一致，一致时标记合并行
        if(cellValue.equals(nextData)){
            if(tempMap.containsKey(cellKey)){
                tempMap.put(cellKey, tempMap.get(cellKey)+1);
            }else{
                tempMap.put(cellKey, 1);
            }
        }else{
            // 执行行合并操作, 当存在需要合并数据时，进行行合并
            if(tempMap.containsKey(cellKey) && !tempMap.get(cellKey).equals(0)){
                sheet.addMergedRegion(new CellRangeAddress(
                        currentRowNum-tempMap.get(cellKey), currentRowNum, currentColNum, currentColNum
                ));
                tempMap.put(cellKey, 0);
            }
        }
    }
}