package cn.gtmap.realestate.common.config.logaop;

import cn.gtmap.gtc.starter.gscas.audit.ZipkinAuditEventRepository;
import cn.gtmap.realestate.common.core.annotations.LogCommonEvent;
import cn.gtmap.realestate.common.util.CommonConstantUtils;
import cn.gtmap.realestate.common.util.CommonUtil;
import cn.gtmap.realestate.common.util.UserManagerUtils;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.actuate.audit.AuditEvent;
import org.springframework.security.core.Authentication;
import org.springframework.security.oauth2.provider.OAuth2Authentication;
import org.springframework.stereotype.Component;
import org.springframework.ui.Model;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;

/**
 * @author <a href="mailto:chenyucheng@gtmap.cn">chenyucheng</a>
 * @version 1.0, 2019/6/3
 * @description
 */
@Component
@Aspect
public class LogCommonEventAspect {

    private static final Logger LOGGER = LoggerFactory.getLogger(LogCommonEventAspect.class);

    @Autowired
    private ZipkinAuditEventRepository zipkinAuditEventRepository;

    @Autowired
    private UserManagerUtils userManagerUtils;

    @Pointcut("@annotation(logEvent)")
    public void logEventPointCut(LogCommonEvent logEvent){

    }

    @AfterReturning(
            returning = "response",
            pointcut = "logEventPointCut(logEvent)"
    )

    /**
     *  记录日志切面
     */
    public void doAfter(JoinPoint joinPoint, LogCommonEvent logEvent, Object response) {
        MethodSignature methodSignature = (MethodSignature)joinPoint.getSignature();
        // 目标函数
        if(StringUtils.isBlank(logEvent.event())) {
            LOGGER.error("方法异常：未指定日志事件！");
        }else {
            // 解析查询参数
            JSONObject paramObj = analysisMethodParam(methodSignature.getParameterNames(), joinPoint.getArgs());
            // 参数
            Map<String, Object> data = getAuditEventParam(logEvent.names(), paramObj);
            // 模块代码
            data.put(CommonConstantUtils.VIEW_TYPE, logEvent.modelDm());
            // 模块名称；
            data.put(CommonConstantUtils.VIEW_TYPE_NAME, logEvent.modelMc());
            // ip
            data.put(CommonConstantUtils.IP, getIpAddress());
            AuditEvent auditEvent = new AuditEvent(userManagerUtils.getCurrentUserName(), logEvent.event(), data);
            // 保存日志
            zipkinAuditEventRepository.add(auditEvent);
        }
    }

    /**
     * version 1.0
     * @description 获取平台日志保存参数
     * @param
     * @return
     * @date 2019/3/14
     * @author <a href ="mailto:wangwei2@gtmap.cn">wangwei2</a>
     */
    private Map<String, Object> getAuditEventParam(String[] names, JSONObject param) {
        Map<String, Object> map = Maps.newHashMap();
        // 配置文件中配置的需要保存到日志中的参数
        Map<String, String> paramConfig = LogParamsTslConstants.PARAM;

        if(MapUtils.isNotEmpty(paramConfig) && !param.isEmpty()) {
            Map<String, Object> tempMap = JSONObject.parseObject(param.toJSONString(), Map.class);
            if(names != null) {
                for(int i = 0; i < names.length; i++) {
                    if(!tempMap.containsKey(names[i])) {
                        tempMap.remove(names[i]);
                    }
                }
            }
            for(Map.Entry<String,Object> entry : tempMap.entrySet()) {
                if(paramConfig.containsKey(entry.getKey().toUpperCase())) {
                    map.put(CommonConstantUtils.PARAM_SUB + entry.getKey(), tempMap.get(entry.getKey()));
                }
            }
            // 台账类型
            if(param.containsKey(CommonConstantUtils.VIEW_TYPE)) {
                map.put(CommonConstantUtils.VIEW_TYPE, tempMap.get(CommonConstantUtils.VIEW_TYPE));
            }
        }
        return map;
    }

    /**
     * version 1.0
     * @description
     * @param paramNames 请求参数名数组
     * @param paramValue 请求参数值数组
     * @return
     * @date 2019/3/15
     * @author <a href ="mailto:wangwei2@gtmap.cn">wangwei2</a>
     */
    private JSONObject analysisMethodParam(String[] paramNames, Object[] paramValue) {
        JSONObject param = new JSONObject();
        if (paramNames != null && paramValue != null) {
            for (int i = 0; i < paramValue.length; i++) {
                if (paramValue[i] != null
                        &&!(paramValue[i] instanceof HttpServletRequest)
                        && !(paramValue[i] instanceof HttpServletResponse)
                        && !(paramValue[i] instanceof Model)
                        && !(paramValue[i] instanceof ModelAndView)
                        && !(paramValue[i] instanceof Authentication)
                        && !(paramValue[i] instanceof OAuth2Authentication)) {
                    if(CommonUtil.isJSONObject(JSONObject.toJSONString(paramValue[i]))) {
                        recursiveMethodParam(param, paramNames[i], JSONObject.toJSONString(paramValue[i]));
                    }else {
                        recursiveMethodParam(param, paramNames[i], paramValue[i].toString());
                    }
                }
            }
        }
        return param;
    }

    /**
     * version 1.0
     * @description 递归查询封装参数 去掉空值
     * @return
     * @date 2019/3/14
     * @author <a href ="mailto:wangwei2@gtmap.cn">wangwei2</a>
     */
    private void recursiveMethodParam(JSONObject result, String paramName, String jsonStr) {
        if(result == null) {
            result = new JSONObject();
        }
        if(StringUtils.isNotBlank(jsonStr)) {
            if(CommonUtil.isJSONObject(jsonStr)) {
                JSONObject temp = JSONObject.parseObject(jsonStr);
                if(temp != null) {
                    for(Map.Entry<String,Object> entry : temp.entrySet()) {
                        recursiveMethodParam(result, entry.getKey(), temp.get(entry.getKey()).toString());
                    }
                }
            }else if(CommonUtil.isJSONOArray(jsonStr)){
                List tempList = JSONArray.parseArray(jsonStr);
                // 标记为查询参数 数组参数
                result.put(paramName, StringUtils.join(tempList, ","));
            }else {
                result.put(paramName, jsonStr);
            }
        }
    }

    /**
     * 获取本机的ip
     *
     * @return
     */
    private static String getIpAddress() {
        try {
            Enumeration<NetworkInterface> allNetInterfaces = NetworkInterface.getNetworkInterfaces();
            InetAddress ip = null;
            while (allNetInterfaces.hasMoreElements()) {
                NetworkInterface netInterface = allNetInterfaces.nextElement();
                if (!netInterface.isLoopback() && !netInterface.isVirtual() && !netInterface.isPointToPoint() &&
                        netInterface.isUp()) {
                    Enumeration<InetAddress> addresses = netInterface.getInetAddresses();
                    while (addresses.hasMoreElements()) {
                        ip = addresses.nextElement();
                        if (ip instanceof Inet4Address) {
                            return ip.getHostAddress();
                        }
                    }
                }
            }
        } catch (Exception e) {
            LOGGER.error("获取本机ip方法异常！");
        }
        return "";
    }

}

