package cn.gtmap.onething.util;

import cn.gtmap.onething.entity.bo.OnethingConfig;
import cn.gtmap.onething.exception.OnethingException;
import com.alibaba.fastjson.JSON;
import com.gtis.config.AppConfig;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;

import java.io.*;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author jiangganzhi <jiangganzhi@gtmap.cn>
 * @date 2023年08月08日
 * @description
 */

public class PropertiesLoadUtil {
    public static final Logger log = Logger.getLogger(PropertiesLoadUtil.class);

    /**
     * @author jiangganzhi <jiangganzhi@gtmap.cn>
     * @date 2023/8/8
     * @descreption 通过当前线程的上下文类加载器获取配置文件，加载一件事相关配置
     */
    public static void loadFromContext() {
        Properties properties = new Properties();
        InputStream proStream = null;
        BufferedReader proBufferReader = null;
        try {
            proStream = Thread.currentThread().getContextClassLoader().getResourceAsStream("onething.properties");
            if (proStream != null) {
                properties.load(proStream);
                // 遍历每个属性
                for (String propertyName : properties.stringPropertyNames()) {
                    String propertyValue = properties.getProperty(propertyName);

                    // 匹配占位符，替换为实际的值
                    Pattern pattern = Pattern.compile("\\$\\{(.+?)\\}");
                    Matcher matcher = pattern.matcher(propertyValue);
                    StringBuffer buffer = new StringBuffer();
                    while (matcher.find()) {
                        String key = matcher.group(1);
                        String value = properties.getProperty(key);
                        if (value == null) {
                            log.error("Missing value for placeholder: " + key);
                        }
                        matcher.appendReplacement(buffer, Matcher.quoteReplacement(value));
                    }
                    matcher.appendTail(buffer);
                    String newValue = buffer.toString();

                    // 将替换后的值重新设置到 Properties 对象中
                    properties.setProperty(propertyName, newValue);
                }

                System.getProperties().putAll(properties);
            }
        } catch (Exception e) {
            throw new OnethingException(9999, "一件事-上下文类加载器-读取配置文件错误", e);
        } finally {
            try {
                if (proStream != null) {
                    proStream.close();
                }
                if (proBufferReader != null) {
                    proBufferReader.close();
                }
            } catch (Exception e) {
                throw new OnethingException(9999, "一件事-上下文类加载器-读取配置文件，流关闭失败", e);
            }
        }
    }

    /**
     * @author jiangganzhi <jiangganzhi@gtmap.cn>
     * @date 2023/8/8
     * @descreption 根据配置文件父路径，加载一件事相关配置
     */
    public static void loadProperties(String propertiesParentPath) {
        Properties properties = new Properties();
        InputStream proStream = null;
        BufferedReader proBufferReader = null;
        try {
            boolean exist = new File(propertiesParentPath + File.separator + "onething.properties").exists();
            if (exist) {
                proStream = new BufferedInputStream(new FileInputStream(propertiesParentPath + File.separator + "onething.properties"));
                proBufferReader = new BufferedReader(new InputStreamReader(proStream, "UTF-8"));
                properties.load(proBufferReader);

                // 遍历每个属性
                for (String propertyName : properties.stringPropertyNames()) {
                    String propertyValue = properties.getProperty(propertyName);

                    // 匹配占位符，替换为实际的值
                    Pattern pattern = Pattern.compile("\\$\\{(.+?)\\}");
                    Matcher matcher = pattern.matcher(propertyValue);
                    StringBuffer buffer = new StringBuffer();
                    while (matcher.find()) {
                        String key = matcher.group(1);
                        String value = properties.getProperty(key);
                        if (value == null) {
                            log.error("Missing value for placeholder: " + key);
                        }
                        matcher.appendReplacement(buffer, Matcher.quoteReplacement(value));
                    }
                    matcher.appendTail(buffer);
                    String newValue = buffer.toString();

                    // 将替换后的值重新设置到 Properties 对象中
                    properties.setProperty(propertyName, newValue);
                }

                System.getProperties().putAll(properties);
            } else {
                loadFromContext();
            }
        } catch (Exception e) {
            throw new OnethingException(9999, "一件事-通过路径-读取配置文件错误", e);
        } finally {
            try {
                if (proStream != null) {
                    proStream.close();
                }
                if (proBufferReader != null) {
                    proBufferReader.close();
                }
            } catch (Exception e) {
                throw new OnethingException(9999, "一件事-通过路径-读取配置文件，流关闭失败", e);
            }
        }
    }

    public static OnethingConfig loadOnethingConfigFromJson() {
        String jsonPath = AppConfig.getProperty("onething.config.json.path");
        OnethingConfig onethingConfig = null;
        if (StringUtils.isBlank(jsonPath)) {
            throw new OnethingException(9999, "一件事-读取配置json文件失败-配置文件未配置json路径");
        } else {
            Boolean existFile = new File(jsonPath).exists();
            if (!existFile) {
                throw new OnethingException(9999, "一件事-读取配置json文件失败-配置json路径没有对应文件");
            }
            String jsonStr = null;
            try {
                Path path = Paths.get(jsonPath);
                jsonStr = new String(Files.readAllBytes(path));
            } catch (IOException e) {
                throw new OnethingException(9999, "一件事-读取配置json文件失败-读取文件失败");
            }
            if (StringUtils.isBlank(jsonStr)) {
                throw new OnethingException(9999, "一件事-读取配置json文件失败-文件内容为空");
            }
            onethingConfig = JSON.parseObject(jsonStr, OnethingConfig.class);
        }
        return onethingConfig;
    }

    public static void setOnethingJsonPath(String path) {
        System.getProperties().put("onething.config.json.path", path);
    }
}
