/*
 * Project:  onemap
 * Module:   server
 * File:     ArcgisServerProxyServiceHandlerImpl.java
 * Modifier: xyang
 * Modified: 2013-05-23 02:51:10
 *
 * Copyright (c) 2013 Gtmap Ltd. All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent or the
 * registration of a utility model, design or code.
 */

package cn.gtmap.onemap.server.handle.service;

import cn.gtmap.onemap.model.Service;
import cn.gtmap.onemap.model.ServiceProvider;
import cn.gtmap.onemap.model.ServiceType;
import cn.gtmap.onemap.server.Constants;
import cn.gtmap.onemap.server.handle.AbstractArcgisHandler;
import cn.gtmap.onemap.server.handle.ServiceHandler;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.InputStreamEntity;
import org.springframework.util.FileCopyUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Enumeration;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 13-4-9
 */
public class ArcgisServerProxyServiceHandlerImpl extends AbstractArcgisHandler implements ServiceHandler {
    public static final Set<String> INPUT_IGRONES = Sets.newHashSet("accept-encoding", "cookie");
    public static final Set<String> OUTPUT_ALLOWS = Sets.newHashSet("cache-control", "etag", "last-modified", "date");

    private HttpClient httpClient;

    public void setHttpClient(HttpClient httpClient) {
        this.httpClient = httpClient;
    }

    @Override
    public boolean accept(String[] paths, ServiceProvider provider, HttpServletRequest request) {
        if (isTileRequest(paths)) {
            if (!provider.hasAttribute(Constants.TILE_INFO)) {
                return false;
            }
        }
        return true;
    }

    @Override
    public void handle(String[] paths, ServiceProvider provider, HttpServletRequest request, HttpServletResponse response) throws Exception {
        String url;
        if (paths.length > 2 && "export".equals(paths[1])) {
            StringBuilder sb = new StringBuilder(32);
            sb.append(StringUtils.substringBefore(provider.getAttribute("url"), "/rest/")).append("/rest");
            for (int i = 2; i < paths.length; i++) {
                sb.append("/").append(paths[i]);
            }
            url = sb.toString();
        } else {
            String queryString = request.getQueryString();
            url = provider.getAttribute("url") + StringUtils.substringAfter(StringUtils.join(paths, "/"), Constants.MAP_SERVER) + (queryString != null ? "?" + queryString : "");
        }
        HttpRequestBase hq;
        if ("post".equalsIgnoreCase(request.getMethod())) {
            hq = new HttpPost(url);
            ((HttpPost) hq).setEntity(new InputStreamEntity(request.getInputStream(), -1));
        } else {
            hq = new HttpGet(url);
        }
        Enumeration hns = request.getHeaderNames();
        while (hns.hasMoreElements()) {
            String name = (String) hns.nextElement();
            if (!INPUT_IGRONES.contains(name.toLowerCase())) {
                hq.setHeader(name, request.getHeader(name));
            }
        }
        try {
            HttpResponse hr = httpClient.execute(hq);
            for (Header header : hr.getAllHeaders()) {
                String name = header.getName();
                if (OUTPUT_ALLOWS.contains(name.toLowerCase())) {
                    response.setHeader(name, header.getValue());
                }
            }
            int code = hr.getStatusLine().getStatusCode();
            response.setStatus(code);
            HttpEntity entity = hr.getEntity();
            if (entity != null) {
                String contentType = entity.getContentType().getValue();
                response.setContentType(contentType);
                String charset = StringUtils.substringAfter(contentType, "charset=");
                if (StringUtils.EMPTY.equals(charset)) {
                    FileCopyUtils.copy(entity.getContent(), response.getOutputStream());
                } else {
                    String mapPath = provider.getMap().getPath();
                    String content = rewriteExport(request.getContextPath() + "/" + Constants.ARCGISREST + "/" + mapPath + "/MapServer", IOUtils.toString(entity.getContent(), charset));
                    content = rewriteContent(mapPath,
                            request.getContextPath(),
                            content,
                            StringUtils.substringBetween(url, "services/", "/MapServer")
                    );
                    response.getWriter().write(content);
                }
            }
        } finally {
            hq.releaseConnection();
        }
    }

    @Override
    public List<Service> getServices(ServiceProvider sp) {
        List<Service> ss = Lists.newArrayListWithCapacity(3);
        Service restService = new Service();
        restService.setServiceType(ServiceType.ARCGIS_REST);
        restService.setUrl(getServiceUrl(sp));
        ss.add(restService);
        Service exportService = restService.clone();
        exportService.setServiceType(ServiceType.ARCGIS_EXPORT);
        ss.add(exportService);
        if (sp.hasAttribute(Constants.TILE_INFO)) {
            Service tileService = restService.clone();
            tileService.setServiceType(ServiceType.ARCGIS_TILE);
            ss.add(tileService);
        }
        return ss;
    }

    private static final Pattern REPLACE_PATTERN = Pattern.compile("(href|src|action)=\"([^\"]+)\"");

    private static String rewriteContent(String mapPath, String ctx, String content, String s) {
        String restUrl = ctx + "/" + Constants.ARCGISREST;
        Matcher m = REPLACE_PATTERN.matcher(content);
        StringBuffer sb = new StringBuffer();
        while (m.find()) {
            String url = m.group(2);
            url = StringUtils.replaceEach(url,
                    new String[]{"/arcgis/rest", "/css", "/static/", "/services", s + "/"},
                    new String[]{"", ctx + "/static/css/main.css", ctx + "/static/css/", restUrl, mapPath + "/"}
            );
            m.appendReplacement(sb, m.group(1) + "=\"" + url + "\"");
        }
        m.appendTail(sb);
        return sb.toString();
    }

    private static final Pattern EXPORT_PATTERN = Pattern.compile("http://.+rest/(directories[^\"]+)");

    private static String rewriteExport(String base, String content) {
        Matcher m = EXPORT_PATTERN.matcher(content);
        StringBuffer sb = new StringBuffer();
        while (m.find()) {
            m.appendReplacement(sb, base + "/export/" + m.group(1));
        }
        m.appendTail(sb);
        return sb.toString();
    }
}
