package cn.gtmap.onemap.platform.service.impl;

import cn.gtmap.onemap.platform.event.GeometryServiceException;
import cn.gtmap.onemap.platform.service.GISManager;
import cn.gtmap.onemap.platform.utils.GeometryUtils;
import cn.gtmap.onemap.platform.utils.SRTransformations;
import cn.gtmap.onemap.service.GeoService;
import com.esri.sde.sdk.pe.PeProjectionException;
import com.vividsolutions.jts.geom.Geometry;
import com.vividsolutions.jts.operation.valid.IsValidOp;
import com.vividsolutions.jts.operation.valid.TopologyValidationError;
import com.vividsolutions.jts.simplify.DouglasPeuckerSimplifier;
import org.geotools.feature.FeatureCollection;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.referencing.crs.CoordinateReferenceSystem;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.File;
import java.util.List;
import java.util.Map;

/**
 * .
 *
 * @author <a href="mailto:lanxy88@gmail.com">NelsonXu</a>
 * @version V1.0, 14-1-2 上午11:13
 */
public class GeoServiceImpl implements GeoService {

    protected final static String SE_OBJECTID_FIELD = "OBJECTID";

    @Autowired
    private GISManager gisManager;

    /**
     * 属性查询接口
     *
     * @param layerName      SDE中空间图层名称
     * @param where          查询条件
     * @param outFields      返回字段，默认null
     * @param returnGeometry 是否返回图形信息，默认true
     * @param dataSource     是否指定图层所在的数据源，默认null
     * @return map list
     */
    @Override
    public List query(String layerName, String where, String[] outFields, boolean returnGeometry, String dataSource) {
        try{
            return gisManager.getGISService().query(layerName, where, outFields, returnGeometry, dataSource);
        }catch (Exception e){
            throw new RuntimeException(" query error , detail : " + e.getLocalizedMessage());
        }
    }

    /**
     * 空间查询接口
     *
     * @param layerName  SDE中空间图层名称
     * @param geometry   查询的空间要素
     * @param outFields  返回字段，默认null
     * @param dataSource 是否指定图层所在的数据源，默认null
     * @return map list
     */
    @Override
    public List query(String layerName, Geometry geometry, String[] outFields, String dataSource) {
        try{
            return gisManager.getGISService().query(layerName, geometry, outFields, dataSource);
        }catch (Exception e){
            throw new RuntimeException(" query error , detail : " + e.getLocalizedMessage());
        }
    }

    /**
     * 空间查询接口
     *
     * @param layerName  SDE中空间图层名称
     * @param geometry   查询的空间要素，为GeoJSON格式
     * @param outFields  返回字段，默认null
     * @param dataSource 是否指定图层所在的数据源，默认null
     * @return map list
     */
    @Override
    public List query(String layerName, String geometry, String[] outFields, String dataSource) {
        try{
            return gisManager.getGISService().query(layerName, gisManager.getGeoService().readGeoJSON(geometry), outFields, dataSource);
        }catch (Exception e){
            throw new RuntimeException(" query error , detail : " + e.getLocalizedMessage());
        }
    }

    /**
     * 数据插入接口
     *
     * @param layerName  SDE中空间图层名称
     * @param feature    待插入要素，GeoJSON格式，可为Geometry、Feature或者FeatureCollection
     * @param dataSource 是否指定图层所在的数据源，默认null
     * @return GeoJSON要素，包含图形与属性，以及插入后OBJECTID属性值
     */
    @Override
    public String insert(String layerName, String feature, String dataSource) {
        try {
            return gisManager.getGISService().insert2(layerName, feature, dataSource);
        } catch (Exception e) {
            throw new RuntimeException(" insert error , detail : " + e.getLocalizedMessage());
        }
    }

    /**
     * 数据插入接口(执行拓扑检查)
     * @param layerName   SDE中空间图层名称
     * @param geoJSON     待插入要素，GeoJSON格式，可为Geometry、Feature或者FeatureCollection
     * @param check       是否执行拓扑检查
     * @param dataSource  是否指定图层所在的数据源，默认null
     * @return
     */
    @Override
    public String insert(String layerName, String geoJSON, boolean check, String dataSource) {
        try {
            return gisManager.getGISService().insert2(layerName, geoJSON,check, dataSource);
        } catch (Exception e) {
            throw new RuntimeException(" insert error , detail : " + e.getLocalizedMessage());
        }
    }

    /**
     * 数据更新接口
     *
     * @param layerName  SDE中空间图层名称
     * @param where      要素更新条件
     * @param feature    GeoJSON格式要素，可单独图形或者属性
     * @param dataSource 是否指定图层所在的数据源，默认null
     * @return
     */
    @Override
    public boolean update(String layerName, String where, String feature, String dataSource) {
        List list = query(layerName, where, null, true, dataSource);
        for (int i = 0; i < list.size(); i++) {
            Map item = (Map) list.get(i);
            String primaryKey = String.valueOf(item.get(SE_OBJECTID_FIELD));
            if (!gisManager.getGISService().update(layerName, primaryKey, feature, dataSource))
                return false;
        }
        return true;
    }

    /**
     * 数据删除接口
     *
     * @param layerName  SDE中空间图层名称
     * @param where      要素删除条件
     * @param dataSource 是否指定图层所在的数据源，默认null
     * @return
     */
    @Override
    public boolean delete(String layerName, String where, String dataSource) {
        List list = query(layerName, where, null, true, dataSource);
        for (int i = 0; i < list.size(); i++) {
            Map item = (Map) list.get(i);
            String primaryKey = String.valueOf(item.get(SE_OBJECTID_FIELD));
            if (!gisManager.getGISService().delete(layerName, primaryKey, dataSource))
                return false;
        }
        return true;
    }

    /**
     * 相交分析接口
     *
     * @param layerName
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public String intersect(String layerName, String geometry, String outFields, String dataSource) {
        String[] fields = "*".equals(outFields) ? null : outFields.split(",");
        List results = gisManager.getGISService().intersect2(layerName, geometry, fields, dataSource);
        FeatureCollection collection = gisManager.getGeoService().list2FeatureCollection(results, null, null);
        return toFeatureJSON(collection);
    }

    /**
     * intersect
     *
     * @param layerName
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public List intersect(String layerName, Geometry geometry, String[] outFields, String dataSource) {
        try {
            return gisManager.getGISService().intersect(layerName, geometry, outFields, dataSource);
        } catch (Exception e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }

    /**
     * 土地规划审查分析
     *
     * @param year
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public Map tdghsc(String year, String geometry, String outFields, String dataSource) {
        String[] fields = "*".equals(outFields) ? null : outFields.split(",");
        return gisManager.getGISService().tdghscResult(gisManager.getGISService().tdghscAnalysis2(year, geometry, fields, dataSource));
    }

    /**
     * 土地利用现状分析
     *
     * @param dltbLayerName
     * @param xzdwLayerName
     * @param geometry
     * @param dataSource
     * @return
     */
    @Override
    public Map tdlyxz(String dltbLayerName, String xzdwLayerName, String geometry, String dataSource) {
        return gisManager.getGISService().tdlyxzAnalysis2(dltbLayerName, xzdwLayerName, geometry, dataSource);
    }

    /**
     * 读取GeoJSON格式，转化为Geometry、GeometryCollection、SimpleFeature、FeatureCollection
     *
     * @param geoJSON 标准化GeoJSON格式
     * @return
     */
    @Override
    public Object readGeoJSON(String geoJSON) {
        return GeometryUtils.parseGeoJSON(geoJSON);
    }

    /**
     * 读取WKT标准格式几何图形
     *
     * @param wkt wkt格式几何描述
     * @return
     */
    @Override
    public Geometry readWKT(String wkt) {
        try{
            return gisManager.getGeoService().readWKT(wkt);
        }catch (Exception e){
            throw new RuntimeException(" readWKT error , detail : " + e.getLocalizedMessage());
        }
    }

    /**
     * 读取空间参考
     *
     * @param geoJson 包含crs信息的geojson格式字符串
     * @return
     */
    @Override
    public CoordinateReferenceSystem readCRS(String geoJson) {
        return gisManager.getGeoService().readFeatureJSONCRS(geoJson);
    }

    /**
     * @param sr wkid or wkt
     * @return
     */
    @Override
    public CoordinateReferenceSystem parseUndefineSR(String sr) {
        return gisManager.getGeoService().parseUndefineSR(sr);
    }

    /**
     * 转化为FeatureJSON格式字符串
     *
     * @param feature
     * @return
     */
    @Override
    public String toFeatureJSON(Object feature) {
        return gisManager.getGeoService().toFeatureJSON(feature);
    }

    /**
     * 转化为GeoJSON格式
     *
     * @param geometry
     * @return
     */
    @Override
    public String toGeoJSON(Object geometry) {
        if (geometry instanceof Geometry) {
            return gisManager.getGeoService().toGeoJSON((Geometry) geometry);
        } else if (geometry instanceof SimpleFeature) {
            return toFeatureJSON(geometry);
        } else if (geometry instanceof FeatureCollection) {
            return toFeatureJSON(geometry);
        } else
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.GEOMETRY_TO_JSON_EXCEPTION, "geometry not be supported");
    }

    /**
     * 几何图形投影转换
     *
     * @param geometry
     * @param sourceCrs
     * @param targetCrs
     * @return
     */
    @Override
    public Geometry project(Geometry geometry, String sourceCrs, String targetCrs) {
        try {
            return SRTransformations.project(geometry, sourceCrs, targetCrs);
        } catch (PeProjectionException e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.PROJECT_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * 几何图形投影转换
     *
     * @param geometry
     * @param inSR
     * @param outSR
     * @return
     */
    @Override
    public Geometry project(Geometry geometry, int inSR, int outSR) {
        return gisManager.getGeoService().project(geometry, readCRS(String.valueOf(inSR)), readCRS(String.valueOf(outSR)));
    }

    /**
     * 强制简化图形
     *
     * @param geometry
     * @param tolerance
     * @return
     */
    @Override
    public Geometry simplify(Geometry geometry, double tolerance) {
        return geometry.isValid() && geometry.isSimple() ? geometry : DouglasPeuckerSimplifier.simplify(geometry, tolerance);
    }

    /**
     * 验证几何图形拓扑关系
     *
     * @param geometry
     * @return
     */
    @Override
    public TopologyValidationError validGeometry(Geometry geometry) {
        IsValidOp isValidOp = new IsValidOp(geometry);
        return isValidOp.getValidationError();
    }

    /**
     * Map value to SimpleFeature
     *
     * @param value   {"SHAPE":"WKT FORMATE","PRO1":"VALUE"}
     * @param srcCRS
     * @param destCRS
     * @return
     */
    @Override
    public String map2SimpleFeature(Map<String, Object> value, CoordinateReferenceSystem srcCRS, CoordinateReferenceSystem destCRS) {
        return toFeatureJSON(gisManager.getGeoService().map2SimpleFeature(value, srcCRS, destCRS));
    }

    /**
     * list values to featureCollection
     *
     * @param value
     * @param srcCRS
     * @param destCRS
     * @return
     */
    @Override
    public String list2FeatureCollection(List<Map<String, Object>> value, CoordinateReferenceSystem srcCRS, CoordinateReferenceSystem destCRS) {
        try{
            return toFeatureJSON(gisManager.getGeoService().list2FeatureCollection(value, srcCRS, destCRS));
        }catch (Exception e){
            throw new RuntimeException(" list values to featureCollection error , detail : " + e.getLocalizedMessage());
        }
    }

    /**
     * SimpleFeature to HashMap
     *
     * @param feature
     * @return
     */
    @Override
    public Map<String, Object> simpleFeature2Map(SimpleFeature feature) {
        return gisManager.getGeoService().simpleFeature2Map(feature);
    }

    /**
     * get crs by x
     *
     * @param x
     * @return
     */
    @Override
    public CoordinateReferenceSystem getCrsByCoordXD(double x) {

        return gisManager.getGeoService().getCrsByCoordXD(x);
    }

    /**
     *
     * @param geoJson GeoJSON格式  eg.{"type":"Feature","crs":{"type":"name","properties":{"name":"EPSG:4610"}},"geometry":{"type":"Polygon","coordinates":[[[118.65,32.83],[118.65,32.82],[118.64,32.83],[118.65,32.83]]]},"properties":{"PRONAME":"xxx"}}
     * @return
     * @throws Exception
     */
    @Override
    public File exportToShp(String geoJson) throws Exception {
        return gisManager.getGeoService().exportToShp(geoJson);
    }

    /**
     *
     * @param layerName  SDE空间图层名
     * @param where      查询条件
     * @param geometry   查询的空间要素，为GeoJSON格式
     * @param outFields  返回字段，默认null
     * @param dataSource sde数据源,默认null
     * @return
     * @throws Exception
     */
    @Override
    public File exportToShp(String layerName, String where, String geometry, String[] outFields, String dataSource) throws Exception {
        return gisManager.getGeoService().exportToShp(layerName,where,geometry,outFields,dataSource);
    }

}
