package cn.gtmap.onemap.platform.service.impl;

import cn.gtmap.onemap.platform.Constant;
import cn.gtmap.onemap.platform.dao.BMarkDao;
import cn.gtmap.onemap.platform.entity.BMark;
import cn.gtmap.onemap.platform.service.BMarkService;
import cn.gtmap.onemap.platform.service.DocumentService;
import cn.gtmap.onemap.platform.service.GISManager;
import cn.gtmap.onemap.platform.service.GeometryService;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.vividsolutions.jts.geom.*;
import com.vividsolutions.jts.operation.valid.TopologyValidationError;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.geotools.geometry.jts.JTSFactoryFinder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.io.IOException;
import java.io.InputStream;
import java.util.*;

/**
 * . 界址点导入/勘测定界
 *
 * @author <a href="mailto:lanxy88@gmail.com">NelsonXu</a>
 * @version V1.0, 13-9-6 上午9:17
 */
//@Service
public class BMarkServiceImpl extends BaseLogger implements BMarkService {

    private static final String OBJECTID = "OBJECTID";

    private static final String NAME = "name";

    private GeometryFactory factory = JTSFactoryFinder.getGeometryFactory();

    enum Pnt {no, rno, x, y}

    enum CoordsTag {
        begin("[地块坐标]"),
        start("@"), point("点号");

        private String label;

        CoordsTag(String value) {
            this.label = value;
        }

        String getLabel() {
            return label;
        }

        public static CoordsTag getTagByValue(String value) {
            for (CoordsTag tag : CoordsTag.values()) {
                if (tag.getLabel().equals(value)) return tag;
            }
            return null;
        }
    }

    @Autowired
    private BMarkDao bMarkDao;

    @Autowired
    private DocumentService documentService;

    @Autowired
    private GISManager gisManager;

    @Autowired
    private GeometryService geometryService;

    /**
     * config
     */
    private Map config;

    /**
     * get all
     *
     * @return
     */
    @Override
    public List<BMark> findAll() {
        return bMarkDao.findAll();
    }

    /**
     * find one
     *
     * @param id
     * @return
     */
    @Override
    public BMark find(String id) {
        return bMarkDao.findOne(id);
    }

    /**
     * insert
     *
     * @param bMark
     * @return
     */
    @Override
    public BMark insert(BMark bMark) {
        Assert.notNull(bMark, getMessage("bm.insert.null"));
        return bMarkDao.save(bMark);
    }

    /**
     * insert bMarks
     *
     * @param bMarks
     * @return
     */
    @Override
    public List<BMark> insert(List<BMark> bMarks) {
        List<BMark> result = new ArrayList<BMark>();
        for (BMark bMark : bMarks) {
            result.add(insert(bMark));
        }
        return result;
    }

    /**
     * 界址点导入
     *
     * @param id
     * @param coords
     * @param type
     * @return
     */
    @Transactional
    @Override
    public boolean insert(String id, List coords, String type) {
        if (isNull(id)) throw new RuntimeException(getMessage("bm.insert.id.null"));
        if (isNull(type)) throw new RuntimeException(getMessage("bm.insert.type.null"));
        BMark bMark = new BMark();
        bMark.setProId(id);
        Geometry geo = parseCoords2Geo(coords);
        TopologyValidationError validationError = geometryService.validGeometry(geo);
        if (!isNull(validationError))
            throw new RuntimeException(getMessage("bm.tp.check.error", validationError.getMessage()));
        bMark.setCoordinate(geo.toText());
        insert2Layer(insert(bMark), getLayerName(type), getLayerProId(type), getDataSource(type));
        return true;
    }

    /**
     * 根据文件流获取 坐标
     *
     * @param inputStream
     * @param type
     * @return
     */
    @Override
    public List getCoordsByFile(InputStream inputStream, Type type) {
        try {
            switch (type) {
                case xls:
                case xlsx:
                    return getCoordsByExcel(documentService.readExcel(inputStream));
                case txt:
                    //
                    break;
            }
            IOUtils.closeQuietly(inputStream);
        } catch (Exception e) {
            throw new RuntimeException(getMessage("bm.parse.coords.error", e.getLocalizedMessage()));
        }
        throw new RuntimeException(getMessage("bm.parse.not.support"));
    }

    /**
     * 关键字快速查找
     *
     * @param key
     * @return
     */
    @Override
    public List query(String key, String type) {
        String title = getLayerTitleField(type);
        List<Map> temp = (List<Map>) gisManager.getGISService().query(getLayerName(type),
                title.concat(" like '%".concat(key).concat("%'")), null, false, getDataSource(type));
        List results = new ArrayList();
        for (Map entry : temp) {
            Map item = new HashMap();
            item.put(OBJECTID, entry.get(OBJECTID));
            item.put(NAME, entry.get(title));
            results.add(item);
        }
        return results;
//        return JSON.parseObject("[{\"name\":\"地块1\"},{\"name\":\"地块1\"},{\"name\":\"地块1\"}]", List.class);
    }

    /**
     * Get Excel Coords list
     *
     * @param excel
     * @return
     */
    private List getCoordsByExcel(List<List> excel) {
        List<List<Map>> coords = new ArrayList<List<Map>>();
        List<List> polygons = new ArrayList<List>();
        List polygon = null;
        boolean begin = false;
        for (List row : excel) {
            if (row.size() == 0) continue;
            if (row.size() == 1) {
                String desc = (String) row.get(0);
                if (CoordsTag.begin.getLabel().equals(desc)) {
                    begin = true;
                    continue;
                }
                if (begin && StringUtils.isNotBlank(desc) && desc.lastIndexOf(CoordsTag.start.getLabel()) == desc.length() - 1) {
                    polygon = new ArrayList();
                    polygons.add(polygon);
                    continue;
                }
            }
            if (begin)
                polygon.add(row);
        }
        List<Map> coord = null;
        for (int i = 0; i < polygons.size(); i++) {
            List<List> p = polygons.get(i);
            for (List r : p) {
                String s = StringUtils.trim(String.valueOf(r.get(0)));
                if (CoordsTag.point.getLabel().equals(s)) {
                    coord = new ArrayList<Map>();
                    continue;
                }
                if (coord != null)
                    coord.add(list2Map(r));
            }
            if (!isNull(coord))
                coords.add(coord);
        }
        return coords;
    }

    /**
     * 行坐标
     *
     * @param value
     * @return
     */
    private Map list2Map(List value) {
        if (value.size() != 4) return null;
        Map map = new HashMap();
        map.put(Pnt.no, value.get(0));
        map.put(Pnt.rno, value.get(1));
        map.put(Pnt.x, value.get(3));
        map.put(Pnt.y, value.get(2));
        return map;
    }

    /**
     * 将界址点生成至图形中
     *
     * @param bMark
     */
    private void insert2Layer(BMark bMark, String layerName, String proId, String dataSource) {
        Assert.notNull(bMark, getMessage("bm.insert.null"));
        Map<String, Object> columns = new HashMap<String, Object>();
        columns.put(proId, bMark.getProId());
        columns.put(Constant.SE_SHAPE_FIELD, bMark.getCoordinate());
        gisManager.getGISService().insert(layerName, columns, dataSource);
    }

    /**
     * parse coords
     *
     * @param coords
     * @return
     */
    private Geometry parseCoords2Geo(List coords) {
        if (isNull(coords)) throw new RuntimeException(getMessage("bm.insert.coords.null"));
        List<Polygon> polygons = new ArrayList<Polygon>();
        for (Object coord : coords) {
            polygons.add(createPolygon((JSONArray) coord));
        }
        if (polygons.size() == 1) return polygons.get(0);
        return factory.createMultiPolygon(polygons.toArray(new Polygon[]{}));
    }

    /**
     * create polygon
     *
     * @param array
     * @return
     */
    private Polygon createPolygon(JSONArray array) {
        Map<Integer, List<Coordinate>> rings = new LinkedHashMap<Integer, List<Coordinate>>();
        for (Object object : array) {
            JSONObject _o = (JSONObject) object;
            int rno = Integer.parseInt(String.valueOf(_o.get(Pnt.rno.name())));
            if (rings.containsKey(rno))
                rings.get(rno).add(new Coordinate(Double.valueOf(String.valueOf(_o.get(Pnt.x.name()))),
                        Double.valueOf(String.valueOf(_o.get(Pnt.y.name())))));
            else {
                List p = new ArrayList();
                p.add(new Coordinate(Double.valueOf(String.valueOf(_o.get(Pnt.x.name()))),
                        Double.valueOf(String.valueOf(_o.get(Pnt.y.name())))));
                rings.put(rno, p);
            }
        }
        LinearRing shell = null;
        List<LinearRing> holes = new ArrayList<LinearRing>();
        for (Map.Entry entry : rings.entrySet()) {
            if (isNull(shell)) {
                shell = factory.createLinearRing(((List<Coordinate>) entry.getValue()).toArray(new Coordinate[]{}));
            } else
                holes.add(factory.createLinearRing(((List<Coordinate>) entry.getValue()).toArray(new Coordinate[]{})));
        }
        return factory.createPolygon(shell, holes.toArray(new LinearRing[]{}));
    }

    /**
     * @param value
     * @return
     */
    @Deprecated
    private List line2List(JSONObject value) {
        Coordinate c = new Coordinate();

        List _r = new ArrayList();
        _r.add(value.get(Pnt.x.name()));
        _r.add(value.get(Pnt.y.name()));
        return _r;
    }

    /**
     * set bmark config
     *
     * @param path
     */
    public void setConfig(Resource path) {
        try {
            this.config = JSON.parseObject(IOUtils.toString(path.getURI(), Constant.UTF_8), Map.class);
        } catch (IOException e) {
            logger.error(getMessage("bm.config.not.found"));
        }
    }

    /**
     * 获取主键
     *
     * @param type
     * @return
     */
    private String getLayerProId(String type) {
        if (!this.config.containsKey(type)) throw new RuntimeException(getMessage("bm.config.type.not.found", type));
        return (String) ((JSONObject) this.config.get(type)).get("idField");
    }

    /**
     * get layer name
     *
     * @param type
     * @return
     */
    private String getLayerName(String type) {
        if (!this.config.containsKey(type)) throw new RuntimeException(getMessage("bm.config.type.not.found", type));
        return (String) ((JSONObject) this.config.get(type)).get("layerName");
    }

    /**
     * get layer title field
     *
     * @param type
     * @return
     */
    private String getLayerTitleField(String type) {
        if (!this.config.containsKey(type)) throw new RuntimeException(getMessage("bm.config.type.not.found", type));
        return (String) ((JSONObject) this.config.get(type)).get("titleField");
    }

    /**
     * get ds
     *
     * @param type
     * @return
     */
    private String getDataSource(String type) {
        if (!this.config.containsKey(type)) throw new RuntimeException(getMessage("bm.config.type.not.found", type));
        return (String) ((JSONObject) this.config.get(type)).get("dataSource");
    }
}
