package cn.gtmap.onemap.platform.controller;

import cn.gtmap.onemap.platform.Constant;
import cn.gtmap.onemap.platform.entity.Configuration;
import cn.gtmap.onemap.platform.entity.LocationMark;
import cn.gtmap.onemap.platform.event.GISDaoException;
import cn.gtmap.onemap.platform.event.JSONMessageException;
import cn.gtmap.onemap.platform.service.*;
import cn.gtmap.onemap.platform.support.spring.BaseController;
import cn.gtmap.onemap.platform.utils.Utils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.esri.sde.sdk.client.SeException;
import com.gtis.common.util.UUIDGenerator;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * . 地图
 *
 * @author <a href="mailto:lanxy88@gmail.com">NelsonXu</a>
 * @version V1.0, 13-3-26 下午2:10
 */
@Controller
@RequestMapping("/map")
public class MapController extends BaseController {

    @Autowired
    private WebMapService webMapService;
    @Autowired
    private MapService mapService;
    @Autowired
    private LocationMarkService locationMarkService;
    @Autowired
    private QueryService queryService;
    @Autowired
    private GISManager gisManager;
    @Autowired
    private SearchService searchService;


    /**
     * 请求map页面
     *
     * @param tpl
     * @param model
     * @return
     */
    @RequestMapping(value = "/{tpl}", method = RequestMethod.GET)
    public String index(@PathVariable String tpl, Model model, HttpServletResponse response) {
        response.setHeader(Constant.ACCESS_CONTROL_ALLOW_ORIGN, "*");
        model.addAttribute("tpl", tpl);
        return "map";
    }


    /**
     * 请求地图配置
     *
     * @param tpl
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/{tpl}/config")
    @ResponseBody
    public Configuration config(@PathVariable String tpl) throws Exception {
        try {
            Assert.notNull(tpl, getMessage("map.tpl.notnull"));
            return webMapService.getConfig(tpl);
//            return mapService.getConfig(tpl);
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 请求行政区列表
     *
     * @param tpl
     * @param regionCode
     * @param sr
     * @param year
     * @return
     */
    @RequestMapping(value = "/{tpl}/regioncode")
    @ResponseBody
    public Map regionCode(@PathVariable String tpl, @RequestParam(value = "regionCode") String regionCode,
                          @RequestParam(value = "sr", required = false) String sr,
                          @RequestParam(value = "year", defaultValue = "current", required = false) String year) {
        try {
            return webMapService.getRegionInfo(Utils.formatRegionCode(regionCode), sr);
//            return mapService.getRegion2(Utils.formatRegionCode(regionCode), sr);
        } catch (Exception e) {
            throw new JSONMessageException(getMessage("map.regioncode.error", e.getLocalizedMessage()));
        }
    }

    /**
     * 请求行政区列表
     *
     * @param tpl
     * @param regionCode
     * @return
     * @deprecated
     */
    @Deprecated
    @RequestMapping(value = "/{tpl}/regionshape")
    @ResponseBody
    public Map regionShape(@PathVariable String tpl, @RequestParam(value = "regionCode") String regionCode) {
        try {
            return webMapService.getRegionInfo(regionCode, tpl);
//            return mapService.getRegionShape(regionCode, tpl);
        } catch (Exception e) {
            throw new JSONMessageException(getMessage("map.regioncode.error", e.getLocalizedMessage()));
        }
    }

    /**
     * 所有服务列表
     *
     * @param tpl
     * @return
     */
    @RequestMapping(value = "/{tpl}/services")
    @ResponseBody
    public Object getServices(@PathVariable String tpl,
                              @RequestParam(value = "simple", required = false) Boolean simple) {
        try {
            return webMapService.getServicesWithClassify(tpl, !isNull(simple) && simple ? WebMapService.SC_MODE.SIMPLIFY : WebMapService.SC_MODE.NORMAL);
        } catch (Exception e) {
            throw new JSONMessageException(getMessage("map.services.error", e.getLocalizedMessage()));
        }

    }

    /**
     * 获取地图标记
     *
     * @param id
     * @return
     */
    @RequestMapping(value = "/mark/{id}")
    @ResponseBody
    public LocationMark getLocationMark(@PathVariable String id) {
        try {
            Assert.notNull(id, getMessage("id.notnull"));
            return locationMarkService.get(id);
        } catch (Exception e) {
            throw new JSONMessageException(getMessage("map.locationMark.error", id, e.getLocalizedMessage()));
        }

    }

    /**
     * 获取属于owner的所有地图标记
     *
     * @param owner
     * @return
     */
    @RequestMapping(value = "/mark/{owner}/marks")
    @ResponseBody
    public List<LocationMark> getLocationMarks(@PathVariable String owner,
                                               @RequestParam(value = "publicity") String publicity) {
        try {
            Assert.notNull(owner, "owner can't be null");
            return locationMarkService.getLocationMarksByOwnerOrPublicity(owner, publicity);
        } catch (Exception e) {
            throw new JSONMessageException(getMessage("map.locationMark.error", owner, e.getLocalizedMessage()));
        }

    }

    /**
     * 保存地图标记
     *
     * @param owner
     * @param data
     */
    @RequestMapping(value = "/mark/{owner}/save")
    @ResponseBody
    public void save(@PathVariable(value = "owner") String owner, @RequestParam(value = "data") String data) {
        try {
            Assert.notNull(owner, "owner can't be null");
            LocationMark locationMark = JSONObject.parseObject(data, LocationMark.class);
            if (StringUtils.isBlank(locationMark.getId()))
                locationMark.setId(UUIDGenerator.generate());
            if (locationMark.getCreateAt() == null)
                locationMark.setCreateAt(new Date());
            locationMarkService.save(locationMark);
        } catch (Exception e) {
            throw new JSONMessageException(getMessage("map.locationMarkSave.error", owner, e.getLocalizedMessage()));
        }
    }

    /**
     * 删除地图标记
     *
     * @param id
     */
    @RequestMapping(value = "/mark/{id}/delete")
    @ResponseBody
    public void delete(@PathVariable String id) {
        try {
            Assert.notNull(id, getMessage("id.notnull"));
            locationMarkService.delete(id);
        } catch (Exception e) {
            throw new JSONMessageException(getMessage("map.locationMarkDelete.error", id, e.getLocalizedMessage()));
        }
    }

    /**
     * @param serverType
     * @param url
     * @return
     */
    @RequestMapping(value = "/query")
    @ResponseBody
    public String query(@RequestParam(value = "serverType") String serverType, @RequestParam(value = "url") String url, HttpServletRequest request) {
        String result = null;
        try {
            Assert.notNull(serverType, "serverType can't be null");
            Assert.notNull(url, "url can't be null");
            logger.debug("do query : [{}]", url);
            if (QueryService.ARCGIS_SERVER.equals(serverType)) {
                result = queryService.executeAgsQuery(url, null, request);
            } else if (QueryService.GTMAP_SERVER.equals(serverType)) {
                result = queryService.executeGtMapQuery(url, null);
            }
            logger.debug("query result [{}]", result);
        } catch (Exception e) {
            throw new JSONMessageException(getMessage("map.query.error", serverType, e.getLocalizedMessage()));
        }
        return result;
    }

    /**
     * @param server
     * @param port
     * @param password
     * @param database
     * @return
     */
    @RequestMapping(value = "/sde/config")
    @ResponseBody
    public Object sdeConfig(@RequestParam(value = "server") String server,
                            @RequestParam(value = "port") int port,
                            @RequestParam(value = "password") String password,
                            @RequestParam(value = "database") String database) {
        try {
            return gisManager.getSdeManager().getConfiguration(server, port, password);
        } catch (SeException e) {
            throw new JSONMessageException(GISDaoException.formateSeError(e.getSeError()));
        }
    }

    /**
     * @param server
     * @param port
     * @param password
     * @param database
     */
    @RequestMapping(value = "/sde/start")
    @ResponseBody
    public void sdeStart(@RequestParam(value = "server") String server,
                         @RequestParam(value = "port") int port,
                         @RequestParam(value = "password") String password,
                         @RequestParam(value = "database") String database) {
        try {
            gisManager.getSdeManager().startInstance(server, port, database, password);
        } catch (SeException e) {
            throw new JSONMessageException(GISDaoException.formateSeError(e.getSeError()));
        }
    }

    /**
     * @param server
     * @param port
     * @param password
     * @param database
     */
    @RequestMapping(value = "/sde/stop")
    @ResponseBody
    public void sdeStop(@RequestParam(value = "server") String server,
                        @RequestParam(value = "port") int port,
                        @RequestParam(value = "password") String password,
                        @RequestParam(value = "database") String database) {
        try {
            gisManager.getSdeManager().stopInstance(server, port, password);
        } catch (SeException e) {
            throw new JSONMessageException(GISDaoException.formateSeError(e.getSeError()));
        }
    }

    /**
     * @param value
     * @param simple
     * @param limit
     * @return
     */
    @RequestMapping(value = "/search")
    @ResponseBody
    public List search(@RequestParam(value = "q", required = true) String value,
                       @RequestParam(value = "s", defaultValue = "true") boolean simple,
                       @RequestParam(value = "l", defaultValue = "10") int limit) {
        return searchService.search(value, limit, simple);
    }

    /**
     * @return
     */
    @RequestMapping(value = "/search/groups")
    @ResponseBody
    public String[] searchGroups() {
        return searchService.groups();
    }

    /**
     * @param layers
     * @return
     */
    @RequestMapping(value = "/agsurl")
    @ResponseBody
    public Map getAgsRealUrl(@RequestParam(value = "layers") String layers) {
        try {
            return mapService.getAGSRealPath(JSON.parseObject(layers, List.class));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * @param tpl
     * @return
     */
    @RequestMapping(value = "/{tpl}/clearServiceCache")
    @ResponseBody
    public boolean clearServiceCache(@PathVariable(value = "tpl") String tpl) {
        try {
            return webMapService.clearServiceCache(tpl);
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * @return
     */
    @RequestMapping(value = "/{tpl}/clearRegionCache")
    @ResponseBody
    public boolean clearRegionCache() {
        try {
            return webMapService.clearRegionCache();
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }


}
