package cn.gtmap.onemap.platform.controller;

import cn.gtmap.onemap.platform.entity.Document;
import cn.gtmap.onemap.platform.event.JSONMessageException;
import cn.gtmap.onemap.platform.service.*;
import cn.gtmap.onemap.platform.support.spring.BaseController;
import com.alibaba.fastjson.JSON;
import com.vividsolutions.jts.geom.Geometry;
import org.apache.commons.lang.StringUtils;
import org.geotools.feature.FeatureCollection;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.referencing.crs.CoordinateReferenceSystem;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * .
 *
 * @author <a href="mailto:lanxy88@gmail.com">NelsonXu</a>
 * @version V1.0, 13-5-20 下午7:21
 */
@Controller
@RequestMapping(value = "/geometryService")
public class GeometryController extends BaseController {

    @Autowired
    private GISManager gisManager;

    @Autowired
    private BMarkService bMarkService;

    @Autowired
    private DictService dictService;

    @Autowired
    private GeometryService geometryService;

    @Autowired
    private DocumentService documentService;


    /**
     * @param layerName
     * @param wkt
     * @param returnFields
     * @param dataSource
     * @return
     */
    @RequestMapping(value = "/intersect")
    @ResponseBody
    public List intersect(@RequestParam("layerName") String layerName, @RequestParam("wkt") String wkt,
                          @RequestParam("returnFields") String returnFields,
                          @RequestParam(value = "dataSource", defaultValue = "", required = false) String dataSource) {
        try {
            return gisManager.getGISService().intersect(layerName, wkt, returnFields.split(","), dataSource);
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 查询接口
     *
     * @param layerName      空间数据图层名称
     * @param where          Where 查询条件
     * @param geometry       GeoJSON 格式图形
     * @param outFields      返回字段
     * @param returnGeometry 是否返回图形
     * @param dataSource     数据源
     * @return
     */
    @RequestMapping(value = "/rest/query")
    @ResponseBody
    public Map query(@RequestParam("layerName") String layerName,
                     @RequestParam(value = "where", required = false) String where,
                     @RequestParam(value = "geometry", required = false) String geometry,
                     @RequestParam(value = "outFields", defaultValue = "*") String outFields,
                     @RequestParam(value = "returnGeometry", defaultValue = "true") boolean returnGeometry,
                     @RequestParam(value = "dataSource", defaultValue = "") String dataSource) {
        try {
            logger.info("[query]:"+geometry);
            String[] fields = "*".equals(outFields) ? null : outFields.split(",");
            List result = null;
            if (StringUtils.isNotBlank(where))
                result = gisManager.getGISService().query(layerName, where, fields, returnGeometry, dataSource);
            else if (StringUtils.isNotBlank(geometry))
                result = gisManager.getGISService().query(layerName, gisManager.getGeoService().readGeoJSON(geometry), fields, dataSource);
            else
                throw new RuntimeException(getMessage("query.condition.missing"));
            FeatureCollection collection = gisManager.getGeoService().list2FeatureCollection(result, null, null);
            logger.info("[query result]:"+gisManager.getGeoService().toFeatureJSON(collection));
            return result(gisManager.getGeoService().toFeatureJSON(collection));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 插入要素
     *
     * @param layerName
     * @param geometry   GeoJSON 格式要素
     * @param dataSource
     * @return
     */
    @RequestMapping(value = "/rest/insert")
    @ResponseBody
    public Map insert(@RequestParam("layerName") String layerName,
                      @RequestParam(value = "geometry", required = false) String geometry,
                      @RequestParam(value = "check", required = false) Boolean check,
                      @RequestParam(value = "dataSource", defaultValue = "") String dataSource) {
        try {
            logger.info("[insert]:"+geometry);
            return result(gisManager.getGISService().insert2(layerName, geometry, check, dataSource));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }


    /**
     *
     * @param layerName
     * @param geometry
     * @param check
     * @param createDate
     * @param dataSource
     * @return
     */
    @RequestMapping(value = "/rest/insert2")
    @ResponseBody
    public Map insert2(@RequestParam("layerName") String layerName,
                       @RequestParam(value = "geometry", required = false) String geometry,
                       @RequestParam(value = "check", required = false) Boolean check,
                       @RequestParam(value = "check", required = false) Boolean createDate,
                       @RequestParam(value = "dataSource", defaultValue = "") String dataSource) {
        try {
            logger.info("[insert]:" + geometry);
            return result(gisManager.getGISService().insert3(layerName, geometry, check, createDate, dataSource));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 更新要素
     *
     * @param layerName
     * @param primaryKey
     * @param geometry   GeoJSON 格式要素
     * @param dataSource
     * @return
     */
    @RequestMapping(value = "/rest/update")
    @ResponseBody
    public Map update(@RequestParam("layerName") String layerName,
                      @RequestParam(value = "primaryKey", required = true) String primaryKey,
                      @RequestParam(value = "geometry", required = true) String geometry,
                      @RequestParam(value = "dataSource", defaultValue = "") String dataSource) {
        try {
            logger.info("[update]:"+geometry);
            return result(gisManager.getGISService().update(layerName, primaryKey, geometry, dataSource));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 删除要素
     *
     * @param layerName
     * @param primaryKey 要素主键
     * @param dataSource
     * @return
     */
    @RequestMapping(value = "/rest/delete")
    @ResponseBody
    public Map delete(@RequestParam("layerName") String layerName,
                      @RequestParam(value = "primaryKey", required = true) String primaryKey,
                      @RequestParam(value = "dataSource", defaultValue = "") String dataSource) {
        try {
            logger.info("[delete]:"+primaryKey);
            return result(gisManager.getGISService().delete(layerName, primaryKey, dataSource));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }


    /**
     * 相交分析
     *
     * @param layerName  空间数据图层名称
     * @param geometry   GeoJSON 格式图形
     * @param outFields  返回字段
     * @param dataSource 数据源
     * @return
     */
    @RequestMapping(value = "/rest/intersect")
    @ResponseBody
    public Map intersect2(@RequestParam("layerName") String layerName,
                          @RequestParam("geometry") String geometry,
                          @RequestParam(value = "outFields", defaultValue = "*") String outFields,
                          @RequestParam(value = "dataSource", defaultValue = "", required = false) String dataSource) {
        try {
            logger.info("[intersect]:"+geometry);
            String[] fields = "*".equals(outFields) ? null : outFields.split(",");
            List results = gisManager.getGISService().intersect3(layerName, geometry, fields, dataSource);
            FeatureCollection collection = geometryService.list2FeatureCollection(results, null, null);
            return result(geometryService.toFeatureJSON(collection));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 土地利用现状分析
     *
     * @param dltb       地类图斑图层
     * @param xzdw       现状地物图层
     * @param regionCode 行政区代码
     * @param geometry   GeoJSON 格式图形
     * @param dataSource
     * @return
     */
    @RequestMapping(value = "/rest/analysis/tdlyxz")
    @ResponseBody
    public Map tdlyxzAnalysis(@RequestParam(value = "dltb", required = false) String dltb,
                              @RequestParam(value = "xzdw", required = false) String xzdw,
                              @RequestParam(value = "regionCode", required = false) String regionCode,
                              @RequestParam(value = "year", defaultValue = "2011") String year,
                              @RequestParam(value = "geometry", required = true) String geometry,
                              @RequestParam(value = "dataSource", defaultValue = "", required = false) String dataSource) {
        try {
            if (StringUtils.isNotBlank(year)) regionCode = year;
            if (StringUtils.isNotBlank(regionCode))
                return result(gisManager.getGISService().tdlyxzAnalysis(regionCode, geometry, dataSource));
            else if (StringUtils.isNotBlank(dltb) && StringUtils.isNotBlank(xzdw))
                return result(gisManager.getGISService().tdlyxzAnalysis2(dltb, xzdw, geometry, dataSource));
            throw new RuntimeException(getMessage("analysis.tdlyxz.param.error"));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 土地利用总体规划审查
     *
     * @param layerType  图层类型
     * @param regionCode 行政区代码
     * @param year       年度，与regionCode二选一，优先选择year
     * @param geometry   GeoJSON格式数据
     * @param outFields  返回字段
     * @param dataSource 数据源
     * @return
     */
    @RequestMapping(value = "/rest/analysis/tdghsc")
    @ResponseBody
    public Map tdghscAnalysis(@RequestParam(value = "layerType", defaultValue = "") String layerType,
                              @RequestParam(value = "regionCode", required = false) String regionCode,
                              @RequestParam(value = "year", defaultValue = "2020") String year,
                              @RequestParam("geometry") String geometry,
                              @RequestParam(value = "outFields", defaultValue = "*") String outFields,
                              @RequestParam(value = "dataSource", defaultValue = "", required = false) String dataSource) {
        try {
            String[] fields = "*".equals(outFields) ? null : outFields.split(",");
            if (StringUtils.isNotBlank(year)) regionCode = year;
            if (StringUtils.isNotBlank(layerType))
                return result(gisManager.getGISService().tdghscAnalysis2(layerType, regionCode, geometry, fields, dataSource));
            else
                return result(gisManager.getGISService().tdghscAnalysis2(regionCode, geometry, fields, dataSource));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * project
     *
     * @param geometry GeoJSON geometry
     * @param inSR     wkid or wkt
     * @param outSR    wkid or wkt
     * @return
     */
    @RequestMapping(value = "/rest/project")
    @ResponseBody
    public Map project(@RequestParam(value = "geometry", required = true) String geometry,
                       @RequestParam(value = "inSR", required = true) String inSR,
                       @RequestParam(value = "outSR", required = true) String outSR) {
        try {
            Object geo = gisManager.getGeoService().readUnTypeGeoJSON(geometry);
            CoordinateReferenceSystem in = gisManager.getGeoService().parseUndefineSR(inSR);
            CoordinateReferenceSystem out = gisManager.getGeoService().parseUndefineSR(outSR);
            if (geo instanceof Geometry) {

                Geometry g = gisManager.getGeoService().project((Geometry) geo, in, out);
                return result(gisManager.getGeoService().toGeoJSON(g));

            }else if((geo instanceof FeatureCollection)||(geo instanceof SimpleFeature))
            {
                Object feature = gisManager.getGeoService().project(geo, in, out);
                return result(gisManager.getGeoService().toFeatureJSON(feature));
            }
            else
                throw new RuntimeException("geometry don't support");
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 返回结果页面
     *
     * @param layerType
     * @param regionCode
     * @param year       年度，与regionCode二选一，优先选择year
     * @param geometry
     * @param outFields
     * @param dataSource
     * @param model
     * @return
     */
    @RequestMapping(value = "/analysis/tdghsc")
    public String tdghscAnalysis(@RequestParam(value = "layerType", defaultValue = "") String layerType,
                                 @RequestParam("regionCode") String regionCode,
                                 @RequestParam(value = "year", defaultValue = "2020") String year,
                                 @RequestParam("geometry") String geometry,
                                 @RequestParam(value = "outFields", defaultValue = "*") String outFields,
                                 @RequestParam(value = "dataSource", defaultValue = "", required = false) String dataSource, Model model) {

        try {
            logger.info("[tdghsc]:"+geometry);
            String[] fields = "*".equals(outFields) ? null : outFields.split(",");
            if (StringUtils.isNotBlank(year)) regionCode = year;
            Map result = new HashMap();
            if (StringUtils.isNotBlank(layerType)) {
                String analysisResult = gisManager.getGISService().tdghscAnalysis2(layerType, regionCode, geometry, fields, dataSource);
                result = gisManager.getGISService().tdghscResult(layerType, analysisResult);
            } else {
                Map analysisMap = gisManager.getGISService().tdghscAnalysis2(regionCode, geometry, fields, dataSource);
                result = gisManager.getGISService().tdghscResult(analysisMap);
            }
            model.addAttribute("result", result);
            model.addAttribute("excelData", JSON.toJSONString(gisManager.getGISService().tdghscExcelData(result)));
            return "analysis/tdghsc";

        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     *  返回页面
     * @param dltb
     * @param xzdw
     * @param regionCode
     * @param geometry
     * @param dataSource
     * @param tpl
     * @param model
     * @return
     */
    @RequestMapping(value = "/analysis/tdlyxz")
    public String tdlyxzAnalysis(@RequestParam(value = "dltb", required = false) String dltb,
                                 @RequestParam(value = "xzdw", required = false) String xzdw,
                                 @RequestParam(value = "regionCode", required = false) String regionCode,
                                 @RequestParam(value = "year",required = false) String year,
                                 @RequestParam(value = "geometry", required = true) String geometry,
                                 @RequestParam(value = "dataSource", defaultValue = "", required = false) String dataSource,
                                 @RequestParam(value = "tpl", required = true) String tpl, Model model) {

        try {
            logger.info("[tdlyxz]:"+geometry);
            Map map = new HashMap();
            if (StringUtils.isNotBlank(year)) regionCode = year;
            if (StringUtils.isNotBlank(regionCode))
                map = result(gisManager.getGISService().tdlyxzAnalysis(regionCode, geometry, dataSource));
            else if (StringUtils.isNotBlank(dltb) && StringUtils.isNotBlank(xzdw))
                map = result(gisManager.getGISService().tdlyxzAnalysis2(dltb, xzdw, geometry, dataSource));
            else
                throw new RuntimeException(getMessage("analysis.tdlyxz.param.error"));
            logger.info("[tdlyxz result]:"+JSON.toJSONString(map.get("result")));
            List<Map> result = gisManager.getGISService().tdlyxzResult((Map) map.get("result"), tpl);
            LinkedHashMap excelData = gisManager.getGISService().tdlyxzExcelData(result);
            model.addAttribute("result", result);
            model.addAttribute("resultStr", JSON.toJSONString(excelData));
            return "analysis/tdlyxz";

        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }
    /**
     * 返回相交分析的结果页面
     * 返回相交分析的一般性页面(只展示组织好的数据)
     * 结果是分组后的带面积的结果
     * @param data
     * @param model
     * @return
     */
    @RequestMapping(value = "/analysis/result/common", method = RequestMethod.POST)
    public String groupedAnalysisResult(@RequestParam(value = "data") String data, Model model) {
        if (StringUtils.isNotBlank(data)) {
            List result = JSON.parseObject(data, List.class);
            model.addAttribute("info", result);
            model.addAttribute("excelData", JSON.toJSONString(gisManager.getGISService().analysisExcelData(result)));
        }
        return "analysis/result-grouped";
    }

    /**
     *  返回管制区用地分析的结果展示页面
     * @param data
     * @param model
     * @return
     */
    @RequestMapping(value = "/analysis/result/gzqyd", method = RequestMethod.POST)
    public String commonAnalysisResult(@RequestParam(value = "data") String data, Model model) {
        if (StringUtils.isNotBlank(data)) {
            List result = JSON.parseObject(data, List.class);
            model.addAttribute("result", result);
            model.addAttribute("excelData", JSON.toJSONString(gisManager.getGISService().gzqydExcelData(result)));
        }
        return "analysis/result-gzqyd";
    }

    /**
     * 规划审查修改分析 返回页面(省厅 订制)
     * @param layerName
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @RequestMapping(value = "/analysis/ghscxg",method = RequestMethod.POST)
    public String ghModifyAnalysis(@RequestParam("layerName") String layerName,
                                   @RequestParam("geometry") String geometry,
                                   @RequestParam(value = "outFields", defaultValue = "*") String outFields,
                                   @RequestParam(value = "showFields", defaultValue = "*") String showFields,
                                   @RequestParam(value = "dataSource", defaultValue = "", required = false) String dataSource,
                                   Model model){


        try {
            logger.info("[analysis]:"+geometry);
            String[] fields = "*".equals(outFields) ? null : outFields.split(",");
            List<Map> results = (List<Map>) gisManager.getGISService().intersect3(layerName, geometry, fields, dataSource);
            Map map = gisManager.getGISService().ghscxgData(results,JSON.parseObject(showFields,List.class));
            model.addAttribute("showFields",JSON.parseObject(showFields,List.class));
            model.addAttribute("results",map.get("result"));
            model.addAttribute("excelData",JSON.toJSONString(map.get("excel")));
            return "analysis/result-ghscxg";

        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw new JSONMessageException(e.getLocalizedMessage());
        }

    }


    /**
     * 界址点导入与关联
     *
     * @return
     */
    @RequestMapping(value = "/bmark")
    public String boundaryMark(@RequestParam(value = "id", required = false) String id,
                               @RequestParam(value = "type", required = false) String type,
                               @RequestParam(value = "btype", defaultValue = "bmi") String btype,
                               Model model) {
        model.addAttribute("id", id);
        model.addAttribute("btype", btype);
        model.addAttribute("type", type);
        return "geo/bmark";
    }

    /**
     * 界址点导入与关联
     *
     * @return
     */
    @RequestMapping(value = "/bmark/coords")
    @ResponseBody
    public List bMarkUpload(@RequestParam(value = "fileData", required = true) MultipartFile file,
                            @RequestParam(value = "fileType", defaultValue = "xls") String fileType) {
        try {
            return bMarkService.getCoordsByFile(file.getInputStream(), BMarkService.Type.getType(fileType));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    @RequestMapping(value = "/bmark/gen", method = RequestMethod.POST)
    @ResponseBody
    public boolean generateShape(@RequestParam(value = "id", required = false) String id,
                                 @RequestParam(value = "coords", required = false) String coords,
                                 @RequestParam(value = "type", required = false) String type) {
        try {
            return bMarkService.insert(id, JSON.parseObject(coords, List.class), type);
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 界址点导入与关联
     *
     * @return
     */
    @RequestMapping(value = "/bmark/dks")
    @ResponseBody
    public List getBMarkDKById(@RequestParam(value = "key", required = false) String key,
                               @RequestParam(value = "type", required = true) String type) {
        try {
            return bMarkService.query(key, type);
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }


    /**
     * 获取分析时分组字段相对应的字典项配置,字典名称:前缀_分组字段名(如,analysis_dlbm)
     *
     * @param tpl
     * @param groupName
     * @return
     */
    @RequestMapping(value = "/analysis/dict")
    @ResponseBody
    public List getAnalysisDict(@RequestParam(value = "tpl", required = true) String tpl,
                                @RequestParam(value = "groupName", required = true) String groupName) {
        return dictService.getAnalysisDict(tpl, groupName);

    }

    /**
     * export data to excel
     *
     * @param data
     */
    @RequestMapping(value = "/export/excel")
    public void export2Excel(@RequestParam("data") String data,
                             @RequestParam(value = "fileName", required = false) String fileName,
                             HttpServletResponse response) {
        try {
            Document document = null;
            if (isNull(fileName)) {
                document = documentService.writeExcel(JSON.parseObject(data, List.class));
                sendFile(new ByteArrayInputStream(document.getContent()), response, document.getFileName());
            } else {
                document = documentService.writeExcel(JSON.parseObject(data, Map.class), fileName);
                sendFile(new ByteArrayInputStream(document.getContent()), response, document.getFileName());
            }
        } catch (Exception e) {
            throw new JSONMessageException(getMessage("doc.excel.export.error", e.getLocalizedMessage()));
        }
    }



    /**
     * 管制区用地分析结果导出excel
     * @param data
     * @param fileName
     * @param response
     */
    @RequestMapping(value = "/export/excel/gzqyd")
    public void export2ExcelGZQ(@RequestParam("data") String data,
                             @RequestParam(value = "fileName", required = true) String fileName,
                             HttpServletResponse response) {
        try {
            Document document = null;
            document = documentService.writeExcelGZQFX(JSON.parseObject(data, List.class), fileName);
            sendFile(new ByteArrayInputStream(document.getContent()), response, document.getFileName());
        } catch (Exception e) {
            throw new JSONMessageException(getMessage("doc.excel.export.error", e.getLocalizedMessage()));
        }
    }
    /**
     * 出图
     * @param where
     * @param scale
     * @param width
     * @param height
     * @param type
     * @param response
     */
    @RequestMapping(value = "/rest/export")
    @ResponseBody
    public void exportMap(@RequestParam(value = "where",required = true)String where,
                          @RequestParam(value = "scale",required = false)String scale,
                          @RequestParam(value = "width",defaultValue = "800")int width,
                          @RequestParam(value = "height",defaultValue = "600")int height,
                          @RequestParam(value = "type",required = true)String type,
                          HttpServletResponse response){

        try {
            InputStream inputStream = gisManager.getGISService().exportMap(where, scale, width, height, type);
            sendFile(inputStream,response, String.valueOf(System.currentTimeMillis()));
        } catch (IOException e) {
            throw new JSONMessageException(getMessage("map.export.error", e.getLocalizedMessage()));
        }
    }

    /**
     * 电子报件导入
     *
     * @return
     */
    @RequestMapping(value = "/bj/upload")
    @ResponseBody
    public void bjUpload(@RequestParam(value = "file") MultipartFile file, HttpServletResponse response) {
        try {
            result(gisManager.getGeoService().getBJCoordinates(file.getInputStream()), response);
        } catch (Exception e) {
            error(getMessage("bj.upload.error", file.getOriginalFilename(), e.getLocalizedMessage()), response);
        }
    }

    /**
     *excel 文件导入
     * @param file
     * @param response
     */
    @RequestMapping(value = "/excel/upload")
    @ResponseBody
    public void xlsUpload(@RequestParam(value = "file") MultipartFile file, HttpServletResponse response) {
        try {
            result(gisManager.getGeoService().getExcelCoordinates(file.getInputStream()),response);
        } catch (Exception e) {
            error(getMessage("bj.upload.error", file.getOriginalFilename(), e.getLocalizedMessage()), response);
        }
    }

    /**
     * shapefile上传解析
     * @param file  shape zip
     * @param response
     */
    @RequestMapping(value = "/shp/upload")
    @ResponseBody
    public void shpUpload(@RequestParam(value = "file",required = false) MultipartFile file,
                         HttpServletResponse response) {
        try {
            result(gisManager.getGeoService().getShpCoordinates(file.getInputStream()),response);
        } catch (Exception e) {
           error(getMessage("shp.upload.error",file.getOriginalFilename(),e.getLocalizedMessage()),response);
        }
    }

    /**
     * shapefile导入2
     * @param filePath zip file 路径
     * @param layerName 插入的空间图层名称
     * @param check
     * @param dataSource
     * @return
     */
    @RequestMapping(value = "/shp/file/upload")
    @ResponseBody
    public Map shpZipUpload(@RequestParam(value = "file", required = true) String filePath,
                             @RequestParam(value = "properties", required = false) String properties,
                             @RequestParam("layerName") String layerName,
                             @RequestParam(value = "check", required = false, defaultValue = "true") Boolean check,
                             @RequestParam(value = "dataSource", defaultValue = "") String dataSource) {
        try {
            File zipFile = new File(filePath);
            if(zipFile.exists())
            {
                String geometry = gisManager.getGeoService().getShpCoordinates(zipFile,properties);
                return result(gisManager.getGISService().insert2(layerName, geometry, check, dataSource));
            }else
                throw new JSONMessageException(getMessage("shp.zipfile.exist.error",filePath));
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw new JSONMessageException(getMessage("shp.upload.error",filePath,e.getLocalizedMessage()));
        }
    }

    /**
     * format json
     *
     * @param json
     * @param pretty
     * @return
     */
    @RequestMapping(value = "/json/format")
    @ResponseBody
    public String formatJSON(@RequestParam(value = "json") String json,
                             @RequestParam(value = "pretty", defaultValue = "true") boolean pretty) {
        try {
            return JSON.toJSONString(JSON.parseObject(json), pretty);
        } catch (Exception e) {
            throw new JSONMessageException(getMessage("json.format.error", e.getLocalizedMessage()));
        }
    }

    /**
     * json editor
     */
    @RequestMapping(value = "/jsoneditor")
    public String jsonEditor() {
        return "jsoneditor";
    }

    /**
     * crs坐标系转为wkt描述
     * @param sr
     * @return
     */
    @RequestMapping(value = "/crs/wkt")
    @ResponseBody
    public Map crsToWkt(@RequestParam(value = "sr", required = true) String sr) {
        try {
            CoordinateReferenceSystem in = gisManager.getGeoService().parseUndefineSR(sr);
            return result(in.toWKT());
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 集成分析，多个图层的综合分析
     * @param geometry   分析的图形
     * @param layers     参与分析的图层配置
     * @param dataSource sde数据源
     * @param tpl        模板名称
     * @param model
     * @return
     * @throws Exception
     */
    @RequestMapping(value = "/analysis/integrated")
    public String integratedAnalysis(@RequestParam("geometry") String geometry,
                                     @RequestParam(value = "layers",required = true)String layers,
                                     @RequestParam(value = "dataSource", defaultValue = "", required = false) String dataSource,
                                     @RequestParam(value = "tpl", required = false) String tpl,
                                     Model model)throws Exception {

        logger.info("[integrated analysis geometry]:"+geometry);

        return "analysis/basic";
    }


}
