package cn.gtmap.onemap.platform.service.impl;

import cn.gtmap.onemap.platform.dao.FileStoreDao;
import cn.gtmap.onemap.platform.entity.FileStore;
import cn.gtmap.onemap.platform.service.FileStoreService;
import cn.gtmap.onemap.platform.utils.FilesUtils;
import com.gtis.generic.util.ImageUtils;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItem;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.FileCleanerCleanup;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.io.FileUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;

import javax.servlet.http.HttpServletRequest;
import java.io.File;
import java.io.IOException;
import java.util.Calendar;
import java.util.Iterator;
import java.util.List;

/**
 * .
 *
 * @author <a href="mailto:lanxy88@gmail.com">NelsonXu</a>
 * @version V1.0, 13-3-25 下午5:38
 */
public class FileStoreServiceImpl extends BaseLogger implements FileStoreService {

    private static final String EGOV_HOME = "${egov.conf}";

    private static final int THRESHOLD = 1024 * 1024 * 10;

    private static final String THUMB_SUFFIX = "_thumb";

    @Autowired
    private FileStoreDao fileStoreDao;

    private Resource baseLocation;
    private Resource location;
    private Resource tempFile;
    private long maxSize;

    /**
     * get one
     *
     * @param id
     * @return
     */
    public FileStore get(String id) {
        return fileStoreDao.findOne(id);
    }

    /**
     * save
     *
     * @param request
     * @param parentId
     * @return
     */
    public FileStore save(HttpServletRequest request, String parentId) {
        Assert.notNull(parentId, getMessage("file.parentId.notnull"));
        FileStore fileStore = null;
        try {
            List<FileItem> fileItems = parseRequest(request);
            for (FileItem item : fileItems) {
                if (!item.isFormField()) {
                    DiskFileItem diskFileItem = (DiskFileItem) item;
                    File newFile = getNewFile(item.getName());
                    if (diskFileItem.isInMemory()) {
                        FileUtils.copyInputStreamToFile(diskFileItem.getInputStream(), newFile);
                    } else {
                        File tmpFile = diskFileItem.getStoreLocation();
                        FileUtils.copyFile(tmpFile, newFile, true);
                    }
                    createThumb(newFile);
                    fileStore = save2DB(newFile, parentId, diskFileItem);
                    logger.debug(getMessage("file.upload.success", item.getName()));
                }
            }
        } catch (FileUploadException e) {
            logger.error(getMessage("file.upload.error", e.getLocalizedMessage()));
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
        }
        return fileStore;
    }

    /**
     * save upload file , parse by CommonsMultipartResolver    <br/>
     *
     * @param request
     * @param parentId
     * @return
     * @see org.springframework.web.multipart.commons.CommonsMultipartResolver
     */
    @Override
    public FileStore save2(MultipartHttpServletRequest request, String parentId) {
        Iterator<String> iterator = request.getFileNames();
        FileStore fileStore = null;
        while (iterator.hasNext()) {
            MultipartFile file = request.getFile(iterator.next());
            String fileName = file.getOriginalFilename();
            logger.debug(" upload file : {0}", fileName);
            try {
                File newFile = getNewFile(fileName);
                FileUtils.copyInputStreamToFile(file.getInputStream(), newFile);
                createThumb(newFile);
                fileStore = save2DB2(newFile, parentId, fileName, file.getSize());
                logger.debug(getMessage("file.upload.success", fileName));
            } catch (IOException e) {
                logger.error(" save upload file error , detail info : [{0}]", e.getLocalizedMessage());
            }
        }
        return fileStore;
    }

    /**
     * save
     *
     * @param fileStore
     * @return
     */
    @Transactional
    public FileStore save(FileStore fileStore) {
        return fileStoreDao.save(fileStore);
    }

    /**
     * delete
     *
     * @return
     */
    @Transactional
    public boolean delete(String id) {
        fileStoreDao.delete(id);
        return true;
    }

    /**
     * delete
     *
     * @param fileStore
     * @return
     */
    @Transactional
    public boolean delete(FileStore fileStore) {
        fileStoreDao.delete(fileStore);
        return true;
    }

    /**
     * @param parentId
     * @return
     */
    public String[] getFileIds(String parentId) {
        List<FileStore> files = fileStoreDao.findByParentId(parentId);
        if (files != null && files.size() > 0) {
            String[] ids = new String[files.size()];
            for (int i = 0; i < files.size(); i++) {
                ids[i] = files.get(i).getId();
            }
            return ids;
        }
        return new String[0];
    }

    /**
     * 根据父主键获取对应关联的文件名
     * @param parentId
     * @return
     */
    @Override
    public String[] getFileNames(String parentId) {
        Assert.notNull(parentId,"查询父ID不可为空！");
        List<FileStore> files = fileStoreDao.findByParentId(parentId);
        if (files != null && files.size() > 0) {
            String[] names = new String[files.size()];
            for (int i = 0; i < files.size(); i++) {
                names[i] = files.get(i).getName();
            }
            return names;
        }
        return new String[0];
    }

    /**
     * get file by id
     *
     * @param id
     * @return
     */
    public File getFile(String id) throws IOException {
        Assert.notNull(id, getMessage("id.notnull"));
        FileStore fileStore = fileStoreDao.findOne(id);
        return new File(getRealPath(fileStore.getPath()));
    }

    /**
     * get thumb
     *
     * @param id
     * @return
     */
    public File getThumb(String id) throws IOException {
        Assert.notNull(id, getMessage("id.notnull"));
        FileStore fileStore = fileStoreDao.findOne(id);
        String path = getRealPath(fileStore.getPath());
        File thumbFile = new File(getThumbPath(path));
        if (thumbFile.exists()) return thumbFile;
        return null;
    }

    /**
     * 缩略图地址
     *
     * @param path
     * @return
     */
    private String getThumbPath(String path) {
        Assert.notNull(path, getMessage("path.notnull"));
        String suffix = path.substring(path.lastIndexOf("."), path.length());
        return path.replace(suffix, THUMB_SUFFIX.concat(suffix));
    }

    /**
     * 解析请求
     *
     * @param servletRequest
     * @return
     * @throws FileUploadException
     */
    private List<FileItem> parseRequest(HttpServletRequest servletRequest) throws FileUploadException, IOException {
        boolean isM = ServletFileUpload.isMultipartContent(servletRequest);

        DiskFileItemFactory fac = createDiskFileItemFactory(tempFile.getURI().getPath(), servletRequest);
        ServletFileUpload upload = new ServletFileUpload(fac);
        upload.setSizeMax(maxSize);
        return upload.parseRequest(servletRequest);
    }

    /**
     * @param saveDir
     * @return
     */
    private DiskFileItemFactory createDiskFileItemFactory(String saveDir, HttpServletRequest request) {
        DiskFileItemFactory fac = new DiskFileItemFactory();
        fac.setSizeThreshold(THRESHOLD);
        if (saveDir != null) {
            fac.setRepository(new File(saveDir));
        }
        fac.setFileCleaningTracker(FileCleanerCleanup.getFileCleaningTracker(request.getSession().getServletContext()));
        return fac;
    }

    /**
     * @param name
     * @return
     */
    private File getNewFile(String name) {
        File file = null;
        try {
            String path = location.getURI().getPath();
            file = new File(path.concat("/" + name));
            if (file.exists()) {
                file = new File(path.concat("/" + reNameFile(name)));
            }
        } catch (IOException e) {
            logger.error(e.getLocalizedMessage());
        }
        return file;
    }

    /**
     * @param name
     * @return
     */
    private String reNameFile(String name) {
        if (name.lastIndexOf(".") > -1) {
            int index = name.lastIndexOf(".");
            String suffix = name.substring(index, name.length());
            return name.replace(suffix, "_" + System.currentTimeMillis() + suffix);
        } else {
            return name + "_" + System.currentTimeMillis();
        }
    }

    /**
     * 生成缩略图
     *
     * @param file
     */
    private void createThumb(File file) {
        Assert.notNull(file, getMessage("file.notexist"));
        try {
            String path = file.getPath();
            String suffix = path.substring(path.lastIndexOf("."), path.length());
            String aPath = path.replace(suffix, THUMB_SUFFIX.concat(suffix));
            File aFile = new File(aPath);
            if (FilesUtils.isImage(file)) {
                ImageUtils.resizeImageWithMaxWidth(file.getPath(), aFile.getPath(), 160);
            }
            logger.debug(getMessage("file.thumb.success", file.getPath()));
        } catch (Exception e) {
            logger.error(getMessage("file.thumb.fail", e.getLocalizedMessage()));
        }
    }

    /**
     * 数据库保存数据
     *
     * @param file
     * @param parentId
     * @param itemInfo
     */
    private FileStore save2DB(File file, String parentId, DiskFileItem itemInfo) {
        try {
            String path = getRealPath(file.getPath());
            FileStore fs = new FileStore();
            fs.setName(itemInfo.getName());
            fs.setPath(path);
            fs.setParentId(parentId);
            fs.setCreateTime(Calendar.getInstance().getTime());
            fs.setFileSize(itemInfo.getSize());
            return fileStoreDao.save(fs);
        } catch (Exception e) {
            throw new RuntimeException(getMessage("file.database.error", e.getLocalizedMessage()));
        }
    }

    /**
     * 数据库保存数据
     *
     * @param file
     * @param parentId
     * @param name
     * @param size
     * @return
     */
    private FileStore save2DB2(File file, String parentId, String name, double size) {
        try {
            String path = getRealPath(file.getPath());
            FileStore fs = new FileStore();
            fs.setName(name);
            fs.setPath(path);
            fs.setParentId(parentId);
            fs.setCreateTime(Calendar.getInstance().getTime());
            fs.setFileSize(size);
            return fileStoreDao.save(fs);
        } catch (Exception e) {
            throw new RuntimeException(getMessage("file.database.error", e.getLocalizedMessage()));
        }
    }

    /**
     * 获取实际地址
     *
     * @param path
     * @return
     * @throws IOException
     */
    private String getRealPath(String path) throws IOException {
        return path.replace(EGOV_HOME, baseLocation.getFile().getPath());
    }


    public void setBaseLocation(Resource baseLocation) {
        this.baseLocation = baseLocation;
    }

    public void setLocation(Resource location) {
        this.location = location;
    }

    public void setTempFile(Resource tempFile) {
        this.tempFile = tempFile;
    }

    public void setMaxSize(long maxSize) {
        this.maxSize = maxSize;
    }
}
