package cn.gtmap.onemap.platform.service;


import cn.gtmap.onemap.platform.entity.LayerRegion;
import cn.gtmap.onemap.platform.event.GeometryServiceException;
import com.esri.sde.sdk.client.SeLayer;
import com.esri.sde.sdk.pe.PeProjectionException;
import com.vividsolutions.jts.geom.Geometry;
import com.vividsolutions.jts.geom.GeometryCollection;
import com.vividsolutions.jts.operation.valid.TopologyValidationError;
import org.geotools.feature.FeatureCollection;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.referencing.crs.CoordinateReferenceSystem;

import java.io.File;
import java.io.InputStream;
import java.util.List;
import java.util.Map;

/**
 * .
 *
 * @author <a href="mailto:lanxy88@gmail.com">NelsonXu</a>
 * @version V1.0, 13-5-17 下午2:25
 */
public interface GeometryService {

    static final String EPSG = "EPSG";

    static final String SHAPE = "SHAPE";

    static final String GEOMETRY = "geometry";

    static final String FEATURE_CRS = "crs";

    static final String FEATURE = "feature";

    /**
     * 读取WKT标准图形
     *
     * @param wkt
     * @return
     */
    Geometry readWKT(String wkt) throws GeometryServiceException;


    /**
     * 解析 GeometryJSON  格式数据 ,单纯包含Geometry
     *
     * @param geoJSON
     * @return
     * @throws GeometryServiceException
     */
    Geometry readGeoJSON(String geoJSON) throws GeometryServiceException;

    /**
     * 解析 GeometryCollection
     *
     * @param geoJSON
     * @return
     * @throws GeometryServiceException
     */
    GeometryCollection readGeoCollectionJSON(String geoJSON) throws GeometryServiceException;

    /**
     * 解析 FeatureJSON 格式数据， 连带属性
     *
     * @param featureJSON
     * @return
     * @throws GeometryServiceException
     */
    SimpleFeature readFeatureJSON(String featureJSON) throws GeometryServiceException;

    /**
     * 解析 FeatureCollectionJSON
     *
     * @param featureJSON
     * @return
     * @throws GeometryServiceException
     */
    FeatureCollection readFeatureCollectionJSON(String featureJSON) throws GeometryServiceException;

    /**
     * 解析未指明GeoJSON
     *
     * @param geoJSON
     * @return
     * @throws GeometryServiceException
     */
    Object readUnTypeGeoJSON(String geoJSON) throws GeometryServiceException;

    /**
     * 读取 FeatureJSON 空间参考信息
     *
     * @param featureJSON
     * @return
     */
    CoordinateReferenceSystem readFeatureJSONCRS(String featureJSON);

    /**
     * 投影转换
     *
     * @param geometry
     * @param sourceCRS
     * @param targetCRS
     * @return
     * @throws GeometryServiceException
     */
    Geometry project(Geometry geometry, CoordinateReferenceSystem sourceCRS, CoordinateReferenceSystem targetCRS) throws GeometryServiceException;

    /**
     * project
     * @param feature simplefeature or FeatureCollection
     * @param sourceCRS
     * @param targetCRS
     * @return
     * @throws GeometryServiceException
     */
    Object project(Object feature, CoordinateReferenceSystem sourceCRS, CoordinateReferenceSystem targetCRS) throws GeometryServiceException;

    /**
     * 投影转换 by arcgis sde
     *
     * @param geometry
     * @param sourceCRS
     * @param targetCRS
     * @return
     */
    Geometry projectByAGS(Geometry geometry, CoordinateReferenceSystem sourceCRS, CoordinateReferenceSystem targetCRS);


    /**
     * 简化图形
     *
     * @param geometry
     * @param tolerance
     * @return
     */
    Geometry simplify(Geometry geometry, double tolerance);

    /**
     * force simplify
     *
     * @param geometry
     * @param tolerance
     * @return
     */
    Geometry forceSimplify(Geometry geometry, double tolerance);

    /**
     * densify
     *
     * @param geometry
     * @param tolerance
     * @return
     */
    Geometry densify(Geometry geometry, double tolerance);

    /**
     * 获取 wkt 格式空间参考
     *
     * @param wktCRS such as "GEOGCS[" + "\"WGS 84\"," + "  DATUM[" + "    \"WGS_1984\","
     *               + "    SPHEROID[\"WGS 84\",6378137,298.257223563,AUTHORITY[\"EPSG\",\"7030\"]],"
     *               + "    TOWGS84[0,0,0,0,0,0,0]," + "    AUTHORITY[\"EPSG\",\"6326\"]],"
     *               + "  PRIMEM[\"Greenwich\",0,AUTHORITY[\"EPSG\",\"8901\"]],"
     *               + "  UNIT[\"DMSH\",0.0174532925199433,AUTHORITY[\"EPSG\",\"9108\"]],"
     *               + "  AXIS[\"Lat\",NORTH]," + "  AXIS[\"Long\",EAST],"
     *               + "  AUTHORITY[\"EPSG\",\"4326\"]]";
     * @return
     */
    CoordinateReferenceSystem getCRSByWKTString(String wktCRS);


    /**
     * 获取常用标准格式空间参考
     *
     * @param crs such as "EPSG:4326" , "urn:ogc:def:ellipsoid:EPSG:6.0:7001" , "AUTO2:42001,"+lat+","+lon
     * @return
     */
    CoordinateReferenceSystem getCRSByCommnonString(String crs);

    /**
     * srid
     *
     * @param srid
     * @return
     */
    CoordinateReferenceSystem getCRSBySRID(String srid);

    /**
     * get SeLayer crs
     *
     * @param layer
     * @return
     */
    CoordinateReferenceSystem getSeLayerCRS(SeLayer layer);

    /**
     * 获取SimpleFeatureType
     *
     * @param value
     * @return
     */
    SimpleFeatureType getFeatureType(Map<String, Object> value);

    /**
     * Map value to SimpleFeature
     *
     * @param value   {"SHAPE":"WKT FORMATE","PRO1":"VALUE"}
     * @param srcCRS
     * @param destCRS
     * @return
     */
    SimpleFeature map2SimpleFeature(Map<String, Object> value, CoordinateReferenceSystem srcCRS, CoordinateReferenceSystem destCRS);

    /**
     * list values to featureCollection
     *
     * @param value
     * @param srcCRS
     * @param destCRS
     * @return
     */
    FeatureCollection list2FeatureCollection(List<Map<String, Object>> value, CoordinateReferenceSystem srcCRS, CoordinateReferenceSystem destCRS);

    /**
     * to featureJSON
     *
     * @param feature
     * @return
     */
    String toFeatureJSON(Object feature);

    /**
     * to geoJSON
     *
     * @param geometry
     * @return
     */
    String toGeoJSON(Geometry geometry);

    /**
     * @param feature
     * @return
     */
    Map<String, Object> simpleFeature2Map(SimpleFeature feature);

    /**
     * @param sr wkid or wkt
     * @return
     */
    CoordinateReferenceSystem parseUndefineSR(String sr);

    /**
     * 验证图形拓扑
     *
     * @param geometry
     * @return
     */
    TopologyValidationError validGeometry(Geometry geometry);

    /**
     * get crs by reset regionCode sets
     *
     * @param regionCode
     * @return
     */
    CoordinateReferenceSystem getCRSByRegionCode(String regionCode);

    /**
     * get region key field
     *
     * @param layerName
     * @return
     */
    LayerRegion getLayerRegion(String layerName);


    /**
     * 行政区字典项配置中是否包含该行政区代码
     *
     * @param regionCode
     * @return
     */
    boolean containsRegionValue(String regionCode);


    /**
     * @param geoJSON
     * @return
     */
    double readGeometryAera(String geoJSON);

    /**
     * 获取简化精度
     *
     * @return
     */
    double getSimplifyTolerance();

    /**
     * 获取电子报件地块坐标
     *
     * @param in
     * @return FeatureJSON
     */
    Map getBJCoordinates(InputStream in) throws Exception;

    /**
     * 获取excel坐标文件地块坐标(geoJson)
     * @param in
     * @return
     * @throws Exception
     */
    String getExcelCoordinates(InputStream in) throws Exception;

    /**
     * 获取shapefile文件坐标以及属性(zip包文件流)
     * @param in
     * @return
     * @throws Exception
     */
    String getShpCoordinates(InputStream in) throws Exception;

    /**
     * 获取shapefile文件坐标以及属性(zip包文件)
     * @param file
     * @return
     * @throws Exception
     */
    String getShpCoordinates(File file) throws Exception;

    /**
     * 带有自定义属性参数
     * @param file
     * @param properties 属性字符串(json格式)
     * @return
     * @throws Exception
     */
    String getShpCoordinates(File file,String properties) throws Exception;

    /**
     *  解析单个shapefile文件流
     * @param in  .shp文件流
     * @return
     * @throws Exception
     */
    String getShpCoordinatesSingle(InputStream in) throws Exception;


    /**
     * get crs by coords x
     * @param x
     * @return
     */
    CoordinateReferenceSystem getCrsByCoordXD(double x);

}
