package cn.gtmap.onemap.platform.service.impl;

import cn.gtmap.onemap.platform.event.GeometryServiceException;
import cn.gtmap.onemap.platform.event.JSONMessageException;
import cn.gtmap.onemap.platform.service.GISManager;
import cn.gtmap.onemap.platform.utils.GeometryUtils;
import cn.gtmap.onemap.platform.utils.SRTransformations;
import cn.gtmap.onemap.service.GeoService;
import com.alibaba.fastjson.JSON;
import com.esri.sde.sdk.pe.PeProjectionException;
import com.vividsolutions.jts.geom.Geometry;
import com.vividsolutions.jts.operation.valid.IsValidOp;
import com.vividsolutions.jts.operation.valid.TopologyValidationError;
import com.vividsolutions.jts.simplify.DouglasPeuckerSimplifier;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.geotools.feature.FeatureCollection;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.referencing.crs.CoordinateReferenceSystem;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.*;
import java.util.*;

/**
 * .
 *
 * @author <a href="mailto:lanxy88@gmail.com">NelsonXu</a>
 * @version V1.0, 14-1-2 上午11:13
 */
public class GeoServiceImpl extends BaseLogger implements GeoService {

    protected final static String SE_OBJECTID_FIELD = "OBJECTID";

    protected final static String EPSG = "EPSG";

    protected final static String SHAPE = "SHAPE";

    protected final static String GEOMETRY = "geometry";

    protected final static String FEATURE_CRS = "crs";

    protected final static String FEATURE = "feature";

    @Autowired
    private GISManager gisManager;

    /**
     * 属性查询接口
     *
     * @param layerName      SDE中空间图层名称
     * @param where          查询条件
     * @param outFields      返回字段，默认null
     * @param returnGeometry 是否返回图形信息，默认true
     * @param dataSource     是否指定图层所在的数据源，默认null
     * @return map list
     */
    @Override
    public List query(String layerName, String where, String[] outFields, boolean returnGeometry, String dataSource) {
        try{
            return gisManager.getGISService().query(layerName, where, outFields, returnGeometry, dataSource);
        }catch (Exception e){
            throw new RuntimeException(" query error , detail : " + e.getLocalizedMessage());
        }
    }

    /**
     * 空间查询接口
     *
     * @param layerName  SDE中空间图层名称
     * @param geometry   查询的空间要素
     * @param outFields  返回字段，默认null
     * @param dataSource 是否指定图层所在的数据源，默认null
     * @return map list
     */
    @Override
    public List query(String layerName, Geometry geometry, String[] outFields, String dataSource) {
        try{
            return gisManager.getGISService().query(layerName, geometry, outFields, dataSource);
        }catch (Exception e){
            throw new RuntimeException(" query error , detail : " + e.getLocalizedMessage());
        }
    }

    /**
     * 空间查询接口
     *
     * @param layerName  SDE中空间图层名称
     * @param geometry   查询的空间要素，为GeoJSON格式
     * @param outFields  返回字段，默认null
     * @param dataSource 是否指定图层所在的数据源，默认null
     * @return map list
     */
    @Override
    public List query(String layerName, String geometry, String[] outFields, String dataSource) {
        try{
            return gisManager.getGISService().query(layerName, gisManager.getGeoService().readGeoJSON(geometry), outFields, dataSource);
        }catch (Exception e){
            throw new RuntimeException(" query error , detail : " + e.getLocalizedMessage());
        }
    }

    /**
     * 数据插入接口
     *
     * @param layerName  SDE中空间图层名称
     * @param feature    待插入要素，GeoJSON格式，可为Geometry、Feature或者FeatureCollection
     * @param dataSource 是否指定图层所在的数据源，默认null
     * @return GeoJSON要素，包含图形与属性，以及插入后OBJECTID属性值
     */
    @Override
    public String insert(String layerName, String feature, String dataSource) {
        try {
            return gisManager.getGISService().insert2(layerName, feature, dataSource);
        } catch (Exception e) {
            throw new RuntimeException(" insert error , detail : " + e.getLocalizedMessage());
        }
    }

    /**
     * 数据插入接口(执行拓扑检查)
     * @param layerName   SDE中空间图层名称
     * @param geoJSON     待插入要素，GeoJSON格式，可为Geometry、Feature或者FeatureCollection
     * @param check       是否执行拓扑检查
     * @param dataSource  是否指定图层所在的数据源，默认null
     * @return
     */
    public String insert(String layerName, String geoJSON, boolean check, String dataSource) {
        try {
            return gisManager.getGISService().insert2(layerName, geoJSON, check, dataSource);
        } catch (Exception e) {
            throw new RuntimeException(" insert error , detail : " + e.getLocalizedMessage());
        }
    }

    /**
     * 数据插入接口(执行拓扑检查,添加数据生成时间)
     * @param layerName      SDE中空间图层名称
     * @param geoJSON        待插入要素，GeoJSON格式，可为Geometry、Feature或者FeatureCollection
     * @param check          是否执行拓扑检查 true/false
     * @param addCreateTime  是否添加数据生成时间
     * @param dataSource     sde数据源
     * @since v1.1.2
     * @return
     */
    @Override
    public String insert(String layerName, String geoJSON, boolean check, boolean addCreateTime, String dataSource) {
        try {
            return gisManager.getGISService().insert3(layerName, geoJSON, check, addCreateTime, dataSource);
        } catch (Exception e) {
            throw new RuntimeException(" insert error , detail : " + e.getLocalizedMessage());
        }
    }

    /**
     * 数据更新接口
     *
     * @param layerName  SDE中空间图层名称
     * @param where      要素更新条件
     * @param feature    GeoJSON格式要素，可单独图形或者属性
     * @param dataSource 是否指定图层所在的数据源，默认null
     * @return
     */
    @Override
    public boolean update(String layerName, String where, String feature, String dataSource) {
        try {
            List list = query(layerName, where, null, true, dataSource);
            for (int i = 0; i < list.size(); i++) {
                Map item = (Map) list.get(i);
                String primaryKey = String.valueOf(item.get(SE_OBJECTID_FIELD));
                if (!gisManager.getGISService().update(layerName, primaryKey, feature, dataSource))
                    return false;
            }
            return true;
        } catch (Exception e) {
            throw new RuntimeException("update error,detail:"+e.getLocalizedMessage());
        }
    }

    /**
     * 数据更新接口 只更新图形的属性信息
     * @param layerName    sde图层名称
     * @param where        要素更新条件
     * @param properties   待更新属性 json格式
     * @param dataSource   sde数据源
     * @since v1.1.2
     * @return
     */
    @Override
    public boolean update2(String layerName, String where, String properties, String dataSource) {
        assert properties!=null;
        try {
            List list = query(layerName, where, null, true, dataSource);
            Map columns = JSON.parseObject(properties, Map.class);
            if (!isNull(list)) {
                for (int i = 0; i < list.size(); i++) {
                    Map item = (Map) list.get(i);
                    String primaryKey = String.valueOf(item.get(SE_OBJECTID_FIELD));
                    if (!gisManager.getGISService().update(layerName, primaryKey, columns, dataSource))
                        return false;
                }
            }
            return true;
        } catch (Exception e) {
            throw new RuntimeException("update error,detail:"+e.getLocalizedMessage());
        }
    }

    /**
     * 数据删除接口
     *
     * @param layerName  SDE中空间图层名称
     * @param where      要素删除条件
     * @param dataSource 是否指定图层所在的数据源，默认null
     * @return
     */
    @Override
    public boolean delete(String layerName, String where, String dataSource) {
        try {
            List list = query(layerName, where, null, true, dataSource);
            for (int i = 0; i < list.size(); i++) {
                Map item = (Map) list.get(i);
                String primaryKey = String.valueOf(item.get(SE_OBJECTID_FIELD));
                if (!gisManager.getGISService().delete(layerName, primaryKey, dataSource))
                    return false;
            }
            return true;
        } catch (Exception e) {
            throw new RuntimeException("delete error,detail:"+e.getLocalizedMessage());
        }
    }

    /**
     * 相交分析接口
     *
     * @param layerName
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public String intersect(String layerName, String geometry, String outFields, String dataSource) {
        String[] fields = "*".equals(outFields) ? null : outFields.split(",");
        List results = gisManager.getGISService().intersect2(layerName, geometry, fields, dataSource);
        FeatureCollection collection = gisManager.getGeoService().list2FeatureCollection(results, null, null);
        return toFeatureJSON(collection);
    }

    /**
     * intersect
     *
     * @param layerName
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public List intersect(String layerName, Geometry geometry, String[] outFields, String dataSource) {
        try {
            return gisManager.getGISService().intersect(layerName, geometry, outFields, dataSource);
        } catch (Exception e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }

    /**
     * difference 分析接口 获取压盖图形中的未相交部分
     * @param layerName   SDE中空间图层名称
     * @param geometry    GeoJSON格式要素
     * @param outFields   分析返回字段，多个字段逗号隔开
     * @param dataSource  分析图层所在的数据源名称 默认null
     * @since v1.1.2
     * @return
     */
    public String difference(String layerName, String geometry, String outFields, String dataSource) {
        try {
            String[] fields = "*".equals(outFields) ? null : outFields.split(",");
            List result = gisManager.getGISService().differenceByGeoJson(layerName,geometry,fields,dataSource);
            FeatureCollection featureCollection  = gisManager.getGeoService().list2FeatureCollection(result, null, null);
            return toFeatureJSON(featureCollection);
        } catch (Exception e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }

    /**
     * 土地规划审查分析
     *
     * @param year
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public Map tdghsc(String year, String geometry, String outFields, String dataSource) {
        String[] fields = "*".equals(outFields) ? null : outFields.split(",");
        return gisManager.getGISService().tdghscResult(gisManager.getGISService().tdghscAnalysis2(year, geometry, fields, dataSource));
    }

    /**
     * 土地利用现状分析
     *
     * @param dltbLayerName
     * @param xzdwLayerName
     * @param geometry
     * @param dataSource
     * @return
     */
    @Override
    public Map tdlyxz(String dltbLayerName, String xzdwLayerName, String geometry, String dataSource) {
        return gisManager.getGISService().tdlyxzAnalysis2(dltbLayerName, xzdwLayerName, geometry, dataSource);
    }

    /**
     * 土地利用现状分析(wcf)
     * @param dltbLayerName  地类图斑图层名
     * @param xzdwLayerName  现状地物图层名
     * @param geometry       分析的图形 geojson格式
     * @return               返回map对象 ｛"analysisArea":map,"analysisAreaDetail"：list｝
     */
    //@Override
    public Map tdlyxzWcf(String dltbLayerName, String xzdwLayerName, String geometry, String dataSource) {
        return gisManager.getGISService().tdlyxzAnalysisByWcf(dltbLayerName, xzdwLayerName, geometry);
    }

    /**
     * 读取GeoJSON格式，转化为Geometry、GeometryCollection、SimpleFeature、FeatureCollection
     *
     * @param geoJSON 标准化GeoJSON格式
     * @return
     */
    @Override
    public Object readGeoJSON(String geoJSON) {
        return GeometryUtils.parseGeoJSON(geoJSON);
    }

    /**
     * 读取WKT标准格式几何图形
     *
     * @param wkt wkt格式几何描述
     * @return
     */
    @Override
    public Geometry readWKT(String wkt) {
        try{
            return gisManager.getGeoService().readWKT(wkt);
        }catch (Exception e){
            throw new RuntimeException(" readWKT error , detail : " + e.getLocalizedMessage());
        }
    }

    /**
     * 读取空间参考
     *
     * @param geoJson 包含crs信息的geojson格式字符串
     * @return
     */
    @Override
    public CoordinateReferenceSystem readCRS(String geoJson) {
        return gisManager.getGeoService().readFeatureJSONCRS(geoJson);
    }

    /**
     * @param sr wkid or wkt
     * @return
     */
    @Override
    public CoordinateReferenceSystem parseUndefineSR(String sr) {
        return gisManager.getGeoService().parseUndefineSR(sr);
    }

    /**
     * 转化为FeatureJSON格式字符串
     *
     * @param feature
     * @return
     */
    @Override
    public String toFeatureJSON(Object feature) {
        return gisManager.getGeoService().toFeatureJSON(feature);
    }

    /**
     * 转化为GeoJSON格式
     *
     * @param geometry
     * @return
     */
    @Override
    public String toGeoJSON(Object geometry) {
        if (geometry instanceof Geometry) {
            return gisManager.getGeoService().toGeoJSON((Geometry) geometry);
        } else if (geometry instanceof SimpleFeature) {
            return toFeatureJSON(geometry);
        } else if (geometry instanceof FeatureCollection) {
            return toFeatureJSON(geometry);
        } else
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.GEOMETRY_TO_JSON_EXCEPTION, "geometry not be supported");
    }

    /**
     * 几何图形投影转换
     *
     * @param geometry
     * @param sourceCrs
     * @param targetCrs
     * @return
     */
    @Override
    public Geometry project(Geometry geometry, String sourceCrs, String targetCrs) {
        try {
            return SRTransformations.project(geometry, sourceCrs, targetCrs);
        } catch (PeProjectionException e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.PROJECT_EXCEPTION, e.getLocalizedMessage());
        }
    }

    /**
     * 几何图形投影转换
     *
     * @param geometry
     * @param inSR
     * @param outSR
     * @return
     */
    @Override
    public Geometry project(Geometry geometry, int inSR, int outSR) {
        return gisManager.getGeoService().project(geometry, readCRS(String.valueOf(inSR)), readCRS(String.valueOf(outSR)));
    }

    /**
     * 强制简化图形
     *
     * @param geometry
     * @param tolerance
     * @return
     */
    @Override
    public Geometry simplify(Geometry geometry, double tolerance) {
        return geometry.isValid() && geometry.isSimple() ? geometry : DouglasPeuckerSimplifier.simplify(geometry, tolerance);
    }

    /**
     * 验证几何图形拓扑关系
     *
     * @param geometry
     * @return
     */
    @Override
    public TopologyValidationError validGeometry(Geometry geometry) {
        IsValidOp isValidOp = new IsValidOp(geometry);
        return isValidOp.getValidationError();
    }

    /**
     * Map value to SimpleFeature
     *
     * @param value   {"SHAPE":"WKT FORMATE","PRO1":"VALUE"}
     * @param srcCRS
     * @param destCRS
     * @return
     */
    @Override
    public String map2SimpleFeature(Map<String, Object> value, CoordinateReferenceSystem srcCRS, CoordinateReferenceSystem destCRS) {
        return toFeatureJSON(gisManager.getGeoService().map2SimpleFeature(value, srcCRS, destCRS));
    }

    /**
     * list values to featureCollection
     *
     * @param value
     * @param srcCRS
     * @param destCRS
     * @return
     */
    @Override
    public String list2FeatureCollection(List<Map<String, Object>> value, CoordinateReferenceSystem srcCRS, CoordinateReferenceSystem destCRS) {
        try{
            return toFeatureJSON(gisManager.getGeoService().list2FeatureCollection(value, srcCRS, destCRS));
        }catch (Exception e){
            throw new RuntimeException(" list values to featureCollection error , detail : " + e.getLocalizedMessage());
        }
    }

    /**
     * SimpleFeature to HashMap
     *
     * @param feature
     * @return
     */
    @Override
    public Map<String, Object> simpleFeature2Map(SimpleFeature feature) {
        return gisManager.getGeoService().simpleFeature2Map(feature);
    }

    /**
     * get crs by x
     *
     * @param x
     * @return
     */
    @Override
    public CoordinateReferenceSystem getCrsByCoordXD(double x) {

        return gisManager.getGeoService().getCrsByCoordXD(x);
    }

    /**
     *解析导入的shp的zip包
     * @param file       zip文件
     * @param properties 额外属性
     * @return
     */
    @Override
    public String getShpCoordinates(File file, String properties) {
        try {
            if (properties != null)
                return gisManager.getGeoService().getShpCoordinates(file, properties);
            else
                return gisManager.getGeoService().getShpCoordinates(file);
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 解析导入的shp的zip包
     * tolerateExp=false时效果与String getShpCoordinates(InputStream stream, String properties)相同
     * tolerateExp=true时，解析过程中遇到异常会跳过异常继续解析
     * @param file       zip文件
     * @param properties 额外属性
     * @param tolerateExp
     * @return
     */
    @Override
    public String getShpCoordinates(File file, String properties, boolean tolerateExp) {
        try {
            if (properties != null)
                return gisManager.getGeoService().getShpCoordinates(file, properties, tolerateExp);
            else
                return gisManager.getGeoService().getShpCoordinates(file, tolerateExp);
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 解析导入的shp的zip包
     * @param stream       zip的文件流
     * @param properties   额外属性
     * @return
     */
    @Override
    public String getShpCoordinates(InputStream stream, String properties) {
        if (isNull(stream)) throw new RuntimeException("stream is null");
        try {
            if (stream.available() > 0) {
                File tmp = new File(System.getProperty("java.io.tmpdir").concat("TEMP_" + System.currentTimeMillis()));
                try {
                    FileOutputStream output = new FileOutputStream(tmp);
                    try {
                        IOUtils.copyLarge(stream, output, 0, stream.available(), new byte[stream.available()]);
                        output.close();
                    } finally {
                        IOUtils.closeQuietly(output);
                    }
                    return getShpCoordinates(tmp,properties);
                } finally {
                    FileUtils.deleteQuietly(tmp);
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 解析导入shp的zip包，
     * tolerateExp=false时效果与String getShpCoordinates(InputStream stream, String properties)相同
     * tolerateExp=true时，解析过程中遇到异常会跳过异常继续解析
     * @param stream
     * @param properties
     * @param tolerateExp
     * @return
     */
    @Override
    public String getShpCoordinates(InputStream stream, String properties, boolean tolerateExp) {
        if (isNull(stream)) throw new RuntimeException("stream is null");
        try {
            if (stream.available() > 0) {
                File tmp = new File(System.getProperty("java.io.tmpdir").concat("TEMP_" + System.currentTimeMillis()));
                try {
                    FileOutputStream output = new FileOutputStream(tmp);
                    try {
                        IOUtils.copyLarge(stream, output, 0, stream.available(), new byte[stream.available()]);
                        output.close();
                    } finally {
                        IOUtils.closeQuietly(output);
                    }
                    return getShpCoordinates(tmp,properties, tolerateExp);
                } finally {
                    FileUtils.deleteQuietly(tmp);
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 解析导入shp的zip包
     * tolerateExp=false时，遇到异常就抛出异常
     * tolerateExp=true时，解析过程中遇到异常(topo异常)会跳过异常继续解析
     * @param filePath
     * @param properties
     * @param tolerateExp
     * @return
     */
    @Override
    public String getShpCoordinates(String filePath, String properties, boolean tolerateExp) {
        File file = new File(filePath);
        if(file.isFile()){
            return getShpCoordinates(file, properties, tolerateExp);
        }
        return "";
    }

    /**
     *
     * @param filePath
     * @param properties
     * @param layerName
     * @param dataSource
     * @param tolerateExp
     * @return
     * @return {"success":boolean,result:JSONString} || ""
     */
    @Override
    public String insertShpCoordinates(String filePath,String properties, String layerName, String dataSource, boolean tolerateExp) throws Exception {
        File file = new File(filePath);
        long date = System.currentTimeMillis();
        if(file.isFile()){
            Map result = gisManager.getGeoService().insertShpCoordinates(file, properties, layerName, dataSource, tolerateExp);
            Map map = new HashMap();
            if(isNull(result)||!((Boolean)result.get("success"))){

                map.put("success", false);
;
                map.put("result",result.get("result"));
            }else{
                String outPath = System.getProperty("java.io.tmpdir").concat("\\TMP_" + System.currentTimeMillis());
                logger.info("插入数据成功，将结果写入文件:" + outPath + "！");
                OutputStream out = new FileOutputStream(outPath);
                BufferedWriter bw = new BufferedWriter(new OutputStreamWriter(out, "UTF-8"));
                bw.write((String)result.get("result"));
                bw.flush();
                logger.info("成功写入数据，结果写入" + outPath + "，总耗时：" + (System.currentTimeMillis() - date) + "ms");
                map.put("success", true);
                map.put("result",outPath);
            }
            return JSON.toJSONString(map);
        }
        Map map = new HashMap(){
            {
                put("success",false);
                put("result","文件不存在");
            }
        };
        return  JSON.toJSONString(map);
    } 
    /**
     *
     * @param geoJson GeoJSON格式  eg.{"type":"Feature","crs":{"type":"name","properties":{"name":"EPSG:4610"}},"geometry":{"type":"Polygon","coordinates":[[[118.65,32.83],[118.65,32.82],[118.64,32.83],[118.65,32.83]]]},"properties":{"PRONAME":"xxx"}}
     * @return
     * @throws Exception
     */
    public File exportToShp(String geoJson) throws Exception {
        return gisManager.getGeoService().exportToShp(geoJson);
    }

    /**
     *
     * @param layerName  SDE空间图层名
     * @param where      查询条件
     * @param geometry   查询的空间要素，为GeoJSON格式
     * @param outFields  返回字段，默认null
     * @param dataSource sde数据源,默认null
     * @return
     * @throws Exception
     */
    public File exportToShp(String layerName, String where, String geometry, String[] outFields, String dataSource) throws Exception {
        return gisManager.getGeoService().exportToShp(layerName,where,geometry,outFields,dataSource);
    }

    /**
     * 解析cad文件 返回geojson字符串
     * @param dwgFile  dwg文件在线地址
     * @param gpUrl    解析dwg文件的ArcGIS gp服务地址
     * @return        返回geojson
     * @throws Exception
     */
    @Override
    public String getDwgCoordinates(String dwgFile, String gpUrl,String properties) throws Exception {
        assert dwgFile != null;
        assert gpUrl != null;
        Map propertyMap = JSON.parseObject(properties,Map.class);
        if(isNull(propertyMap))
            logger.info("properties is null");
        if (!gpUrl.endsWith("/execute"))
            gpUrl = gpUrl.concat("/execute");
        HttpClient httpclient = new DefaultHttpClient();
        HttpPost httpRequest = new HttpPost(gpUrl);
        List<BasicNameValuePair> params = new ArrayList<BasicNameValuePair>();
        params.add(new BasicNameValuePair("input_cad_file", dwgFile));
        params.add(new BasicNameValuePair("f", "json"));
        HttpEntity httpentity;
        try {
            httpentity = new UrlEncodedFormEntity(params, "UTF-8");
            httpRequest.setEntity(httpentity);
            HttpResponse httpResponse = httpclient.execute(httpRequest);
            if (httpResponse.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
                String responseBody = EntityUtils.toString(httpResponse.getEntity());
                logger.info("[调用gp服务返回结果:]" + responseBody);
                Map map = JSON.parseObject(responseBody, Map.class);
                List<Map> results = (List<Map>) map.get("results");
                List<Map> list = (List<Map>) results.get(0).get("value");

                List<Map<String, Object>> features = new ArrayList<Map<String, Object>>();
                LinkedHashMap featureCollection = new LinkedHashMap();
                for (Map item:list)
                {
//                    CoordinateReferenceSystem crs = null;
//                    Map sr = MapUtils.getMap(item, "spatialReference");
//                    if (sr.containsKey("wkt") && sr.get("wkt") != null)
//                        crs = gisManager.getGeoService().parseUndefineSR(MapUtils.getString(sr, "wkt"));
//                    if (sr.containsKey("wkid") && sr.get("wkid") != null)
//                        crs = gisManager.getGeoService().parseUndefineSR(MapUtils.getString(sr, "wkid"));
                    List<Map> polygons = (List<Map>) item.get("features");
                    for (Map polygon:polygons)
                    {
                        LinkedHashMap<String, Object> feature = new LinkedHashMap<String, Object>();
                        LinkedHashMap<String, Object> geometry = new LinkedHashMap<String, Object>();
                        Map proMap= new HashMap();
                        geometry.put("type","Polygon");
//                        Map attributes =MapUtils.getMap(polygon,"attributes");
                        Map geo = MapUtils.getMap(polygon,"geometry");
                        List rings = (List) geo.get("rings");
                        geometry.put("coordinates", rings);

                        feature.put("type","Feature");
                        feature.put("geometry", geometry);
                        if(!isNull(propertyMap))
                        {
                            proMap.putAll(propertyMap);
                            feature.put("properties",proMap);
                        }
                        features.add(feature);
                    }
                    featureCollection.put("type","FeatureCollection");
                    featureCollection.put("features",features);
                    logger.info(JSON.toJSONString(featureCollection));
                }
                return JSON.toJSONString(featureCollection);
            } else {
                logger.error("调用gp服务异常");
                return null;
            }
        } catch (ClientProtocolException e) {
            throw new RuntimeException(e.getMessage());
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException(e.getMessage());
        } catch (IllegalArgumentException e) {
            throw new RuntimeException(e.getMessage());
        } catch (IOException e) {
            throw new RuntimeException(e.getMessage());
        }
    }

}
