package cn.gtmap.onemap.platform.service.impl;

import cn.gtmap.onemap.core.support.hibernate.UUIDHexGenerator;
import cn.gtmap.onemap.platform.Constant;
import cn.gtmap.onemap.platform.dao.SpatialDao;
import cn.gtmap.onemap.platform.dao.impl.ArcSDEDaoImpl;
import cn.gtmap.onemap.platform.dao.impl.OracleSpatialDaoImpl;
import cn.gtmap.onemap.platform.entity.Configuration;
import cn.gtmap.onemap.platform.entity.FileStore;
import cn.gtmap.onemap.platform.entity.LayerRegion;
import cn.gtmap.onemap.platform.entity.dict.Dict;
import cn.gtmap.onemap.platform.entity.dict.Item;
import cn.gtmap.onemap.platform.event.GISDaoException;
import cn.gtmap.onemap.platform.event.GeometryServiceException;
import cn.gtmap.onemap.platform.event.JSONMessageException;
import cn.gtmap.onemap.platform.service.*;
import cn.gtmap.onemap.platform.support.spring.ApplicationContextHelper;
import cn.gtmap.onemap.platform.utils.*;
import cn.gtmap.onemap.platform.utils.ArrayUtils;
import cn.gtmap.onemap.platform.utils.EnumUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.esri.sde.sdk.client.SeLayer;
import com.gtis.config.AppConfig;
import com.vividsolutions.jts.geom.*;
import com.vividsolutions.jts.geom.Point;
import com.vividsolutions.jts.geom.Polygon;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.collections.keyvalue.MultiKey;
import org.apache.commons.collections.map.HashedMap;
import org.apache.commons.collections.map.MultiKeyMap;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.codehaus.xfire.client.Client;
import org.geotools.data.simple.SimpleFeatureCollection;
import org.geotools.feature.FeatureCollection;
import org.geotools.feature.FeatureIterator;
import org.geotools.geometry.jts.Geometries;
import org.opengis.feature.Property;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.referencing.crs.CoordinateReferenceSystem;
import org.opengis.referencing.crs.GeographicCRS;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.util.Assert;

import javax.imageio.ImageIO;
import javax.naming.NameParser;
import java.awt.*;
import java.awt.geom.Arc2D;
import java.awt.image.BufferedImage;
import java.io.*;
import java.math.BigDecimal;
import java.net.URL;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.List;
import java.text.DecimalFormat;
import java.util.regex.Pattern;

/**
 * .
 *
 * @author <a href="mailto:lanxy88@gmail.com">NelsonXu</a>
 * @version V1.0, 13-5-20 下午5:25
 */
public class GISServiceImpl extends BaseLogger implements GISService {

    private final String SE_SHAPE_FIELD = Constant.SE_SHAPE_FIELD;

    private final String SE_SHAPE_AREA = Constant.SE_SHAPE_AREA;

    private final String OG_PRO_PERFIX = "OG_PRO_";

    private final String SEARCH_LAYERS = "layers";

    private final String EXPORT_TYPE_GH = "gh";

    private final String EXPORT_TYPE_XZ = "xz";

    /**
     * 分析展示结果模板文件位置
     */
    private final String ANALYSIS_FTL_DIR = "analysis/template/";

    private final String TPL_SUFFIX = ".ftl";

    private final String ANALSYIS_DEFAULT_TPL = "default.ftl";


    /**
     * 规划图层命名中部特征码
     */
    private static final String LAYER_MIDDLE_FIX_E = "_E_";

    /**
     * 现状图层命名中部特征码
     */
    private static final String LAYER_MIDDLE_FIX_H = "_H_";

    static final SimpleDateFormat sdf = new SimpleDateFormat(Constant.DEFAULT_DATE_FORMATE);

    /**
     * 地类名称字段
     */
    private String DLMC = "DLMC";
    /**
     * 地类编码字段
     */
    private String DLBM = "DLBM";

    /**
     * 耕地地类编码
     */
    private String[] GDDLBM = {
            "011", "012", "013"
    };

    /**
     * 耕地地类
     */
    private Map<String, String> GDMAP = new HashMap<String, String>() {
        {
            put("011", "水田");
            put("012", "水浇地");
            put("013", "旱地");
        }
    };

    /**
     * 可调整地类编码
     */
    private String[] KTZYDDLBM = {
            "021", "022", "023", "031", "033", "042", "114"
    };
    /**
     * 可调整地类
     */
    private Map<String, String> KTZMAP = new HashMap<String, String>() {
        {
            put("021", "");
            put("022", "");
            put("023", "");
            put("031", "");
            put("033", "");
            put("042", "");
            put("114", "");
        }
    };

    private String KTZYDDLBZ = "K";

    /**
     * 基本农田保护图斑 叠加分析 所需字段
     */
    private String[] JBNTBHTB = {
            DLMC, DLBM, "DLBZ"
            //DLMC,DLBM,"QSXZ","QSDWDM","QSDWMC","ZLDWDM","ZLDWMC","GDLX","DLBZ","JBNTMJ","TBMJ"
    };

    /**
     * 土地用途总体规划审查
     */
    private enum GHSC {

        TDYTQ("土地用途分区"),
        JSYDGZQ("建设用地管制区"),
        GHJBNTTZ("规划基本农田调整"),
        MZZDJSXM("重点建设项目");

        private String label;

        private GHSC(String value) {
            this.label = value;
        }

        private String getLabel() {
            return label;
        }
    }

    /**
     * 土地利用现状
     */
    private enum TDXZ {
        DLTB, XZDW
    }

    private enum Tag {
        YES, NO;
    }

    /**
     * wcf用到的tag
     */
    private enum WCFTag {
        AnalyseGeoJSON, TBLayerName, XWLayerName, LXLayerName, UseGlobalArea, TBDLMJ, XZDWMJ, LXDWMJ, TKMJ, KCDLBM,
        Summary, LXFeatures, XWFeatures, TBFeatures, SummaryDetail, SummaryTBs, SummaryXWs, SummaryLXs,
        DLBM, QSDWMC, QSDWMC1, QSDWDM, QSDWDM1, QSXZ, AREA, ZLDWDM, ZLDWMC, DLMJ, DLMC, geometry
    }

    /**
     * 面积单位
     */
    private enum UNITS {
        SQUARE(1, "平方米"),
        ACRES(0.0015, "亩"),
        HECTARE(0.0001, "公顷");

        private double conv;
        private String alias;

        private UNITS(double value, String cvalue) {
            this.conv = value;
            this.alias = cvalue;
        }

        private double getConv() {
            return conv;
        }

        private String getAlias() {
            return alias;
        }
    }

    private SpatialDao spatialDao;

    @Autowired
    private GeometryService geometryService;
    @Autowired
    private AgsGeoemtryService agsGeoemtryService;

    @Autowired
    private WebMapService webMapService;

    @Autowired
    private FileStoreService fileStoreService;

    @Autowired
    private TemplateService templateService;

    private Map searchConfig;

    private Map exportConfig;
    /**
     * 分析功能用到的设置项
     */
    private Map analysisSet;

    /**
     * 分析字段映射集合
     */
    private Map<String, ?> analysisFieldsMap;

    /**
     * 综合分析funid与别名之间映射集合
     */
    private Map<String, ?> analysisAliasMap;

    private String insertDateField;

    private String insertRegionField;
    /**
     * 是否自动插入生成时间
     */
    private boolean createDateAuto = false;
    /**
     * 是否自动插入行政区代码
     */
    private boolean regionCodeAuto = false;

    public void setSearchConfig(Resource path) {
        try {
            searchConfig = JSON.parseObject(IOUtils.toString(path.getURI(), Constant.UTF_8), Map.class);
        } catch (IOException e) {
            logger.error(" search config can't found ");
        }
    }

    public void setExportConfig(Resource path) {
        try {
            exportConfig = JSON.parseObject(IOUtils.toString(path.getURI(), Constant.UTF_8), Map.class);
        } catch (IOException e) {
            logger.error(" export config can't found ");
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
        }
    }

    public void setAnalysisSet(Resource path) {
        try {
            this.analysisSet = JSON.parseObject(IOUtils.toString(path.getURI(), Constant.UTF_8), Map.class);
            if (analysisSet.containsKey("analysis-fields-mapping")) {
                analysisFieldsMap = (Map<String, ?>) analysisSet.get("analysis-fields-mapping");
            }
            if (analysisSet.containsKey("analysis-alias-mapping")) {
                analysisAliasMap = (Map<String, ?>) analysisSet.get("analysis-alias-mapping");
            }
        } catch (IOException e) {
            logger.error("gis analysis set not found!");
        }
    }

    /**
     * 初始化
     */
    @Override
    public void initialize(Constant.SpatialType type) {
        switch (type) {
            case ARC_SDE:
                spatialDao = (SpatialDao) ApplicationContextHelper.createBean(ArcSDEDaoImpl.class);
                break;
            case ORACLE_SPATIAL:
                spatialDao = (SpatialDao) ApplicationContextHelper.createBean(OracleSpatialDaoImpl.class);
                break;
        }
        insertDateField = isNull(AppPropertyUtils.getAppEnv("insert.date.field")) ? null : String.valueOf(AppPropertyUtils.getAppEnv("insert.date.field")).trim();
        createDateAuto = isNull(AppPropertyUtils.getAppEnv("insert.create.date")) ? false : Boolean.valueOf(String.valueOf(AppPropertyUtils.getAppEnv("insert.create.date")));
        insertRegionField = isNull(AppPropertyUtils.getAppEnv("insert.region.field")) ? null : String.valueOf(AppPropertyUtils.getAppEnv("insert.region.field")).trim();
        regionCodeAuto = isNull(AppPropertyUtils.getAppEnv("insert.create.regioncode")) ? false : Boolean.valueOf(String.valueOf(AppPropertyUtils.getAppEnv("insert.create.regioncode")));
    }

    /**
     * 属性查询
     *
     * @param layerName
     * @param where
     * @param columns
     * @param returnGeometry
     * @param dataSource
     * @return
     */
    @Override
    public List<?> query(String layerName, String where, String[] columns, boolean returnGeometry, String dataSource) {
        Assert.notNull(layerName, getMessage("layer.name.notnull"));
        Assert.notNull(where, getMessage("query.where.notnull"));
        List result = null;
        try {
            result = spatialDao.query(layerName, where, columns, returnGeometry, dataSource);
        } catch (Exception e) {
            logger.info(getMessage("query.result.null", layerName), e.getLocalizedMessage());
        }
        return result;
    }

    /**
     * 空间查询
     *
     * @param layerName
     * @param wkt
     * @param columns
     * @param dataSource
     * @return
     */
    @Override
    public List<?> query(String layerName, String wkt, String[] columns, String dataSource) {
        Assert.notNull(layerName, getMessage("layer.name.notnull"));
        Assert.notNull(wkt, getMessage("query.geometry.notnull"));
        return spatialDao.query(layerName, wkt, columns, dataSource);
    }

    /**
     * 空间查询
     *
     * @param layerName
     * @param geometry
     * @param columns
     * @param dataSource
     * @return
     */
    @Override
    public List<?> query(String layerName, Geometry geometry, String[] columns, String dataSource) {
        return query(layerName, geometry.toText(), columns, dataSource);
    }

    /**
     * 空间查询(进行投影转换)
     *
     * @param layerName
     * @param feature
     * @param columns
     * @param dataSource
     * @return
     */
    @Override
    public List<?> query(String layerName, SimpleFeature feature, String[] columns, String dataSource) {
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        CoordinateReferenceSystem sourceCRS = feature.getFeatureType().getCoordinateReferenceSystem();
        Geometry geometry = (Geometry) feature.getDefaultGeometry();
        if (sourceCRS == null)
            sourceCRS = geometry.getSRID() != 0 ?
                    geometryService.getCRSBySRID(String.valueOf(geometry.getSRID())) : null;
        if (sourceCRS != null && !sourceCRS.equals(layerCRS))
            geometry = geometryService.project(geometry, sourceCRS, layerCRS);
        return query(layerName, geometry.toText(), columns, dataSource);
    }


    /**
     * 相交分析
     *
     * @param layerName
     * @param wktPlygon
     * @param returnFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersect(String layerName, String wktPlygon, String[] returnFields, String dataSource) {
        Assert.notNull(layerName, getMessage("layer.name.notnull"));
        return spatialDao.intersect(layerName, wktPlygon, returnFields, dataSource);
    }

    /**
     * 相交分析
     *
     * @param layerName
     * @param polygon
     * @param returnFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersect(String layerName, Polygon polygon, String[] returnFields, String dataSource) {
        return intersect(layerName, polygon.toText(), returnFields, dataSource);
    }

    /**
     * 相交分析
     *
     * @param layerName
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersect(String layerName, Geometry geometry, CoordinateReferenceSystem sourceCRS,
                             String[] outFields, String dataSource) {
        Assert.notNull(geometry, getMessage("geometry.notnull"));
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        if (sourceCRS == null)
            sourceCRS = geometry.getSRID() != 0 ?
                    geometryService.getCRSBySRID(String.valueOf(geometry.getSRID())) : null;
        if (sourceCRS != null) geometry = geometryService.project(geometry, sourceCRS, layerCRS);
        List<Map<String, Object>> result = new ArrayList<Map<String, Object>>();
        if (geometry instanceof GeometryCollection) {
            for (int i = 0; i < geometry.getNumGeometries(); i++) {
                try {
                    Geometry geo = geometry.getGeometryN(i);
                    result.addAll((Collection<? extends Map<String, Object>>)
                            intersect(layerName, geo.toText(), outFields, dataSource));
                    addGeoProperty2List(result, geo);
                } catch (Exception e) {
                    logger.error(e.getLocalizedMessage());
                }
            }
        } else {
            result.addAll((Collection<? extends Map<String, Object>>)
                    intersect(layerName, geometry.toText(), outFields, dataSource));
            addGeoProperty2List(result, geometry);
        }
        if (sourceCRS != null) {
            for (Map<String, Object> item : result) {
                Geometry geo = geometryService.readWKT((String) item.get(SE_SHAPE_FIELD));
                item.put(Constant.SE_SHAPE_FIELD, geometryService.project(geo, layerCRS, sourceCRS).toText());
            }
        }
        return result;
    }

    /**
     * 相交分析(New)
     *
     * @param layerName
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersect(String layerName, Geometry geometry, String[] outFields, String dataSource) {

        List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
        CoordinateReferenceSystem sourceCRS = null;
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        sourceCRS = geometry.getSRID() != 0 ? geometryService.getCRSBySRID(String.valueOf(geometry.getSRID())) : null;
        String regionField = null;
        if (layerCRS instanceof GeographicCRS) {
            LayerRegion layerRegion = geometryService.getLayerRegion(layerName);
            if (!isNull(layerRegion.getSourceLayerCRS()))
                sourceCRS = layerRegion.getSourceLayerCRS();
            else if (!isNull(layerRegion.getRegionField())) {
                regionField = layerRegion.getRegionField();
                if (!checkFieldInLayer(regionField, layerName, dataSource))
                    throw new RuntimeException(getMessage("field.not.in.layer", regionField, layerName));
                if (!isNull(outFields) && !ArrayUtils.contains(outFields, regionField, true))
                    outFields = ArrayUtils.add2Arrays(outFields, regionField);
            }
        }

        if (geometry instanceof GeometryCollection) {
            for (int i = 0; i < geometry.getNumGeometries(); i++) {
                try {
                    Geometry geo = geometry.getGeometryN(i);
                    List queryResults = query(layerName, geo, outFields, dataSource);

                    if (layerCRS instanceof GeographicCRS) {
                        if (isNull(sourceCRS) && queryResults.size() > 0)
                            sourceCRS = geometryService.getCRSByRegionCode(String.valueOf(((Map) queryResults.get(0)).get(regionField)));
                        geo = geometryService.project(geo, layerCRS, isNull(sourceCRS) ? layerCRS : sourceCRS);
                    }
                    Map<String, Object> result = null;
                    for (int j = 0; j < queryResults.size(); j++) {
                        Map map = (Map) queryResults.get(j);
                        Geometry geo1 = geometryService.readWKT(map.get(SE_SHAPE_FIELD).toString());
                        if (layerCRS instanceof GeographicCRS) {
                            geo1 = geometryService.project(geo1, layerCRS, sourceCRS);
                        }
                        Geometry geometryResult = null;
                        try {
                            if (isNull(sourceCRS))
                                sourceCRS = layerCRS;
                            geometryResult = geometryService.readWKT(agsGeoemtryService.intersection(geo1.toText(), geo.toText(), isNull(sourceCRS) ? null : sourceCRS.toWKT()));
                        } catch (Exception e) {
                            if (!geo.isValid() || !geo.isSimple()) {
                                logger.error(" geometry is invalid,[{}]", e.getLocalizedMessage());
                                geometryResult = geometryService.forceSimplify(geo1, geometryService.getSimplifyTolerance()).intersection(geo);
                            } else
                                logger.error("intersection error:", e.getMessage());
                        }
                        if (geometryResult.isEmpty())
                            continue;
                        result = new HashMap<String, Object>();
                        result.putAll(map);
                        result.put(SE_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geometryResult.toText()));
                        if (layerCRS instanceof GeographicCRS) {
                            try {
                                geometryResult = geometryService.project(geometryResult, sourceCRS, layerCRS);
                            } catch (GeometryServiceException e) {
                                geometryResult = geometryService.simplify(geometryResult, geometryService.getSimplifyTolerance());
                                geometryResult = geometryService.project(geometryResult, sourceCRS, layerCRS);
                            }
                        }
                        result.put(SE_SHAPE_FIELD, geometryResult.toText());
                        results.add(result);
                    }
                } catch (Exception e) {
                    logger.error(e.getLocalizedMessage());
                }
            }
        } else {
            List queryResults = query(layerName, geometry, outFields, dataSource);

            if (layerCRS instanceof GeographicCRS) {
                if (isNull(sourceCRS) && queryResults.size() > 0)
                    sourceCRS = geometryService.getCRSByRegionCode(String.valueOf(((Map) queryResults.get(0)).get(regionField))); //
                geometry = geometryService.project(geometry, layerCRS, isNull(sourceCRS) ? layerCRS : sourceCRS);
            }
            Map<String, Object> result = null;
            for (int j = 0; j < queryResults.size(); j++) {
                Map map = (Map) queryResults.get(j);
                Geometry geo1 = geometryService.readWKT(map.get(SE_SHAPE_FIELD).toString());
                if (layerCRS instanceof GeographicCRS) {
                    geo1 = geometryService.project(geo1, layerCRS, sourceCRS);
                }
                Geometry geometryResult = null;
                try {
                    if (isNull(sourceCRS))
                        sourceCRS = layerCRS;
                    geometryResult = geometryService.readWKT(agsGeoemtryService.intersection(geo1.toText(), geometry.toText(), sourceCRS.toWKT()));
                } catch (Exception e) {
                    if (!geometry.isValid() || !geometry.isSimple()) {
                        logger.error(" geometry is invalid,[{}]", e.getLocalizedMessage());
                        geometryResult = geometryService.forceSimplify(geo1, geometryService.getSimplifyTolerance()).intersection(geometry);
                    } else
                        logger.error("intersection error", e.getMessage());
                }
                if (geometryResult.isEmpty())
                    continue;
                result = new HashMap<String, Object>();
                result.putAll(map);
                result.put(SE_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geometryResult.toText()));
                if (layerCRS instanceof GeographicCRS) {
                    try {
                        geometryResult = geometryService.project(geometryResult, sourceCRS, layerCRS);
                    } catch (GeometryServiceException e) {
                        geometryResult = geometryService.simplify(geometryResult, geometryService.getSimplifyTolerance());
                        geometryResult = geometryService.project(geometryResult, sourceCRS, layerCRS);
                    }
                }
                result.put(SE_SHAPE_FIELD, geometryResult.toText());
                results.add(result);
            }
        }

        return results;
    }

    /**
     * @param layerName
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> difference(String layerName, Geometry geometry, String[] outFields, String dataSource) {
        try {
            List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
            List<Geometry> diffGeos = new ArrayList<Geometry>();
            CoordinateReferenceSystem sourceCRS = null;
            CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
            sourceCRS = geometry.getSRID() != 0 ? geometryService.getCRSBySRID(String.valueOf(geometry.getSRID())) : null;
            String regionField = null;
            if (layerCRS instanceof GeographicCRS) {
                LayerRegion layerRegion = geometryService.getLayerRegion(layerName);
                if (!isNull(layerRegion.getSourceLayerCRS()))
                    sourceCRS = layerRegion.getSourceLayerCRS();
                else if (!isNull(layerRegion.getRegionField())) {
                    regionField = layerRegion.getRegionField();
                    if (!checkFieldInLayer(regionField, layerName, dataSource))
                        throw new RuntimeException(getMessage("field.not.in.layer", regionField, layerName));
                    if (!isNull(outFields) && !ArrayUtils.contains(outFields, regionField, true))
                        outFields = ArrayUtils.add2Arrays(outFields, regionField);
                }
            }
            if (geometry instanceof Geometry) {
                List queryResults = query(layerName, geometry, outFields, dataSource);
                if (layerCRS instanceof GeographicCRS) {
                    if (isNull(sourceCRS) && queryResults.size() > 0)
                        sourceCRS = geometryService.getCRSByRegionCode(String.valueOf(((Map) queryResults.get(0)).get(regionField))); //
                    geometry = geometryService.project(geometry, layerCRS, isNull(sourceCRS) ? layerCRS : sourceCRS);
                }
                Map<String, Object> result = null;
                if (!isNull(queryResults) && queryResults.size() > 0) {
                    for (int j = 0; j < queryResults.size(); j++) {
                        Map map = (Map) queryResults.get(j);
                        Geometry geo1 = geometryService.readWKT(map.get(SE_SHAPE_FIELD).toString());
                        if (layerCRS instanceof GeographicCRS) {
                            geo1 = geometryService.project(geo1, layerCRS, sourceCRS);
                        }
                        List<Geometry> others = new ArrayList<Geometry>();
                        if (isGeometryCollection(geo1)) {
                            others.addAll(decomposeGC((GeometryCollection) geo1));
                        } else
                            others.add(geo1);
                        for (Geometry _geo : others) {                          //保证difference的是geometry
                            Geometry geometryResult = geometry.difference(_geo);//获得结果属于geometry 不属于geo1的部分
                            if (isNull(geometryResult))
                                continue;
                            if (geometryResult.isEmpty())
                                continue;
                            diffGeos.add(geometryResult);
                            result = new HashMap<String, Object>();
                            result.putAll(map);
                            result.put(SE_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geometryResult.toText()));
                            if (layerCRS instanceof GeographicCRS) {
                                try {
                                    geometryResult = geometryService.project(geometryResult, sourceCRS, layerCRS);
                                } catch (GeometryServiceException e) {
                                    geometryResult = geometryService.simplify(geometryResult, geometryService.getSimplifyTolerance());
                                    geometryResult = geometryService.project(geometryResult, sourceCRS, layerCRS);
                                }
                            }
                            result.put(SE_SHAPE_FIELD, geometryResult.toText());
                            result.put(Constant.ORIGINAL_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geo1.toText()));
                            results.add(result);
                        }
                    }
                } else {
                    diffGeos.add(geometry);//如果没有相交 则返回原先的图形
                }
                //已存在difference结果
                if (results.size() > 0) {
                    List<Geometry> _list = new ArrayList<Geometry>();
                    //updated by yxf on 2015/07/30
                    if (results.size() > 1) {
                        List<Geometry> filterList = new ArrayList<Geometry>();
                        for (Geometry g : diffGeos) {
                            if (isGeometryCollection(g)) continue;
                            filterList.add(g);
                        }
                        if (filterList.size() > 0) {
                            Geometry dGeo = getDuplicatedGeo(filterList, getLayerCRS(layerName, dataSource));
                            if (dGeo.isEmpty()) {
                                results.clear();
                                return results;
                            }
                        }
                    }

                    for (Geometry _g : diffGeos) {
                        if (isGeometryCollection(_g))
                            _list.addAll(decomposeGC((GeometryCollection) _g));
                        else
                            _list.add(_g);
                    }
                    //difference 之后若存在多个结果geometry 应该取相互重叠部分
                    Geometry duplicatedGeo = getDuplicatedGeo(_list, getLayerCRS(layerName, dataSource));
                    if (!duplicatedGeo.isEmpty()) {
                        results.clear();
                        Map<String, Object> tmp = new HashMap<String, Object>();
                        tmp.put(SE_SHAPE_AREA, agsGeoemtryService.getGeometryArea(duplicatedGeo.toText()));
                        tmp.put(SE_SHAPE_FIELD, duplicatedGeo.toText());
                        results.add(tmp);
                    }

                } else {
                    if (diffGeos.size() > 0) {
                        Map<String, Object> tmp = new HashMap<String, Object>();
                        tmp.put(SE_SHAPE_AREA, agsGeoemtryService.getGeometryArea(diffGeos.get(0).toText()));
                        tmp.put(SE_SHAPE_FIELD, diffGeos.get(0).toText());
                        results.add(tmp);
                    }
                }
            } else {
                throw new RuntimeException("only support geometry!");
            }
            return results;
        } catch (RuntimeException e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getMessage());
        }
    }

    /**
     * 相交分析
     *
     * @param layerName
     * @param geoJson
     * @param outField
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersectByGeoJSON(String layerName, String geoJson, String[] outField, String dataSource) {
        Object geo = geometryService.readUnTypeGeoJSON(geoJson);
        CoordinateReferenceSystem sourceCRS = null;
        List results = null;
        if (geo instanceof Geometry) {
            Geometry geometry = (Geometry) geo;
            if (!geometry.isValid())
                geometry = createValidGeometry(geometry);
            sourceCRS = geometry.getSRID() != 0 ? geometryService.getCRSBySRID(String.valueOf(geometry.getSRID())) : null;
            results = intersect(layerName, geometry, sourceCRS, outField, dataSource);
        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            results = addFeaturePros2List((List<Map>) intersectBySimpleFeature(layerName, feature, outField, dataSource), feature);
        } else if (geo instanceof FeatureCollection) {
            FeatureCollection collection = (FeatureCollection) geo;
            results = new ArrayList();
            FeatureIterator iterator = ((FeatureCollection) geo).features();
            while (iterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) iterator.next();
                results.addAll(addFeaturePros2List((List<Map>) intersectBySimpleFeature(layerName, feature, outField, dataSource), feature));
            }
        }
        return results;
    }


    /**
     * 改进的相交分析，为了减少土地利用现状分析的误差，需考虑将原始的分析区域的坐标保留，便于后续在平面参考下进行相切
     *
     * @param layerName
     * @param geoJson
     * @param outField
     * @param dataSource
     * @return
     */
    public List<?> intersect4(String layerName, String geoJson, String[] outField, String dataSource) {
        Object geo = geometryService.readUnTypeGeoJSON(geoJson);
        CoordinateReferenceSystem sourceCrs = null;
        CoordinateReferenceSystem layerCrs = spatialDao.getLayerCRS(layerName, dataSource);
        List results = null;
        if (geo instanceof Geometry) {
            Geometry geometry = (Geometry) geo;
            if (!geometry.isValid())
                geometry = createValidGeometry(geometry);
            results = bufferIntersect(layerName, geometry, layerCrs, outField, dataSource);
        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            Geometry geometry = (Geometry) feature.getDefaultGeometry();
            if (!geometry.isValid())
                geometry = createValidGeometry(geometry);
            sourceCrs = feature.getFeatureType().getCoordinateReferenceSystem();
            setFeaturePros2Geo(feature, geometry);
            results = addFeaturePros2List((List<Map>) bufferIntersect(layerName, geometry, sourceCrs, outField, dataSource), feature);
        } else if (geo instanceof FeatureCollection) {
            results = new ArrayList();
            FeatureIterator iterator = ((FeatureCollection) geo).features();
            while (iterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) iterator.next();
                Geometry geometry = (Geometry) feature.getDefaultGeometry();
                if (!geometry.isValid())
                    geometry = createValidGeometry(geometry);
                sourceCrs = feature.getFeatureType().getCoordinateReferenceSystem();
                setFeaturePros2Geo(feature, geometry);
                results.addAll(addFeaturePros2List((List<Map>) bufferIntersect(layerName, geometry, sourceCrs, outField, dataSource), feature));

            }
        }
        return results;
    }

    /**
     * 为了提高土地利用现状分析的精确度，将经纬度进行相交搜索的时候，做了buffer，扩大搜索范围
     *
     * @param layerName
     * @param sourceGeometry
     * @param sourceCRS
     * @param outFields
     * @param dataSource
     * @return
     */
    public List<?> bufferIntersect(String layerName, Geometry sourceGeometry, CoordinateReferenceSystem sourceCRS, String[] outFields, String dataSource) {

        List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        String regionField = null;
        if (layerCRS instanceof GeographicCRS) {
            LayerRegion layerRegion = geometryService.getLayerRegion(layerName);
            if (!isNull(layerRegion.getSourceLayerCRS()))
                sourceCRS = layerRegion.getSourceLayerCRS();
            else if (!isNull(layerRegion.getRegionField())) {
                regionField = layerRegion.getRegionField();
                if (!checkFieldInLayer(regionField, layerName, dataSource))
                    throw new RuntimeException(getMessage("field.not.in.layer", regionField, layerName));
                if (!isNull(outFields) && !ArrayUtils.contains(outFields, regionField, true))
                    outFields = ArrayUtils.add2Arrays(outFields, regionField);
            }
        }

        if (sourceGeometry instanceof GeometryCollection) {
            for (int i = 0; i < sourceGeometry.getNumGeometries(); i++) {
                try {
                    Geometry beforeGeo = sourceGeometry.getGeometryN(i);
                    Geometry afterGeometry = null;
                    if (!isNull(layerCRS))
                        afterGeometry = geometryService.project(beforeGeo, sourceCRS, layerCRS);
                    else
                        afterGeometry = beforeGeo;
                    Geometry tmpGeo = afterGeometry.buffer(0.002);
                    List queryResults = query(layerName, tmpGeo, outFields, dataSource);

                    CoordinateReferenceSystem tmpProjectedCRS = null;
                    if (sourceCRS instanceof GeographicCRS && layerCRS instanceof GeographicCRS) {

                        if (queryResults.size() > 0)
                            tmpProjectedCRS = geometryService.getCRSByRegionCode(String.valueOf(((Map) queryResults.get(0)).get(regionField)));
                        beforeGeo = geometryService.project(beforeGeo, layerCRS, isNull(tmpProjectedCRS) ? layerCRS : tmpProjectedCRS);
                    }

                    Map<String, Object> result = null;
                    for (int j = 0; j < queryResults.size(); j++) {
                        Map map = (Map) queryResults.get(j);
                        Geometry geo1 = geometryService.readWKT(map.get(SE_SHAPE_FIELD).toString());
                        if (layerCRS instanceof GeographicCRS) {
                            geo1 = geometryService.project(geo1, layerCRS, isNull(tmpProjectedCRS) ? sourceCRS : tmpProjectedCRS);
                        }
                        Geometry geometryResult = null;
                        try {
                            geometryResult = geometryService.readWKT(agsGeoemtryService.intersection(geo1.toText(), beforeGeo.toText(), sourceCRS.toWKT()));
                        } catch (Exception e) {
                            logger.error(" geometry is invalid,[{}]", e.getLocalizedMessage());
                            geometryResult = geometryService.forceSimplify(geo1, geometryService.getSimplifyTolerance()).intersection(beforeGeo);
                        }
                        if (geometryResult.isEmpty())
                            continue;
                        result = new HashMap<String, Object>();
                        result.putAll(map);
                        result.put(SE_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geometryResult.toText()));
                        if (layerCRS instanceof GeographicCRS) {
                            try {
                                geometryResult = geometryService.project(geometryResult, isNull(tmpProjectedCRS) ? sourceCRS : tmpProjectedCRS, layerCRS);
                            } catch (GeometryServiceException e) {
                                geometryResult = geometryService.simplify(geometryResult, geometryService.getSimplifyTolerance());
                                geometryResult = geometryService.project(geometryResult, isNull(tmpProjectedCRS) ? sourceCRS : tmpProjectedCRS, layerCRS);
                            }
                        }
                        result.put(SE_SHAPE_FIELD, geometryResult.toText());
                        result.put(Constant.ORIGINAL_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geo1.toText()));
                        results.add(result);
                    }
                } catch (Exception e) {
                    logger.error(e.getLocalizedMessage());
                }
            }
        } else {
            Geometry afterGeometry = null;
            if (!isNull(layerCRS))
                afterGeometry = geometryService.project(sourceGeometry, sourceCRS, layerCRS);
            else
                afterGeometry = sourceGeometry;
            Geometry tmpGeo = afterGeometry.buffer(0.002);
            List queryResults = query(layerName, tmpGeo, outFields, dataSource);

            CoordinateReferenceSystem tmpProjectedCRS = null;
            if (sourceCRS instanceof GeographicCRS && layerCRS instanceof GeographicCRS) {
                if (queryResults.size() > 0)
                    tmpProjectedCRS = geometryService.getCRSByRegionCode(String.valueOf(((Map) queryResults.get(0)).get(regionField))); //
                sourceGeometry = geometryService.project(sourceGeometry, layerCRS, isNull(tmpProjectedCRS) ? layerCRS : tmpProjectedCRS);
            }
            Map<String, Object> result = null;
            for (int j = 0; j < queryResults.size(); j++) {
                Map map = (Map) queryResults.get(j);
                Geometry geo1 = geometryService.readWKT(map.get(SE_SHAPE_FIELD).toString());
                if (layerCRS instanceof GeographicCRS) {
                    geo1 = geometryService.project(geo1, layerCRS, isNull(tmpProjectedCRS) ? sourceCRS : tmpProjectedCRS);
                }
                Geometry geometryResult = null;
                try {
                    geometryResult = geometryService.readWKT(agsGeoemtryService.intersection(geo1.toText(), sourceGeometry.toText(), sourceCRS.toWKT()));
                } catch (Exception e) {
                    logger.error(" geometry is invalid,[{}]", e.getLocalizedMessage());
                    geometryResult = geometryService.forceSimplify(geo1, geometryService.getSimplifyTolerance()).intersection(sourceGeometry);
                }
                if (geometryResult.isEmpty())
                    continue;
                result = new HashMap<String, Object>();
                result.putAll(map);
                result.put(SE_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geometryResult.toText()));
                if (layerCRS instanceof GeographicCRS) {
                    try {
                        geometryResult = geometryService.project(geometryResult, isNull(tmpProjectedCRS) ? sourceCRS : tmpProjectedCRS, layerCRS);
                    } catch (GeometryServiceException e) {
                        geometryResult = geometryService.simplify(geometryResult, geometryService.getSimplifyTolerance());
                        geometryResult = geometryService.project(geometryResult, isNull(tmpProjectedCRS) ? sourceCRS : tmpProjectedCRS, layerCRS);
                    }
                }
                result.put(SE_SHAPE_FIELD, geometryResult.toText());
                result.put(Constant.ORIGINAL_SHAPE_AREA, agsGeoemtryService.getGeometryArea(geo1.toText()));
                results.add(result);
            }
        }

        return results;
    }


    /**
     * @param layerName
     * @param feature
     * @param outField
     * @param dataSource
     * @return
     */
    private List<?> intersectBySimpleFeature(String layerName, SimpleFeature feature, String[] outField, String dataSource) {
        CoordinateReferenceSystem sourceCRS = null;
        Geometry geometry = (Geometry) feature.getDefaultGeometry();
        if (!geometry.isValid())
            geometry = createValidGeometry(geometry);
        setFeaturePros2Geo(feature, geometry);
        try {
            sourceCRS = feature.getFeatureType().getCoordinateReferenceSystem();
            return intersect(layerName, geometry, sourceCRS, outField, dataSource);
        } catch (Exception e) {
            logger.error("intersect analysis - feature json crs not define [{}]", e.getLocalizedMessage());
        }
        return null;
    }

    /**
     * 相交分析
     *
     * @param layerName
     * @param geoJson
     * @param outField
     * @param dataSource
     * @return GeoJSON 格式要素集 {"type":"FeatureCollection","features":[{"type":"Feature","geometry":{"type":"Polygon","coordinates":[[]]},"properties":{}}}
     */
    @Override
    public List<?> intersect2(String layerName, String geoJson, String[] outField, String dataSource) {
        List results = intersectByGeoJSON(layerName, geoJson, outField, dataSource);
//        FeatureCollection collection = geometryService.list2FeatureCollection(results, null, null);
        return results;
    }

    /**
     * 相交分析(New)
     *
     * @param layerName
     * @param geoJson
     * @param outField
     * @param dataSource
     * @return
     */
    @Override
    public List<?> intersect3(String layerName, String geoJson, String[] outField, String dataSource) {
        Object geo = geometryService.readUnTypeGeoJSON(geoJson);
        CoordinateReferenceSystem sourceCrs = null;
        CoordinateReferenceSystem layerCrs = spatialDao.getLayerCRS(layerName, dataSource);
        List results = null;
        if (geo instanceof Geometry) {
            Geometry geometry = (Geometry) geo;
            if (!geometry.isValid())
                geometry = createValidGeometry(geometry);
            results = intersect(layerName, geometry, outField, dataSource);
        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            Geometry geometry = (Geometry) feature.getDefaultGeometry();
            if (!geometry.isValid() && !isNull(geometryService.validGeometry(geometry)))
                geometry = createValidGeometry(geometry);
            sourceCrs = feature.getFeatureType().getCoordinateReferenceSystem();
            if (!isNull(sourceCrs) && !isNull(layerCrs) && !sourceCrs.equals(layerCrs))
                geometry = geometryService.project(geometry, sourceCrs, layerCrs);
            setFeaturePros2Geo(feature, geometry);
            results = addFeaturePros2List((List<Map>) intersect(layerName, geometry, outField, dataSource), feature);
        } else if (geo instanceof FeatureCollection) {
            results = new ArrayList();
            FeatureIterator iterator = ((FeatureCollection) geo).features();
            while (iterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) iterator.next();
                Geometry geometry = (Geometry) feature.getDefaultGeometry();
                if (!geometry.isValid() && !isNull(geometryService.validGeometry(geometry)))
                    geometry = createValidGeometry(geometry);
                sourceCrs = feature.getFeatureType().getCoordinateReferenceSystem();
                if (!isNull(sourceCrs) && !isNull(layerCrs) && !sourceCrs.equals(layerCrs))
                    geometry = geometryService.project(geometry, sourceCrs, layerCrs);
                setFeaturePros2Geo(feature, geometry);
                results.addAll(addFeaturePros2List((List<Map>) intersect(layerName, geometry, outField, dataSource), feature));
//                results.addAll(intersect(layerName, geometry, outField, dataSource));
            }
        }
        return results;
    }

    /**
     * Computes a Geometry representing the points making up this Geometry that do not make up other.
     *
     * @param layerName
     * @param geoJson
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> differenceByGeoJson(String layerName, String geoJson, String[] outFields, String dataSource) {
        Object geo = geometryService.readUnTypeGeoJSON(geoJson);
        CoordinateReferenceSystem sourceCrs = null;
        CoordinateReferenceSystem layerCrs = spatialDao.getLayerCRS(layerName, dataSource);
        List results = null;
        if (geo instanceof Geometry) {
            Geometry geometry = (Geometry) geo;
            if (!geometry.isValid())
                geometry = createValidGeometry(geometry);
            results = difference(layerName, geometry, outFields, dataSource);
        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            Geometry geometry = (Geometry) feature.getDefaultGeometry();
            if (!geometry.isValid())
                geometry = createValidGeometry(geometry);
            sourceCrs = feature.getFeatureType().getCoordinateReferenceSystem();
            if (!isNull(sourceCrs) && !isNull(layerCrs) && !sourceCrs.equals(layerCrs))
                geometry = geometryService.project(geometry, sourceCrs, layerCrs);
            setFeaturePros2Geo(feature, geometry);
            results = addFeaturePros2List((List<Map>) difference(layerName, geometry, outFields, dataSource), feature);
        } else if (geo instanceof FeatureCollection) {
            results = new ArrayList();
            FeatureIterator iterator = ((FeatureCollection) geo).features();
            while (iterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) iterator.next();
                Geometry geometry = (Geometry) feature.getDefaultGeometry();
                if (!geometry.isValid())
                    geometry = createValidGeometry(geometry);
                sourceCrs = feature.getFeatureType().getCoordinateReferenceSystem();
                if (!isNull(sourceCrs) && !isNull(layerCrs) && !sourceCrs.equals(layerCrs))
                    geometry = geometryService.project(geometry, sourceCrs, layerCrs);
                setFeaturePros2Geo(feature, geometry);
                results.addAll(addFeaturePros2List((List<Map>) difference(layerName, geometry, outFields, dataSource), feature));
            }
        }
        return results;
    }

    /**
     * 创建新的有效的geometry
     *
     * @param geometry
     * @return
     */
    public Geometry createValidGeometry(Geometry geometry) {
        try {
            if (geometry instanceof Polygon) {
                Polygon polygon = (Polygon) geometry;
                GeometryFactory factory = geometry.getFactory();
                List<Polygon> polygons = new ArrayList<Polygon>();
                Polygon exteriorPolygon = new Polygon((LinearRing) polygon.getExteriorRing(), null, factory);
                polygons.add(exteriorPolygon);
                for (int i = 0; i < polygon.getNumInteriorRing(); i++) {
                    LinearRing interiorRing = (LinearRing) polygon.getInteriorRingN(i);
                    Polygon interiorPolygon = new Polygon(interiorRing, null, factory);
                    polygons.add(interiorPolygon);
                }
                List<Polygon> newPolygons = new ArrayList<Polygon>();
                List<Polygon> excludePolygons = new ArrayList<Polygon>();
                for (Polygon temp : polygons) {
                    temp = getPolygonWithHoles(polygons, excludePolygons, temp);
                    if (!excludePolygons.contains(temp))
                        newPolygons.add(temp);
                }
                return new MultiPolygon(newPolygons.toArray(new Polygon[0]), factory);
            } else if (geometry instanceof MultiPolygon) {
                MultiPolygon multiPolygon = (MultiPolygon) geometry;
                List<Polygon> polygonList = new ArrayList<Polygon>();
                for (int j = 0; j < multiPolygon.getNumGeometries(); j++) {

                    Polygon polygon = (Polygon) multiPolygon.getGeometryN(j);
                    if (!polygon.isValid()) {
                        MultiPolygon tempMultiPolygon = (MultiPolygon) createValidGeometry(polygon);
                        for (int k = 0; k < tempMultiPolygon.getNumGeometries(); k++) {
                            polygonList.add((Polygon) tempMultiPolygon.getGeometryN(k));
                        }
                    } else {
                        polygonList.add(polygon);

                    }
                }
                return new MultiPolygon(polygonList.toArray(new Polygon[0]), multiPolygon.getFactory());
            } else {
                logger.info("geometryType has not been supported yet");
            }
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
        }
        return null;
    }

    /**
     * 判断一个polygon在polygon组中是否是含有环的polygon
     *
     * @param srcPolygons     原始polygon组
     * @param excludePolygons 内环polygon需排除
     * @param polygon
     * @return
     */
    private Polygon getPolygonWithHoles(List<Polygon> srcPolygons, List<Polygon> excludePolygons, Polygon polygon) {
        List<LinearRing> holes = new ArrayList<LinearRing>();
        for (Polygon item : srcPolygons) {
            if (polygon.equals(item) || excludePolygons.contains(polygon))
                continue;
            if (polygon.contains(item)) {
                holes.add((LinearRing) item.getExteriorRing());
                excludePolygons.add(item);
            }
        }
        if (holes.size() > 0)
            return new Polygon((LinearRing) polygon.getExteriorRing(), holes.toArray(new LinearRing[0]), polygon.getFactory());
        else
            return polygon;
    }


    /**
     * @param value
     * @param feature
     * @return
     */
    private List addFeaturePros2List(List<Map> value, SimpleFeature feature) {
        for (Map item : value) {
            for (Property p : feature.getProperties()) {
                if (p.getName().getLocalPart().equals("geometry") || p.getName().getLocalPart().equals("crs")) continue;
                item.put(OG_PRO_PERFIX.concat(p.getName().getLocalPart()), p.getValue());
            }
        }
        return value;
    }

    @Override
    public int insertRows(String layerName, List<Map<String, Object>> rows, String dataSource) {
        LayerRegion layerRegion = geometryService.getLayerRegion(layerName);
        List<Map<String, Object>> regionRows = new ArrayList<Map<String, Object>>();
        int result = 0;
        if (!isNull(layerRegion.getRegionField())) {
            int valid = 0;
            for (Map<String, Object> columns : rows) {
                boolean contains = false;
                for (Map.Entry entry : columns.entrySet()) {
                    if (layerRegion.getRegionField().equalsIgnoreCase((String) entry.getKey())) {
                        contains = true;
                        if (geometryService.containsRegionValue(String.valueOf(entry.getValue()))) {
                            regionRows.add(columns);
                            break;
                        } else {
                            logger.warn(getMessage("insert.region.value.not.exist", layerName, layerRegion.getRegionField(), String.valueOf(entry.getValue())));
                            valid++;
                            break;
                        }
                    }
                }
                if (!contains) {
                    throw new RuntimeException(getMessage("insert.region.field.not.null", layerName, layerRegion.getRegionField()));
                }
            }
            if (regionRows.size() > 0) {
                logger.info("跳过" + valid + "条非法数据，实际准备插入" + regionRows.size() + "条数据！");
                result = spatialDao.insertRows(layerName, regionRows, dataSource);
                logger.info("插入" + result + "条数据！");
            } else {
                logger.warn("跳过" + valid + "条非法数据，实际准备插入" + regionRows.size() + "条数据！");
            }
        }
        return result;
    }

    /**
     * 插入数据
     *
     * @param layerName
     * @param columns    包含图形、属性数据
     * @param dataSource
     * @return
     */
    @Override
    public String insert(String layerName, Map<String, Object> columns, String dataSource) {
        LayerRegion layerRegion = geometryService.getLayerRegion(layerName);
        if (!isNull(layerRegion.getRegionField())) {
            boolean contains = false;
            for (Map.Entry entry : columns.entrySet()) {
                if (layerRegion.getRegionField().equalsIgnoreCase((String) entry.getKey())) {
                    if (geometryService.containsRegionValue(String.valueOf(entry.getValue()))) {
                        contains = true;
                        break;
                    } else
                        throw new RuntimeException(getMessage("insert.region.value.not.exist", layerName, layerRegion.getRegionField(), String.valueOf(entry.getValue())));
                }
            }
            if (!contains)
                throw new RuntimeException(getMessage("insert.region.field.not.null", layerName, layerRegion.getRegionField()));
        }
        return spatialDao.insert(layerName, columns, dataSource);
    }

    /**
     * 插入数据
     *
     * @param layerName
     * @param geoJSON    GeoJSON 格式
     * @param dataSource
     * @return primaryKey
     */
    @Override
    public String insert(String layerName, String geoJSON, String dataSource) {
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        Object geo = geometryService.readUnTypeGeoJSON(geoJSON);
        try {
            if (geo instanceof FeatureCollection) {
                Map<String, Object> columns = null;
                SimpleFeatureCollection featureCollection = (SimpleFeatureCollection) geo;
                FeatureIterator featureIterator = featureCollection.features();
                while (featureIterator.hasNext()) {
                    SimpleFeature feature = (SimpleFeature) featureIterator.next();
                    CoordinateReferenceSystem sourceCRS = geometryService.readFeatureJSONCRS(geometryService.toFeatureJSON(feature));
                    columns = geometryService.simpleFeature2Map(feature); //转map
                    for (Map.Entry<String, Object> entry : columns.entrySet()) {
                        if (entry.getValue() instanceof Geometry) {
                            Geometry geometry = (Geometry) entry.getValue();
                            if (layerCRS != null && sourceCRS != null)
                                geometry = geometryService.project(geometry, sourceCRS, layerCRS);
                            columns.put(SE_SHAPE_FIELD, geometry.toText());
                            columns.remove(entry.getKey());
                        }
                    }
                    if (featureIterator.hasNext())
                        insert(layerName, columns, dataSource);
                    else
                        return insert(layerName, columns, dataSource);
                }
            }
        } catch (Exception e) {
            logger.info(getMessage("insert.sde.false"), e.getLocalizedMessage());
        }
        return insert(layerName, geoJSON2Map(geoJSON, layerCRS), dataSource);
    }

    /**
     * 插入数据（返回geojson）
     *
     * @param layerName
     * @param geoJSON
     * @param dataSource
     * @return
     */
    @Override
    public String insert2(String layerName, String geoJSON, String dataSource) {
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        Object geo = geometryService.readUnTypeGeoJSON(geoJSON);
        if (geo instanceof FeatureCollection) {
            LinkedHashMap<String, Object> resultFc = new LinkedHashMap<String, Object>();
            resultFc.put("type", "FeatureCollection");
            List<LinkedHashMap> resultFeatures = new ArrayList();
            Map<String, Object> columns = null;
            SimpleFeatureCollection featureCollection = (SimpleFeatureCollection) geo;
            FeatureIterator featureIterator = featureCollection.features();
            while (featureIterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) featureIterator.next();
                CoordinateReferenceSystem sourceCRS;
                sourceCRS = feature.getFeatureType().getCoordinateReferenceSystem();
                LinkedHashMap featureMap = JSON.parseObject(geometryService.toFeatureJSON(feature), LinkedHashMap.class);
                JSONObject properties = (JSONObject) featureMap.get("properties");
                if (isNull(sourceCRS))
                    sourceCRS = geometryService.readFeatureJSONCRS(geometryService.toFeatureJSON(feature));
                columns = geometryService.simpleFeature2Map(feature);
                for (String key : columns.keySet()) {
                    if (columns.get(key) instanceof Geometry) {
                        Geometry geometry = (Geometry) columns.get(key);
                        if (layerCRS != null && sourceCRS != null)
                            geometry = geometryService.project(geometry, sourceCRS, layerCRS);
                        columns.put(SE_SHAPE_FIELD, geometry.toText());
                        columns.remove(key);
                        break;
                    }
                }
                if (createDateAuto) {
                    if (isNull(insertDateField)) {
                        logger.error(getMessage("insert.createAt.field.null"));
                        throw new RuntimeException(getMessage("insert.createAt.field.null"));
                    } else {
                        columns.put(insertDateField, new Date());
                        properties.put(insertDateField, sdf.format(new Date()));
                    }
                }
                if (regionCodeAuto) {
                    if (isNull(insertRegionField))
                        throw new RuntimeException(getMessage("insert.region.field.null"));
                    else {
                        String regionValue = findXzqdm(feature, String.valueOf(AppPropertyUtils.getAppEnv("insert.region.layer")));
                        if (!isNull(regionValue)) {
                            List<String> rFields = Arrays.asList(insertRegionField.split(","));
                            for (String f : rFields) {
                                columns.put(f, regionValue);
                                properties.put(f, regionValue);
                            }
                        }
                    }
                }
                String objectId = insert(layerName, columns, dataSource);
                properties.put("OBJECTID", objectId);
                resultFeatures.add(featureMap);
            }
            resultFc.put("features", resultFeatures);
            return JSON.toJSONString(resultFc);

        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            LinkedHashMap featureMap = JSON.parseObject(geometryService.toFeatureJSON(feature), LinkedHashMap.class);
            JSONObject properties = (JSONObject) featureMap.get("properties");
            Map<String, Object> columns = geoJSON2Map(geoJSON, layerCRS);
            if (createDateAuto) {
                if (isNull(insertDateField)) {
                    logger.error(getMessage("insert.createAt.field.null"));
                    throw new RuntimeException(getMessage("insert.createAt.field.null"));
                } else {
                    columns.put(insertDateField, new Date());
                    properties.put(insertDateField, sdf.format(new Date()));
                }
            }
            if (regionCodeAuto) {
                if (isNull(insertRegionField))
                    throw new RuntimeException(getMessage("insert.region.field.null"));
                else {
                    String regionValue = findXzqdm(feature, String.valueOf(AppPropertyUtils.getAppEnv("insert.region.layer")));
                    if (!isNull(regionValue)) {
                        List<String> rFields = Arrays.asList(insertRegionField.split(","));
                        for (String f : rFields) {
                            columns.put(f, regionValue);
                            properties.put(f, regionValue);
                        }
                    }
                }
            }
            String objectId = insert(layerName, columns, dataSource);
            properties.put("OBJECTID", objectId);

            return JSON.toJSONString(featureMap);
        }
        return null;

    }

    /**
     * @param layerName
     * @param geoJSON
     * @param check      是否执行拓扑检查
     * @param dataSource
     * @return
     */
    @Override
    public String insert2(String layerName, String geoJSON, boolean check, String dataSource) {
        if (check) {
            CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
            Object geo = geometryService.readUnTypeGeoJSON(geoJSON);
            if (geo instanceof FeatureCollection) {
                LinkedHashMap<String, Object> resultFc = new LinkedHashMap<String, Object>();
                resultFc.put("type", "FeatureCollection");
                List<LinkedHashMap> resultFeatures = new ArrayList();
                Map<String, Object> columns = null;
                SimpleFeatureCollection featureCollection = (SimpleFeatureCollection) geo;
                FeatureIterator featureIterator = featureCollection.features();
                while (featureIterator.hasNext()) {
                    SimpleFeature feature = (SimpleFeature) featureIterator.next();
                    List list = query(layerName, feature, null, dataSource);
                    if (list.size() > 0) {
                        logger.error(getMessage("insert.check.fail", geometryService.toFeatureJSON(feature)));
                        if (featureIterator.hasNext())   //提示错误后 继续下面的insert
                            continue;
                        else
                            throw new GISDaoException(GISDaoException.Method.INSERT, getMessage("insert.check.fail", geometryService.toFeatureJSON(feature)), GISDaoException.Type.ARC_SDE);
                    }
                    logger.debug("[insert-- start getting sourceCRS...]");
                    CoordinateReferenceSystem sourceCRS = feature.getFeatureType().getCoordinateReferenceSystem();
                    LinkedHashMap featureMap = JSON.parseObject(geometryService.toFeatureJSON(feature), LinkedHashMap.class);
                    JSONObject properties = (JSONObject) featureMap.get("properties");
                    if (isNull(sourceCRS))
                        sourceCRS = geometryService.readFeatureJSONCRS(geometryService.toFeatureJSON(feature));
                    if (isNull(sourceCRS))
                        logger.debug("[insert sourceCRS:]null");
                    else
                        logger.debug("[insert sourceCRS:]" + sourceCRS.toWKT());
                    columns = geometryService.simpleFeature2Map(feature);
                    try {
                        for (String key : columns.keySet()) {
                            if (columns.get(key) instanceof Geometry) {
                                Geometry geometry = (Geometry) columns.get(key);
                                if (layerCRS != null && sourceCRS != null)
                                    geometry = geometryService.project(geometry, sourceCRS, layerCRS);
                                columns.put(SE_SHAPE_FIELD, geometry.toText());
                                columns.remove(key);
                                break;
                            }
                        }
                    } catch (Exception e) {
                        logger.error(e.getLocalizedMessage());
                        throw new RuntimeException(e.getLocalizedMessage());
                    }
                    if (createDateAuto) {
                        if (isNull(insertDateField)) {
                            logger.error(getMessage("insert.createAt.field.null"));
                            throw new RuntimeException(getMessage("insert.createAt.field.null"));
                        } else {
                            columns.put(insertDateField, new Date());
                            properties.put(insertDateField, sdf.format(new Date()));
                        }
                    }
                    if (regionCodeAuto) {
                        if (isNull(insertRegionField))
                            throw new RuntimeException(getMessage("insert.region.field.null"));
                        else {
                            String regionValue = findXzqdm(feature, String.valueOf(AppPropertyUtils.getAppEnv("insert.region.layer")));
                            if (isNotNull(regionValue)) {
                                List<String> rFields = Arrays.asList(insertRegionField.split(","));
                                for (String f : rFields) {
                                    columns.put(f, regionValue);
                                    properties.put(f, regionValue);
                                }
                            } else {
                                logger.error(getMessage("insert.region.value.null"));
                                throw new RuntimeException(getMessage("insert.region.value.null"));
                            }
                        }
                    }
                    String objectId = insert(layerName, columns, dataSource);
                    properties.put("OBJECTID", objectId);
                    resultFeatures.add(featureMap);
                }
                resultFc.put("features", resultFeatures);
                return JSON.toJSONString(resultFc);

            } else if (geo instanceof SimpleFeature) {

                SimpleFeature feature = (SimpleFeature) geo;
                List list = query(layerName, feature, null, dataSource);
                if (list.size() > 0) {
                    logger.error("insert.check.fail", geoJSON);
                    throw new GISDaoException(GISDaoException.Method.INSERT, getMessage("insert.check.fail", geoJSON), GISDaoException.Type.ARC_SDE);
                }
                LinkedHashMap featureMap = JSON.parseObject(geometryService.toFeatureJSON(feature), LinkedHashMap.class);
                JSONObject properties = (JSONObject) featureMap.get("properties");
                Map<String, Object> columns = geoJSON2Map(geoJSON, layerCRS);
                if (createDateAuto) {
                    if (isNull(insertDateField)) {
                        logger.error(getMessage("insert.createAt.field.null"));
                        throw new RuntimeException(getMessage("insert.createAt.field.null"));
                    } else {
                        columns.put(insertDateField, new Date());
                        properties.put(insertDateField, sdf.format(new Date()));
                    }
                }
                if (regionCodeAuto) {
                    if (isNull(insertRegionField))
                        throw new RuntimeException(getMessage("insert.region.field.null"));
                    else {
                        String regionValue = findXzqdm(feature, String.valueOf(AppPropertyUtils.getAppEnv("insert.region.layer")));
                        if (!isNull(regionValue)) {
                            List<String> rFields = Arrays.asList(insertRegionField.split(","));
                            for (String f : rFields) {
                                columns.put(f, regionValue);
                                properties.put(f, regionValue);
                            }
                        } else {
                            logger.error(getMessage("insert.region.value.null"));
                            throw new RuntimeException(getMessage("insert.region.value.null"));
                        }
                    }
                }
                String objectId = insert(layerName, columns, dataSource);
                properties.put("OBJECTID", objectId);
                return JSON.toJSONString(featureMap);
            }

        } else {
            return insert2(layerName, geoJSON, dataSource);
        }

        return null;
    }

    /**
     * 插入要素3
     *
     * @param layerName
     * @param geoJSON
     * @param createAt
     * @param dataSource
     * @return
     */
    @Deprecated
    public String insert3(String layerName, String geoJSON, boolean createAt, String dataSource) {

        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        Object geo = geometryService.readUnTypeGeoJSON(geoJSON);
        if (geo instanceof FeatureCollection) {
            LinkedHashMap<String, Object> resultFc = new LinkedHashMap<String, Object>();
            resultFc.put("type", "FeatureCollection");
            List<LinkedHashMap> resultFeatures = new ArrayList();
            Map<String, Object> columns = null;
            SimpleFeatureCollection featureCollection = (SimpleFeatureCollection) geo;
            FeatureIterator featureIterator = featureCollection.features();
            while (featureIterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) featureIterator.next();
                CoordinateReferenceSystem sourceCRS;
                sourceCRS = feature.getFeatureType().getCoordinateReferenceSystem();
                if (isNull(sourceCRS))
                    sourceCRS = geometryService.readFeatureJSONCRS(geometryService.toFeatureJSON(feature));
                columns = geometryService.simpleFeature2Map(feature);
                for (String key : columns.keySet()) {
                    if (columns.get(key) instanceof Geometry) {
                        Geometry geometry = (Geometry) columns.get(key);
                        if (layerCRS != null && sourceCRS != null)
                            geometry = geometryService.project(geometry, sourceCRS, layerCRS);
                        columns.put(SE_SHAPE_FIELD, geometry.toText());
                        columns.remove(key);
                        break;
                    }
                }
                if (createAt) {
                    if (isNull(insertDateField)) {
                        logger.error(getMessage("insert.createAt.field.null"));
                        throw new RuntimeException(getMessage("insert.createAt.field.null"));
                    } else {
                        columns.put(insertDateField, new Date());
                    }
                }
                String objectId = insert(layerName, columns, dataSource);
                LinkedHashMap featureMap = JSON.parseObject(geometryService.toFeatureJSON(feature), LinkedHashMap.class);
                JSONObject properties = (JSONObject) featureMap.get("properties");
                properties.put("OBJECTID", objectId);
                resultFeatures.add(featureMap);
            }
            resultFc.put("features", resultFeatures);
            return JSON.toJSONString(resultFc);

        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            LinkedHashMap featureMap = JSON.parseObject(geometryService.toFeatureJSON(feature), LinkedHashMap.class);
            JSONObject properties = (JSONObject) featureMap.get("properties");
            Map<String, Object> columns = geoJSON2Map(geoJSON, layerCRS);
            if (createAt) {
                if (isNull(insertDateField)) {
                    logger.error(getMessage("insert.createAt.field.null"));
                    throw new RuntimeException(getMessage("insert.createAt.field.null"));
                } else {
                    columns.put(insertDateField, new Date());
                }
            }
            String objectId = insert(layerName, columns, dataSource);
            properties.put("OBJECTID", objectId);
            return JSON.toJSONString(featureMap);
        }
        return null;
    }

    /**
     * 插入要素3
     *
     * @param layerName  sde图层名
     * @param geoJSON    插入要素的geojson
     * @param check      是否进行拓扑检查
     * @param createAt   是否增加要素插入时间
     * @param dataSource sde数据源
     * @return
     */
    @Override
    public String insert3(String layerName, String geoJSON, boolean check, boolean createAt, String dataSource) {
        if (isNull(check))
            check = false;
        if (isNull(createAt))
            createAt = false;
        if (check) {
            CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
            Object geo = geometryService.readUnTypeGeoJSON(geoJSON);
            if (geo instanceof FeatureCollection) {

                LinkedHashMap<String, Object> resultFc = new LinkedHashMap<String, Object>();
                resultFc.put("type", "FeatureCollection");
                List<LinkedHashMap> resultFeatures = new ArrayList();
                Map<String, Object> columns = null;
                SimpleFeatureCollection featureCollection = (SimpleFeatureCollection) geo;
                FeatureIterator featureIterator = featureCollection.features();
                while (featureIterator.hasNext()) {
                    SimpleFeature feature = (SimpleFeature) featureIterator.next();
                    List list = query(layerName, feature, null, dataSource);
                    if (list.size() > 0) {
                        logger.error(getMessage("insert.check.fail", geometryService.toFeatureJSON(feature)));
                        if (featureIterator.hasNext())   //提示错误后 继续下面的insert
                            continue;
                        else
                            throw new GISDaoException(GISDaoException.Method.INSERT, getMessage("insert.check.fail", geometryService.toFeatureJSON(feature)), GISDaoException.Type.ARC_SDE);
                    }
                    CoordinateReferenceSystem sourceCRS = feature.getFeatureType().getCoordinateReferenceSystem();
                    if (isNull(sourceCRS))
                        sourceCRS = geometryService.readFeatureJSONCRS(geometryService.toFeatureJSON(feature));
                    if (isNull(sourceCRS))
                        logger.debug("[insert sourceCRS:]null");
                    columns = geometryService.simpleFeature2Map(feature);
                    try {
                        for (String key : columns.keySet()) {
                            if (columns.get(key) instanceof Geometry) {
                                Geometry geometry = (Geometry) columns.get(key);
                                if (layerCRS != null && sourceCRS != null)
                                    geometry = geometryService.project(geometry, sourceCRS, layerCRS);
                                columns.put(SE_SHAPE_FIELD, geometry.toText());
                                columns.remove(key);
                                break;
                            }
                        }
                    } catch (Exception e) {
                        logger.error(getMessage("insert.shape.error", e.getLocalizedMessage()));
                        throw new RuntimeException(e.getLocalizedMessage());
                    }
                    if (createAt) {
                        if (isNull(insertDateField)) {
                            logger.error(getMessage("insert.createAt.field.null"));
                            throw new RuntimeException(getMessage("insert.createAt.field.null"));
                        } else {
                            columns.put(insertDateField, new Date());
                        }
                    }
                    String objectId = insert(layerName, columns, dataSource);
                    LinkedHashMap featureMap = JSON.parseObject(geometryService.toFeatureJSON(feature), LinkedHashMap.class);
                    JSONObject properties = (JSONObject) featureMap.get("properties");
                    properties.put("OBJECTID", objectId);
                    resultFeatures.add(featureMap);
                }
                resultFc.put("features", resultFeatures);
                return JSON.toJSONString(resultFc);

            } else if (geo instanceof SimpleFeature) {

                SimpleFeature feature = (SimpleFeature) geo;
                List list = query(layerName, feature, null, dataSource);
                if (list.size() > 0) {
                    logger.error("insert.check.fail", geoJSON);
                    throw new GISDaoException(GISDaoException.Method.INSERT, getMessage("insert.check.fail", geoJSON), GISDaoException.Type.ARC_SDE);
                }
                LinkedHashMap featureMap = JSON.parseObject(geometryService.toFeatureJSON(feature), LinkedHashMap.class);
                JSONObject properties = (JSONObject) featureMap.get("properties");
                Map<String, Object> columns = geoJSON2Map(geoJSON, layerCRS);
                if (createAt) {
                    if (isNull(insertDateField)) {
                        logger.error(getMessage("insert.createAt.field.null"));
                        throw new RuntimeException(getMessage("insert.createAt.field.null"));
                    } else {
                        columns.put(insertDateField, new Date());
                    }
                }
                String objectId = insert(layerName, columns, dataSource);
                properties.put("OBJECTID", objectId);
                return JSON.toJSONString(featureMap);
            }

        } else {
            return insert3(layerName, geoJSON, createAt, dataSource);
        }

        return null;
    }

    /**
     * 更新数据
     *
     * @param layerName
     * @param primaryKey
     * @param columns
     * @param dataSource
     * @return
     */
    @Override
    public boolean update(String layerName, String primaryKey, Map<String, Object> columns, String dataSource) {
        return spatialDao.update(layerName, primaryKey, columns, dataSource);
    }

    /**
     * 更新数据
     *
     * @param layerName
     * @param primaryKey
     * @param geoJSON
     * @param dataSource
     * @return
     */
    @Override
    public boolean update(String layerName, String primaryKey, String geoJSON, String dataSource) {
        CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
        return update(layerName, primaryKey, geoJSON2Map(geoJSON, layerCRS), dataSource);
    }

    /**
     * 删除
     *
     * @param layerName
     * @param primaryKey
     * @param dataSource
     * @return
     */
    @Override
    public boolean delete(String layerName, String primaryKey, String dataSource) {
        return spatialDao.delete(layerName, primaryKey, dataSource);
    }

    @Override
    public List<Map> djAnalysis(String geometry, List analysisLayers, String dataSource, Map unit) {
        if (isNull(analysisLayers)) throw new RuntimeException("no analysis Layers");
        if (isNull(geometry)) throw new JSONMessageException("geometry is null");
        if (isNull(unit)) throw new JSONMessageException("unit is null");
        List results = null;
        Object geo = geometryService.readUnTypeGeoJSON(geometry);
        if (geo instanceof FeatureCollection) {
            results = new ArrayList();
            FeatureIterator iterator = ((FeatureCollection) geo).features();
            while (iterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) iterator.next();
                Map map = singleDjAnalysis(feature, analysisLayers, dataSource);
                map.put("title", feature.getAttribute("title"));
                results.add(map);
            }
        } else if (geo instanceof SimpleFeature) {
            results = new ArrayList();
            results.add(singleDjAnalysis((SimpleFeature) geo, analysisLayers, dataSource));
        }
        if (results != null) {
            results = mergeResults(results, unit);
        }
        return results;
    }

    public List<Map> mergeResults(List results, Map unit) {
        Map rMap = new HashMap();
        double conv = UNITS.ACRES.getConv();
        DecimalFormat df_dj = new DecimalFormat("0.0000");//公顷保留4位小数
        if (!isNull(unit)) {
            df_dj.applyPattern(MapUtils.getString(unit, "format", "0.0"));
            String ualias = MapUtils.getString(unit, "alias");
            for (UNITS us : UNITS.values()) {
                if (us.getAlias().equals(ualias)) {
                    conv = us.getConv();
                    break;
                }
            }
        }
        for (int i = 0; i < results.size(); i++) {
            Map map = (Map) results.get(i);
            Map areasMap = (Map) map.get("JBNT");
            Map gdAreasMap = (Map) areasMap.get("gdmap");
            if (!rMap.containsKey("gdmap")) {
                rMap.put("gdmap", gdAreasMap);
            } else {
                Map gdold = (Map) rMap.get("gdmap");
                // 统计耕地
                Iterator iter = gdAreasMap.entrySet().iterator();
                while (iter.hasNext()) {
                    Map.Entry entry = (Map.Entry) iter.next();
                    gdold.put(entry.getKey(), (Double) gdold.get(entry.getKey()) + (Double) entry.getValue());
                }
                rMap.put("gdmap", gdold);
            }
            Map kAreasMap = (Map) areasMap.get("ktmap");
            if (!rMap.containsKey("ktmap")) {
                rMap.put("ktmap", kAreasMap);
            } else {
                // 统计可调整
                Map gdold = (Map) rMap.get("ktmap");
                // 统计耕地
                Iterator iter = kAreasMap.entrySet().iterator();
                while (iter.hasNext()) {
                    Map.Entry entry = (Map.Entry) iter.next();
                    gdold.put(entry.getKey(), (Double) gdold.get(entry.getKey()) + (Double) entry.getValue());
                }
                rMap.put("ktmap", gdold);
            }
        }
        Map gdAreasMap = (Map) rMap.get("gdmap");
        Iterator iter = gdAreasMap.entrySet().iterator();
        Double gdtotal = 0.0;
        while (iter.hasNext()) {
            Map.Entry entry = (Map.Entry) iter.next();
            Double val = Double.parseDouble(df_dj.format(entry.getValue()));
            gdtotal += val;
            gdAreasMap.put(entry.getKey(), val == 0.0 ? "" : df_dj.format(val));
        }
        gdAreasMap.put("total", gdtotal == 0.0 ? "" : df_dj.format(gdtotal));
        rMap.put("gdmap", gdAreasMap);

        Map kAreasMap = (Map) rMap.get("ktmap");
        Iterator iter1 = kAreasMap.entrySet().iterator();
        Double ktotal = 0.0;
        while (iter1.hasNext()) {
            Map.Entry entry = (Map.Entry) iter1.next();
            Double val = Double.parseDouble(df_dj.format(entry.getValue()));
            ktotal += val;
            kAreasMap.put(entry.getKey(), val == 0.0 ? "" : df_dj.format(val));
        }
        kAreasMap.put("total", ktotal == 0.0 ? "" : df_dj.format(ktotal));
        rMap.put("ktmap", kAreasMap);
        rMap.put("total", ktotal + gdtotal == 0.0 ? "" : df_dj.format(ktotal + gdtotal));

        List<Map> result = new ArrayList<Map>();
        result.add(rMap);
        return result;
    }

    public Map singleDjAnalysis(SimpleFeature feature, List analysisLayers, String dataSource) {
        Map analysisMap = new HashMap();
        String jbntLyr = null;
        for (int i = 0; i < analysisLayers.size(); i++) {
            Map layer = (Map) analysisLayers.get(i);
            String lyrId = layer.get("fid").toString();
            if (lyrId.toLowerCase().equals("jbnt")) {
                jbntLyr = String.valueOf(layer.get("layerName"));
                continue;
            }
            String lyrName = (String) layer.get("layerName");
            String outFields = layer.get("fields").toString();
            String[] fields = "*".equals(outFields) ? null : outFields.split(",");
            List<Map> analysisList = (List<Map>) intersectBySimpleFeature(lyrName, feature, fields, dataSource);
            analysisMap.put(lyrId.toUpperCase(), analysisList);
            logger.debug("[" + lyrId.toUpperCase() + "]分析结果数:" + analysisList.size());
        }
        if (isNotNull(jbntLyr)) {
            analysisMap.put("jbnt".toUpperCase(), jbntdjAnalysis(jbntLyr, feature, JBNTBHTB, dataSource));
        }
        return analysisMap;
    }

    public Map jbntdjAnalysis(String layerName, SimpleFeature feature, String[] fields, String dataSource) {
        if (StringUtils.isBlank(layerName))
            throw new RuntimeException(getMessage("analysis.jbntdj.params.error", "基本农田保护图斑图层名称为空！"));

        List<Map> jbntList = (List<Map>) intersectBySimpleFeature(layerName, feature, fields, dataSource);
        Map gdAreasMap = new HashMap();
        Map kAreasMap = new HashMap();
        Map oAreasMap = new HashMap();
        // 统计所有编码的面积
        for (int i = 0; i < GDDLBM.length; i++) {
            gdAreasMap.put(GDDLBM[i], 0.0);
        }
        for (int i = 0; i < KTZYDDLBM.length; i++) {
            kAreasMap.put(KTZYDDLBM[i], 0.0);
        }
        for (int i = 0; i < jbntList.size(); i++) {
            HashMap<String, Object> analyMap = (HashMap<String, Object>) jbntList.get(i);
            Double SHAPE_AREA = MapUtils.getDouble(analyMap, "SHAPE_AREA");
//            Double JBNTMJ = MapUtils.getDouble(analyMap, "JBNTMJ");
//            String OBJECTID = MapUtils.getString(analyMap, "OBJECTID");
            if (SHAPE_AREA != null && SHAPE_AREA > 0.0) {
                String dlmc = MapUtils.getString(analyMap, DLMC);
                String dlbm = MapUtils.getString(analyMap, DLBM);
                String dlbz = MapUtils.getString(analyMap, "DLBZ");
                String key = dlbm + "_" + dlbz;
                if (ArrayUtils.contains(GDDLBM, dlbm, false) && !StringUtils.equalsIgnoreCase(dlbz, KTZYDDLBZ)) {
                    // 水田、旱地、水浇地
                    if (gdAreasMap.containsKey(dlbm)) {
                        SHAPE_AREA += (Double) gdAreasMap.get(dlbm);
                    }
                    gdAreasMap.put(dlbm, SHAPE_AREA);
                } else if (ArrayUtils.contains(KTZYDDLBM, dlbm, false) && StringUtils.equalsIgnoreCase(dlbz, KTZYDDLBZ)) {
                    // 可调整原地
                    if (kAreasMap.containsKey(dlbm)) {
                        SHAPE_AREA += (Double) kAreasMap.get(dlbm);
                    }
                    kAreasMap.put(dlbm, SHAPE_AREA);
                } else {
                    // 登记但不展示，在计算总面积时也不使用
                    if (oAreasMap.containsKey(key)) {
                        SHAPE_AREA += (Double) oAreasMap.get(key);
                    }
                    oAreasMap.put(key, SHAPE_AREA);
                }
            }
        }
        Map rMap = new HashMap();
        rMap.put("gdmap", gdAreasMap);
        rMap.put("ktmap", kAreasMap);
        rMap.put("omap", oAreasMap);
        return rMap;
    }

    public List<Map> hcAnalysisByArea(String geometry, List analysisLayers, String dataSource) {
        if (isNull(geometry)) throw new JSONMessageException("geometry is null");
        if (isNull(analysisLayers)) throw new RuntimeException("no analysis Layers");
        if (isNull(dataSource)) throw new JSONMessageException("dataSource is null");
        List results = null;
        Object geo = geometryService.readUnTypeGeoJSON(geometry);
        if (geo instanceof FeatureCollection) {
            results = new ArrayList();
            FeatureIterator iterator = ((FeatureCollection) geo).features();
            while (iterator.hasNext()) {
                SimpleFeature feature = (SimpleFeature) iterator.next();
                Map map = singleHcAnalysisByArea(feature, analysisLayers, dataSource);
                map.put("title", feature.getAttribute("title"));
                results.add(map);
            }
        } else if (geo instanceof SimpleFeature) {
            results = new ArrayList();
            results.add(singleHcAnalysisByArea((SimpleFeature) geo, analysisLayers, dataSource));
        }

        return results;
    }

    public Map singleHcAnalysisByArea(SimpleFeature feature, List analysisLayers, String dataSource) {
        Map analysisMap = new HashMap() {
            {
                put("hfydmj", 0); // 合法用地面积
                put("hfydper", 0);// 合法用地比率
                put("wfydmj", 0); // 违法用地面积
                put("wfydper", 0);
            }
        };
        // 单个图斑合法面积
        double hfPercent = 0;
        String jbntLyr = null;
        for (int i = 0; i < analysisLayers.size(); i++) {
            Map layer = (Map) analysisLayers.get(i);
            String lyrId = layer.get("fid").toString();
            if (lyrId.toLowerCase().equals("jbnt")) {
                jbntLyr = String.valueOf(layer.get("layerName"));
                continue;
            }
            boolean hf = (Boolean) layer.get("hf");
            String lyrName = layer.get("layerName").toString();
            String outFields = layer.get("fields").toString();
            String[] fields = "*".equals(outFields) ? null : outFields.split(",");
            List<Map> analysisList = (List<Map>) intersectBySimpleFeature(lyrName, feature, fields, dataSource);
            // 计算重叠面积
            double percent = 0;
            double mj = 0;
            if (analysisList.size() > 0) {
                // 同一类重叠相加，算总数
                for (int j = 0; j < analysisList.size(); j++) {
                    //SE_SHAPE_AREA、OG_SHAPE_AREA、INPUT_SHAPE_AREA
                    Map map = analysisList.get(j);
                    double se_area = Double.parseDouble(map.get(Constant.SE_SHAPE_AREA).toString());
                    double input_area = Double.parseDouble(map.get(Constant.INPUT_SHAPE_AREA).toString());
                    percent += se_area / input_area;
                    mj += se_area;
                }
            }
            analysisMap.put(hf ? "hfydmj" : "wfydmj", Double.parseDouble(analysisMap.get(hf ? "hfydmj" : "wfydmj").toString()) + mj);
            analysisMap.put(hf ? "hfydper" : "wfydper", Double.parseDouble(analysisMap.get(hf ? "hfydper" : "wfydper").toString()) + mj);
            analysisMap.put(lyrId + "ydmj", analysisMap.containsKey(lyrId + "ydmj") ? Double.parseDouble(analysisMap.get(lyrId + "ydmj").toString()) + mj : mj);
            analysisMap.put(lyrId + "ydper", analysisMap.containsKey(lyrId + "ydper") ? Double.parseDouble(analysisMap.get(lyrId + "ydper").toString()) + percent : percent);
        }
        if (isNotNull(jbntLyr)) {
            analysisMap.put("jbnt".toUpperCase(), jbntdjAnalysis(jbntLyr, feature, JBNTBHTB, dataSource));
        }
        return analysisMap;
    }

    /**
     * 监测图斑分析
     *
     * @param geometry
     * @param analysisLayers
     * @param dataSource
     * @param unit
     * @return
     */
    @Override
    public List<Map> jctbAnalysis(String geometry, List analysisLayers, String dataSource, Map unit) {
        if (isNull(analysisLayers)) throw new RuntimeException("no analysis Layers");
        if (isNull(geometry)) throw new JSONMessageException("geometry is null");
        if (isNull(unit)) throw new JSONMessageException("unit is null");
        List results = null;
        try {
            Object geo = geometryService.readUnTypeGeoJSON(geometry);
            if (geo instanceof FeatureCollection) {
                results = new ArrayList();
                FeatureIterator iterator = ((FeatureCollection) geo).features();
                while (iterator.hasNext()) {
                    SimpleFeature feature = (SimpleFeature) iterator.next();
                    Map map = singleJctbAnalysis(feature, analysisLayers, dataSource, unit);
                    results.add(map);
                }
            } else if (geo instanceof SimpleFeature) {
                results = new ArrayList();
                results.add(singleJctbAnalysis((SimpleFeature) geo, analysisLayers, dataSource, unit));
            }
        } catch (GeometryServiceException e) {
            throw new GeometryServiceException(GeometryServiceException.ExceptionType.GEOJSON_PARSE_EXCEPTION, e.getLocalizedMessage());
        } catch (NoSuchElementException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return results;
    }

    /**
     * 单个feature的监测图斑分析处理
     *
     * @param feature
     * @param analysisLayers
     * @param dataSource
     * @return
     */
    public Map singleJctbAnalysis(SimpleFeature feature, List analysisLayers, String dataSource, Map unit) {
        Map result = null;
        double conv = UNITS.ACRES.getConv();
        Map<String, List<Map>> analysisMap = new HashMap<String, List<Map>>();
        DecimalFormat df_jctb = new DecimalFormat("0.0");
        if (!isNull(unit)) {
            df_jctb.applyPattern(MapUtils.getString(unit, "format", "0.0"));
            String ualias = MapUtils.getString(unit, "alias");
            for (UNITS us : UNITS.values()) {
                if (us.getAlias().equals(ualias)) {
                    conv = us.getConv();
                    break;
                }
            }
        }
        try {
            String dltbLyr = null;
            String xzdwLyr = null;
            String bpLyr = null;
            CoordinateReferenceSystem crs = null;
            for (int i = 0; i < analysisLayers.size(); i++) {
                Map layer = (Map) analysisLayers.get(i);
                String lyrId = layer.get("fid").toString();
                if (lyrId.equals("dltb")) {
                    dltbLyr = String.valueOf(layer.get("layerName"));
                    continue;
                }
                if (lyrId.equals("xzdw")) {
                    xzdwLyr = String.valueOf(layer.get("layerName"));
                    continue;
                }
                if (lyrId.equals("bpdk"))
                    bpLyr = MapUtils.getString(layer, "layerName");
                String lyrName = (String) layer.get("layerName");
                String outFields = layer.get("fields").toString();
                String[] fields = "*".equals(outFields) ? null : outFields.split(",");
                if (isNull(crs))
                    crs = getLayerCRS(lyrName, dataSource);// 获取图层的crs
                List<Map> analysisList = (List<Map>) intersect3(lyrName, geometryService.toFeatureJSON(feature), fields, dataSource);
                analysisMap.put(lyrId.toUpperCase(), analysisList);
                if (analysisList != null) {
                    logger.debug("[" + lyrId.toUpperCase() + "]分析结果数:" + analysisList.size());
                } else {
                    logger.debug("[" + lyrId.toUpperCase() + "]没有分析结果");
                }

            }
            Map tdlyxzMap = tdlyxzAnalysis2(dltbLyr, xzdwLyr, geometryService.toFeatureJSON(feature), dataSource);
            Map tdlyxzAnalysisArea = (Map) tdlyxzMap.get("analysisArea");
            List tdlyxzAnalysisDetail = new ArrayList();
            if (tdlyxzMap.containsKey("analysisAreaDetail") && tdlyxzMap.get("analysisAreaDetail") != null)
                tdlyxzAnalysisDetail = (List) tdlyxzMap.get("analysisAreaDetail");

            List<Map> bpList = analysisMap.containsKey(JCTB.BPDK.name()) ? analysisMap.get(JCTB.BPDK.name()) : new ArrayList<Map>();
            List<Map> gdList = analysisMap.containsKey(JCTB.GDDK.name()) ? analysisMap.get(JCTB.GDDK.name()) : new ArrayList<Map>();
            List<Map> jsydgzqList = analysisMap.containsKey(JCTB.JSYDGZQ.name()) ? analysisMap.get(JCTB.JSYDGZQ.name()) : new ArrayList<Map>();
            List<Map> tdytqList = analysisMap.containsKey(JCTB.TDYTQ.name()) ? analysisMap.get(JCTB.TDYTQ.name()) : new ArrayList<Map>();
            List<Map> ssnydList = analysisMap.containsKey(JCTB.SSNYD.name()) ? analysisMap.get(JCTB.SSNYD.name()) : new ArrayList<Map>();
            List<Map> lsydList = analysisMap.containsKey(JCTB.LSYD.name()) ? analysisMap.get(JCTB.LSYD.name()) : new ArrayList<Map>();

            //mas
            List<Map> yxjsList = analysisMap.containsKey(JCTB.YXJS.name()) ? analysisMap.get(JCTB.YXJS.name()) : new ArrayList<Map>();
            List<Map> ytjjsList = analysisMap.containsKey(JCTB.YTJJS.name()) ? analysisMap.get(JCTB.YTJJS.name()) : new ArrayList<Map>();
            List<Map> byxjsList = analysisMap.containsKey(JCTB.BYXJS.name()) ? analysisMap.get(JCTB.BYXJS.name()) : new ArrayList<Map>();
            List<Map> jbntList = analysisMap.containsKey(JCTB.JBNT.name()) ? analysisMap.get(JCTB.JBNT.name()) : new ArrayList<Map>();

            double lsydBpArea = getIntersectArea(lsydList, bpLyr, dataSource) * conv;
            double ssnydBpArea = getIntersectArea(ssnydList, bpLyr, dataSource) * conv;

            result = new HashMap();
            for (Property p : feature.getProperties()) {
                if (p.getName().getLocalPart().equals("geometry") || p.getName().getLocalPart().equals("crs")) continue;
                result.put(OG_PRO_PERFIX.concat(p.getName().getLocalPart()), p.getValue());
            }

            /** 把图形简化后带上 用于页面的定位(for mas)**/
            Geometry simpleGeo = geometryService.simplify((Geometry) feature.getDefaultGeometry(), geometryService.getSimplifyTolerance());
            result.put(OG_PRO_PERFIX.concat(SE_SHAPE_FIELD), geometryService.toGeoJSON(simpleGeo));

            double geoArea = MapUtils.getDoubleValue(result, OG_PRO_PERFIX.concat("JCMJ"), 0) * 666.6666667 * conv; //jcmj单位为亩 转为㎡后计算
            if (geoArea == 0)
                geoArea = geometryService.getGeoArea(feature, null) * conv;
            result.put(OG_PRO_PERFIX.concat("JCMJ"), df_jctb.format(geoArea));

            result.put(JTag.JC_GD_AREA.name(), df_jctb.format((MapUtils.getDouble(tdlyxzAnalysisArea, "水田", 0.0) + MapUtils.getDouble(tdlyxzAnalysisArea, "水浇地", 0.0) + MapUtils.getDouble(tdlyxzAnalysisArea, "旱地", 0.0)) * conv));
            result.put(JTag.BP_AREA.name(), df_jctb.format(getAreaByList(bpList, true, conv, crs)));
            result.put(JTag.BP_GD_AREA.name(), df_jctb.format(getGdArea(bpList, dltbLyr, xzdwLyr, dataSource) * conv));
            result.put(JTag.YG_AREA.name(), df_jctb.format(getAreaByList(gdList, true, conv, crs)));
            result.put(JTag.WG_AREA.name(), df_jctb.format((geoArea - MapUtils.getDouble(result, JTag.YG_AREA.name(), 0.0)) < 0 ? 0 : geoArea - MapUtils.getDouble(result, JTag.YG_AREA.name(), 0.0)));
            result.put(JTag.WPYJ_AREA.name(), df_jctb.format((geoArea - MapUtils.getDouble(result, JTag.BP_AREA.name(), 0.0)) < 0 ? 0 : geoArea - MapUtils.getDouble(result, JTag.BP_AREA.name(), 0.0)));
            result.put(JTag.WPYJ_GD_AREA.name(), df_jctb.format(MapUtils.getDouble(result, JTag.JC_GD_AREA.name(), 0.0) - MapUtils.getDouble(result, JTag.BP_GD_AREA.name(), 0.0) < 0 ? 0 : MapUtils.getDouble(result, JTag.JC_GD_AREA.name(), 0.0) - MapUtils.getDouble(result, JTag.BP_GD_AREA.name(), 0.0)));
            result.put(JTag.YXJSQ_AREA.name(), df_jctb.format(getAreaByField(jsydgzqList, "GZQLXDM", "010", conv)));
            result.put(JTag.JBNT_AREA.name(), df_jctb.format(getAreaByField(tdytqList, "TDYTQLXDM", "010", conv)));
            result.put(JTag.YBNTQ_AREA.name(), df_jctb.format(getAreaByField(tdytqList, "TDYTQLXDM", "020", conv)));

            result.put(JTag.PCMC.name(), getStrValueByField(bpList, "PCJC"));
            result.put(JTag.NZYPW.name(), getStrValueByField(bpList, "PZWH"));
            result.put(JTag.GDBH.name(), getStrValueByField(gdList, "XMBH"));

            result.put(JTag.LSYD_AREA.name(), df_jctb.format(getAreaByList(lsydList, false, conv, crs) - lsydBpArea));
            result.put(JTag.LSYD_GD_AREA.name(), df_jctb.format(getGdArea(lsydList, dltbLyr, xzdwLyr, dataSource) * conv));
            result.put(JTag.SSNYD_AREA.name(), df_jctb.format(getAreaByList(ssnydList, false, conv, crs) - ssnydBpArea));
            result.put(JTag.SSSNYD_GD_AREA.name(), df_jctb.format(getGdArea(ssnydList, dltbLyr, xzdwLyr, dataSource) * conv));
            result.put(JTag.LSYD_BH.name(), getStrValueByField(lsydList, "BH"));
            result.put(JTag.SSNYD_BH.name(), getStrValueByField(ssnydList, "BH"));

            //for mas
            result.put(JTagMAS.YXJS_AREA.name(), df_jctb.format(getAreaByList(yxjsList, true, conv, crs)));
            result.put(JTagMAS.YTJJS_AREA.name(), df_jctb.format(getAreaByList(ytjjsList, true, conv, crs)));
            result.put(JTagMAS.BYXJS_AREA.name(), df_jctb.format(getAreaByList(byxjsList, true, conv, crs)));
            result.put(JTagMAS.JBNT_MAS_AREA.name(), df_jctb.format(getAreaByList(jbntList, true, conv, crs)));
            result.put(JTagMAS.NYD_AREA.name(), df_jctb.format(getAreaByField(tdlyxzAnalysisDetail, "DLBM", "011,012,013,021,022,023,031,032,033,041,042,104,114,117,122,123", "CCMJ", conv)));
            result.put(JTagMAS.GD_AREA.name(), df_jctb.format(getAreaByField(tdlyxzAnalysisDetail, "DLBM", "011,012,013", "CCMJ", conv)));
            result.put(JTagMAS.JSYD_AREA.name(), df_jctb.format(getAreaByField(tdlyxzAnalysisDetail, "DLBM", "051,052,053,054,061,062,063,071,072,081,082,083,084,085,086,087,088,091,092,093,094,095,101,102,103,105,106,107,113,118,121,201,202,203,204,205", "CCMJ", conv)));
            result.put(JTagMAS.WLYD_AREA.name(), df_jctb.format(getAreaByField(tdlyxzAnalysisDetail, "DLBM", "111,112,115,116,119,043,124,125,126,127", "CCMJ", conv)));
            result.put(JTagMAS.PCMC_MAS.name(), getStrValueByField(bpList, new String[]{"PWSJ", "PWBH", "XMMC"}, "-"));
            result.put(JTagMAS.NZYPW_MAS.name(), getStrValueByField(bpList, "PZWH"));
            result.put(JTagMAS.HPDH_MAS.name(), getStrValueByField(bpList, "DAH"));
            result.put(JTagMAS.GDBH_MAS.name(), getStrValueByField(gdList, "CRBH"));
            result.put(JTagMAS.GDDH_MAS.name(), getStrValueByField(gdList, "DAH"));
            result.put(JTagMAS.XMMC_MAS.name(), getStrValueByField(gdList, "XMMC"));
            result.put(JTagMAS.YDDW_MAS.name(), getStrValueByField(gdList, "YDDW"));

        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return result;
    }

    /**
     * 土地利用现状分析，只扣除线物
     *
     * @param dltbLayerName
     * @param xzdwLayerName
     * @param geometry      GeoJSON format
     * @param outFields
     * @return
     */
    @Override
    public Map tdlyxzAnalysis(String dltbLayerName, String xzdwLayerName, String geometry, String[] outFields, String dataSource) {
        Assert.notNull(geometry, getMessage("geometry.notnull"));
        Map<String, Double> analysisResult = new HashMap<String, Double>();
        Map<String, Double> analysisTemp = new HashMap<String, Double>();
        /**
         * 地类图斑相交
         */
        String[] dltbOutFields = ArrayUtils.add2Arrays(outFields, new String[]{"DLMC", "ZLDWDM", "TBBH"});
        List<Map<String, Object>> results = (List<Map<String, Object>>) intersectByGeoJSON(dltbLayerName, geometry, dltbOutFields, dataSource);
        for (Map<String, Object> result : results) {
            Double area = Double.parseDouble(String.valueOf(result.get(SE_SHAPE_AREA)));
            if (area > 0.4) {
                String dlmc = String.valueOf(result.get("DLMC"));
                analysisResult.put(dlmc, (analysisResult.containsKey(dlmc) ? analysisResult.get(dlmc) : 0) + area);
                analysisTemp.put(String.valueOf(result.get("ZLDWDM")) + String.valueOf(result.get("TBBH")), area);
            }
        }
        /**
         * 现状地物相交并扣除
         */
        String[] xzdwOutFields = ArrayUtils.add2Arrays(outFields, new String[]{"DLMC", "KD", "KCTBDWDM2", "KCTBDWDM1", "KCTBBH1", "KCTBBH2"});
        List<Map<String, Object>> xzResults = (List<Map<String, Object>>) intersectByGeoJSON(xzdwLayerName, geometry, xzdwOutFields, dataSource);
        for (Map<String, Object> result : xzResults) {
            Double length = Double.parseDouble(String.valueOf(result.get(SE_SHAPE_AREA)));
            if (length > 0.08) {
                String dlmc = String.valueOf(result.get("DLMC"));
                Double kd = Double.parseDouble(String.valueOf(result.get("KD")));
                analysisResult.put(dlmc, (analysisResult.containsKey(dlmc) ? analysisResult.get(dlmc) : 0) + length * kd);
                //
                String where = "ZLDWDM='" + result.get("KCTBDWDM1") + "' and TBBH='" + result.get("KCTBBH1") + "'";
                List<Map<String, Object>> temps = new ArrayList<Map<String, Object>>();
                try {
                    temps = (List<Map<String, Object>>) query(dltbLayerName, where,
                            ArrayUtils.add2Arrays(outFields, new String[]{"DLMC"}), true, dataSource);
                } catch (Exception e) {
                    logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
                }
                String kctb1Key = String.valueOf(result.get("KCTBDWDM1")) + String.valueOf(result.get("KCTBBH1"));
                //
                if (StringUtils.isBlank((String) result.get("KCTBDWDM2"))) {
                    for (Map<String, Object> temp : temps) {
                        String tmpDlmc = String.valueOf(temp.get("DLMC"));
                        analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - length * kd);
                        analysisTemp.put(kctb1Key, analysisTemp.get(kctb1Key) - length * kd);
                    }
                } else {
                    //
                    Tag tag = Tag.NO;
                    String tmpDlmc = "";
                    String tmpDlmc1 = "";
                    Double tmpXzdwKd = 0.0;
                    String tmpXzdwKctbdwdm1 = "";
                    String tmpXzdwKctbbh1 = "";
                    for (Map<String, Object> tmp : temps) {
                        tmpDlmc = String.valueOf(tmp.get("DLMC"));
                        if (analysisTemp.containsKey(kctb1Key)) {
                            Double tmpValue = analysisTemp.get(kctb1Key);
                            if (tmpValue - (length / 2.0) * kd > 0.0) {
                                analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - (length / 2.0) * kd);
                                tmpXzdwKd = (length / 2.0) * kd;
                                tmpDlmc1 = tmpDlmc;
                                tmpXzdwKctbbh1 = String.valueOf(result.get("KCTBBH1"));
                                tmpXzdwKctbdwdm1 = String.valueOf(result.get("KCTBDWDM1"));
                                analysisTemp.put(tmpXzdwKctbdwdm1 + tmpXzdwKctbbh1, tmpValue - (length / 2.0) * kd);
                            } else
                                tag = Tag.YES;
                        } else
                            tag = Tag.YES;
                    }
                    //
                    where = "ZLDWDM='" + result.get("KCTBDWDM2") + "' and TBBH='" + result.get("KCTBBH2") + "'";
                    try {
                        temps = (List<Map<String, Object>>) query(dltbLayerName, where, new String[]{"DLMC"}, true, dataSource);
                    } catch (Exception e) {
                        logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
                        temps.clear();
                    }
                    String kctb2Key = String.valueOf(result.get("KCTBDWDM2")) + String.valueOf(result.get("KCTBBH2"));
                    for (Map<String, Object> tmp : temps) {
                        tmpDlmc = String.valueOf(tmp.get("DLMC"));
                        if (analysisTemp.containsKey(kctb2Key)) {
                            Double tmpValue = analysisTemp.get(kctb2Key);
                            if (tmpValue - (length / 2.0) * kd > 0.0) {
                                if (!Tag.YES.equals(tag)) {
                                    analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - (length / 2.0) * kd);
                                    analysisTemp.put(kctb2Key, tmpValue - length * kd);
                                } else {
                                    analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - (length / 2.0) * kd);
                                    analysisTemp.put(kctb2Key, tmpValue - length * kd);
                                }
                            } else {
                                tmpDlmc = tmpDlmc1;
                                analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - tmpXzdwKd);
                                analysisTemp.put(kctb1Key, analysisTemp.get(kctb1Key) - tmpXzdwKd);
                            }
                        } else {
                            tmpDlmc = tmpDlmc1;
                            analysisResult.put(tmpDlmc, analysisResult.get(tmpDlmc) - tmpXzdwKd);
                            analysisTemp.put(kctb1Key, analysisTemp.get(kctb1Key) - tmpXzdwKd);
                        }
                    }
                }
            }
        }
        return analysisResult;
    }

    /**
     * 土地利用现状分析，只扣除线物
     *
     * @param regionCode
     * @param geometry   GeoJSON format
     * @param dataSource
     * @return
     */
    @Override
    public Map tdlyxzAnalysis(String regionCode, String geometry, String dataSource) {
        regionCode = Utils.formatRegionCode(regionCode);
        String dltb = TDXZ.DLTB.name().concat(LAYER_MIDDLE_FIX_H).concat(regionCode);
        String xzdw = TDXZ.XZDW.name().concat(LAYER_MIDDLE_FIX_H).concat(regionCode);
        return tdlyxzAnalysis2(dltb, xzdw, geometry, dataSource);
    }

    /**
     * 土地利用现状分析
     *
     * @param dltbLayerName
     * @param xzdwLayerName
     * @param geometry
     * @param dataSource
     * @return
     */
    @Override
    public Map tdlyxzAnalysis2(String dltbLayerName, String xzdwLayerName, String geometry, String dataSource) {
        if (StringUtils.isBlank(dltbLayerName))
            throw new RuntimeException(getMessage("analysis.tdlyxz.params.error", "地类图斑图层名称为空！"));
        if (StringUtils.isBlank(xzdwLayerName))
            logger.warn(getMessage("analysis.tdlyxz.params.error", "线状地物图层名称为空,只会分析地类图斑图层！"));
        if (StringUtils.isBlank(geometry))
            throw new RuntimeException(getMessage("analysis.tdlyxz.params.error", "分析地块坐标为空！"));
        //是否开启wcf分析
        boolean useWcf = AppConfig.getBooleanProperty("analysis.useWcf");
        if (useWcf) {
            String wcfUrl = AppConfig.getProperty("wcfUrl");
            String wcfMethod = AppConfig.getProperty("wcfMethod");
            if (StringUtils.isBlank(wcfUrl))
                throw new RuntimeException(getMessage("analysis.wcf.url.null"));
            if (StringUtils.isBlank(wcfMethod))
                throw new RuntimeException(getMessage("analysis.wcf.method.null"));
            return tdlyxzAnalysisByWcf(geometry, dltbLayerName, xzdwLayerName, dataSource);
        } else {
            Map<String, Double> dlMap = new HashMap<String, Double>();   //分析地类面积结果
            Map<String, Double> bhMap = new HashMap<String, Double>();
            MultiKeyMap analysisDetailMap = MultiKeyMap.decorate(new HashedMap()); //分析地类面积明细
            String[] columns = {"DLMC"};
            List<Map<String, Object>> xzdwDltbResult = new ArrayList<Map<String, Object>>();
            List<Map<String, Object>> dltbAnalysisResult = new ArrayList<Map<String, Object>>();
            List<Map<String, Object>> xzdwAnalysisResult = new ArrayList<Map<String, Object>>();
            //与地类图斑层进行相交分析
            try {
                dltbAnalysisResult = (List<Map<String, Object>>) intersect3(dltbLayerName, geometry, "OBJECTID;BSM;DLMC;ZLDWDM;TBBH;ZLDWMC;QSXZ;DLBM;TBMJ;QSDWDM;QSDWMC".split(";"), dataSource);
            } catch (Exception e) {
                logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
            }
            if (dltbAnalysisResult.size() > 0) {
                for (int i = 0; i < dltbAnalysisResult.size(); i++) {
                    Double tmpArea = Double.parseDouble(String.valueOf(dltbAnalysisResult.get(i).get("SHAPE_AREA"))); //相交部分的面积
                    if (tmpArea > 0.4) {
                        if (dlMap.containsKey(dltbAnalysisResult.get(i).get("DLMC")))
                            dlMap.put(String.valueOf(dltbAnalysisResult.get(i).get("DLMC")), dlMap.get(dltbAnalysisResult.get(i).get("DLMC")) + tmpArea);
                        else
                            dlMap.put(String.valueOf(dltbAnalysisResult.get(i).get("DLMC")), tmpArea);
                        String tmpZldwdmTbbh = String.valueOf(dltbAnalysisResult.get(i).get("ZLDWDM")) + String.valueOf(dltbAnalysisResult.get(i).get("TBBH"));

                        if (bhMap.containsKey(tmpZldwdmTbbh))
                            bhMap.put(tmpZldwdmTbbh, bhMap.get(tmpZldwdmTbbh) + tmpArea);
                        else
                            bhMap.put(tmpZldwdmTbbh, tmpArea);

                        //图斑明细
                        if (analysisDetailMap.containsKey(String.valueOf(dltbAnalysisResult.get(i).get("ZLDWDM")), String.valueOf(dltbAnalysisResult.get(i).get("TBBH")))) {
                            Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(dltbAnalysisResult.get(i).get("ZLDWDM")), String.valueOf(dltbAnalysisResult.get(i).get("TBBH")));
                            detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) + tmpArea);
                        } else {
                            Map<String, Object> detailValue = new HashMap<String, Object>();
                            detailValue.put("ZLDWDM", String.valueOf(dltbAnalysisResult.get(i).get("ZLDWDM")));
                            detailValue.put("ZLDWMC", String.valueOf(dltbAnalysisResult.get(i).get("ZLDWMC")));
                            detailValue.put("TBBH", String.valueOf(dltbAnalysisResult.get(i).get("TBBH")));
                            detailValue.put("QSXZ", String.valueOf(dltbAnalysisResult.get(i).get("QSXZ")));
                            detailValue.put("DLMC", String.valueOf(dltbAnalysisResult.get(i).get("DLMC")));
                            detailValue.put("DLBM", String.valueOf(dltbAnalysisResult.get(i).get("DLBM")));
                            detailValue.put("TBMJ", String.valueOf(dltbAnalysisResult.get(i).get("TBMJ")));
                            detailValue.put("QSDWDM", String.valueOf(dltbAnalysisResult.get(i).get("QSDWDM")));
                            detailValue.put("QSDWMC", String.valueOf(dltbAnalysisResult.get(i).get("QSDWMC")));
                            detailValue.put("CCMJ", tmpArea);
                            detailValue.put(SE_SHAPE_FIELD, MapUtils.getString(dltbAnalysisResult.get(i), "SHAPE"));
                            analysisDetailMap.put(String.valueOf(dltbAnalysisResult.get(i).get("ZLDWDM")), String.valueOf(dltbAnalysisResult.get(i).get("TBBH")), detailValue);
                        }

                    }
                }
            }

            //与线状地物层进行相交分析，并扣除
            try {
                xzdwAnalysisResult = (List<Map<String, Object>>) intersect3(xzdwLayerName, geometry, "OBJECTID;BSM;DLMC;KD;KCTBDWDM2;KCTBDWDM1;KCTBBH1;KCTBBH2;QSDWDM1;XZDWBH;QSXZ;DLBM;XZDWMJ;QSDWMC1".split(";"), dataSource);
            } catch (Exception e) {
                logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
            }
            if (xzdwAnalysisResult.size() > 0) {
                for (int i = 0; i < xzdwAnalysisResult.size(); i++) {
                    Double tmpLength = Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("SHAPE_AREA")));
                    if (tmpLength > 0.08) {
                        if (dlMap.containsKey(xzdwAnalysisResult.get(i).get("DLMC"))) {
                            dlMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC")), dlMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC"))) + (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                        } else
                            dlMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC")), tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));

                        //加入线物明细
                        if (analysisDetailMap.containsKey(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))) {
                            Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")));
                            detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) + tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                        } else {
                            Map<String, Object> detailValue = new HashMap<String, Object>();
                            detailValue.put("ZLDWDM", String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH"))));
                            detailValue.put("ZLDWMC", String.valueOf(xzdwAnalysisResult.get(i).get("")));
                            detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")));
                            detailValue.put("QSXZ", String.valueOf(xzdwAnalysisResult.get(i).get("QSXZ")));
                            detailValue.put("DLMC", String.valueOf(xzdwAnalysisResult.get(i).get("DLMC")));
                            detailValue.put("DLBM", String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")));
                            detailValue.put("TBMJ", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWMJ")));
                            detailValue.put("QSDWDM", String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")));
                            detailValue.put("QSDWMC", String.valueOf(xzdwAnalysisResult.get(i).get("QSDWMC1")));
                            detailValue.put("CCMJ", tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                            detailValue.put(SE_SHAPE_FIELD, MapUtils.getString(xzdwAnalysisResult.get(i), "SHAPE"));
                            analysisDetailMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), detailValue);
                        }
                        //开始分情况扣除
                        if (StringUtils.isBlank(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2"))) || String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")).equalsIgnoreCase("null")) {
                            String tmpWhereClause = "ZLDWDM='" + xzdwAnalysisResult.get(i).get("KCTBDWDM1") + "' and TBBH='" + xzdwAnalysisResult.get(i).get("KCTBBH1") + "'";
                            try {
                                xzdwDltbResult = (List<Map<String, Object>>) query(dltbLayerName, tmpWhereClause, columns, true, dataSource);
                            } catch (Exception e) {
                                logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
                            }
                            String tmpKey = String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1"));
                            if (!isNull(xzdwDltbResult) && xzdwDltbResult.size() > 0 && bhMap.containsKey(tmpKey)) {
                                for (int j = 0; j < xzdwDltbResult.size(); j++) {

                                    //需要判断bhMap中的图斑面积是否小于线物面积，也就是出现负数。如果为负数，则扣到0，同时dlMap中要还回一部分线物的地类面积
                                    Double tmpCCMJ = 0.0;
                                    if (bhMap.get(tmpKey) >= (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))))) {
                                        dlMap.put(String.valueOf(xzdwDltbResult.get(j).get("DLMC")), dlMap.get(String.valueOf(xzdwDltbResult.get(j).get("DLMC"))) - (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                        bhMap.put(tmpKey, bhMap.get(tmpKey) - (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                        tmpCCMJ = tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")));
                                    } else {
                                        Double tmpKouchuMj = bhMap.get(tmpKey);
                                        Double tmpSyMj = (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))) - tmpKouchuMj;
                                        //只扣除到0
                                        dlMap.put(String.valueOf(xzdwDltbResult.get(j).get("DLMC")), dlMap.get(String.valueOf(xzdwDltbResult.get(j).get("DLMC"))) - tmpKouchuMj);
                                        bhMap.put(tmpKey, 0.0);
                                        //还回未扣除部分线物面积
                                        dlMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC")), dlMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC"))) - tmpSyMj);
                                        //线物明细中需要扣除的面积
                                        tmpCCMJ = tmpKouchuMj;
                                        //线物明细需要还回的面积
                                        Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")));
                                        detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - tmpSyMj);
                                    }

                                    //扣除线物明细
                                    if (analysisDetailMap.containsKey(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")))) {
                                        Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                        detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - tmpCCMJ);
                                    } else {
                                        logger.error((getMessage("analysis.tdlyxz.kc.error", "ERROR", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")),
                                                String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")), String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))
                                                , xzdwAnalysisResult.get(i).get("KCTBDWDM1"), xzdwAnalysisResult.get(i).get("KCTBDWDM2"), xzdwAnalysisResult.get(i).get("XZDWMJ"), tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))))));
                                        Map<String, Object> detailValue = new HashMap<String, Object>();
                                        detailValue.put("ZLDWDM", "error" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
                                        detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                        analysisDetailMap.put("error" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), detailValue);
                                    }
                                }
                            } else {

                                //应该还回线物面积
                                dlMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC")), dlMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC"))) - (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                //从明细中还回去
                                Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")));
                                detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                //如果找不到要扣除的图斑，则报错
//                                throw new RuntimeException(getMessage("analysis.tdlyxz.kc.error", "根据相交线物查找到的扣除图斑不在地块范围之内",
//                                        String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), String.valueOf(xzdwAnalysisResult.get(i).get("DLMC")),xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
                            }
                        } else {
                            //记录扣除状态
                            String kouchuTag = "no";
                            //记录扣除面积
                            double kouchuMj = 0.0;
                            //记录扣除地类
                            String kouchuDlmc = "";
                            //扣除ZLDWDM
                            String kouchuZldwdm = "";
                            //扣除tbbh
                            String kouchuTbbh = "";
                            String tmpWhereClause = "ZLDWDM='" + xzdwAnalysisResult.get(i).get("KCTBDWDM1") + "' and TBBH='" + xzdwAnalysisResult.get(i).get("KCTBBH1") + "'";

                            try {
                                xzdwDltbResult = (List<Map<String, Object>>) query(dltbLayerName, tmpWhereClause, columns, true, dataSource);
                            } catch (Exception e) {

                                logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
                            }
                            if (!isNull(xzdwDltbResult) && xzdwDltbResult.size() > 0) {
                                for (int j = 0; j < xzdwDltbResult.size(); j++) {
                                    String tmpDlmcKey = String.valueOf(xzdwDltbResult.get(j).get("DLMC"));
                                    if (bhMap.containsKey(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")))) {
                                        Double tmpValue = bhMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                        if (tmpValue - ((tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))) > 0.0) {
                                            dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - (tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                            kouchuMj = (tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")));
                                            kouchuDlmc = tmpDlmcKey;
                                            kouchuZldwdm = String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1"));
                                            kouchuTbbh = String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1"));
                                            bhMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), tmpValue - ((tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                            //扣除线物明细
                                            Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                            detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - kouchuMj);

                                        } else {
                                            kouchuTag = "yes";
                                            logger.error(getMessage("analysis.tdlyxz.kc.error", "面积过小", 2, xzdwAnalysisResult.get(i).get("DLBM"), xzdwAnalysisResult.get(i).get("QSDWDM1"), xzdwAnalysisResult.get(i).get("KCTBDWDM1"), xzdwAnalysisResult.get(i).get("KCTBDWDM2")));
                                            Map<String, Object> detailValue = new HashMap<String, Object>();
                                            detailValue.put("ZLDWDM", "error面积过小" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
                                            detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                            detailValue.put("TBMJ", tmpValue);
                                            detailValue.put("CCMJ", (tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                            analysisDetailMap.put("error面积过小" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), detailValue);
                                        }
                                    } else {
                                        kouchuTag = "yes";
                                        logger.error(getMessage("analysis.tdlyxz.kc.error", "没有占用", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")),
                                                String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")), String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))
                                                , xzdwAnalysisResult.get(i).get("KCTBDWDM1"), xzdwAnalysisResult.get(i).get("KCTBDWDM2"), xzdwAnalysisResult.get(i).get("XZDWMJ"), tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                        Map<String, Object> detailValue = new HashMap<String, Object>();
                                        detailValue.put("ZLDWDM", "error没有占用" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
                                        detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                        analysisDetailMap.put("error没有占用" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), detailValue);
                                    }
                                }
                            } else {
                                kouchuTag = "yes";
                                logger.error(getMessage("analysis.tdlyxz.kc.error", "没有找到", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")),
                                        String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")), String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))
                                        , xzdwAnalysisResult.get(i).get("KCTBDWDM1"), xzdwAnalysisResult.get(i).get("KCTBDWDM2"), xzdwAnalysisResult.get(i).get("XZDWMJ"), tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                Map<String, Object> detailValue = new HashMap<String, Object>();
                                detailValue.put("ZLDWDM", "error,没有找到" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")));
                                detailValue.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")));
                                analysisDetailMap.put("error,没有找到" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM1")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH1")), detailValue);
                            }

                            tmpWhereClause = "ZLDWDM='" + xzdwAnalysisResult.get(i).get("KCTBDWDM2") + "' and TBBH='" + xzdwAnalysisResult.get(i).get("KCTBBH2") + "'";
                            try {
                                xzdwDltbResult = (List<Map<String, Object>>) query(dltbLayerName, tmpWhereClause, columns, true, dataSource);
                            } catch (Exception e) {
                                logger.error("tdlyxzAnalysis error [{}] ", e.getLocalizedMessage());
                            }
                            if (!isNull(xzdwDltbResult) && xzdwDltbResult.size() > 0) {
                                for (int j = 0; j < xzdwDltbResult.size(); j++) {
                                    String tmpDlmcKey = String.valueOf(xzdwDltbResult.get(j).get("DLMC"));
                                    if (bhMap.containsKey(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")))) {
                                        Double tmpValue = bhMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
                                        if (tmpValue - ((tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))) > 0.0) {
                                            if (!kouchuTag.equals("yes")) {
                                                dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - ((tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));

                                                //扣除线物明细
                                                Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
                                                detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - (tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));

                                                bhMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")), tmpValue - ((tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                            } else {
                                                Double tmpCCMJ = 0.0;
                                                //需要判断bhMap中的图斑面积是否小于线物面积，也就是否会出现负数。如果为负数，则扣到0，同时dlMap中要还回一部分线物的地类面积
                                                if (tmpValue >= (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))))) {
                                                    dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                                    bhMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")), tmpValue - (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                                    tmpCCMJ = tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")));
                                                } else {
                                                    Double tmpKouchuMj = bhMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
                                                    Double tmpSyMj = (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))) - tmpKouchuMj;
                                                    //只扣除到0
                                                    dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - tmpKouchuMj);
                                                    bhMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")) + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")), 0.0);
                                                    //还回未扣除部分线物面积
                                                    dlMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC")), dlMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC"))) - tmpSyMj);
                                                    tmpCCMJ = tmpKouchuMj;

                                                    //线物明细需要还回的面积
                                                    Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")));
                                                    detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - tmpSyMj);
                                                }

                                                //扣除线物明细
                                                Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
                                                detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - tmpCCMJ);

                                            }
                                        } else {
                                            if (StringUtils.isNotBlank(kouchuDlmc)) {
                                                Double tmpCCMJ = 0.0;
                                                try {

                                                    //需要判断bhMap中的图斑面积是否小于扣除线物面积，也就是否会出现负数。如果为负数，则扣到0，同时dlMap中要还回一部分线物的地类面积
                                                    tmpDlmcKey = kouchuDlmc;
                                                    if (bhMap.get(kouchuZldwdm + kouchuTbbh) >= kouchuMj) {
                                                        dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - kouchuMj);
                                                        bhMap.put(kouchuZldwdm + kouchuTbbh, bhMap.get(kouchuZldwdm + kouchuTbbh) - kouchuMj);
                                                        tmpCCMJ = kouchuMj;
                                                    } else {
                                                        Double tmpKouchuMj = bhMap.get(kouchuZldwdm + kouchuTbbh);
                                                        Double tmpSyMj = kouchuMj - tmpKouchuMj;
                                                        //只扣除到0
                                                        dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - tmpKouchuMj);
                                                        bhMap.put(kouchuZldwdm + kouchuTbbh, 0.0);
                                                        //还未扣除部分线物面积
                                                        dlMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC")), dlMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC"))) - tmpSyMj);
                                                        tmpCCMJ = tmpKouchuMj;
                                                        //线物明细需要还回的面积
                                                        Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")));
                                                        detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - tmpSyMj);
                                                    }
                                                } catch (Exception ex) {
                                                    logger.error(getMessage("analysis.tdlyxz.kc.error", "面积扣除错误，出现无法扣除情况，请检查，请用实地勘测坐标叠加到图形上！!", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")),
                                                            String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")), String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))
                                                            , xzdwAnalysisResult.get(i).get("KCTBDWDM1"), xzdwAnalysisResult.get(i).get("KCTBDWDM2"), xzdwAnalysisResult.get(i).get("XZDWMJ"), tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                                }


                                                Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(kouchuZldwdm, kouchuTbbh);
                                                detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - tmpCCMJ);

                                                logger.error(getMessage("analysis.tdlyxz.kc.error", "面积过小", 2, xzdwAnalysisResult.get(i).get("DLBM"), xzdwAnalysisResult.get(i).get("QSDWDM1"), xzdwAnalysisResult.get(i).get("KCTBDWDM1"), xzdwAnalysisResult.get(i).get("KCTBDWDM2")));
                                                Map<String, Object> detailValue1 = new HashMap<String, Object>();
                                                detailValue1.put("ZLDWDM", "error面积过小" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")));
                                                detailValue1.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
                                                detailValue1.put("TBMJ", tmpValue);
                                                detailValue1.put("CCMJ", (tmpLength / 2.0) * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                                analysisDetailMap.put("error面积过小" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")), detailValue1);
                                            } else {
                                                //这个地方是KCTBDWDM1不够扣或者找不到，而KCTBDWDM2又不够扣，那么应该还回线物面积
                                                dlMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC")), dlMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC"))) - (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                                //从明细中还回去
                                                Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")));
                                                detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                            }
                                        }
                                    } else {
                                        if (StringUtils.isNotBlank(kouchuDlmc)) {
                                            Double tmpCCMJ = 0.0;
                                            try {
                                                tmpDlmcKey = kouchuDlmc;
                                                if (bhMap.get(kouchuZldwdm + kouchuTbbh) >= kouchuMj) {
                                                    dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - kouchuMj);
                                                    bhMap.put(kouchuZldwdm + kouchuTbbh, bhMap.get(kouchuZldwdm + kouchuTbbh) - kouchuMj);
                                                    tmpCCMJ = kouchuMj;
                                                } else {
                                                    Double tmpKouchuMj = bhMap.get(kouchuZldwdm + kouchuTbbh);
                                                    Double tmpSyMj = kouchuMj - tmpKouchuMj;
                                                    //只扣除到0
                                                    dlMap.put(tmpDlmcKey, dlMap.get(tmpDlmcKey) - tmpKouchuMj);
                                                    bhMap.put(kouchuZldwdm + kouchuTbbh, 0.0);
                                                    //还回未扣除部分线物面积
                                                    dlMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC")), dlMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC"))) - tmpSyMj);
                                                    tmpCCMJ = tmpKouchuMj;

                                                    //线物明细需要还回的面积
                                                    Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")));
                                                    detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - tmpSyMj);
                                                }
                                            } catch (Exception ex) {
                                                logger.error(getMessage("analysis.tdlyxz.kc.error", "土地利用现状底图中现状地物扣除有误，请检查!", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")),
                                                        String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")), String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))
                                                        , xzdwAnalysisResult.get(i).get("KCTBDWDM1"), xzdwAnalysisResult.get(i).get("KCTBDWDM2"), xzdwAnalysisResult.get(i).get("XZDWMJ"), tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                            }

                                            Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(kouchuZldwdm, kouchuTbbh);
                                            detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - tmpCCMJ);

                                            logger.error(getMessage("analysis.tdlyxz.kc.error", "没有占用", String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")),
                                                    String.valueOf(xzdwAnalysisResult.get(i).get("DLBM")), String.valueOf(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")))
                                                    , xzdwAnalysisResult.get(i).get("KCTBDWDM1"), xzdwAnalysisResult.get(i).get("KCTBDWDM2"), xzdwAnalysisResult.get(i).get("XZDWMJ"), tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));

                                            Map<String, Object> detailValue1 = new HashMap<String, Object>();
                                            detailValue1.put("ZLDWDM", "error没有占用" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")));
                                            detailValue1.put("TBBH", String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")));
                                            analysisDetailMap.put("error没有占用" + String.valueOf(xzdwAnalysisResult.get(i).get("KCTBDWDM2")), String.valueOf(xzdwAnalysisResult.get(i).get("KCTBBH2")), detailValue1);
                                        } else {
                                            //这个地方是KCTBDWDM1不够扣或者找不到，而KCTBDWDM2又找不到扣除图斑，那么应该还回线物面积
                                            dlMap.put(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC")), dlMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("DLMC"))) - (tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD")))));
                                            //从明细中去掉
                                            Map<String, Object> detailValue = (Map<String, Object>) analysisDetailMap.get(String.valueOf(xzdwAnalysisResult.get(i).get("QSDWDM1")) + String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")), String.valueOf(xzdwAnalysisResult.get(i).get("XZDWBH")));
                                            detailValue.put("CCMJ", Double.valueOf(String.valueOf(detailValue.get("CCMJ"))) - tmpLength * Double.parseDouble(String.valueOf(xzdwAnalysisResult.get(i).get("KD"))));
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            Map resultMap = new HashMap();
            resultMap.put("analysisArea", dlMap);
            List resultDetailList = new ArrayList();
            Iterator keyIterator = analysisDetailMap.keySet().iterator();
            while (keyIterator.hasNext()) {
                MultiKey tmpKey = (MultiKey) keyIterator.next();
                if (!isNull(tmpKey) && tmpKey.toString().indexOf("error") > -1) continue;//去除异常的数据
                resultDetailList.add(analysisDetailMap.get(tmpKey));
            }
            resultMap.put("analysisAreaDetail", resultDetailList);
            return resultMap;
        }
    }


    /**
     * 调用cs发布的webservice进行分析
     *
     * @param dltbLayerName 地类图斑图层名称
     * @param xzdwLayerName 现状地物图层名称
     * @param geometry      geojson格式的图形
     * @return
     */
    @Override
    public Map tdlyxzAnalysisByWcf(String geometry, String dltbLayerName, String xzdwLayerName, String dataSource) {
        Object geo = geometryService.readUnTypeGeoJSON(geometry);
        CoordinateReferenceSystem sourceCrs = null;
        CoordinateReferenceSystem layerCrs = spatialDao.getLayerCRS(dltbLayerName, dataSource);
        if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            sourceCrs = feature.getFeatureType().getCoordinateReferenceSystem();
            Geometry tmpGeo = (Geometry) feature.getDefaultGeometry();
            if (!tmpGeo.isValid() && !isNull(geometryService.validGeometry(tmpGeo)))
                throw new RuntimeException("图形不是有效图形:" + geometryService.validGeometry(tmpGeo).getMessage());
        } else if (geo instanceof FeatureCollection) {
            FeatureIterator iterator = ((FeatureCollection) geo).features();
            SimpleFeature feature = (SimpleFeature) iterator.next();
            sourceCrs = feature.getFeatureType().getCoordinateReferenceSystem();
            Geometry tmpGeo = (Geometry) feature.getDefaultGeometry();
            if (!tmpGeo.isValid() && !isNull(geometryService.validGeometry(tmpGeo)))
                throw new RuntimeException("图形不是有效图形:" + geometryService.validGeometry(tmpGeo).getMessage());
        }
        if (!isNull(sourceCrs) && !isNull(layerCrs) && !sourceCrs.equals(layerCrs)) {
            Object projectGeo = geometryService.project(geo, sourceCrs, layerCrs);
            geometry = geometryService.toFeatureJSON(projectGeo);
        }
        String wcfUrl = String.valueOf(AppPropertyUtils.getAppEnv("wcfUrl"));
        String wcfMethod = String.valueOf(AppPropertyUtils.getAppEnv("wcfMethod"));
        if (StringUtils.isBlank(dltbLayerName))
            throw new RuntimeException(getMessage("analysis.tdlyxz.params.error", "地类图斑图层名称为空！"));
        if (StringUtils.isBlank(xzdwLayerName))
            throw new RuntimeException(getMessage("analysis.tdlyxz.params.error", "线状地物图层名称为空！"));
        if (StringUtils.isBlank(geometry))
            throw new RuntimeException(getMessage("analysis.tdlyxz.params.error", "分析地块坐标为空！"));
        if (StringUtils.isBlank(wcfUrl))
            throw new RuntimeException(getMessage("tdlyxz.wcf.url.null"));
        if (StringUtils.isBlank(wcfMethod))
            throw new RuntimeException(getMessage("tdlyxz.wcf.method.null"));
        try {
            logger.debug("----启用webservice进行土地利用现状分析----");
            logger.debug("[wcf地址:]" + wcfUrl);
            Client client = new Client(new URL(wcfUrl));
            JSONObject params = new JSONObject();
            params.put(WCFTag.AnalyseGeoJSON.name(), geometry);
            params.put(WCFTag.TBLayerName.name(), dltbLayerName);
            params.put(WCFTag.XWLayerName.name(), xzdwLayerName);
            params.put(WCFTag.UseGlobalArea.name(), false);
            Object[] objects;
            try {
                objects = client.invoke(wcfMethod, new String[]{JSON.toJSONString(params)});
            } catch (Exception e) {
                logger.error(getMessage("tdlyxz.wcf.error", e.getLocalizedMessage()));
                throw new RuntimeException(getMessage("tdlyxz.wcf.error", e.getLocalizedMessage()));
            }
            Map analysisMap = JSON.parseObject(String.valueOf(objects[0]), Map.class);
            if (!analysisMap.isEmpty())
                logger.debug("[wcf 分析结果:]" + JSON.toJSONString(analysisMap));
            List<Map> features = new ArrayList<Map>();
            List<Map> summary = (List<Map>) MapUtils.getObject(analysisMap, WCFTag.Summary.name());
            List<Map> detail = (List<Map>) MapUtils.getObject(analysisMap, WCFTag.SummaryDetail.name());
            for (int i = 0; i < detail.size(); i++) {
                features.addAll((Collection<? extends Map>) detail.get(i).get(WCFTag.SummaryTBs.name()));
                features.addAll((Collection<? extends Map>) detail.get(i).get(WCFTag.SummaryXWs.name()));
            }
            Map dlbmSummary = (Map) summary.get(0).get("SummaryByDLDM");

            List<Map> detailList = new ArrayList<Map>();
            Map analysisArea = new HashMap();
            for (Map tmp : features) {
                Map detailMap = new HashMap();
                double area = MapUtils.getDoubleValue(tmp, WCFTag.DLMJ.name(), 0);
                String zldwmc = MapUtils.getString(tmp, WCFTag.ZLDWMC.name());
                String zldwdm = MapUtils.getString(tmp, WCFTag.ZLDWDM.name());
                String qsdwdm = MapUtils.getString(tmp, WCFTag.QSDWDM.name());
                String qsdwmc = MapUtils.getString(tmp, WCFTag.QSDWMC.name());
                /** 加上图形 用于后面的shp导出所用 modified by yxf **/
                if (tmp.containsKey(WCFTag.geometry.name())) {
                    Map geoMap = (Map) MapUtils.getObject(tmp, WCFTag.geometry.name());
                    Geometry geoEntity = geometryService.readGeoJSON(JSON.toJSONString(geoMap));
                    if (!isNull(geoEntity) && !geoEntity.isEmpty() && geoEntity.isValid())
                        detailMap.put(SE_SHAPE_FIELD, geoEntity.toText());
                }
                if (area <= 0) continue;
                detailMap.put("CCMJ", area);
                detailMap.put("QSDWDM", qsdwdm);
                detailMap.put("QSDWMC", qsdwmc);
                detailMap.put("ZLDWDM", StringUtils.isBlank(zldwdm) ? qsdwdm : zldwdm);
                detailMap.put("ZLDWMC", StringUtils.isBlank(zldwmc) ? qsdwmc : zldwmc);
                detailMap.put("DLBM", MapUtils.getString(tmp, WCFTag.DLBM.name()));
                if (tmp.containsKey(WCFTag.DLMC.name()))
                    detailMap.put("DLMC", MapUtils.getString(tmp, WCFTag.DLMC.name()));
                else
                    detailMap.put("DLMC", EnumUtils.findByDlbm(MapUtils.getString(tmp, WCFTag.DLBM.name())));
                detailMap.put("QSXZ", MapUtils.getString(tmp, WCFTag.QSXZ.name()));
                detailList.add(detailMap);
            }
            for (Object key : dlbmSummary.keySet()) {
                String dlbm = String.valueOf(key).substring(4);
                double area = MapUtils.getDouble(dlbmSummary, key, 0.0);
                if (area <= 0) continue;
                analysisArea.put(EnumUtils.findByDlbm(dlbm), area);
            }
            Map result = new HashMap();
            result.put("analysisAreaDetail", detailList);
            result.put("analysisArea", analysisArea);
            return result;

        } catch (Exception e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }

    /**
     * @param year
     * @param geometry
     * @return
     */
    @Override
    public Map tdlyxzAnalysisByWcf(String geometry, String year, String datasSource) {
        String dltb = TDXZ.DLTB.name().concat(LAYER_MIDDLE_FIX_H).concat(year);
        String xzdw = TDXZ.XZDW.name().concat(LAYER_MIDDLE_FIX_H).concat(year);
        return tdlyxzAnalysisByWcf(geometry, dltb, xzdw, datasSource);
    }

    /**
     * @param list
     * @param dm
     * @return
     */
    public String getDlmc(List<Map> list, String dm) {
        Iterator iterator = list.iterator();
        while (iterator.hasNext()) {
            Map entry = (Map) iterator.next();
            String dlbm = MapUtils.getString(entry, "DLBM");
            if (dlbm.equals(dm))
                return MapUtils.getString(entry, "DLMC");
        }
        return dm;
    }

    /**
     * 土地利用总体规划审查 , 单一类型分析
     *
     * @param layerType
     * @param regionCode
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public List<?> tdghscAnalysis(String layerType, String regionCode, String geometry, String[] outFields, String dataSource) {
        if (StringUtils.isBlank(layerType))
            throw new RuntimeException(getMessage("layer.type.notnull"));
        String layerName = layerType.concat(LAYER_MIDDLE_FIX_E).concat(Utils.formatRegionCode(regionCode));
        SeLayer layer = spatialDao.detectLayer(layerName, dataSource);
        if (layer == null) {
            logger.error(getMessage("analysis.tdghsc.layer.not.found", layerName));
            return null;
        }
        return intersect3(layerName, geometry, outFields, dataSource);
    }

    /**
     * 土地利用总体规划审查
     *
     * @param regionCode
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public Map tdghscAnalysis(String regionCode, String geometry, String[] outFields, String dataSource) {
        Map<String, Object> result = new HashMap<String, Object>();
        for (GHSC item : GHSC.values()) {
            result.put(item.getLabel(), tdghscAnalysis(item.name(), regionCode, geometry, outFields, dataSource));
        }
        return result;
    }

    /**
     * 土地利用总体规划审查 , 单一类型分析
     *
     * @param layerType
     * @param regionCode
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public String tdghscAnalysis2(String layerType, String regionCode, String geometry, String[] outFields, String dataSource) {
        List result = tdghscAnalysis(layerType, regionCode, geometry, outFields, dataSource);
        FeatureCollection collection = geometryService.list2FeatureCollection(result, null, null);
        return geometryService.toFeatureJSON(collection);
    }

    /**
     * 土地利用总体规划审查
     *
     * @param regionCode
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Override
    public Map tdghscAnalysis2(String regionCode, String geometry, String[] outFields, String dataSource) {
        Map<String, Object> result = new HashMap<String, Object>();
        for (GHSC item : GHSC.values()) {
            result.put(item.getLabel(), tdghscAnalysis2(item.name(), regionCode, geometry, outFields, dataSource));
        }
        return result;
    }

    /**
     * 组织土地规划分析的结果（所有图层类型）
     *
     * @param analysisResult
     * @return
     */
    @Override
    public Map tdghscResult(Map analysisResult) {

        Map resultMap = new HashMap();
        List<Map> detailList = new ArrayList<Map>();
        try {
            for (GHSC item : GHSC.values()) {
                List list = new ArrayList();
                Map total = new HashMap();
                Map ghMap = new HashMap();
                double totalArea = 0;
                JSONObject singleObj = JSON.parseObject(analysisResult.get(item.getLabel()).toString());
                JSONArray features = JSON.parseArray(singleObj.get("features").toString());
                if (features != null && features.size() > 0) {
                    totalArea = getTotalArea(features);
                    total.put("LXMC", "地块总面积");
                    total.put("AREA", totalArea);
                    list.add(total);
                    switch (item.ordinal()) {
                        case 0:
                            for (EnumUtils.TDYTQ obj : EnumUtils.TDYTQ.values()) {
                                List detail = new ArrayList();
                                double area = getAreaByLxdm("TDYTQLXDM", features, obj.getLxdm(), false);
                                double per = 0;
                                if (totalArea > 0)
                                    per = area / totalArea * 100;
                                Map temp = new HashMap();
                                temp.put("LXMC", obj.name());
                                temp.put("AREA", area);
                                temp.put("PER", per);
                                for (int i = 0; i < features.size(); i++) {
                                    LinkedHashMap detailMap = new LinkedHashMap();
                                    JSONObject feature = (JSONObject) features.get(i);
                                    Map properties = (Map) feature.get("properties");
                                    if (obj.getLxdm().equals(properties.get("TDYTQLXDM"))) {
                                        detailMap.put("类型名称", obj.name());
                                        detailMap.put("类型代码", properties.get("TDYTQLXDM"));
                                        detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                        detailMap.put("标识码", properties.get("BSM"));
                                        detailMap.put("面积", properties.get("SHAPE_AREA"));
                                        for (Object key : properties.keySet()) {
                                            if (String.valueOf(key).startsWith(OG_PRO_PERFIX))
                                                detailMap.put(key, properties.get(key));
                                        }
                                        detail.add(detailMap);
                                    }
                                }
                                temp.put("detail", detail);
                                detailList.addAll(detail);
                                list.add(temp);
                                logger.debug(obj.name() + area);
                            }
                            break;
                        case 1:
                            for (EnumUtils.JSYDGZQ obj : EnumUtils.JSYDGZQ.values()) {
                                List detail = new ArrayList();
                                double area = getAreaByLxdm("GZQLXDM", features, obj.getLxdm(), true);
                                double per = 0;
                                if (totalArea > 0)
                                    per = area / totalArea * 100;
                                Map temp = new HashMap();
                                temp.put("LXMC", obj.name());
                                temp.put("AREA", area);
                                temp.put("PER", per);
                                for (int i = 0; i < features.size(); i++) {
                                    LinkedHashMap detailMap = new LinkedHashMap();
                                    JSONObject feature = (JSONObject) features.get(i);
                                    Map properties = (Map) feature.get("properties");
                                    if (obj.getLxdm().equals(properties.get("GZQLXDM"))) {
                                        detailMap.put("类型名称", obj.name());
                                        detailMap.put("类型代码", properties.get("GZQLXDM"));
                                        detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                        detailMap.put("标识码", properties.get("BSM"));
                                        detailMap.put("面积", properties.get("SHAPE_AREA"));
                                        for (Object key : properties.keySet()) {
                                            if (String.valueOf(key).startsWith(OG_PRO_PERFIX))
                                                detailMap.put(key, properties.get(key));
                                        }
                                        detail.add(detailMap);
                                    }
                                }
                                temp.put("detail", detail);
                                detailList.addAll(detail);
                                list.add(temp);
                                logger.debug(obj.name() + area);
                            }
                            break;
                        case 2:
                            for (EnumUtils.GHJBNTTZ obj : EnumUtils.GHJBNTTZ.values()) {
                                List detail = new ArrayList();
                                double area = getAreaByLxdm("TZLXDM", features, obj.getLxdm(), false);
                                double per = 0;
                                if (totalArea > 0)
                                    per = area / totalArea * 100;
                                Map temp = new HashMap();
                                temp.put("LXMC", obj.name());
                                temp.put("AREA", area);
                                temp.put("PER", per);
                                for (int i = 0; i < features.size(); i++) {
                                    LinkedHashMap detailMap = new LinkedHashMap();
                                    JSONObject feature = (JSONObject) features.get(i);
                                    Map properties = (Map) feature.get("properties");
                                    if (obj.getLxdm().equals(properties.get("TZLXDM"))) {
                                        detailMap.put("类型名称", obj.name());
                                        detailMap.put("类型代码", properties.get("TZLXDM"));
                                        detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                        detailMap.put("标识码", properties.get("BSM"));
                                        detailMap.put("面积", properties.get("SHAPE_AREA"));
                                        for (Object key : properties.keySet()) {
                                            if (String.valueOf(key).startsWith(OG_PRO_PERFIX))
                                                detailMap.put(key, properties.get(key));
                                        }
                                        detail.add(detailMap);
                                    }
                                }
                                temp.put("detail", detail);
                                detailList.addAll(detail);
                                list.add(temp);
                                logger.debug(obj.name());
                            }
                            break;
                        case 3:
                            for (EnumUtils.MZZDJSXM obj : EnumUtils.MZZDJSXM.values()) {
                                List detail = new ArrayList();
                                double area = getAreaByLxdm("XMLXDM", features, obj.getLxdm(), false);
                                double per = 0;
                                if (totalArea > 0)
                                    per = area / totalArea * 100;
                                Map temp = new HashMap();
                                temp.put("LXMC", obj.name());
                                temp.put("AREA", area);
                                temp.put("PER", per);

                                for (int i = 0; i < features.size(); i++) {
                                    LinkedHashMap detailMap = new LinkedHashMap();
                                    JSONObject feature = (JSONObject) features.get(i);
                                    Map properties = (Map) feature.get("properties");
                                    if (obj.getLxdm().equals(properties.get("XMLXDM"))) {
                                        detailMap.put("类型名称", obj.name());
                                        detailMap.put("类型代码", properties.get("XMLXDM"));
                                        detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                        detailMap.put("标识码", properties.get("BSM"));
                                        detailMap.put("面积", properties.get("SHAPE_AREA"));
                                        for (Object key : properties.keySet()) {
                                            if (String.valueOf(key).startsWith(OG_PRO_PERFIX))
                                                detailMap.put(key, properties.get(key));
                                        }
                                        detail.add(detailMap);
                                    }
                                }
                                temp.put("detail", detail);
                                detailList.addAll(detail);
                                list.add(temp);
                                logger.debug(obj.getLxdm());
                            }
                            break;
                    }
                    ghMap.put("info", list);
                    resultMap.put(item.getLabel(), ghMap);
                }
            }
            if (detailList.size() > 0)
                resultMap.put("detail", JSON.toJSONString(detailList));
        } catch (Exception e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return resultMap;
    }

    /**
     * 组织土地规划分析的结果（单个图层类型）
     *
     * @param layerType
     * @param analysisResult
     * @return
     */
    @Override
    public Map tdghscResult(String layerType, String analysisResult) {
        Map resultMap = new HashMap();
        try {

            JSONObject fc = JSON.parseObject(analysisResult);
            JSONArray features = (JSONArray) fc.get("features");

            List list = new ArrayList();
            Map total = new HashMap();
            Map ghMap = new HashMap();
            double totalArea = 0;
            totalArea = getTotalArea(features);
            total.put("LXMC", "地块总面积");
            total.put("AREA", totalArea);
            list.add(total);
            for (GHSC item : GHSC.values()) {

                if (item.name().equals(layerType)) {
                    switch (item.ordinal()) {
                        case 0:
                            for (EnumUtils.TDYTQ obj : EnumUtils.TDYTQ.values()) {
                                List detail = new ArrayList();
                                double area = getAreaByLxdm("TDYTQLXDM", features, obj.getLxdm(), false);
                                double per = 0;
                                if (totalArea > 0)
                                    per = area / totalArea * 100;
                                Map temp = new HashMap();
                                temp.put("LXMC", obj.name());
                                temp.put("AREA", area);
                                temp.put("PER", per);
                                for (int i = 0; i < features.size(); i++) {
                                    Map detailMap = new HashMap();
                                    JSONObject feature = (JSONObject) features.get(i);
                                    Map properties = (Map) feature.get("properties");
                                    if (obj.getLxdm().equals(properties.get("TDYTQLXDM"))) {
                                        detailMap.put("类型名称", obj.name());
                                        detailMap.put("类型代码", properties.get("TDYTQLXDM"));
//                                        detailMap.put("ghsc", item.getLabel());
//                                        detailMap.put("地块名称", properties.get("G_plotName"));
//                                        detailMap.put("地块编号", properties.get("G_plotId"));
                                        detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                        detailMap.put("标识码", properties.get("BSM"));
                                        detailMap.put("面积", properties.get("SHAPE_AREA"));
                                        detail.add(detailMap);
                                    }
                                }
                                temp.put("detail", detail);
                                list.add(temp);
                                logger.debug(obj.name() + area);
                            }
                            break;
                        case 1:
                            for (EnumUtils.JSYDGZQ obj : EnumUtils.JSYDGZQ.values()) {
                                List detail = new ArrayList();
                                double area = getAreaByLxdm("GZQLXDM", features, obj.getLxdm(), true);
                                double per = 0;
                                if (totalArea > 0)
                                    per = area / totalArea * 100;
                                Map temp = new HashMap();
                                temp.put("LXMC", obj.name());
                                temp.put("AREA", area);
                                temp.put("PER", per);
                                for (int i = 0; i < features.size(); i++) {
                                    LinkedHashMap detailMap = new LinkedHashMap();
                                    JSONObject feature = (JSONObject) features.get(i);
                                    Map properties = (Map) feature.get("properties");
                                    if (obj.getLxdm().equals(properties.get("GZQLXDM"))) {
                                        detailMap.put("类型名称", obj.name());
//                                        detailMap.put("ghsc", item.getLabel());
                                        detailMap.put("类型代码", properties.get("GZQLXDM"));
//                                        detailMap.put("地块名称", properties.get("G_plotName"));
//                                        detailMap.put("地块编号", properties.get("G_plotId"));
                                        detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                        detailMap.put("标识码", properties.get("BSM"));
                                        detailMap.put("面积", properties.get("SHAPE_AREA"));
                                        detail.add(detailMap);
                                    }
                                }
                                temp.put("detail", detail);
                                list.add(temp);
                                logger.debug(obj.name() + area);
                            }
                            break;
                        case 2:
                            for (EnumUtils.GHJBNTTZ obj : EnumUtils.GHJBNTTZ.values()) {
                                List detail = new ArrayList();
                                double area = getAreaByLxdm("TZLXDM", features, obj.getLxdm(), false);
                                double per = 0;
                                if (totalArea > 0)
                                    per = area / totalArea * 100;
                                Map temp = new HashMap();
                                temp.put("LXMC", obj.name());
                                temp.put("AREA", area);
                                temp.put("PER", per);
                                for (int i = 0; i < features.size(); i++) {
                                    LinkedHashMap detailMap = new LinkedHashMap();
                                    JSONObject feature = (JSONObject) features.get(i);
                                    Map properties = (Map) feature.get("properties");
                                    if (obj.getLxdm().equals(properties.get("TZLXDM"))) {
                                        detailMap.put("类型名称", obj.name());
                                        detailMap.put("类型代码", properties.get("TZLXDM"));
//                                        detailMap.put("地块名称", properties.get("G_plotName"));
//                                        detailMap.put("地块编号", properties.get("G_plotId"));
                                        detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                        detailMap.put("标识码", properties.get("BSM"));
                                        detailMap.put("面积", properties.get("SHAPE_AREA"));
                                        detail.add(detailMap);
                                    }
                                }
                                temp.put("detail", detail);
                                list.add(temp);
                                logger.debug(obj.name());
                            }
                            break;
                        case 3:
                            for (EnumUtils.MZZDJSXM obj : EnumUtils.MZZDJSXM.values()) {
                                List detail = new ArrayList();
                                double area = getAreaByLxdm("XMLXDM", features, obj.getLxdm(), false);
                                double per = 0;
                                if (totalArea > 0)
                                    per = area / totalArea * 100;
                                Map temp = new HashMap();
                                temp.put("LXMC", obj.name());
                                temp.put("AREA", area);
                                temp.put("PER", per);

                                for (int i = 0; i < features.size(); i++) {
                                    LinkedHashMap detailMap = new LinkedHashMap();
                                    JSONObject feature = (JSONObject) features.get(i);
                                    Map properties = (Map) feature.get("properties");
                                    if (obj.getLxdm().equals(properties.get("XMLXDM"))) {
                                        detailMap.put("类型名称", obj.name());
                                        detailMap.put("类型代码", properties.get("XMLXDM"));
//                                        detailMap.put("ghsc", item.getLabel());
//                                        detailMap.put("地块名称", properties.get("G_plotName"));
//                                        detailMap.put("地块编号", properties.get("G_plotId"));
                                        detailMap.put("地块面积", properties.get("OG_SHAPE_AREA"));
                                        detailMap.put("标识码", properties.get("BSM"));
                                        detailMap.put("面积", properties.get("SHAPE_AREA"));
                                        detail.add(detailMap);
                                    }
                                }
                                temp.put("detail", detail);
                                list.add(temp);
                                logger.debug(obj.getLxdm());
                            }
                            break;
                    }
                    ghMap.put("info", list);
                    resultMap.put(item.getLabel(), ghMap);
                }
            }

        } catch (Exception e) {

            throw new RuntimeException(e.getLocalizedMessage());
        }
        return resultMap;
    }

    @Override
    public LinkedHashMap tdghscExcelData(Map result) {
        LinkedHashMap map = new LinkedHashMap();
        for (Object key : result.keySet()) {
            if ("detail".equals(key)) continue;
            List list = new ArrayList();
            List keyList = (List) ((Map) result.get(key)).get("info");
            for (int i = 1; i < keyList.size(); i++) {
                Map temp = (Map) keyList.get(i);
                List tempList = new ArrayList();
                tempList.add(temp.get("LXMC"));
                tempList.add(Math.round(Double.valueOf(temp.get("AREA").toString())));
                list.add(tempList);
            }
            map.put(key, list);
        }
        return map;
    }

    /**
     * 组织综合分析的导出excel数据
     *
     * @param data
     * @return
     */
    public LinkedHashMap multiExcelData(Map<String, Map> data) {
        LinkedHashMap excelMap = new LinkedHashMap();
        DecimalFormat df = new DecimalFormat("0.####");
        for (String key : data.keySet()) {
            List list = new ArrayList();
            List headerList = new ArrayList();
            Map aMap = data.get(key);
            String alias = MapUtils.getString(aMap, "alias");
            if (EnumUtils.MULTI_ANALYZE_TYPE.bp.name().equals(key.toString()) || EnumUtils.MULTI_ANALYZE_TYPE.gd.name().equals(key.toString())) {
                Map<String, Object> bpmap = (Map<String, Object>) aMap.get("result");
                List bpTotal = (List) bpmap.get("info");
                List bpDesc = (List) bpmap.get("detail");
                List list2 = new ArrayList();
                for (int i = 0; i < bpTotal.size(); i++) {
                    Map temp1 = (Map) bpTotal.get(i);
                    List tempList = new ArrayList();
                    tempList.add(temp1.get("type"));
                    tempList.add(df.format(Double.valueOf(temp1.get("area").toString())));
                    tempList.add(df.format(Double.valueOf(temp1.get("area_gq").toString())));
                    tempList.add(df.format(Double.valueOf(temp1.get("area_m").toString())));
                    list.add(tempList);
                }
                if (bpDesc.size() > 0) {
                    Map headMap = (Map) bpDesc.get(0);
                    for (Object k : headMap.keySet()) {
                        headerList.add(k);
                    }
                    list2.add(headerList);
                    for (int i = 0; i < bpDesc.size(); i++) {
                        Map temp2 = (Map) bpDesc.get(i);
                        List tempList = new ArrayList();
                        for (Object fieldkey : temp2.keySet()) {
                            Object value = temp2.get(fieldkey);
                            if (value instanceof Double)
                                tempList.add(df.format(MapUtils.getDoubleValue(temp2, fieldkey, 0)));
                            else
                                tempList.add(temp2.get(fieldkey));
                        }
                        list2.add(tempList);
                    }
                }
                if ("bp".equals(key.toString())) {
                    excelMap.put("报批汇总", list);
                    excelMap.put("报批详情", list2);
                } else {
                    excelMap.put("供地汇总", list);
                    excelMap.put("供地详情", list2);
                }
            } else {
                List keyList = new ArrayList();
                Object obj = (aMap).get("result");
                if (obj instanceof Map)
                    keyList = (List) ((Map) obj).get("detail");
                else if (obj instanceof List)
                    keyList = (List) obj;

                if (keyList.size() == 0)
                    return excelMap;
                headerList.addAll(ArrayUtils.mapConvertList((Map) keyList.get(0), ArrayUtils.TYPE.key));
                list.add(headerList);

                for (int i = 0; i < keyList.size(); i++) {
                    Map temp = (Map) keyList.get(i);
//                    List dataList = new ArrayList();
//                    for (Object item : temp.entrySet()) {
//                        Map.Entry entry = (Map.Entry) item;
//                        if (entry.getValue() instanceof Double)
//                            dataList.add(df.format(MapUtils.getDoubleValue(temp, entry.getKey(), 0)));
//                        else
//                            dataList.add(entry.getValue());
//                    }
                    list.add(ArrayUtils.mapConvertList(temp, ArrayUtils.TYPE.value));
                }
                excelMap.put(alias, list);
            }
        }
        return excelMap;
    }

    /**
     * 组织土地利用现状分析的结果
     *
     * @param analysisResult
     * @param tpl
     * @param unit
     * @return
     */
    @Override
    public List<Map> tdlyxzResult(Map analysisResult, String tpl, String unit) {
        List<Map> resultList = new ArrayList<Map>();
        double conv = 1.0;
        if (!isNull(unit)) {
            if (unit.equalsIgnoreCase(UNITS.ACRES.name()))
                conv = UNITS.ACRES.getConv();
            else if (unit.equalsIgnoreCase(UNITS.HECTARE.name()))
                conv = UNITS.HECTARE.getConv();
        }
        try {
            if (!isNull(analysisResult) && !analysisResult.isEmpty()) {
                List detailList = (List) analysisResult.get("analysisAreaDetail");
                if (!isNull(detailList) && detailList.size() > 0) {
                    List<Map> dwdmList = new ArrayList<Map>();
                    for (int i = 0; i < detailList.size(); i++) {
                        Map temp = new HashMap();
                        List list = new ArrayList();
                        int tag = 0;

                        Map map = (Map) detailList.get(i);
                        String zldwdm = MapUtils.getString(map, "ZLDWDM", "");  //坐落单位代码
                        String zldwmc = MapUtils.getString(map, "ZLDWMC", ""); //坐落单位名称
                        String qsdwmc = MapUtils.getString(map, "QSDWMC", ""); //权属单位名称
                        String qsdwdm = MapUtils.getString(map, "QSDWDM", ""); //权属单位代码

                        String xzqdm = "";

                        if (zldwdm.length() > 12)
                            xzqdm = zldwdm.substring(0, 12);
                        else
                            xzqdm = zldwdm;
                        if (xzqdm.indexOf("error") > -1) continue;

                        list.add(map);
                        temp.put("xzqdm", xzqdm);
                        temp.put("xzqmc", zldwmc);
                        temp.put("detail", list);
                        /***
                         * added  by yingxiufeng
                         * @since v2.1.4
                         */
                        temp.put("qsdwmc", qsdwmc);
                        temp.put("qsdwdm", qsdwdm);

                        if (dwdmList.size() > 0) {
                            for (Map dwdmMap : dwdmList) {
                                if (dwdmMap.get("xzqdm").toString().equals(xzqdm)) {
                                    List tempList = (List) dwdmMap.get("detail");
                                    tempList.add(map);
                                    dwdmMap.put("detail", tempList);
                                    tag = 1;
                                }
                            }
                        }
                        if (tag == 0) {
                            dwdmList.add(temp);
                        }
                    }
                    List<Map> dictList = getTdlyxzDictList(tpl);
                    for (Map dwMap : dwdmList) {
                        Map<String, List<Map>> dlList = ArrayUtils.listConvertMap((List<Map>) dwMap.get("detail"), "DLBM");
                        List<Map> dlbms = new ArrayList<Map>();

                        Iterator iterator = dlList.entrySet().iterator();
                        while (iterator.hasNext()) {
                            Map.Entry entry = (Map.Entry) iterator.next();
                            String key = String.valueOf(entry.getKey()).trim();
                            //属于某个dlbm的集合
                            List<Map> maps = (List<Map>) entry.getValue();

                            Map dlMap = new HashMap();
                            dlMap.put("dlbm", key);
                            dlMap.put("area", getDouValueByField(maps, "CCMJ") * conv);
                            dlMap.put("area_jt", getSumByQueryField(maps, "QSXZ", Arrays.asList(new String[]{"30", "31", "32", "33", "34", "40"}), "CCMJ") * conv);
                            dlMap.put("area_gy", getSumByQueryField(maps, "QSXZ", Arrays.asList(new String[]{"10", "20"}), "CCMJ") * conv);
                            dlbms.add(dlMap);
                        }
                        Map resultMap = new HashMap();
                        resultMap.put("xzqmc", MapUtils.getString(dwMap, "xzqmc"));
                        resultMap.put("xzqdm", MapUtils.getString(dwMap, "xzqdm"));
                        resultMap.put("qsdwdm", MapUtils.getString(dwMap, "qsdwdm"));
                        resultMap.put("qsdwmc", MapUtils.getString(dwMap, "qsdwmc"));
                        resultMap.put("sumArea", getDouValueByField(dlbms, "area"));
                        resultMap.put("sumAreaJt", getDouValueByField(dlbms, "area_jt"));
                        resultMap.put("sumAreaGy", getDouValueByField(dlbms, "area_gy"));
                        resultMap.put("categoryA", AnalysisUtils.getTdlyCategoryByGrade(dlbms, dictList, AnalysisUtils.TDLYXZ_GRADE.valueOf("A")));
                        resultMap.put("categoryB", AnalysisUtils.getTdlyCategoryByGrade(dlbms, dictList, AnalysisUtils.TDLYXZ_GRADE.valueOf("B")).get(0));
                        resultList.add(resultMap);
                    }
                    Map totalMap = new HashMap();
                    totalMap.put("xzqmc", "合计");
                    totalMap.put("xzqdm", "合计");
                    totalMap.put("qsdwdm", "合计");
                    totalMap.put("qsdwmc", "合计");
                    double sumArea = 0.0;
                    double sumJt = 0.0;
                    double sumGy = 0.0;
                    for (Map item : resultList) {
                        sumArea += MapUtils.getDoubleValue(item, "sumArea", 0.0);
                        sumJt += MapUtils.getDoubleValue(item, "sumAreaJt", 0.0);
                        sumGy += MapUtils.getDoubleValue(item, "sumAreaGy", 0.0);
                    }
                    totalMap.put("sumArea", sumArea);
                    totalMap.put("sumAreaJt", sumJt);
                    totalMap.put("sumAreaGy", sumGy);
                    totalMap.put("categoryA", getSumCategoryA(resultList));
                    totalMap.put("categoryB", getSumCategoryB(resultList));

                    resultList.add(totalMap);
                }
            }
        } catch (Exception e) {
            logger.error(getMessage("analysis.tdlyxz.parse.error", e.getLocalizedMessage()));
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return resultList;
    }

    /**
     * 组织土地利用现状的报件结果 面积单位都是公顷
     *
     * @param totalMap
     * @param report
     * @return
     */
    @Override
    public Map tdlyxzReport(Map totalMap, Map report) {
        assert totalMap != null;
        assert report != null;
        try {
            Map result = new HashMap();
            Map sumB = (Map) totalMap.get("categoryB");
            double aSumArea = MapUtils.getDoubleValue(totalMap, "sumArea", 0.0);

            double aNydArea = MapUtils.getDoubleValue(sumB, "农用地", 0.0);
            double aGdArea = MapUtils.getDoubleValue(sumB, "01", 0.0);
            double aJsydArea = MapUtils.getDoubleValue(sumB, "建设用地", 0.0);
            double aWlydArea = MapUtils.getDoubleValue(sumB, "未利用地", 0.0);

            double rSumArea = MapUtils.getDoubleValue(report, "area", 0.0);
            double rNydArea = MapUtils.getDoubleValue(report, "nydArea", 0.0);
            double rGdArea = MapUtils.getDoubleValue(report, "gdArea", 0.0);
            double rJsydArea = MapUtils.getDoubleValue(report, "jsydArea", 0.0);
            double rWlydArea = MapUtils.getDoubleValue(report, "wlydArea", 0.0);

            result.put("rArea", rSumArea);
            result.put("rNydArea", rNydArea);
            result.put("rGdArea", rGdArea);
            result.put("rJsydArea", rJsydArea);
            result.put("rWlydArea", rWlydArea);

            result.put("sumResult", (rSumArea - aSumArea) * 10000);// 比较结果 报件减去分析结果 单位换为㎡
            result.put("nydResult", (rNydArea - aNydArea) * 10000);
            result.put("gdResult", (rGdArea - aGdArea) * 10000);
            result.put("jsydResult", (rJsydArea - aJsydArea) * 10000);
            result.put("wlydResult", (rWlydArea - aWlydArea) * 10000);

            result.put("sumMistake", (rSumArea - aSumArea) == 0 ? 0 : (rSumArea - aSumArea) / rSumArea);  //误差
            result.put("nydMistake", (rNydArea - aNydArea) == 0 ? 0 : (rNydArea - aNydArea) / rNydArea);
            result.put("gdMistake", (rGdArea - aGdArea) == 0 ? 0 : (rGdArea - aGdArea) / rGdArea);
            result.put("jsydMistake", (rJsydArea - aJsydArea) == 0 ? 0 : (rJsydArea - aJsydArea) / rJsydArea);
            result.put("wlydMistake", (rWlydArea - aWlydArea) == 0 ? 0 : (rWlydArea - aWlydArea) / rWlydArea);
            return result;
        } catch (Exception e) {
            throw new RuntimeException(e.getMessage());
        }
    }

    @Override
    public List<LinkedHashMap> analysisExcelData(List list) {
        List<LinkedHashMap> result = new ArrayList<LinkedHashMap>();
        for (int i = 0; i < list.size(); i++) {
            LinkedHashMap map = new LinkedHashMap();
            List headerList = new ArrayList();
            List dataList = new ArrayList();
            JSONObject obj = (JSONObject) list.get(i);
            JSONArray children = (JSONArray) obj.get("children");
            for (int j = 0; j < children.size(); j++) {
                JSONArray child = (JSONArray) children.get(j);
                List temp = new ArrayList();

                for (int m = 0; m < child.size(); m++) {
                    JSONObject field = (JSONObject) child.get(m);
                    if (j == 0)
                        headerList.add(field.get("alias") == null ? field.get("name") : field.get("alias"));
                    if (field.get("value") instanceof BigDecimal)
                        temp.add(processValue(((BigDecimal) field.get("value")).doubleValue(), 3));
                    else
                        temp.add(field.get("value"));
                }
                dataList.add(temp);
            }
            map.put("name", obj.get("alias"));
            map.put("header", headerList);
            map.put("data", dataList);
            result.add(map);
        }
        return result;
    }

    /**
     * 一个sheet展示所有记录
     *
     * @param list
     * @return
     */
    @Override
    public List<LinkedHashMap> analysisExcelList(List list) {
        List<LinkedHashMap> result = new ArrayList<LinkedHashMap>();
        LinkedHashMap sheetMap = new LinkedHashMap();
        List headerList = new ArrayList();
        List dataList = new ArrayList();
        for (int i = 0; i < list.size(); i++) {
            Map item = (Map) list.get(i);
            JSONArray children = (JSONArray) item.get("children");
            for (int j = 0; j < children.size(); j++) {
                List<Map> child = (List<Map>) children.get(j);
                List temp = new ArrayList();
                for (int m = 0; m < child.size(); m++) {
                    JSONObject field = (JSONObject) child.get(m);
                    if (i == 0 && j == 0)
                        headerList.add(field.get("alias") == null ? field.get("name") : field.get("alias"));
                    if (field.get("value") instanceof BigDecimal)
                        temp.add(processValue(((BigDecimal) field.get("value")).doubleValue(), 3));
                    else
                        temp.add(field.get("value"));
                }
                dataList.add(temp);
            }
        }
        sheetMap.put("name", "sheet1");
        sheetMap.put("header", headerList);
        sheetMap.put("data", dataList);
        result.add(sheetMap);
        return result;
    }

    /**
     * @param list
     * @return
     */
    @Override
    public List<Map> gzqydExcelData(List list) {
        try {
            List<Map> result = new ArrayList<Map>();
            for (int i = 0; i < list.size(); i++) {
                Map tempMap = new HashMap();
                List<List> tempList = new ArrayList<List>();
                Map item = (Map) list.get(0);
                List<Map> values = (List<Map>) item.get("value");
                for (Map value : values) {
                    List temp = new ArrayList(8);
                    temp.add(0, String.valueOf(value.get("OG_PRO_XZQMC")));
                    temp.add(1, String.valueOf(value.get("OG_PRO_REGIONCODE")));
                    temp.add(2, String.valueOf(value.get("OG_PRO_PRONAME")));
                    temp.add(3, Double.valueOf(String.valueOf(value.get("011"))));
                    temp.add(3, Double.valueOf(String.valueOf(value.get("012"))));
                    temp.add(3, Double.valueOf(String.valueOf(value.get("020"))));
                    temp.add(3, Double.valueOf(String.valueOf(value.get("030"))));
                    temp.add(3, Double.valueOf(String.valueOf(value.get("040"))));
                    tempList.add(temp);
                }
                tempMap.put("value", tempList);
                tempMap.put("type", (String) item.get("type"));
                result.add(tempMap);
            }
            return result;
        } catch (NumberFormatException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }


    /**
     * 获取合计面积一级分类
     *
     * @param list
     * @return
     */
    public List<Map> getSumCategoryA(List<Map> list) {
        assert list != null;
        List<Map> result = null;
        if (list.size() > 0) {
            result = new ArrayList<Map>();
            List<Map> firstList = (List<Map>) list.get(0).get("categoryA");
            for (Map map : firstList) {
                Map temp = new HashMap();
                temp.put("dlbm", map.get("dlbm"));
                temp.put("dlmc", map.get("dlmc"));
                temp.put("area", 0);
                temp.put("area_jt", 0);
                temp.put("area_gy", 0);
                for (Map item : list) {
                    List<Map> groupList = (List<Map>) item.get("categoryA");
                    for (Map groupMap : groupList) {
                        if (temp.get("dlmc").equals(groupMap.get("dlmc"))) {
                            temp.put("area", MapUtils.getDouble(temp, "area", 0.0) + MapUtils.getDouble(groupMap, "area", 0.0));
                            temp.put("area_jt", MapUtils.getDoubleValue(temp, "area_jt", 0.0) + MapUtils.getDoubleValue(groupMap, "area_jt", 0.0));
                            temp.put("area_gy", MapUtils.getDoubleValue(temp, "area_gy", 0.0) + MapUtils.getDoubleValue(groupMap, "area_gy", 0.0));
                        }
                    }
                }
                result.add(temp);
            }
        }
        return result;
    }

    /**
     * 获取合计面积二级分类
     *
     * @param list
     * @return
     */
    public Map getSumCategoryB(List<Map> list) {
        assert list != null;
        Map result = null;
        if (list.size() > 0) {
            result = new HashMap();
            for (Map item : list) {
                Map categoryB = (Map) item.get("categoryB");
                for (Object key : categoryB.keySet()) {
                    result.put(key, MapUtils.getDouble(result, key, 0.0) + MapUtils.getDouble(categoryB, key, 0.0));
                }
            }
        }
        return result;
    }


    /**
     * 处理double类型数据,返回处理后的新数据
     *
     * @param value
     * @param precision
     * @return
     */
    public double processValue(Double value, int precision) {

        switch (precision) {
            case 1:
                return Math.round(value * 100) / 100.0;
            case 2:
                return Math.round(value * 100) / 100.00;
            case 3:
                return Math.round(value * 100) / 100.000;
            default:
                return Math.round(value * 100) / 100;

        }
    }

    /**
     * 快速检索
     *
     * @param value
     * @return
     */
    @Override
    public List search(String value, boolean returnGeometry) {
        if (searchConfig == null) throw new RuntimeException(getMessage("search.config.not.found"));
        List result = new ArrayList();
        List<Map> layers = (List) searchConfig.get(SEARCH_LAYERS);
        for (Map layer : layers) {
            String name = (String) layer.get("layer");
            String[] fields = ((String) layer.get("fields")).split(",");
            String where = getWhereClause(fields, value);
            result.addAll(spatialDao.query(name, where, fields, returnGeometry, 10, null));
        }
        return result;
    }

    /**
     * 根据相关条件输出图片
     *
     * @param where    查询子句
     * @param mapScale 出图的比例尺
     * @param width    图片的width
     * @param height   图片的height
     * @param type     出图类型(xz,gh)
     * @return
     */
    @Override
    public InputStream exportMap(String where, String mapScale, int width, int height, String type) {
        if (exportConfig == null) {
            logger.info("无配置信息");
            return null;
        }
        int bgWidth = width + 10;
        int bgHeight = height + 10;
        String imageSize = width + "," + height;
        double scale = 0;
        if (StringUtils.isNotBlank(mapScale))
            scale = Double.valueOf(mapScale);
        InputStream inputStream;
        List<BufferedImage> baseImages = new ArrayList<BufferedImage>();
        BufferedImage topImage = null;
        BufferedImage resultImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_BGR);
        Envelope envelope = null;
        Map operaLayer = (Map) exportConfig.get("operaLayer");
        List<Map> ghLayers = (List<Map>) exportConfig.get("ghLayers");
        List<Map> xzLayers = (List<Map>) exportConfig.get("xzLayers");
        String layerName = (String) operaLayer.get("layerName");
        String dataSource = (String) operaLayer.get("dataSource");
        String url = (String) operaLayer.get("url");
        List list = query(layerName, where, null, true, dataSource);
        if (list.size() > 0) {
            FeatureCollection featureCollection = geometryService.list2FeatureCollection(list, null, null);
            envelope = featureCollection.getBounds();
            topImage = getExportImage(envelope, url, imageSize, scale);

            if (type.equals(EXPORT_TYPE_GH)) {
                for (Map item : ghLayers) {
                    BufferedImage tempImage = getExportImage(envelope, (String) item.get("url"), bgWidth + "," + bgHeight, scale);
                    baseImages.add(tempImage);
                }
            } else if (type.equals(EXPORT_TYPE_XZ)) {
                for (Map item : xzLayers) {
                    BufferedImage tempImage = getExportImage(envelope, (String) item.get("url"), bgWidth + "," + bgHeight, scale);
                    baseImages.add(tempImage);
                }
            } else {
                logger.error("type is not found");
            }
            Graphics2D graphics2D = resultImage.createGraphics();
            resultImage = graphics2D.getDeviceConfiguration().createCompatibleImage(width, height, Transparency.TRANSLUCENT);
            graphics2D.dispose();
            graphics2D = resultImage.createGraphics();
            if (baseImages.size() > 0) {
                for (BufferedImage tempImage : baseImages) {
                    graphics2D.drawImage(tempImage, 0, 0, tempImage.getWidth(), tempImage.getHeight(), null);
                }
            }
            graphics2D.drawImage(topImage, 0, 0, topImage.getWidth(), topImage.getHeight(), null);
            graphics2D.dispose();
        } else {
            logger.info("未查询到地块!");
            return null;
        }
        try {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            ImageIO.write(resultImage, "png", out);
            inputStream = new ByteArrayInputStream(out.toByteArray());
        } catch (IOException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return inputStream;
    }

    private enum JCTB {
        BPDK, GDDK, DLTB, XZDW, JSYDGZQ, TDYTQ, LSYD, SSNYD,
        YXJS, YTJJS, BYXJS, JBNT //mas
    }

    private enum JTag {

        JC_GD_AREA, BP_AREA, BP_GD_AREA, YG_AREA, WG_AREA, WPYJ_AREA, WPYJ_GD_AREA, YXJSQ_AREA, JBNT_AREA, YBNTQ_AREA, PCMC, NZYPW, GDBH, SSNYD_AREA, LSYD_AREA, SSNYD_BH, LSYD_BH, SSSNYD_GD_AREA, LSYD_GD_AREA
    }

    // for mas
    private enum JTagMAS {
        YXJS_AREA, YTJJS_AREA, BYXJS_AREA, JBNT_MAS_AREA, NYD_AREA, GD_AREA, JSYD_AREA, WLYD_AREA, PCMC_MAS, NZYPW_MAS, GDBH_MAS, XMMC_MAS, YDDW_MAS, GDDH_MAS, HPDH_MAS
    }


    /**
     * 组织规划审查修改分析的结果数据
     *
     * @param data
     * @param fields
     * @return
     */
    @Override
    public Map ghscxgData(List<Map> data, List<Map> fields) {

        Map result = new HashMap();
        List<Map> showResult = new ArrayList<Map>();
        Map excelResult = new HashMap();
        List<Map> sheet = new ArrayList<Map>();

        try {
            List headerList = new ArrayList();
            List<List> dataList = new ArrayList<List>();
            for (int i = 0; i < fields.size(); i++) {
                headerList.add(fields.get(i).get("alias"));
            }
            Iterator<Map> dataIterator = data.iterator();
            while (dataIterator.hasNext()) {
                Map map = dataIterator.next();
                List list = new ArrayList();
                Iterator<Map> fieldIterator = fields.iterator();
                while (fieldIterator.hasNext()) {
                    Map f = fieldIterator.next();
                    if (f.containsKey("type")) {
                        String type = (String) f.get("type");
                        if (type.toLowerCase().equals("date")) {
                            String val = MapUtils.getString(map, f.get("name"));
                            if (val.indexOf("T") > 0) {
                                SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd\'T\'");
                                Date date = simpleDateFormat.parse(val);
                                map.put(f.get("name"), sdf.format(date));
                            } else
                                map.put(f.get("name"), DateFormat.getDateInstance().format(val));
                        }

                    }
                    list.add(isNull(map.get(f.get("name"))) ? "" : map.get(f.get("name")));
                }
                list.add(0, data.indexOf(map) + 1);
                double area = MapUtils.getDouble(map, SE_SHAPE_AREA);
                list.add(area);
                list.add(area * 0.0001);
                showResult.add(map);
                dataList.add(list);
            }
            headerList.add(0, "ID");
            headerList.add("重叠面积（平方米）");
            headerList.add("重叠面积（公顷）");
            excelResult.put("header", headerList);
            excelResult.put("data", dataList);
            sheet.add(excelResult);
            result.put("excel", sheet);
            result.put("result", showResult);
            return result;
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }

    /**
     * 综合分析 new
     *
     * @param multiParams
     * @param geometry
     * @param level       多级别控制 地方性版本会有所差异 默认:standard,eg. wuzhong .etc
     * @param tpl
     * @return
     */
    @Override
    public LinkedHashMap<String, Object> multiAnalyze(List<Map> multiParams, String geometry, String level, String tpl) {
        if (isNull(multiParams)) throw new RuntimeException("params cannot be null");
        if (isNull(geometry)) throw new RuntimeException("geometry cannot be null");
        LinkedHashMap<String, Object> analysisResult = new LinkedHashMap<String, Object>();
        try {
            Map layerMap = new HashMap();
            for (Map paraMap : multiParams) {
                String funid = MapUtils.getString(paraMap, "funid");
                Boolean isOpen = MapUtils.getBoolean(paraMap, "visible", false);
                String alias = paraMap.containsKey("alias") ? MapUtils.getString(paraMap, "alias") : MapUtils.getString(analysisAliasMap, funid);
                String condition = MapUtils.getString(paraMap, "condition", null); //分析的筛选条件
                if (paraMap.containsKey("layerName"))
                    layerMap.put(funid, MapUtils.getString(paraMap, "layerName"));
                if (isOpen) {
                    String dataSource = MapUtils.getString(paraMap, "dataSource");  //分析数据源
                    Map<String, Object> singleResult = new HashMap<String, Object>();              //存储各项的分析结果
                    singleResult.put("alias", alias);                                //分析结果显示的名称

                    if (EnumUtils.MULTI_ANALYZE_TYPE.xz.name().equals(funid)) {
                        Map map = new HashMap();
                        String dltb = MapUtils.getString(paraMap, "dltb");
                        String xzdw = MapUtils.getString(paraMap, "xzdw");
                        Boolean exportable = MapUtils.getBoolean(paraMap, "exportable", false);
                        List<Map> xzResult = new ArrayList<Map>();                       //组织好的现状分析结果
                        if (EnumUtils.MULTI_ANALYZE_LEVEL.mas.name().equalsIgnoreCase(level)) {
                            SimpleFeature feature;
                            Object analysisGeo = geometryService.readUnTypeGeoJSON(geometry);
                            if (analysisGeo instanceof SimpleFeature) {
                                map = tdlyxzAnalysis2(dltb, xzdw, geometry, dataSource);
                                Map rMap = processTdlyxzResultMas(map, tpl);
                                rMap.put("xzqmc", "地块1");  //增加分析地块的属性
                                xzResult.add(rMap);
                            } else if (analysisGeo instanceof FeatureCollection) {
                                FeatureIterator iterator = ((FeatureCollection) analysisGeo).features();
                                int i = 0;
                                while (iterator.hasNext()) {
                                    ++i;
                                    feature = (SimpleFeature) iterator.next();
                                    map = tdlyxzAnalysis2(dltb, xzdw, geometryService.toFeatureJSON(feature), dataSource);
                                    Map rMap = processTdlyxzResultMas(map, tpl);
                                    rMap.put("xzqmc", "地块".concat(Integer.toString(i))); //增加分析地块的属性
                                    xzResult.add(rMap);
                                }
                            }
                            if (isNotNull(xzResult) && xzResult.size() > 1) {
                                Map totalMap = new HashMap();
                                totalMap.put("xzqmc", "合计");
                                double sumArea = 0.0;
                                double sumJt = 0.0;
                                double sumGy = 0.0;
                                for (Map item : xzResult) {
                                    sumArea += MapUtils.getDoubleValue(item, "sumArea", 0.0);
                                    sumJt += MapUtils.getDoubleValue(item, "sumAreaJt", 0.0);
                                    sumGy += MapUtils.getDoubleValue(item, "sumAreaGy", 0.0);
                                }
                                totalMap.put("sumArea", sumArea);
                                totalMap.put("sumAreaJt", sumJt);
                                totalMap.put("sumAreaGy", sumGy);
                                totalMap.put("categoryA", getSumCategoryA(xzResult));
                                totalMap.put("categoryB", getSumCategoryB(xzResult));
                                xzResult.add(totalMap);
                            }
                        } else {
                            map = tdlyxzAnalysis2(dltb, xzdw, geometry, dataSource);
                            xzResult = tdlyxzResult(map, tpl, null);
                        }

                        //处理导出图形等信息
                        if (exportable && !isNull(dltb)) {
                            List<Map> detail = (List<Map>) map.get("analysisAreaDetail");
                            List exportData = new ArrayList<Map>();
                            for (Map item : detail) {
                                String wkt = MapUtils.getString(item, SE_SHAPE_FIELD);
                                if (geometryService.readWKT(wkt) instanceof MultiLineString) continue;
                                exportData.add(item);
                            }
                            if (exportData.size() > 0) {
                                FeatureCollection featureCollection = geometryService.list2FeatureCollection(exportData, null, null);
                                File shpFile = geometryService.exportToShp(geometryService.toFeatureJSON(featureCollection), getLayerCRS(dltb, dataSource));
                                if (shpFile.exists()) {
                                    FileStore fileStore = fileStoreService.save3(shpFile, UUIDHexGenerator.generate());
                                    singleResult.put("shpId", fileStore.getId());
                                }
                            }
                        }
                        singleResult.put("result", xzResult);
                        singleResult.put("unit", null);
                        singleResult.put("totalResult", xzResult.size() > 0 ? xzResult.get(xzResult.size() - 1) : new HashMap());
                        Map excelMap = new HashMap();
                        excelMap.put("result", xzResult);
                        excelMap.put("unit", "SQUARE");
                        singleResult.put("resultStr", JSON.toJSONString(excelMap));
                        analysisResult.put(funid, singleResult);

                    } else if (EnumUtils.MULTI_ANALYZE_TYPE.gh.name().equals(funid)) {
                        Map map = new HashMap();
                        String regionCode = "";
                        String layerType = MapUtils.getString(paraMap, "layerType");
                        String outFields = isNull(MapUtils.getString(paraMap, "outFields")) ? "*" : MapUtils.getString(paraMap, "outFields");
                        String[] fields = "*".equals(outFields) ? null : outFields.split(",");
                        String year = MapUtils.getString(paraMap, "year");
                        if (StringUtils.isNotBlank(year)) regionCode = year;
                        if (StringUtils.isNotBlank(layerType)) {
                            String tdghResult = tdghscAnalysis2(layerType, regionCode, geometry, fields, dataSource);
                            map = tdghscResult(layerType, tdghResult);
                        } else {
                            Map analysisMap = tdghscAnalysis2(regionCode, geometry, fields, dataSource);
                            map = tdghscResult(analysisMap);
                        }
                        singleResult.put("result", map);
                        singleResult.put("excelData", JSON.toJSONString(tdghscExcelData(map)));
                        analysisResult.put(funid, singleResult);
                    } else if (EnumUtils.MULTI_ANALYZE_TYPE.sgbwm.name().equals(funid)) {
                        String lyr = MapUtils.getString(paraMap, "layerName");

                        List<String> lyrsForDiff = new ArrayList<String>();
                        String _zd = MapUtils.getString(layerMap, EnumUtils.WZ_ANALYZE_TYPE.zd.name(), "");   //宗地
                        String _gd = MapUtils.getString(layerMap, EnumUtils.WZ_ANALYZE_TYPE.gd.name(), "");  //供地
                        String _zhd = MapUtils.getString(layerMap, EnumUtils.WZ_ANALYZE_TYPE.zhd.name(), ""); //征地
                        if (StringUtils.isNotBlank(_zd))
                            lyrsForDiff.add(_zd);
                        if (StringUtils.isNotBlank(_gd))
                            lyrsForDiff.add(_gd);
                        if (StringUtils.isNotBlank(_zhd))
                            lyrsForDiff.add(_zhd);
                        if (lyrsForDiff.size() > 0) {
                            String geometryNew = getDifferenceGeo(geometry, dataSource, lyrsForDiff, lyr);
                            if (isNotNull(geometryNew)) {
                                String outFields = isNull(MapUtils.getString(paraMap, "returnFields")) ? "*" : MapUtils.getString(paraMap, "returnFields");
                                String[] fields = "*".equals(outFields) ? null : outFields.split(",");
                                List<Map> intersectResult = (List<Map>) intersect3(lyr, geometryNew, fields, dataSource);

                                if (intersectResult.size() > 0) {
                                    List<Map<String, Object>> sgbwmResult = (List<Map<String, Object>>) multiAnalyResult(funid, lyr, fields, intersectResult);
                                    singleResult.put("result", sgbwmResult);
                                    analysisResult.put(funid, singleResult);
                                }
                            }
                        } else {
                            String outFields = isNull(MapUtils.getString(paraMap, "returnFields")) ? "*" : MapUtils.getString(paraMap, "returnFields");
                            String[] fields = "*".equals(outFields) ? null : outFields.split(",");
                            List<Map> intersectResult = (List<Map>) intersect3(lyr, geometry, fields, dataSource);
                            if (intersectResult.size() > 0) {
                                List<Map<String, Object>> sgbwmResult = (List<Map<String, Object>>) multiAnalyResult(funid, lyr, fields, intersectResult);
                                List<Map> _result = new ArrayList<Map>();
                                for (Map<String, Object> item : sgbwmResult) {
                                    _result.add(createConvertedMap(item));
                                }
                                singleResult.put("result", _result);
                                analysisResult.put(funid, singleResult);
                            }
                        }
                    } else {
                        if (EnumUtils.MULTI_ANALYZE_TYPE.tdlygh.name().equalsIgnoreCase(funid)
                                && EnumUtils.MULTI_ANALYZE_LEVEL.mas.name().equalsIgnoreCase(level)) {

                            /** 马鞍山订制版本(土地利用规划分析)**/
                            List<String> tdlyghLayers = Arrays.asList(MapUtils.getString(paraMap, "layerName", "").split(","));
                            /** 默认第一个是中心城区图层 **/
                            String criticalLyr = tdlyghLayers.get(0);
                            String fhghLyr;    //符合规划面图层
                            String ytjLyr;     //有条件建设区图层
                            String bfhghLyr;   //不符合规划面图层
                            String[] fields = new String[]{"OBJECTID"};
                            double geoArea = geometryService.getGeoArea(geometryService.readUnTypeGeoJSON(geometry), getLayerCRS(criticalLyr, dataSource));
                            List<Map> criticalResult = (List<Map>) intersect3(criticalLyr, geometry, fields, dataSource);

                            Map<String, Object> result = new HashMap<String, Object>();

                            if (isNotNull(criticalResult) && criticalResult.size() > 0) {
                                fhghLyr = tdlyghLayers.get(1);
                                ytjLyr = tdlyghLayers.get(2);
                                bfhghLyr = tdlyghLayers.get(3);
                                fields = ArrayUtils.add2Arrays(fields, "SFHRJBNT");
                            } else {
                                fhghLyr = tdlyghLayers.get(4);
                                ytjLyr = tdlyghLayers.get(5);
                                bfhghLyr = tdlyghLayers.get(6);
                            }
                            fields = ArrayUtils.add2Arrays(fields, new String[]{SE_SHAPE_FIELD, DLBM, DLMC});
                            List<Map<String, Object>> fhghIntersect = (List<Map<String, Object>>) intersect3(fhghLyr, geometry, fields, dataSource);
                            List<Map<String, Object>> bfhghIntersect = (List<Map<String, Object>>) intersect3(bfhghLyr, geometry, fields, dataSource);
                            List<Map<String, Object>> ytjIntersect = (List<Map<String, Object>>) intersect3(ytjLyr, geometry, fields, dataSource);

                            Map fhghMap = new HashMap();
                            Map bfhghMap = new HashMap();
                            Map ytjMap = new HashMap();

                            fhghMap.put("detail", fhghIntersect);
                            fhghMap.put("total", processTdlyghForMas(fhghIntersect));
                            bfhghMap.put("detail", bfhghIntersect);
                            bfhghMap.put("total", processTdlyghForMas(bfhghIntersect));
                            ytjMap.put("detail", ytjIntersect);
                            ytjMap.put("total", processTdlyghForMas(ytjIntersect));


                            if (bfhghIntersect.size() > 0) {
                                FeatureCollection featureCollection = geometryService.list2FeatureCollection(bfhghIntersect, null, null);
                                File shpFile = geometryService.exportToShp(geometryService.toFeatureJSON(featureCollection), getLayerCRS(bfhghLyr, dataSource));
                                if (shpFile.exists()) {
                                    FileStore fileStore = fileStoreService.save3(shpFile, UUIDHexGenerator.generate());
                                    bfhghMap.put("shpId", fileStore.getId());
                                }
                            }
                            if (ytjIntersect.size() > 0) {
                                FeatureCollection featureCollection = geometryService.list2FeatureCollection(ytjIntersect, null, null);
                                File shpFile = geometryService.exportToShp(geometryService.toFeatureJSON(featureCollection), getLayerCRS(ytjLyr, dataSource));
                                if (shpFile.exists()) {
                                    FileStore fileStore = fileStoreService.save3(shpFile, UUIDHexGenerator.generate());
                                    ytjMap.put("shpId", fileStore.getId());
                                }
                            }
                            result.put("fhgh", fhghMap);
                            result.put("bfhgh", bfhghMap);
                            result.put("ytj", ytjMap);
                            result.put("geoArea", geoArea);
                            singleResult.put("result", result);
                            analysisResult.put(funid, singleResult);

                        } else {
                            Boolean exportable = MapUtils.getBoolean(paraMap, "exportable", false);
                            String outFields = isNull(MapUtils.getString(paraMap, "returnFields")) ? "*" : MapUtils.getString(paraMap, "returnFields");
                            String[] fields = "*".equals(outFields) ? null : outFields.split(",");
                            /** 马鞍山定制版本需要保留相交图形 **/
                            if (EnumUtils.MULTI_ANALYZE_LEVEL.valueOf(level) == EnumUtils.MULTI_ANALYZE_LEVEL.mas &&
                                    !ArrayUtils.contains(fields, SE_SHAPE_FIELD, true))
                                fields = ArrayUtils.add2Arrays(fields, SE_SHAPE_FIELD);
                            String hyperlink = MapUtils.getString(paraMap, "hyperlink");
                            List<String> layers = Arrays.asList(MapUtils.getString(paraMap, "layerName", "").split(",")); // layers need to analyze

                            /** 存放多个图层的分析结果(单个分析类别)**/
                            List<Map<String, Object>> multiLists = new ArrayList<Map<String, Object>>();
                            List<Map> multiHyperlinks = new ArrayList<Map>(); //存放最终的超链接集合
                            List<Map> detailList = new ArrayList<Map>();
                            /** 存放多个图层的相交图形结果(单个分析类别)**/
                            List<Map<String, Object>> iGeos = new ArrayList<Map<String, Object>>();

                            /** 取出hyperlink 重新组织成[{key:url}...] **/
                            List<Map<String, String>> hyperLinkHolders = new ArrayList<Map<String, String>>();
                            if (!isNull(hyperlink)) {
                                List<String> hls = Arrays.asList(hyperlink.split(","));
                                for (String hl : hls) {
                                    String[] nameUrl = hl.split("\\|");
                                    Map<String, String> tmp = new HashMap<String, String>();
                                    tmp.put("key", nameUrl[0]);
                                    tmp.put("url", nameUrl[1]);
                                    hyperLinkHolders.add(tmp);
                                }
                            }

                            List<Map<String, Object>> intersectResult = new ArrayList<Map<String, Object>>();
                            List<Map> multiHyper = new ArrayList<Map>();
                            for (String layerName : layers) {
                                intersectResult = (List<Map<String, Object>>) intersect3(layerName, geometry, fields, dataSource);

                                /** 若存在筛选条件则进行筛选 仅支持mas版本**/
                                if (EnumUtils.MULTI_ANALYZE_LEVEL.valueOf(level) == EnumUtils.MULTI_ANALYZE_LEVEL.mas
                                        && StringUtils.isNotBlank(condition)) {
                                    Map cMap = AnalysisUtils.parseAnalysisCondition(condition);
                                    String operator = MapUtils.getString(cMap, "operator");
                                    String key = MapUtils.getString(cMap, "key");
                                    String value = MapUtils.getString(cMap, "value");
                                    for (Map map : intersectResult) {
                                        Object _val = map.get(key);
                                        if ("=".equals(operator)) {
                                            if (_val.equals(value))
                                                continue;
                                            else
                                                intersectResult.remove(map);
                                        }
                                    }
                                }

                                /** 对于吴中等地方性的特殊版本做单独处理(扣除相交部分的面积) **/
                                if (EnumUtils.MULTI_ANALYZE_LEVEL.wuzhong.name().equalsIgnoreCase(level)) {
                                    List<String> lyrsForDiff = new ArrayList<String>();
                                    if (EnumUtils.WZ_ANALYZE_TYPE.gd.name().equalsIgnoreCase(funid)) {
                                        String _zd = MapUtils.getString(layerMap, EnumUtils.WZ_ANALYZE_TYPE.zd.name(), "");
                                        if (StringUtils.isNotBlank(_zd))
                                            lyrsForDiff.addAll(Arrays.asList(_zd.split(",")));
                                    } else if (EnumUtils.WZ_ANALYZE_TYPE.zhd.name().equalsIgnoreCase(funid)) {
                                        lyrsForDiff.addAll(Arrays.asList(MapUtils.getString(layerMap, EnumUtils.WZ_ANALYZE_TYPE.zd.name(), "").split(",")));
                                    }
                                    if (lyrsForDiff.size() > 0) {
                                        for (Map i : intersectResult) {
                                            double newArea = getDifferenceArea(geometryService.readWKT(MapUtils.getString(i, SE_SHAPE_FIELD)), dataSource, lyrsForDiff);
                                            i.put(SE_SHAPE_AREA, newArea);
                                        }
                                    }
                                }
                                if (intersectResult.size() > 0) {
                                    List<Map<String, Object>> multiResult = (List<Map<String, Object>>) multiAnalyResult(funid, layerName, fields, intersectResult);
                                    multiLists.addAll(multiResult);
                                    /** 处理相交的图形 **/
                                    for (Map item : intersectResult) {
                                        Map<String, Object> temp = new HashMap<String, Object>();
                                        temp.putAll(item);
                                        iGeos.add(temp);
                                    }
                                }
                                for (Map hyperLinkHolder : hyperLinkHolders) {
                                    String key = MapUtils.getString(hyperLinkHolder, "key");  //xmmc
                                    String url = MapUtils.getString(hyperLinkHolder, "url"); //http://xxx?q={xmbh}
                                    List<Map> values = new ArrayList<Map>();
                                    for (Map r : intersectResult) {
                                        if (r.containsKey(key))
                                            values.add(Utils.updateMap(r.get(key), Utils.replaceHolder("{", "}", url, r), null));
                                    }
                                    multiHyper.add(Utils.updateMap(key, values, null));
                                }
                            }

                            for (Map item : multiHyper) {
                                multiHyperlinks.add(createConvertedMap(item, "GQ", "M"));
                            }
                            double geoArea = geometryService.getGeoArea(geometryService.readUnTypeGeoJSON(geometry), null);       //分析图形面积
                            if (EnumUtils.MULTI_ANALYZE_TYPE.bp.name().equals(funid) || EnumUtils.MULTI_ANALYZE_TYPE.gd.name().equals(funid)) {
                                Map<String, Object> bpResult = new HashMap<String, Object>();
                                Map<String, Object> bpResult1 = new HashMap<String, Object>();
                                Map<String, Object> bpResult2 = new HashMap<String, Object>();
                                Map<String, Object> bpResult3 = new HashMap<String, Object>();
                                double bpArea = getDouValueByField(intersectResult, SE_SHAPE_AREA); //占用面积
                                List<Map<String, Object>> infoList = new ArrayList<Map<String, Object>>();
                                bpResult1.put("type", EnumUtils.MULTI_ANALYZE_TYPE.bp.name().equals(funid) ? "已报" : "已供");
                                bpResult1.put("area", bpArea);
                                bpResult1.put("area_gq", bpArea * UNITS.HECTARE.getConv());
                                bpResult1.put("area_m", bpArea * UNITS.ACRES.getConv());
                                infoList.add(bpResult1);
                                bpResult2.put("type", EnumUtils.MULTI_ANALYZE_TYPE.bp.name().equals(funid) ? "未报" : "未供");
                                bpResult2.put("area", geoArea - bpArea);
                                bpResult2.put("area_gq", (geoArea - bpArea) * UNITS.HECTARE.getConv());
                                bpResult2.put("area_m", (geoArea - bpArea) * UNITS.ACRES.getConv());
                                infoList.add(bpResult2);
                                bpResult3.put("type", "合计");
                                bpResult3.put("area", geoArea);
                                bpResult3.put("area_gq", geoArea * UNITS.HECTARE.getConv());
                                bpResult3.put("area_m", geoArea * UNITS.ACRES.getConv());
                                infoList.add(bpResult3);
                                bpResult.put("info", infoList);

                                for (Map item : multiLists) {
                                    detailList.add(createConvertedMap(item, "GQ", "M"));
                                }
                                bpResult.put("detail", detailList);
                                singleResult.put("result", bpResult);
                            } else {
                                /** dataUrl 用于向某些业务系统请求关联数据 <br />
                                 ** 并结合现有数据 在综合分析的页面上展示 <br />
                                 ** modified by yxf on 2015/02/11 **/
                                String dataUrl = MapUtils.getString(paraMap, "dataUrl");
                                for (Map item : multiLists) {
                                    if (isNull(dataUrl))
                                        detailList.add(createConvertedMap(item, "GQ", "M"));
                                    else {
                                        String data = HttpRequest.sendRequest2(Utils.urlReplaceHolder(dataUrl, item), null);
                                        if (!isNull(data))
                                            item.putAll(JSON.parseObject(data, Map.class));
                                        detailList.add(createConvertedMap(item, "GQ", "M"));
                                    }
                                }
                                singleResult.put("result", detailList);
                                /** 对马鞍山的分析结果进行统一的统计(压盖面积，未压盖面积 结果个数等) **/
                                if (EnumUtils.MULTI_ANALYZE_LEVEL.mas.equals(EnumUtils.MULTI_ANALYZE_LEVEL.valueOf(level))) {
                                    Map statMap = new HashMap();
                                    String areaAlias = findAnalysisFieldAlias(Constant.SE_SHAPE_AREA);
                                    double coverArea = getDouValueByField(detailList, areaAlias);
                                    statMap.put("coverArea", coverArea);
                                    statMap.put("unCoverArea", geoArea - coverArea);
                                    statMap.put("count", detailList.size());
                                    if (!statMap.isEmpty())
                                        singleResult.put("stat", statMap);
                                }
                            }
                            singleResult.put("hyperlinks", multiHyperlinks);
                            singleResult.put("layer", layers);

                            /**  处理相交的图形 组织成featureCollection **/
                            if (iGeos.size() > 0) {
                                FeatureCollection fc = geometryService.list2FeatureCollection(iGeos, geometryService.getLayerCRS(layers.get(0), dataSource), null);
                                if (isNotNull(fc))
                                    singleResult.put("geojson", geometryService.toFeatureJSON(fc));
                            }
                            if (exportable && intersectResult.size() > 0) {
                                FeatureCollection featureCollection = geometryService.list2FeatureCollection(intersectResult, null, null);
//                                    File shpFile = geometryService.exportToShp(geometryService.toFeatureJSON(featureCollection), getLayerCRS(layers.get(0), dataSource));
                                File shpFile = geometryService.exportToShp(getLayerCRS(layers.get(0), dataSource), geometry, geometryService.toFeatureJSON(featureCollection));
                                if (shpFile.exists()) {
                                    FileStore fileStore = fileStoreService.save3(shpFile, UUIDHexGenerator.generate());
                                    singleResult.put("shpId", fileStore.getId());
                                }
                            }
                            /** 组织普通分析导出excel数据**/
                            Map<String, Map> _temp = new HashMap<String, Map>();
                            _temp.put(funid, singleResult);
                            for (Object res : _temp.keySet()) {
                                Map _r = _temp.get(res);
                                _r.put("excelData", JSON.toJSONString(multiExcelData(_temp)));
                            }
                            analysisResult.putAll(_temp);
                        }
                    }
                }
            }
            /** 对分析结果页面tpl 进行渲染**/
            if (!analysisResult.isEmpty()) {
                for (String key : analysisResult.keySet()) {
                    Map result = (Map) analysisResult.get(key);
                    result.put("level", level);
                    String analysisTpl = renderAnalysisTpl(result, key);
                    result.put("tpl", analysisTpl);
                }
                /** 生成分析报告的内容记录 **/
                generateSummaryReport(analysisResult);
            }
            return analysisResult;
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }

    /**
     * 处理土地利用现状分析结果(马鞍山版本,按照分析地块分组统计结果)
     *
     * @param analysisMap
     * @return
     */
    private Map processTdlyxzResultMas(Map analysisMap, String tpl) {
        Map result = new HashMap();
        if (analysisMap != null && !analysisMap.isEmpty()) {
            Map analysisArea = (Map) analysisMap.get("analysisArea");
            List<Map> detailList = (List) analysisMap.get("analysisAreaDetail");
            if (!isNull(detailList) && detailList.size() > 0) {
                List<Map> dictList = getTdlyxzDictList(tpl);
                //按照dlbm分组
                Map<String, List<Map>> dlList = ArrayUtils.listConvertMap(detailList, "DLBM");
                List<Map> dlbms = new ArrayList<Map>();

                Iterator iterator = dlList.entrySet().iterator();
                while (iterator.hasNext()) {
                    Map.Entry entry = (Map.Entry) iterator.next();
                    String key = String.valueOf(entry.getKey()).trim();
                    //属于某个dlbm的集合
                    List<Map> maps = (List<Map>) entry.getValue();

                    Map dlMap = new HashMap();
                    dlMap.put("dlbm", key);
                    dlMap.put("area", getDouValueByField(maps, "CCMJ"));
                    dlMap.put("area_jt", getSumByQueryField(maps, "QSXZ", Arrays.asList(new String[]{"30", "31", "32", "33", "34", "40"}), "CCMJ"));
                    dlMap.put("area_gy", getSumByQueryField(maps, "QSXZ", Arrays.asList(new String[]{"10", "20"}), "CCMJ"));
                    dlbms.add(dlMap);
                }
                result.put("sumArea", getDouValueByField(dlbms, "area"));
                result.put("sumAreaJt", getDouValueByField(dlbms, "area_jt"));
                result.put("sumAreaGy", getDouValueByField(dlbms, "area_gy"));
                result.put("categoryA", AnalysisUtils.getTdlyCategoryByGrade(dlbms, dictList, AnalysisUtils.TDLYXZ_GRADE.valueOf("A")));
                result.put("categoryB", AnalysisUtils.getTdlyCategoryByGrade(dlbms, dictList, AnalysisUtils.TDLYXZ_GRADE.valueOf("B")).get(0));
            }
        }
        return result;
    }

    /**
     * 处理土地利用规划分析结果(for mas)
     *
     * @param analysisDetail
     * @return
     */
    private Map processTdlyghForMas(List<Map<String, Object>> analysisDetail) {
        Map total = new HashMap();
        double areaUnused = 0;
        double areaJsyd = 0;
        double areaNyd = 0;
        double areaJbnt = 0;
        double sumArea = 0;
        for (Map<String, Object> item : analysisDetail) {
            String dlbm = MapUtils.getString(item, DLBM);
            if (dlbm.length() == 2) dlbm = dlbm.concat("0");
            double area = MapUtils.getDoubleValue(item, SE_SHAPE_AREA, 0.0);
            sumArea += area;
            if (item.containsKey("SFHRJBNT") && "1".equals(MapUtils.getString(item, "SFHRJBNT"))) {
                areaJbnt += area;
            } else {

                if (EnumUtils.TDLYXZ_THREE_TYPE.FARM.isContained(dlbm))
                    areaNyd += area;
                else if (EnumUtils.TDLYXZ_THREE_TYPE.BUILD.isContained(dlbm))
                    areaJsyd += area;
                else
                    areaUnused += area;
            }
        }
        total.put("unused", areaUnused);
        total.put("jsyd", areaJsyd);
        total.put("nyd", areaNyd);
        total.put("jbnt", areaJbnt);
        total.put("sum", sumArea);
        return total;
    }

    /**
     * 根据综合分析的结果组织分析报告
     *
     * @param map
     * @return
     */
    private void generateSummaryReport(LinkedHashMap<String, Object> map) {
        Map summary;
        DecimalFormat df = new DecimalFormat("#.####");
        String areaAlias = findAnalysisFieldAlias(Constant.SE_SHAPE_AREA);
        if (isNotNull(map) && !map.isEmpty()) {
            for (String key : map.keySet()) {
                Map item = (Map) map.get(key);
                summary = new HashMap();

                //获取分析类别的别名
                String alias = MapUtils.getString(item, "alias");
                summary.put("alias", alias.substring(0, alias.length() - 2));

                //获取分析后的结果 map/list
                List<Map> analysisList = new ArrayList<Map>();
                Map analysisMap = new HashMap();
                if (item.get("result") instanceof Map)
                    analysisMap = (Map) item.get("result");
                else if (item.get("result") instanceof List)
                    analysisList = (List<Map>) item.get("result");

                List<Map> info;
                try {
                    switch (EnumUtils.MULTI_ANALYZE_TYPE.valueOf(key)) {
                        case xz:
                            Map totalResult = (Map) item.get("totalResult");
                            List<Map> categoryA = (List<Map>) totalResult.get("categoryA");
                            summary.put("total", MapUtils.getDoubleValue(totalResult, "sumArea", 0.0));
                            for (Map _m : categoryA) {
                                String dlmc = MapUtils.getString(_m, "dlmc");
                                String area = df.format(MapUtils.getDoubleValue(_m, "area", 0.0));
                                if ("农用地".equals(dlmc))
                                    summary.put("nyd", area);
                                else if ("建设用地".equals(dlmc))
                                    summary.put("jsyd", area);
                                else if ("未利用地".equals(dlmc))
                                    summary.put("wlyd", area);
                                else
                                    continue;
                            }
                            break;
                        case bp:
                            info = (List) analysisMap.get("info");
                            summary.put("yb", df.format(MapUtils.getDoubleValue(info.get(0), "area", 0.0)));
                            summary.put("wb", df.format(MapUtils.getDoubleValue(info.get(1), "area", 0.0)));
                            summary.put("total", MapUtils.getDoubleValue(info.get(2), "area", 0.0));
                            break;
                        case gd:
                            info = (List) analysisMap.get("info");
                            summary.put("yg", df.format(MapUtils.getDoubleValue(info.get(0), "area", 0.0)));
                            summary.put("wg", df.format(MapUtils.getDoubleValue(info.get(1), "area", 0.0)));
                            summary.put("total", MapUtils.getDoubleValue(info.get(2), "area", 0.0));
                            break;
                        case cl:
                        case sp:
                            summary.put("total", analysisList.size());
                            break;
                        case tdlygh:
                            Map fhghMap = (Map) analysisMap.get("fhgh");
                            Map bfhghMap = (Map) analysisMap.get("bfhgh");
                            Map ytjMap = (Map) analysisMap.get("ytj");
                            summary.put("fhgh", ((Map) fhghMap.get("total")).get("sum"));
                            summary.put("bfhgh", ((Map) bfhghMap.get("total")).get("sum"));
                            summary.put("ytj", ((Map) ytjMap.get("total")).get("sum"));
                            summary.put("total", MapUtils.getDoubleValue(summary, "fhgh", 0.0) + MapUtils.getDoubleValue(summary, "bfhgh", 0.0) +
                                    MapUtils.getDoubleValue(summary, "ytj", 0.0));
                            break;
                        default:
                            double total = 0;
                            for (Map _m : analysisList) {
                                if (_m.containsKey(areaAlias))
                                    total += MapUtils.getDoubleValue(_m, areaAlias, 0.0);
                            }
                            summary.put("total", total);
                            break;
                    }

                } catch (Exception e) {
                    double total = 0;
                    for (Map _m : analysisList) {
                        if (_m.containsKey(areaAlias))
                            total += MapUtils.getDoubleValue(_m, areaAlias, 0.0);
                    }
                    summary.put("total", total);
                }
                item.put("summary", summary);
            }
        }
    }

    /**
     * 获取配置的分析字段对应的别名
     *
     * @param key
     * @return
     */
    private String findAnalysisFieldAlias(String key) {
        if (isNotNull(analysisFieldsMap) && !analysisFieldsMap.isEmpty()) {
            List<String> values = getAnalysisFieldAlias(key);
            if (isNotNull(values) && values.size() > 0) {
                return values.get(values.size() - 1);
            }
        }
        return key;
    }

    /**
     * 根据配置的tpl模板 渲染分析结果
     *
     * @param data
     * @param tplName
     * @return
     */
    private String renderAnalysisTpl(Map data, String tplName) {
        try {
            Map map = new HashMap();
            map.put("tplData", data);
            List<String> tpls = templateService.listTplNames(ANALYSIS_FTL_DIR);
            if (!isNull(tpls) && tpls.size() > 0) {
                if (tpls.contains(tplName.concat(TPL_SUFFIX)))
                    return templateService.getTemplate(map, ANALYSIS_FTL_DIR.concat(tplName.concat(TPL_SUFFIX)));
                else if (tpls.contains(ANALSYIS_DEFAULT_TPL))
                    return templateService.getTemplate(map, ANALYSIS_FTL_DIR.concat(ANALSYIS_DEFAULT_TPL));
            }
            return getMessage("template.not.exist", tplName.concat(TPL_SUFFIX));
        } catch (Exception e) {
            logger.error("render analysis tpl error:" + e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }

    }

    /**
     * 一个图形和多个图层的difference 部分的图形(geojson)
     *
     * @param geometry
     * @param dataSource
     * @param layerNames
     * @param targetLayer
     * @return
     */
    private String getDifferenceGeo(String geometry, String dataSource, List<String> layerNames, String targetLayer) {
        List<Geometry> dfGeos = new ArrayList<Geometry>();                                   //和所有图层difference之后的图形集合
        for (String lyr : layerNames) {
            List<Map> dfResult = (List<Map>) differenceByGeoJson(lyr, geometry, new String[]{SE_SHAPE_FIELD}, dataSource);
            if (isNull(dfResult) || (isNotNull(dfResult) && dfResult.size() == 0))
                return null;                        //有一个图层difference的结果是空 就表明这个图层完全覆盖分析图形 则表示不需要进行分析,返回null
            for (Map tmp : dfResult) {
                dfGeos.add(geometryService.readWKT(MapUtils.getString(tmp, SE_SHAPE_FIELD)));
            }
        }
        /** 对于存在geometryCollection的 进行分解 组织成新的geometry的list**/
        List<Geometry> list = new ArrayList<Geometry>();
        for (Geometry _g : dfGeos) {
            if (isGeometryCollection(_g))
                list.addAll(decomposeGC((GeometryCollection) _g));
            else
                list.add(_g);
        }
        Geometry lastGeo = getDuplicatedGeo(list, getLayerCRS(targetLayer, dataSource)); //对difference出来的结果再取相交部分
        if (!isNull(lastGeo) && !lastGeo.isEmpty() && lastGeo.isValid()) {
            return geometryService.toGeoJSON(lastGeo);
        }
        return null;
    }

    /**
     * 一个图形和多个图层的difference 部分的面积
     *
     * @param geometry
     * @param dataSource
     * @param layerNames
     * @return
     */
    private double getDifferenceArea(Geometry geometry, String dataSource, List<String> layerNames) {
        try {
            List<Geometry> geos = new ArrayList<Geometry>();
            List<Map> list = new ArrayList<Map>();
            double totArea = 0;
            for (String layerName : layerNames) {
                list.addAll((List<Map>) difference(layerName, geometry, null, dataSource));
            }
            if (!isNull(list) && list.size() > 0) {
                for (Map map : list) {
                    geos.add(geometryService.readWKT(MapUtils.getString(map, SE_SHAPE_FIELD)));
                }
            }
            if (geos.size() > 1) {
                /** 对于存在geometryCollection的 进行分解 组织成新的geometry的list**/
                List<Geometry> _list = new ArrayList<Geometry>();
                for (Geometry _g : geos) {
                    if (isGeometryCollection(_g))
                        _list.addAll(decomposeGC((GeometryCollection) _g));
                    else
                        _list.add(_g);
                }
                Geometry duplicatedGeo = getDuplicatedGeo(_list, null);
                if (!isNull(duplicatedGeo) && !duplicatedGeo.isEmpty())
                    totArea = getAreaByGeometry(duplicatedGeo, 1);
            } else {
                for (Map map : list) {
                    totArea += getAreaByGeometry(geometryService.readWKT(MapUtils.getString(map, SE_SHAPE_FIELD)), 1);
                }
            }
            return totArea;
        } catch (GeometryServiceException e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }

    /**
     * 综合分析结果处理
     *
     * @param funid
     * @param layerName
     * @param fields
     * @param multiResult
     * @return
     */
    @Override
    public List<?> multiAnalyResult(String funid, String layerName, String[] fields, List<?> multiResult) {
        try {
            if (multiResult.size() < 1) return null;
            List<String> rf = new ArrayList<String>();
            if (fields != null)
                rf = Arrays.asList(fields);
            List<Map<String, Object>> multiLists = new ArrayList<Map<String, Object>>();
            /** 单独处理sgbwm的分析结果 **/
            if (EnumUtils.MULTI_ANALYZE_TYPE.sgbwm.name().equals(funid)) {
                Map<String, List<Map>> xzqMap = ArrayUtils.listConvertMap((List<Map>) multiResult, "SSQY"); //按所属区域分组
                for (Map.Entry entry : xzqMap.entrySet()) {
                    String xzqdm = (String) entry.getKey();
                    String town = "";     //乡镇
                    String village = "";  //村
                    double totalArea = 0;
                    List<Map> value = (List<Map>) entry.getValue();
                    LinkedHashMap<String, Object> multiItem = new LinkedHashMap<String, Object>(); //存储最终的结果
                    if (isNotNull(xzqdm)) {
                        switch (xzqdm.length()) {
                            case 9:
                                town = xzqdm;
                                break;
                            case 12:
                                town = xzqdm.substring(0, 9);
                                village = xzqdm;
                                break;
                        }
                        multiItem.put("town", town);
                        multiItem.put("village", village);
                        Map<String, List<Map>> lbMap = ArrayUtils.listConvertMap(value, "SGBWMLB");
                        for (Map.Entry _e : lbMap.entrySet()) {
                            String lb = (String) _e.getKey();
                            double area = getAreaByList((List<Map>) _e.getValue(), false, 1, null);
                            multiItem.put(lb, area);
                            totalArea += area;
                        }
                        multiItem.put("total", totalArea);
                        multiLists.add(multiItem);
                    }
                }
            } else {
                for (int i = 0; i < multiResult.size(); i++) {
                    Map<String, Object> getMap = (Map<String, Object>) multiResult.get(i);
                    LinkedHashMap<String, Object> setMap = new LinkedHashMap<String, Object>();
                    /** 按照返回字段顺序排序 **/
                    for (String f : rf) {
                        if (getMap.containsKey(f)) {
                            setMap.put(f, getMap.get(f));
                        }
                    }
                    double area = MapUtils.getDoubleValue(getMap, SE_SHAPE_AREA, 0);
                    String wkt = MapUtils.getString(getMap, SE_SHAPE_FIELD);
                    Geometry geo = geometryService.readWKT(wkt);
                    if (geo instanceof Polygon || geo instanceof MultiPolygon)
                        setMap.put(SE_SHAPE_AREA, area);
                    /** 方便页面展示 此处转为geojson格式 **/
                    if (setMap.containsKey(SE_SHAPE_FIELD))
                        setMap.put(SE_SHAPE_FIELD, geometryService.toGeoJSON(geo));
                    setMap.put("GQ", area * UNITS.HECTARE.getConv());
                    setMap.put("M", area * UNITS.ACRES.getConv());
                    if ("SDE.CKQ".equals(layerName.toUpperCase())) {
                        setMap.put("KCTYPE", "采矿权");
                    } else if ("SDE.TKQ".equals(layerName.toUpperCase())) {
                        setMap.put("KCTYPE", "探矿权");
                    }
                    multiLists.add(setMap);
                }
            }
            return multiLists;
        } catch (Exception e) {
            logger.error(getMessage("analysis.multi.result.error", e.getLocalizedMessage()));
        }
        return null;
    }

    /**
     * get layer crs by name and ds
     *
     * @param layerName
     * @param dataSource
     * @return
     */
    @Override
    public CoordinateReferenceSystem getLayerCRS(String layerName, String dataSource) {
        try {
            return spatialDao.getLayerCRS(layerName, dataSource);
        } catch (Exception e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }

    /**
     * 转换分析后得到的map 将字段名转换为对应的字段别名
     *
     * @param map
     * @param excludeFields 需要排除的字段 即返回的map里不包含这些key
     * @return
     * @since v2.1.0
     */
    @Override
    public Map createConvertedMap(Map map, String... excludeFields) {
        if (isNull(analysisFieldsMap)) throw new RuntimeException("analysis fields map not found");
        assert map != null;
        try {
            Map result;
            if (map instanceof LinkedHashMap)
                result = new LinkedHashMap();
            else
                result = new HashMap();
            for (Object object : map.entrySet()) {
                Map.Entry entry = (Map.Entry) object;
                String key = entry.getKey().toString();
                boolean contains = false;    //是否属于排除field集合
                if (!isNull(excludeFields)) {
                    for (int i = 0; i < excludeFields.length; i++) {
                        if (key.equals(excludeFields[i])) {
                            contains = true;
                            break;
                        }
                    }
                }
                if (contains) continue;
                List<String> aliases = getAnalysisFieldAlias(key);
                if (aliases.size() > 0)
                    key = aliases.get(aliases.size() - 1);
                result.put(key, entry.getValue());
            }
            return result;
        } catch (Exception e) {
            throw new RuntimeException(e.getMessage());
        }
    }

    /**
     * 根据geometry分析获取其行政区代码
     *
     * @param feature
     * @param layerName
     * @return
     */
    public String findXzqdm(SimpleFeature feature, String layerName) {
        assert feature != null;
        assert layerName != null;
        String dataSource = null;
        Point center = null;
        try {
            if (layerName.contains(".")) {
                String[] array = layerName.split("\\.");
                dataSource = array[0] != null ? array[0].toLowerCase() : null;
            }
            Geometry geometry = (Geometry) feature.getDefaultGeometry();
            CoordinateReferenceSystem layerCRS = spatialDao.getLayerCRS(layerName, dataSource);
            CoordinateReferenceSystem sourceCRS = feature.getFeatureType().getCoordinateReferenceSystem();
            if (isNull(sourceCRS))
                logger.error("source crs is null");
            if (!isNull(sourceCRS) && !isNull(layerCRS) && !layerCRS.equals(sourceCRS))
                geometry = geometryService.project(geometry, sourceCRS, layerCRS);
            center = geometryService.getGeometryCentre(geometry);
            List<Map> list = (List<Map>) intersect(layerName, center, null, dataSource);
            if (!isNull(list) && list.size() > 0) {
                Map map = list.get(0);
                if (map.containsKey("XZQDM"))
                    return MapUtils.getString(map, "XZQDM");
            }
            return null;
        } catch (Exception e) {
            throw new RuntimeException(getMessage("xzqdm.find.error", e.getMessage(), layerName, isNull(center) ? null : center.toText()));
        }
    }

    /**
     * 根据配置文件中的映射关系获取对应分析结果字段对应的别名
     *
     * @param key
     * @return
     * @since v2.1.0
     */
    private List<String> getAnalysisFieldAlias(String key) {
        List<String> fields = new ArrayList<String>();
        for (Object property : analysisFieldsMap.entrySet()) {
            if (Pattern.compile(((Map.Entry) property).getKey().toString())
                    .matcher(key).matches()) {
                fields.add(((Map.Entry) property).getValue().toString());
            }
        }
        return fields;
    }


    /**
     * sum area of list
     *
     * @param list
     * @param removeDuplicated 是否去除重叠部分的面积
     * @param conv
     * @return
     */
    public double getAreaByList(List<Map> list, boolean removeDuplicated, double conv, CoordinateReferenceSystem crs) {
        if (isNull(conv)) conv = 1;
        double area = 0;
        double duplicatedArea = 0;
        List<Geometry> geos = new ArrayList<Geometry>();
        try {
            for (Map map : list) {
                String wktPolygon = MapUtils.getString(map, SE_SHAPE_FIELD);
                geos.add(geometryService.readWKT(wktPolygon));
                double value = MapUtils.getDoubleValue(map, SE_SHAPE_AREA, 0.0);
                area += value;
            }
        } catch (GeometryServiceException e) {
            throw new RuntimeException(e.getMessage());
        }
        if (geos.size() > 1 && removeDuplicated) {
            /** 对于存在geometryCollection的 进行分解 组织成新的geometry的list**/
            List<Geometry> _list = new ArrayList<Geometry>();
            for (Geometry _g : geos) {
                if (isGeometryCollection(_g))
                    _list.addAll(decomposeGC((GeometryCollection) _g));
                else
                    _list.add(_g);
            }
            Geometry duplicatedGeo = getDuplicatedGeo(_list, crs);
            if (duplicatedGeo != null && !duplicatedGeo.isEmpty())
                duplicatedArea = duplicatedGeo.getArea();
        }
        return (area - duplicatedArea) * conv;
    }

    /**
     * 从一系列geometry中找出所有重叠的部分
     *
     * @param geos [geometry] 必须都是geometry 不支持geometryCollection
     * @return
     */
    public Geometry getDuplicatedGeo(List<Geometry> geos, CoordinateReferenceSystem crs) {
        try {
            if (!isNull(geos) && geos.size() > 0) {
                Geometry geo0 = geos.get(0);
                geos.remove(0);
                if (geos.size() == 0)
                    return geo0;
                else {
                    if (isNull(crs))
                        return geo0.intersection(getDuplicatedGeo(geos, null));
                    else
                        return geometryService.readWKT(agsGeoemtryService.intersection(getDuplicatedGeo(geos, crs).toText(), geo0.toText(), crs.toWKT()));
                }
            }
            return null;
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }


    /**
     * is geometry Collection
     *
     * @param g
     * @return
     */
    private boolean isGeometryCollection(Geometry g) {
        return ("com.vividsolutions.jts.geom.GeometryCollection").equals(g.getClass().getName());
    }

    /**
     * 分解一个geometryCollection
     *
     * @param geometryCollection
     * @return
     */
    public List<Geometry> decomposeGC(GeometryCollection geometryCollection) {
        assert geometryCollection != null;
        try {
            List<Geometry> list = new ArrayList<Geometry>();
            for (int i = 0; i < geometryCollection.getNumGeometries(); i++) {
                Geometry singleGeo = geometryCollection.getGeometryN(i);
                if (singleGeo instanceof GeometryCollection)
                    list.addAll(decomposeGC((GeometryCollection) singleGeo));
                else {
                    if (isNotNull(singleGeo) && !singleGeo.isEmpty() && singleGeo.isSimple())
                        list.add(singleGeo);
                }
            }
            return list;
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
    }

    /**
     * get feature's area
     *
     * @param feature
     * @return
     */
    @Deprecated
    public double getAreaByFeature(SimpleFeature feature, double conv) {
        if (isNull(conv)) conv = 1;
        double area = 0;
        try {
            CoordinateReferenceSystem sourceCRS = feature.getFeatureType().getCoordinateReferenceSystem();
            CoordinateReferenceSystem targetCrs = geometryService.parseUndefineSR("2364");
            Geometry geometry = (Geometry) feature.getDefaultGeometry();
            if (sourceCRS != null && sourceCRS instanceof GeographicCRS) {
                geometry = geometryService.project((Geometry) feature.getDefaultGeometry(), sourceCRS, targetCrs);
            }
            if (isPolygon(geometry))
                area = geometry.getArea();
            else
                area = geometry.getLength();
        } catch (GeometryServiceException e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return area * conv;
    }

    /**
     * get area by some field
     *
     * @param list
     * @param field
     * @param value 多个值逗号隔开
     * @param conv
     * @return
     */
    public double getAreaByField(List<Map> list, String field, String value, double conv) {
        if (isNull(conv)) conv = 1;
        assert field != null;
        assert value != null;
        double area = 0;
        List<String> array = Arrays.asList(value.split(","));
        for (Map map : list) {
            if (array.contains(map.get(field)))
                area += (Double) map.get(SE_SHAPE_AREA);
        }
        return area * conv;
    }

    /**
     * @param list
     * @param field
     * @param value
     * @param areaField SHAPE_AREA or other field name
     * @param conv
     * @return
     */
    public double getAreaByField(List<Map> list, String field, String value, String areaField, double conv) {
        if (isNull(conv)) conv = 1;
        assert field != null;
        assert value != null;
        double area = 0;
        List<String> array = Arrays.asList(value.split(","));
        for (Map map : list) {
            if (array.contains(map.get(field)))
                area += (Double) map.get(areaField);
        }
        return area * conv;
    }

    /**
     * get string value by some field
     *
     * @param list
     * @param field
     * @return
     */
    public String getStrValueByField(List<Map> list, String field) {
        StringBuilder sb = new StringBuilder();
        assert field != null;
        for (Map map : list) {
            if (map.containsKey(field)) {
                if (StringUtils.isNotBlank(sb.toString()))
                    sb.append(",");
                if (!isNull(map.get(field)))
                    sb.append(map.get(field));
            }
        }
        return sb.toString();
    }

    /**
     * 获取字符串 多个逗号隔开 .eg xxx-11-33,cc-dd-vv
     *
     * @param list
     * @param fields
     * @param hyphen 连字符
     * @return
     */
    public String getStrValueByField(List<Map> list, String[] fields, String hyphen) {

        StringBuilder sb = new StringBuilder();
        assert fields != null;
        List<String> array = Arrays.asList(fields);
        for (Map map : list) {
            if (StringUtils.isNotBlank(sb.toString()))
                sb.append(",");
            for (String f : array) {
                if (map.containsKey(f)) {
                    if (!isNull(map.get(f))) {
                        if (array.indexOf(f) != 0)
                            sb.append(hyphen);
                        sb.append(map.get(f));
                    }
                }
            }
        }
        return sb.toString();
    }

    /**
     * get double value by some field
     *
     * @param list
     * @param field
     * @return
     */
    public double getDouValueByField(List list, String field) {
        double value = 0;
        assert field != null;
        for (Object _item : list) {
            Map map = (Map) _item;
            if (isNull(map.get(field))) continue;
            value += MapUtils.getDoubleValue(map, field, 0.0);
        }
        return value;
    }

    /**
     * @param list
     * @param queryField
     * @param queryValue
     * @param sumField
     * @return
     */
    public double getSumByQueryField(List<Map> list, String queryField, List queryValue, String sumField) {
        double result = 0;
        assert queryField != null;
        assert queryValue != null;
        for (Map map : list) {
            if (queryValue.contains(MapUtils.getString(map, queryField))) {
                result += MapUtils.getDoubleValue(map, sumField, 0.0);
            }
        }
        return result;
    }


    /**
     * 根据bbox范围和相关参数,获取输出图片
     *
     * @param envelope
     * @param url
     * @param imageSize
     * @param mapScale
     * @return
     */
    private BufferedImage getExportImage(Envelope envelope, String url, String imageSize, double mapScale) {
        BufferedImage image = null;
        String bbox = "bbox=" + envelope.getMinX() + "," + envelope.getMinY() + "," + envelope.getMaxX() + "," + envelope.getMaxY();
        String size = "size=" + imageSize;
        StringBuilder exportUrl = new StringBuilder();
        exportUrl.append(url);
        exportUrl.append("/export?");
        exportUrl.append(bbox);
        exportUrl.append("&");
        exportUrl.append(size);
        exportUrl.append("&");
        exportUrl.append("&");
        if (mapScale > 0)
            exportUrl.append("mapScale=" + mapScale);
        exportUrl.append("&");
        exportUrl.append("transparent=true&");
        exportUrl.append("format=png&");
        exportUrl.append("f=image");
        HttpClient httpClient = new HttpClient();
        GetMethod getMethod = new GetMethod(exportUrl.toString());
        try {
            httpClient.executeMethod(getMethod);
            InputStream inputStream = new ByteArrayInputStream(getMethod.getResponseBody());
            if (inputStream.available() > 0)
                image = ImageIO.read(inputStream);
        } catch (IOException e) {
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return image;
    }


    /**
     * 统计数组的面积值
     *
     * @param features
     * @return
     */
    private double getTotalArea(JSONArray features) {

        double area = 0;
        for (int i = 0; i < features.size(); i++) {
            JSONObject object = (JSONObject) features.get(i);
            JSONObject property = (JSONObject) object.get("properties");
            area = area + Double.parseDouble(property.get("SHAPE_AREA").toString());
        }
        return area;
    }

    /**
     * 根据类型代码的值统计面积
     *
     * @param field
     * @param features
     * @param lxdm
     * @param fuzzyMatch 是否模糊匹配 匹配前两位代码
     * @return
     */
    private double getAreaByLxdm(String field, JSONArray features, String lxdm, boolean fuzzyMatch) {
        double area = 0;
        for (int i = 0; i < features.size(); i++) {
            JSONObject object = (JSONObject) features.get(i);
            JSONObject property = (JSONObject) object.get("properties");
            boolean matched;
            if (property.get(field) == null)
                return 0;
            else {
                String val = String.valueOf(property.get(field));
                matched = fuzzyMatch ? val.equals(lxdm) || val.startsWith(lxdm.substring(0, 2)) : val.equals(lxdm);
            }
            if (matched)
                area = area + Double.parseDouble(property.get("SHAPE_AREA").toString());
        }
        return area;
    }

    /**
     * @param fields
     * @param keyword
     * @return
     */
    private String getWhereClause(String[] fields, String keyword) {
        String or = " or ";
        StringBuilder b = new StringBuilder();
        for (String field : fields) {
            b.append(field);
            b.append(" like '");
            b.append(keyword);
            b.append("%'");
            b.append(or);
        }
        b.delete(b.length() - or.length(), b.length());
        return b.toString();
    }


    /**
     * GeoJSON to Map
     *
     * @param geoJSON single geometry
     * @return
     */
    private Map<String, Object> geoJSON2Map(String geoJSON, CoordinateReferenceSystem targetCRS) {
        Object geo = geometryService.readUnTypeGeoJSON(geoJSON);
        Map<String, Object> columns = null;
        if (geo instanceof Geometry) {
            columns = new HashMap<String, Object>();
            columns.put(SE_SHAPE_FIELD, ((Geometry) geo).toText());
        } else if (geo instanceof SimpleFeature) {
            SimpleFeature feature = (SimpleFeature) geo;
            CoordinateReferenceSystem sourceCRS;
            sourceCRS = feature.getFeatureType().getCoordinateReferenceSystem();
            if (isNull(sourceCRS))
                sourceCRS = geometryService.readFeatureJSONCRS(geoJSON);
            columns = geometryService.simpleFeature2Map(feature);
            try {
                for (String key : columns.keySet()) {
                    if (columns.get(key) instanceof Geometry) {
                        Geometry geometry = (Geometry) columns.get(key);
                        if (targetCRS != null && sourceCRS != null)
                            geometry = geometryService.project(geometry, sourceCRS, targetCRS);
                        columns.put(SE_SHAPE_FIELD, geometry.toText());
                        columns.remove(key);
                        break;
                    }
                }
            } catch (Exception e) {
                logger.error(e.getLocalizedMessage());
                throw new RuntimeException(e.getLocalizedMessage());
            }
        } else if (geo instanceof GeometryCollection) {
            throw new RuntimeException("current omp version don't support GeometryCollection ");
        } else if (geo instanceof FeatureCollection) {
            throw new RuntimeException("current omp version don't support FeatureCollection ");
        } else {
            throw new RuntimeException(getMessage("geometry.undefined"));
        }
        return columns;
    }

    /**
     * feature to geometry
     *
     * @param feature
     * @param geometry
     * @return
     */
    private Geometry setFeaturePros2Geo(SimpleFeature feature, Geometry geometry) {
        Map<String, Object> map = null;
        if (geometry instanceof GeometryCollection) {
            for (int i = 0; i < geometry.getNumGeometries(); i++) {
                Geometry geo = geometry.getGeometryN(i);
                map = new HashMap<String, Object>();
                for (Property p : feature.getProperties()) {
                    if (p.getName().equals("geometry")) continue;
                    map.put(p.getName().getLocalPart(), p.getValue());
                }
                geo.setUserData(map);
            }
        } else {
            map = new HashMap<String, Object>();
            for (Property p : feature.getProperties()) {
                if (p.getName().getLocalPart().equals("geometry")) continue;
                map.put(p.getName().getLocalPart(), p.getValue());
            }
            geometry.setUserData(map);
        }
        return geometry;
    }

    /**
     * add pros 2 map  list
     *
     * @param list
     * @param geometry
     * @return
     */
    private List<?> addGeoProperty2List(List<Map<String, Object>> list, Geometry geometry) {
        if (geometry.getUserData() == null) return list;
        for (Map<String, Object> item : list) {
            Map<String, Object> pros = (Map<String, Object>) geometry.getUserData();
            for (Map.Entry entry : pros.entrySet()) {
                item.put("G_".concat((String) entry.getKey()), entry.getValue());
            }
        }
        return list;
    }

    /**
     * @param tpl
     * @return
     */
    private List<Map> getTdlyxzDictList(String tpl) {
        List<Map> groupList = new ArrayList<Map>();
        Configuration configuration = webMapService.getConfig(tpl);
        List<Dict> dicts = configuration.getDicts();
        for (Dict dict : dicts) {
            if (dict.getName().toString().equals("tdlyxz")) {
                List<Item> dictItems = dict.getDictItems();
                for (Item item : dictItems) {
                    Map temp = new HashMap();
                    temp.put("dlbm", item.getName());
                    temp.put("dlmc", item.getValue());
                    temp.put("area", 0);
                    temp.put("jtArea", 0);
                    temp.put("gyArea", 0);
                    groupList.add(temp);
                }
            }
        }
        if (groupList.size() == 0)
            throw new RuntimeException(getMessage("get.dict.error", tpl));
        return groupList;
    }

    /**
     * check field is in layer
     *
     * @param field
     * @param layerName
     * @return
     */
    private boolean checkFieldInLayer(String field, String layerName, String dataSource) {
        return ArrayUtils.contains(spatialDao.getLayerColumns(layerName, dataSource), field, true);
    }


    private boolean isPolygon(Geometry geometry) {
        String geometryType = geometry.getGeometryType();
        if (geometryType.equals(Geometries.POLYGON.getName()) || geometryType.equals(Geometries.MULTIPOLYGON.getName()))
            return true;
        else if (geometryType.equals(Geometries.LINESTRING.getName()) || geometryType.equals(Geometries.MULTILINESTRING.getName()))
            return false;
        else if (geometryType.equals(Geometries.GEOMETRYCOLLECTION.getName())) {
            return geometry.getArea() > 0 ? true : false;
        } else
            return false;
//        return (geometry.getGeometryType().equals(Geometries.POLYGON.getName()) || geometry.getGeometryType().equals(Geometries.MULTIPOLYGON.getName())) ? true : false;
    }

    /**
     * 获取和某个图层相交分析的面积
     *
     * @param list
     * @param layerName
     * @param dataSource
     * @return
     */
    public double getIntersectArea(List<Map> list, String layerName, String dataSource) {
        double area = 0;
        try {
            for (Map map : list) {
                String wktPolygon = MapUtils.getString(map, SE_SHAPE_FIELD);
                List<Map> intersectResult = (List<Map>) intersect(layerName, wktPolygon, null, dataSource);
                if (!isNull(intersectResult) && intersectResult.size() > 0)
                    area += getAreaByList(intersectResult, false, 1, getLayerCRS(layerName, dataSource));
            }
        } catch (GeometryServiceException e) {

            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return area;
    }

    /**
     * 获取耕地面积
     *
     * @param list 报批地块集合
     * @param dltb 地类图斑图层等配置
     * @param xzdw 现状地物图层等配置
     * @return
     */
    public double getGdArea(List<Map> list, String dltb, String xzdw, String dataSource) {
        double area = 0;
        try {
            for (Map map : list) {
                String wktPolygon = String.valueOf(map.get(SE_SHAPE_FIELD));
                Map tdlyxzMap = tdlyxzAnalysis2(dltb, xzdw, geometryService.toGeoJSON(geometryService.readWKT(wktPolygon)), dataSource);
                Map tdlyxzAnalysisArea = (Map) tdlyxzMap.get("analysisArea");
                area += MapUtils.getDouble(tdlyxzAnalysisArea, "水田", 0.0) + MapUtils.getDouble(tdlyxzAnalysisArea, "水浇地", 0.0) + MapUtils.getDouble(tdlyxzAnalysisArea, "旱地", 0.0);
            }
        } catch (GeometryServiceException e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return area;
    }

    /**
     * get feature's area
     *
     * @param geo
     * @param conv
     * @return
     */
    public double getAreaByGeometry(Object geo, double conv) {
        if (isNull(conv)) conv = 1;
        double area = 0;
        try {
            SimpleFeature feature = null;
            CoordinateReferenceSystem targetCrs = geometryService.parseUndefineSR("2364");
            if (geo instanceof SimpleFeature) {
                feature = (SimpleFeature) geo;
                CoordinateReferenceSystem sourceCRS = feature.getFeatureType().getCoordinateReferenceSystem();
                Geometry geometry = (Geometry) feature.getDefaultGeometry();
                if (sourceCRS != null && sourceCRS instanceof GeographicCRS) {
                    geometry = geometryService.project((Geometry) feature.getDefaultGeometry(), sourceCRS, targetCrs);
                }
                if (isPolygon(geometry))
                    area = geometry.getArea();
                else
                    area = geometry.getLength();
            } else if (geo instanceof FeatureCollection) {

                FeatureIterator iterator = ((FeatureCollection) geo).features();
                while (iterator.hasNext()) {
                    feature = (SimpleFeature) iterator.next();
                    area += getAreaByGeometry(feature, conv);
                }
            } else if (geo instanceof Geometry) {
                Geometry temp = (Geometry) geo;
                Geometry geometry;
                CoordinateReferenceSystem sCrs = geometryService.getCrsByCoordXD(temp.getCentroid().getX());
                if (sCrs != null && sCrs instanceof GeographicCRS) {
                    geometry = geometryService.project(temp, sCrs, targetCrs);
                } else
                    geometry = temp;
                if (isPolygon(geometry))
                    area = geometry.getArea();
                else
                    area = geometry.getLength();
            } else {
                if (geo instanceof GeometryCollection) {
                    GeometryCollection geometryCollection = (GeometryCollection) geo;
                    for (int i = 0; i < geometryCollection.getNumGeometries(); i++) {
                        Geometry geometry = geometryCollection.getGeometryN(i);
                        if (isPolygon(geometry))
                            area += getAreaByGeometry(geometry, conv);
                    }
                }
            }
        } catch (GeometryServiceException e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return area * conv;
    }

}
