package cn.gtmap.onemap.platform.service.impl;

import cn.gtmap.onemap.model.*;
import cn.gtmap.onemap.platform.dao.GTPLDao;
import cn.gtmap.onemap.platform.dao.TplDao;
import cn.gtmap.onemap.platform.entity.Configuration;
import cn.gtmap.onemap.platform.entity.Service;
import cn.gtmap.onemap.platform.entity.ThematicMap;
import cn.gtmap.onemap.platform.entity.dict.Dict;
import cn.gtmap.onemap.platform.service.ConfigService;
import cn.gtmap.onemap.platform.service.ThematicMapService;
import cn.gtmap.onemap.service.MetadataService;
import com.google.common.collect.Lists;
import org.apache.commons.lang.ObjectUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.Map;

/**
 * Created with IntelliJ IDEA.
 * User: Administrator
 * Date: 13-10-24  Time: 上午9:55
 * Version: v1.0
 */
@org.springframework.stereotype.Service
public class ConfigServiceImpl extends BaseLogger implements ConfigService {


    @Autowired
    private GTPLDao gtplDao;

    @Autowired
    private MetadataService metadataService;

    @Autowired
    private TplDao tplDao;
    @Autowired
    private ThematicMapService thematicMapService;

    /**
     * @return
     */
    @Override
    public List<Map> getThumbTplInfos() {
        List<Map> info = new ArrayList<Map>();
        Configuration configuration = null;
        for (String tplName : tplDao.getTplNames()) {
            try {
                configuration = getConfiguration(tplName);
                Service service = configuration.getMap().getBaseLayers().size() > 0 ? configuration.getMap().getBaseLayers().get(0) :
                        (configuration.getMap().getOperationalLayers().size() > 0 ? configuration.getMap().getOperationalLayers().get(0) : null);
                Map item = new HashMap();
                item.put("tpl", tplName);
                item.put("name", configuration.getName());
                item.put("description", configuration.getDescription());
                item.put("createAt", configuration.getCreateAt());
                item.put("thumbnail", isNull(service) || isNull(service.getId()) ? null : "/oms/thumbnail/".concat(service.getId()));
                info.add(item);
            } catch (Exception e) {
                logger.error(e.getLocalizedMessage());
            }
        }
        return info;
    }

    /**
     *  get thematic maps
     * @return
     */
    @Override
    public List<ThematicMap> getThematicMaps() {
        return thematicMapService.getAll();
    }

    /**
     * delete thematic map
     * @param id
     */
    @Override
    public void deleteThematicMap(String id) {
        thematicMapService.delete(id);
    }

    /**
     * save or upddate thematic map
     * @param thematicMap
     * @return
     */
    @Override
    public ThematicMap updateThematicMap(ThematicMap thematicMap) {
        return (ThematicMap) thematicMapService.update(thematicMap);
    }

    /**
     * get config by tpl
     *
     * @param tplName
     * @return
     */
    @Override
    public Configuration getConfiguration(String tplName) {
        return tplDao.getConfiguration(tplName);
    }

    @Override
    public List getAllService(String tpl) {
        return tplDao.getAllServices(tpl);
    }

    @Override
    public List<Map> getLayerFields(String mapId, String layerName) {
        Layer layer = metadataService.getLayerByName(mapId, layerName);
        List<Field> fieldList = metadataService.getFields(layer.getId());
        List<Map> list = new ArrayList<Map>();
        for (Field field : fieldList) {
            Map temp = new HashMap();
            temp.put("name", field.getName());
            temp.put("alias", field.getAlias());
            temp.put("type", field.getType());
            list.add(temp);
        }
        return list;
    }

    @Override
    public List<Map> getLayersByService(Service service) {
        List<Map> list = new ArrayList<Map>();
        List<Layer> layers = metadataService.getLayers(service.getId());
        for (int i = 0; i < layers.size(); i++) {
            Layer layer = layers.get(i);
            List<Map> fields = getLayerFields(service.getId(), layer.getName());
            Map temp = new HashMap();
            temp.put("serviceId", service.getId());
            temp.put("layerName", layer.getName());
            temp.put("layerIndex", layer.getIndex());
            temp.put("layerUrl", service.getUrl() + "/" + layer.getIndex());
            temp.put("fields", fields);
            list.add(temp);
        }
        return list;
    }

    /**
     * get layers in tpl
     *
     * @param tplName
     * @return
     */
    @Override
    public List<Map> getLayers(String tplName) {
        List<Map> list = new ArrayList<Map>();
        List<Service> services = getAllService(tplName);
        for (Service service : services) {
            List<Layer> layers = metadataService.getLayers(service.getId());
            Map temp = new HashMap();
            temp.put("id", service.getId());
            temp.put("name", service.getName());
            temp.put("alias", service.getAlias());
            temp.put("group", service.getGroup());
            temp.put("category", service.getCategory());
            temp.put("url", service.getUrl());
            temp.put("layers", layers);
            list.add(temp);
        }
        return list;
    }

    /**
     * create new tpl config
     *
     * @param tplName
     * @return
     */
    @Override
    public Configuration createTpl(String tplName) {
        return tplDao.createTpl(tplName);
    }

    /**
     * create tpl by some info
     *
     * @param tpl
     * @param name
     * @param description
     * @return
     */
    @Override
    public Configuration createTpl(String tpl, String name, String description,String thematicMap,String parentTpl) {
        Configuration configuration = createTpl(tpl);
        configuration.setName(name);
        configuration.setDescription(description);
        configuration.setThematicMap(thematicMap);
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        configuration.setCreateAt(dateFormat.format(new Date()));
        if (!isNull(parentTpl)) {
            Configuration pConfiguration = getConfiguration(parentTpl);
            configuration.setCoordinateVisible(pConfiguration.isCoordinateVisible());
            configuration.setLogoVisible(pConfiguration.isLogoVisible());
            configuration.setLogo(pConfiguration.getLogo());
            configuration.setDicts(pConfiguration.getDicts());
            configuration.setWidgets(pConfiguration.getWidgets());
            configuration.setDefaultRegionCode(pConfiguration.getDefaultRegionCode());
            configuration.setDefaultYear(pConfiguration.getDefaultYear());
            configuration.setGeometryService(pConfiguration.getGeometryService());
            configuration.setWidgetContainer(pConfiguration.getWidgetContainer());
            configuration.setMap(pConfiguration.getMap());
            configuration.setPrintService(pConfiguration.getPrintService());
            configuration.setTitle(pConfiguration.getTitle());
        }
        return tplDao.saveConfiguration(tpl, configuration);
    }

    /**
     * delete config
     *
     * @param tplName
     */
    @Override
    public void deleteTpl(String tplName) {
        tplDao.deleteTpl(tplName);
    }

    /**
     * save all services
     *
     * @param tplName
     * @param services
     * @return
     */
    @Override
    public List<Service> saveAllServices(String tplName, List<Service> services) {
        return tplDao.saveAllServices(tplName, services);
    }

    /**
     * insert service
     *
     * @param tplName
     * @param service
     * @return
     */
    @Override
    public Service insertService(String tplName, Service service) {
        return tplDao.saveOrUpateService(tplName, service);
    }

    /**
     * update service
     *
     * @param tplName
     * @param service
     * @return
     */
    @Override
    public Service updateService(String tplName, Service service) {
        return tplDao.saveOrUpateService(tplName, service);
    }

    /**
     * delete service
     *
     * @param tplName
     */
    @Override
    public void deleteServices(String tplName, String[] serviceIds) {
        tplDao.deleteServices(tplName, serviceIds);
    }

    /**
     * get services by ids
     *
     * @param tplName
     * @param serviceIds
     * @return
     */
    @Override
    public List<Service> getServicesByIds(String tplName, String[] serviceIds) {
        return tplDao.getServicesByIds(tplName, serviceIds);
    }

    /**
     * save base layers
     *
     * @param tplName
     * @param services
     * @return
     */
    @Override
    public List<Service> saveBaseLayers(String tplName, List<Service> services) {
        return tplDao.saveBaseLayers(tplName, services);
    }

    /**
     * delete base layers
     *
     * @param tplName
     * @param serviceIds
     */
    @Override
    public void deleteBaseLayer(String tplName, String[] serviceIds) {
        tplDao.deleteBaseLayer(tplName, serviceIds);
    }

    /**
     * get widget collection in tpl
     *
     * @param tplName
     * @return
     */
    @Override
    public Map<String, ?> getWidgetCollection(String tplName) {
        Configuration configuration = getConfiguration(tplName);
        Map<String, Object> collection = new HashMap<String, Object>();
        collection.put("dockWidgets", configuration.getWidgets());
        collection.put("widgets", configuration.getWidgetContainer().getWidgets());
        collection.put("widgetsGroup", configuration.getWidgetContainer().getWidgetsGroup());
        if(configuration.getMap().getBaseLayers().size()>0)
        collection.put("baseLayer", configuration.getMap().getBaseLayers());
        else if(configuration.getMap().getOperationalLayers().size()>0)
        collection.put("baseLayer",configuration.getMap().getOperationalLayers().get(0));
        return collection;
    }

    /**
     * get dock widgets
     *
     * @param tplName
     * @return
     */
    @Override
    public List<Configuration.Widget> getAllDockWidgets(String tplName) {
        return getConfiguration(tplName).getWidgets();
    }

    /**
     * get widgets in widgetContainer
     *
     * @param tplName
     * @return
     */
    @Override
    public List<Configuration.Widget> getAllWidgets(String tplName) {
        return getConfiguration(tplName).getWidgetContainer().getWidgets();
    }

    /**
     * get widgetGroup
     *
     * @param tplName
     * @return
     */
    @Override
    public List<Configuration.WidgetsGroup> getAllWidgetGroups(String tplName) {
        return getConfiguration(tplName).getWidgetContainer().getWidgetsGroup();
    }

    /**
     * update dock widget
     *
     * @param tplName
     * @param widget
     * @return
     */
    @Override
    public Configuration.Widget updateDockWidget(String tplName, Configuration.Widget widget) {
        return tplDao.saveOrUpdateDockWidget(tplName, widget);
    }

    /**
     * delete dock widget
     * @param tplName
     * @param widget
     */
    @Override
    public void deleteDockWidget(String tplName, Configuration.Widget widget) {
        tplDao.deleteDockWidget(tplName,widget);
    }


    /**
     * insert new widget
     *
     * @param tplName
     * @param widget
     * @return
     */
    @Override
    public Configuration.Widget insertWidget(String tplName, Configuration.Widget widget) {
        return tplDao.saveOrUpdateWidget(tplName, widget);
    }

    /**
     * update widget
     *
     * @param tplName
     * @param widget
     * @return
     */
    @Override
    public Configuration.Widget updateWidget(String tplName, Configuration.Widget widget) {
        return tplDao.saveOrUpdateWidget(tplName, widget);
    }

    /**
     * delete widget
     *
     * @param tplName
     * @param widget
     */
    @Override
    public void deleteWidget(String tplName, Configuration.Widget widget) {
        tplDao.deleteWidget(tplName, widget);
    }

    /**
     * insert widgetGroup
     *
     * @param tplName
     * @param widgetsGroup
     * @param widget
     * @return
     */
    @Override
    public Configuration.WidgetsGroup insertWidgetGroup(String tplName, Configuration.WidgetsGroup widgetsGroup, Configuration.Widget widget) {
        return tplDao.saveOrUpdateWidgetGroup(tplName, widgetsGroup, widget);
    }

    /**
     * update widgetGroup
     *
     * @param tplName
     * @param widgetsGroup
     * @param widget
     * @return
     */
    @Override
    public Configuration.WidgetsGroup updateWidgetGroup(String tplName, Configuration.WidgetsGroup widgetsGroup, Configuration.Widget widget) {
        return tplDao.saveOrUpdateWidgetGroup(tplName, widgetsGroup, widget);
    }

    /**
     * delete widgetsGroup or widget in widgetsGroup
     *
     * @param tplName
     * @param widgetsGroup
     * @param widget
     */
    @Override
    public void deleteWidgetsGroup(String tplName, Configuration.WidgetsGroup widgetsGroup, Configuration.Widget widget) {
        tplDao.deleteWidgetsGroup(tplName, widgetsGroup, widget);
    }

    /**
     * update map init extent
     *
     * @param tplName
     * @param extent
     * @return
     */
    @Override
    public Map updateMapInitExtent(String tplName, Map extent) {
        return tplDao.updateMapInitExtent(tplName, extent);
    }

    @Override
    public double updateDefaultScale(String tplName, double scale) {
        return tplDao.updateDefaultScale(tplName,scale);
    }

    /**
     * uodate map lods
     *
     * @param tplName
     * @param lods
     * @return
     */
    @Override
    public List updateMapLods(String tplName, List lods) {
        return tplDao.updateMapLods(tplName, lods);
    }

    /**
     * get global setting
     *
     * @param tplName
     * @return
     */
    @Override
    public Configuration getGlobalConfiguration(String tplName) {
        Configuration configuration = getConfiguration(tplName);
        configuration.setWidgets(null);
        configuration.setWidgetContainer(null);
        configuration.getMap().setBaseLayers(null);
        configuration.getMap().setOperationalLayers(null);
        return configuration;
    }

    /**
     * update global config <br/>
     * <p/>
     * <p>include :           <br/>
     * name                   <br/>
     * title                  <br/>
     * coordinateVisible      <br/>
     * logo                  <br/>
     * logoVisible           <br/>
     * geometryService      <br/>
     * printService         <br/>
     * </p>
     *
     * @param tplName
     * @param configuration
     * @return
     */
    @Override
    public Configuration updateGlobalConfiguration(String tplName, Configuration configuration) {
        return tplDao.updateGlobalConfiguration(tplName, configuration);
    }

    /**
     * get all services from oms, include root and group path
     *
     * @return
     */
    @Override
    public Map getAllSevices() {
        List<MapGroup> mapGroups = metadataService.getAllMapGroups();
        List<Map> groupsList = new ArrayList<Map>();
        Map<String, Object> result = new LinkedHashMap<String, Object>();
        result.put("default", getServicesByGroupId(null));
        for (MapGroup mapGroup : mapGroups) {
            Map temp = new HashMap();
            temp.put("id", mapGroup.getId());
            temp.put("name", mapGroup.getName());
            temp.put("services", null);
            groupsList.add(temp);
        }
        result.put("groups", groupsList);
        return result;
    }

    /**
     * get service from oms
     *
     * @param groupId
     * @return
     */
    @Override
    public List getServicesByGroupId(String groupId) {
        try {
            List<cn.gtmap.onemap.model.Map> maps = metadataService.findMaps(null, MapQuery.query(groupId), null).getContent();
            List<Service> services = Lists.newArrayList();
            for (cn.gtmap.onemap.model.Map map : maps) {
                services.add(Service.fromMap(map, metadataService.getServices(map.getId()), groupId));
            }
            return services;
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
          throw new RuntimeException(e.getLocalizedMessage());
        }
    }

    /**
     *
     * @return
     */
    @Override
    public List getServicesWithClassify() {
        List<MapGroup> mapGroupList = metadataService.getChildrenMapGroups(null, true);
        return organizeGroups(mapGroupList);
    }

    public List organizeGroups(List<MapGroup> groups) {

        List<Map<String, Object>> list = new ArrayList<Map<String, Object>>();
        for (MapGroup group:groups){
            Map<String, Object> item = new HashMap<String, Object>();
            item.put("id", group.getId());
            item.put("name", group.getName());
            List children = new ArrayList();
            if (group.getChildren() != null && group.getChildren().size() > 0)
                children.addAll(organizeGroups(group.getChildren()));
            item.put("children", children.size() > 0 ? children : null);
            list.add(item);
        }
        return list;
    }


    /**
     * get all common widgets from widgets.tpl
     *
     * @return
     */
    @Override
    public List<Configuration.Widget> getCommonWidgets() {
        return tplDao.getCommonWidgets();
    }

    /**
     * delete a common widget
     *
     * @param id
     */
    @Override
    public void deleteCommonWidget(String id) {
        tplDao.deleteCommonWidget(id);
    }

    /**
     * insert widget into commonWidgets
     *
     * @param widget
     * @return
     */
    @Override
    public Configuration.Widget insertCommonWidget(Configuration.Widget widget) {
        return tplDao.insertCommonWidget(widget);
    }

    @Override
    public List<Configuration.Widget> getPublicWidgets() {
        return tplDao.getPublicWidgets();
    }

    @Override
    public void deletePublicWidget(String id) {
        tplDao.deletePublicWidgets(id);
    }

    @Override
    public Configuration.Widget insertPublicWidget(Configuration.Widget widget) {
        return tplDao.insertPublicWidgets(widget);
    }

    /**
     * 获取全文检索配置
     *
     * @return
     */
    @Override
    public Map getSearchConfig() {
        return tplDao.getSearchConfig();
    }

    /**
     * 更新全文检索配置
     *
     * @param content
     */
    @Override
    public void updateSearchConfig(String content) {
        tplDao.updateSearchConfig(content);
    }

    /**
     * save or update Dict
     * @param tplName
     * @param dict
     * @return
     */
    @Override
    public Dict saveDict(String tplName, Dict dict) {
        return tplDao.saveOrUpdateDict(tplName,dict);
    }

    /**
     *delete dict by id
     * @param tplName
     * @param id
     */
    @Override
    public void deleteDict(String tplName, String id) {
        tplDao.deleteDict(tplName,id);
    }

    /***
     * 获取分析相关配置
     * @return
     */
    @Override
    public Map getAnalysizConfig() {
        return tplDao.getAnalysisConfig();
    }


}
