package cn.gtmap.onemap.platform.controller;

import cn.gtmap.onemap.core.support.hibernate.UUIDHexGenerator;
import cn.gtmap.onemap.platform.entity.Document;
import cn.gtmap.onemap.platform.entity.FileStore;
import cn.gtmap.onemap.platform.event.JSONMessageException;
import cn.gtmap.onemap.platform.service.*;
import cn.gtmap.onemap.platform.support.spring.BaseController;
import cn.gtmap.onemap.platform.utils.EnumUtils;
import com.alibaba.fastjson.JSON;
import com.vividsolutions.jts.geom.Geometry;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.geotools.feature.FeatureCollection;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.referencing.crs.CoordinateReferenceSystem;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.util.*;
/**
 * .
 *
 * @author <a href="mailto:lanxy88@gmail.com">NelsonXu</a>
 * @version V1.0, 13-5-20 下午7:21
 */
@Controller
@RequestMapping(value = "/geometryService")
public class GeometryController extends BaseController {

    @Autowired
    private GISManager gisManager;

    @Autowired
    private BMarkService bMarkService;

    @Autowired
    private DictService dictService;

    @Autowired
    private GeometryService geometryService;

    @Autowired
    private DocumentService documentService;
    @Autowired
    private FileStoreService fileStoreService;


    /**
     * @param layerName
     * @param wkt
     * @param returnFields
     * @param dataSource
     * @return
     */
    @RequestMapping(value = "/intersect")
    @ResponseBody
    public List intersect(@RequestParam("layerName") String layerName,
                          @RequestParam("wkt") String wkt,
                          @RequestParam("returnFields") String returnFields,
                          @RequestParam(value = "dataSource", defaultValue = "", required = false) String dataSource) {
        try {
            return gisManager.getGISService().intersect(layerName, wkt, returnFields.split(","), dataSource);
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 查询接口
     *
     * @param layerName      空间数据图层名称
     * @param where          Where 查询条件
     * @param geometry       GeoJSON 格式图形
     * @param outFields      返回字段
     * @param returnGeometry 是否返回图形
     * @param dataSource     数据源
     * @return
     */
    @RequestMapping(value = "/rest/query")
    @ResponseBody
    public Map query(@RequestParam("layerName") String layerName,
                     @RequestParam(value = "where", required = false) String where,
                     @RequestParam(value = "geometry", required = false) String geometry,
                     @RequestParam(value = "outFields", defaultValue = "*") String outFields,
                     @RequestParam(value = "returnGeometry", defaultValue = "true") boolean returnGeometry,
                     @RequestParam(value = "dataSource", defaultValue = "") String dataSource) {
        try {
            logger.debug("[query]:"+geometry);
            String[] fields = "*".equals(outFields) ? null : outFields.split(",");
            List result = null;
            if (StringUtils.isNotBlank(where))
                result = gisManager.getGISService().query(layerName, where, fields, returnGeometry, dataSource);
            else if (StringUtils.isNotBlank(geometry))
                result = gisManager.getGISService().query(layerName, gisManager.getGeoService().readGeoJSON(geometry), fields, dataSource);
            else
                throw new RuntimeException(getMessage("query.condition.missing"));
            FeatureCollection collection = gisManager.getGeoService().list2FeatureCollection(result, null, null);
            logger.debug("[query result]:"+gisManager.getGeoService().toFeatureJSON(collection));
            return result(gisManager.getGeoService().toFeatureJSON(collection));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 插入要素
     *
     * @param layerName
     * @param geometry   GeoJSON 格式要素
     * @param dataSource
     * @return
     */
    @RequestMapping(value = "/rest/insert")
    @ResponseBody
    public Map insert(@RequestParam("layerName") String layerName,
                      @RequestParam(value = "geometry", required = false) String geometry,
                      @RequestParam(value = "check", required = false) Boolean check,
                      @RequestParam(value = "dataSource", defaultValue = "") String dataSource) {
        try {
            logger.debug("[insert]:"+geometry);
            return result(gisManager.getGISService().insert2(layerName, geometry, check, dataSource));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 插入要素2 带有要素生成时间
     * @param layerName
     * @param geometry
     * @param check
     * @param dataSource
     * @return
     */
    @RequestMapping(value = "/rest/insert2")
    @ResponseBody
    public Map insert2(@RequestParam("layerName") String layerName,
                       @RequestParam(value = "geometry", required = false) String geometry,
                       @RequestParam(value = "check", required = false) Boolean check,
                       @RequestParam(value = "dataSource", defaultValue = "") String dataSource) {
        try {
            logger.debug("[insert]:" + geometry);
            return result(gisManager.getGISService().insert3(layerName, geometry, check, true, dataSource));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 更新要素
     *
     * @param layerName
     * @param primaryKey
     * @param geometry   GeoJSON 格式要素
     * @param dataSource
     * @return
     */
    @RequestMapping(value = "/rest/update")
    @ResponseBody
    public Map update(@RequestParam("layerName") String layerName,
                      @RequestParam(value = "primaryKey", required = true) String primaryKey,
                      @RequestParam(value = "geometry", required = true) String geometry,
                      @RequestParam(value = "dataSource", defaultValue = "") String dataSource) {
        try {
            logger.debug("[update]:"+geometry);
            return result(gisManager.getGISService().update(layerName, primaryKey, geometry, dataSource));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 删除要素
     *
     * @param layerName
     * @param primaryKey 要素主键
     * @param dataSource
     * @return
     */
    @RequestMapping(value = "/rest/delete")
    @ResponseBody
    public Map delete(@RequestParam("layerName") String layerName,
                      @RequestParam(value = "primaryKey", required = true) String primaryKey,
                      @RequestParam(value = "dataSource", defaultValue = "") String dataSource) {
        try {
            logger.debug("[delete]:"+primaryKey);
            return result(gisManager.getGISService().delete(layerName, primaryKey, dataSource));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }


    /**
     * 相交分析
     *
     * @param layerName  空间数据图层名称
     * @param geometry   GeoJSON 格式图形
     * @param outFields  返回字段
     * @param dataSource 数据源
     * @return
     */
    @RequestMapping(value = "/rest/intersect")
    @ResponseBody
    public Map intersect2(@RequestParam("layerName") String layerName,
                          @RequestParam("geometry") String geometry,
                          @RequestParam(value = "outFields", defaultValue = "*") String outFields,
                          @RequestParam(value = "dataSource", defaultValue = "", required = false) String dataSource) {
        try {
            logger.debug("[intersect]:"+geometry);
            String[] fields = "*".equals(outFields) ? null : outFields.split(",");
            List results = gisManager.getGISService().intersect3(layerName, geometry, fields, dataSource);
            FeatureCollection collection = geometryService.list2FeatureCollection(results, null,null);//geometryService.parseUndefineSR("4610")
            return result(geometryService.toFeatureJSON(collection));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 土地利用现状分析
     *
     * @param dltb       地类图斑图层
     * @param xzdw       现状地物图层
     * @param regionCode 行政区代码
     * @param geometry   GeoJSON 格式图形
     * @param dataSource
     * @return
     */
    @RequestMapping(value = "/rest/analysis/tdlyxz")
    @ResponseBody
    public Map tdlyxzAnalysis(@RequestParam(value = "dltb", required = false) String dltb,
                              @RequestParam(value = "xzdw", required = false) String xzdw,
                              @RequestParam(value = "regionCode", required = false) String regionCode,
                              @RequestParam(value = "year", required = false) String year,
                              @RequestParam(value = "geometry", required = true) String geometry,
                              @RequestParam(value = "dataSource", defaultValue = "", required = false) String dataSource) {
        try {
            if (StringUtils.isNotBlank(year)) regionCode = year;
            if (StringUtils.isNotBlank(regionCode))
                return result(gisManager.getGISService().tdlyxzAnalysis(regionCode, geometry, dataSource));
            else if (StringUtils.isNotBlank(dltb) && StringUtils.isNotBlank(xzdw))
                return result(gisManager.getGISService().tdlyxzAnalysis2(dltb, xzdw, geometry, dataSource));
            throw new RuntimeException(getMessage("analysis.tdlyxz.param.error"));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 土地利用总体规划审查
     *
     * @param layerType  图层类型
     * @param regionCode 行政区代码
     * @param year       年度，与regionCode二选一，优先选择year
     * @param geometry   GeoJSON格式数据
     * @param outFields  返回字段
     * @param dataSource 数据源
     * @return
     */
    @RequestMapping(value = "/rest/analysis/tdghsc")
    @ResponseBody
    public Map tdghscAnalysis(@RequestParam(value = "layerType", defaultValue = "") String layerType,
                              @RequestParam(value = "regionCode", required = false) String regionCode,
                              @RequestParam(value = "year", defaultValue = "2020") String year,
                              @RequestParam("geometry") String geometry,
                              @RequestParam(value = "outFields", defaultValue = "*") String outFields,
                              @RequestParam(value = "dataSource", defaultValue = "", required = false) String dataSource) {
        try {
            String[] fields = "*".equals(outFields) ? null : outFields.split(",");
            if (StringUtils.isNotBlank(year)) regionCode = year;
            if (StringUtils.isNotBlank(layerType))
                return result(gisManager.getGISService().tdghscAnalysis2(layerType, regionCode, geometry, fields, dataSource));
            else
                return result(gisManager.getGISService().tdghscAnalysis2(regionCode, geometry, fields, dataSource));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * project
     *
     * @param geometry GeoJSON geometry
     * @param inSR     wkid or wkt
     * @param outSR    wkid or wkt
     * @return
     */
    @RequestMapping(value = "/rest/project")
    @ResponseBody
    public void project(@RequestParam(value = "geometry", required = true) String geometry,
                       @RequestParam(value = "inSR", required = true) String inSR,
                       @RequestParam(value = "outSR", required = true) String outSR,
                       HttpServletResponse response) {
        try {
            Object geo = gisManager.getGeoService().readUnTypeGeoJSON(geometry);
            CoordinateReferenceSystem in = gisManager.getGeoService().parseUndefineSR(inSR);
            CoordinateReferenceSystem out = gisManager.getGeoService().parseUndefineSR(outSR);
            if (geo instanceof Geometry) {
                Geometry g = gisManager.getGeoService().project((Geometry) geo, in, out);
                result(gisManager.getGeoService().toGeoJSON(g),response);
            } else if ((geo instanceof FeatureCollection) || (geo instanceof SimpleFeature)) {
                Object feature = gisManager.getGeoService().project(geo, in, out);
                result(gisManager.getGeoService().toFeatureJSON(feature),response);
            } else
                throw new RuntimeException("geometry don't support");
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /***
     * 各种分析的中转页面
     * @param request
     * @param model
     * @return
     */
    @RequestMapping(value = "/analysis/hub")
    public String analysisHub(HttpServletRequest request, Model model) {
        model.addAttribute("type", request.getSession().getAttribute("type"));
        model.addAttribute("title", request.getSession().getAttribute("title"));
        model.addAttribute("params", request.getSession().getAttribute("params"));
        model.addAttribute("geometry", request.getSession().getAttribute("geometry"));
        return "analysis/hub";
    }

    /**
     * 返回结果页面
     *
     * @param layerType
     * @param regionCode
     * @param year       年度，与regionCode二选一，优先选择year
     * @param geometry
     * @param outFields
     * @param dataSource
     * @param model
     * @return
     */
    @RequestMapping(value = "/analysis/tdghsc")
    public String tdghscAnalysis(@RequestParam(value = "layerType", defaultValue = "") String layerType,
                                 @RequestParam(value = "regionCode",required = false) String regionCode,
                                 @RequestParam(value = "year", defaultValue = "2020") String year,
                                 @RequestParam("geometry") String geometry,
                                 @RequestParam(value = "outFields", defaultValue = "*") String outFields,
                                 @RequestParam(value = "dataSource", defaultValue = "", required = false) String dataSource, Model model) {

        try {
            logger.debug("[tdghsc]:"+geometry);
            String[] fields = "*".equals(outFields) ? null : outFields.split(",");
            if (StringUtils.isNotBlank(year)) regionCode = year;
            Map result = new HashMap();
            if (StringUtils.isNotBlank(layerType)) {
                String analysisResult = gisManager.getGISService().tdghscAnalysis2(layerType, regionCode, geometry, fields, dataSource);
                result = gisManager.getGISService().tdghscResult(layerType, analysisResult);
            } else {
                Map analysisMap = gisManager.getGISService().tdghscAnalysis2(regionCode, geometry, fields, dataSource);
                result = gisManager.getGISService().tdghscResult(analysisMap);
            }
            model.addAttribute("result", result);
            model.addAttribute("excelData", JSON.toJSONString(gisManager.getGISService().tdghscExcelData(result)));
            return "analysis/tdghsc";
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     *  土地利用现状分析(返回页面)
     * @param dltb        地类图斑图层名
     * @param xzdw        线状地物图层名
     * @param regionCode  行政区代码
     * @param year        年份
     * @param geometry    分析的图形geojson
     * @param dataSource  sde数据源
     * @param unit        分析结果展示面积单位
     * @param tpl         模板名称
     * @param report      是否来自电子报件 true/false
     * @param exportable  是否可以将分析结果导出为shp或者cad文件  @since  v2.0.14
     * @param model
     * @return
     */
    @RequestMapping(value = "/analysis/tdlyxz")
    public String tdlyxzAnalysis(@RequestParam(value = "dltb", required = false) String dltb,
                                 @RequestParam(value = "xzdw", required = false) String xzdw,
                                 @RequestParam(value = "regionCode", required = false) String regionCode,
                                 @RequestParam(value = "year", required = false) String year,
                                 @RequestParam(value = "geometry", required = true) String geometry,
                                 @RequestParam(value = "dataSource", defaultValue = "", required = false) String dataSource,
                                 @RequestParam(value = "unit", defaultValue = "SQUARE", required = false) String unit,
                                 @RequestParam(value = "tpl", required = true) String tpl,
                                 @RequestParam(value = "report", required = false) String report,
                                 @RequestParam(value = "exportable", required = false, defaultValue = "false") Boolean exportable,
                                 Model model) {

        try {
            logger.debug("[tdlyxz geometry]:"+geometry);
            Map map;
            if (StringUtils.isNotBlank(year)) regionCode = year;
            if (StringUtils.isNotBlank(regionCode))
                map = gisManager.getGISService().tdlyxzAnalysis(regionCode, geometry, dataSource);
            else if (StringUtils.isNotBlank(dltb) && StringUtils.isNotBlank(xzdw))
                map = gisManager.getGISService().tdlyxzAnalysis2(dltb, xzdw, geometry, dataSource);
            else
                throw new RuntimeException(getMessage("analysis.tdlyxz.param.error"));

            logger.debug("[tdlyxz result]:" + JSON.toJSONString(map));
            if (exportable && !isNull(dltb) && !isNull(map) && map.containsKey("analysisAreaDetail") && map.get("analysisAreaDetail") != null) {
                List detail = (List) map.get("analysisAreaDetail");
                FeatureCollection featureCollection = gisManager.getGeoService().list2FeatureCollection(detail, null, null);
                File shpFile = gisManager.getGeoService().exportToShp(gisManager.getGeoService().toFeatureJSON(featureCollection), gisManager.getGISService().getLayerCRS(dltb, dataSource));
                if (shpFile.exists()) {
                    FileStore fileStore = fileStoreService.save3(shpFile, UUIDHexGenerator.generate());
                    model.addAttribute("shpId", fileStore.getId());
                }
            }
            List<Map> result = gisManager.getGISService().tdlyxzResult(map, tpl, unit);
            model.addAttribute("result", result);
            model.addAttribute("unit",unit);
            model.addAttribute("exportable", exportable);

            model.addAttribute("resultStr", JSON.toJSONString(model.asMap()));
            model.addAttribute("ogGeo",geometry);//added by yxf for zjgg

            if (StringUtils.isNotBlank(report) && result.size() > 0) {
                model.addAttribute("totalResult", result.get(result.size() - 1));
                Map reportMap = gisManager.getGISService().tdlyxzReport(result.get(result.size() - 1), JSON.parseObject(report, Map.class));
                model.addAttribute("report", reportMap);
                Map xlsMap = new HashMap();
                xlsMap.put("report", reportMap);
                xlsMap.put("totalResult", result.get(result.size() - 1));
                xlsMap.put("unit",unit);
                model.addAttribute("reportXls", JSON.toJSONString(xlsMap));
            }
            return "analysis/tdlyxz";

        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /***
     * 叠加分析
     * @param geometry
     * @param analysisLayers
     * @param unit
     * @param dataSource
     * @param template
     * @param model
     * @return
     */
    @RequestMapping(value = "/analysis/dj")
    public String djAnalysis(@RequestParam(value="geometry") String geometry,
                             @RequestParam("analysisLayers") String analysisLayers,
                             @RequestParam(value = "unit",required = true)String unit,
                             @RequestParam(value = "dataSource", defaultValue = "", required = false) String dataSource,
                             @RequestParam(value = "template",defaultValue = "result-dj",required = false)String template,
                             Model model){
        try{
            if(StringUtils.isNotBlank(analysisLayers))
            {
                List layers = JSON.parseObject(analysisLayers,List.class);
                Map areaUnit = JSON.parseObject(unit,Map.class);
                List<Map> result = gisManager.getGISService().djAnalysis(geometry,layers,dataSource,areaUnit);
                model.addAttribute("unit",areaUnit);
                model.addAttribute("results",result);
                return "analysis/".concat(template);
            }
        }catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return null;
    }

    /**
     * 卫片监察核查
     * 图斑与报批、供地、违法用地图层做对比，分析。
     * 报批、可以相重合的，是合法的，
     * 违法用地图层相重合的，是已经违法的,
     * 和报批、供地、违法用地图层都叠加不上的为疑似违法地块,
     * 重叠的分比例来确定是否合法，违法,超过等于70%为全图斑认定,小于等于30%剩余部分为全图斑认定为疑似。
     * @return
     */
    @RequestMapping(value = "/analysis/wpjchc")
    public String wpjchcAnalysis(@RequestParam(value = "dataSource", required = false) String dataSource,
                                 @RequestParam(value = "wpjctb", required = false) String wpjctb,
                                 @RequestParam(value = "bpdk", required = false) String bpdk,
                                 @RequestParam(value = "gddk", required = false) String gddk,
                                 @RequestParam(value = "wfyd", required = false) String wfyd,
                                 @RequestParam(value = "template", required = false, defaultValue = "result-wpjc") String template,
                                 Model model){
        return "";
    }

    /**
     * 监测图斑分析
     * @param year           图斑年度
     * @param geometry       分析图形
     * @param analysisLayers 分析图层集合
     * @param unit           结果面积单位
     * @param dataSource     sde数据源
     * @param template       展示页面模板名称
     * @param model
     * @return
     */
    @RequestMapping(value = "/analysis/jctb")
    public String jctbAnalysis(@RequestParam(value = "year") String year,
                       @RequestParam("geometry") String geometry,
                       @RequestParam("analysisLayers") String analysisLayers,
                       @RequestParam(value = "unit",required = true)String unit,
                       @RequestParam(value = "dataSource", defaultValue = "", required = false) String dataSource,
                       @RequestParam(value = "template",defaultValue = "result-jctb",required = false)String template,
                       Model model) {
        try {
            if(StringUtils.isNotBlank(analysisLayers))
            {
                List layers = JSON.parseObject(analysisLayers, List.class);
                Map areaUnit = JSON.parseObject(unit, Map.class);
                List<Map> result = gisManager.getGISService().jctbAnalysis(geometry, layers, dataSource, areaUnit);
                List<Map> xlsData = new ArrayList<Map>();
                for (Map item : result) {
                    Map tmp = new HashMap();
                    for (Object k : item.keySet()) {
                        if (k.equals("OG_PRO_SHAPE")) continue;
                        tmp.put(k, item.get(k));
                    }
                    xlsData.add(tmp);
                }
                model.addAttribute("year", year);
                model.addAttribute("result", result);
                model.addAttribute("unit", areaUnit);
                Map xlsMap = new HashMap();
                xlsMap.put("year", year);
                xlsMap.put("unit", areaUnit);
                xlsMap.put("xlsData", xlsData);
                model.addAttribute("resultStr", JSON.toJSONString(xlsMap));
                return "analysis/".concat(template);
            }
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw new RuntimeException(e.getLocalizedMessage());
        }
        return null;
    }

    /**
     * 规划审查修改分析 返回页面(省厅 订制)
     * @param layerName
     * @param geometry
     * @param outFields
     * @param dataSource
     * @return
     */
    @Deprecated
    @RequestMapping(value = "/analysis/ghscxg",method = RequestMethod.POST)
    public String ghModifyAnalysis(@RequestParam("layerName") String layerName,
                                   @RequestParam("geometry") String geometry,
                                   @RequestParam(value = "outFields", defaultValue = "*") String outFields,
                                   @RequestParam(value = "showFields", defaultValue = "*") String showFields,
                                   @RequestParam(value = "dataSource", defaultValue = "", required = false) String dataSource,
                                   Model model){


        try {
            logger.debug("[analysis]:"+geometry);
            String[] fields = "*".equals(outFields) ? null : outFields.split(",");
            List<Map> results = (List<Map>) gisManager.getGISService().intersect3(layerName, geometry, fields, dataSource);
            Map map = gisManager.getGISService().ghscxgData(results,JSON.parseObject(showFields,List.class));
            model.addAttribute("showFields",JSON.parseObject(showFields,List.class));
            model.addAttribute("results",map.get("result"));
            model.addAttribute("excelData",JSON.toJSONString(map.get("excel")));
            return "analysis/result-ghscxg";

        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw new JSONMessageException(e.getLocalizedMessage());
        }

    }

    /**
     * 综合分析
     * @param jsonParams
     * @param geometry
     * @param tpl
     * @param level  standard / wuzhong / .etc                   <b>since v2.1.2<b/>
     * @param link   用于显示分析结果的超链接(其他业务系统负责展示页面) <b>since v2.1.2<b/>
     * @param model
     * @return
     */
    @RequestMapping(value = "/analysis/multiAnalysis",method = {RequestMethod.POST,RequestMethod.GET})
    public String multiAnalysis(@RequestParam("jsonParams") String jsonParams,
                                @RequestParam("geometry") String geometry,
                                @RequestParam(value = "tpl") String tpl,
                                @RequestParam(value = "level",defaultValue = "standard")String level,
                                @RequestParam(value = "link",required = false)String link,
                                Model model) {

        try {
            List<Map> jsonValues = JSON.parseObject(jsonParams, List.class);
            Map<String, Object> analysisResult = gisManager.getGISService().multiAnalyze(jsonValues, geometry,level, tpl);
            model.addAttribute("result", analysisResult);
            model.addAttribute("geometry",geometry);

            Map<String,Object> mapData = new HashMap<String, Object>();
            Map<String,Object> summaryData = new HashMap<String, Object>();
            for (Map.Entry entry : analysisResult.entrySet()) {
                Map value = (Map) entry.getValue();
                if ("mas".equals(level)) {
                    List<Map> analysisList;
                    Map analysisMap;
                    Object r = value.get("result");
                    List<Map> nList;
                    if (r instanceof Map) {
                        analysisMap = (Map) r;
                        Map nMap = new HashMap(); //组织新的map 去除detail中的shape字段
                        nList = new ArrayList<Map>();
                        nMap.put("info", analysisMap.get("info"));
                        List<Map> detail;
                        if(analysisMap.containsKey("detail")){
                            detail = (List<Map>) analysisMap.get("detail");
                            for (Map _d : detail) {
                                Map item = new HashMap();
                                for (Object o : _d.keySet()) {
                                    String k = String.valueOf(o);
                                    if (k == "SHAPE") continue;
                                    item.put(o, _d.get(o));
                                }
                                nList.add(item);
                            }
                            nMap.put("detail", nList);
                            mapData.put(String.valueOf(entry.getKey()), nMap);
                        }
                    } else if (r instanceof List) {
                        analysisList = (List<Map>) r;
                        nList = new ArrayList<Map>();
                        for (Map _m : analysisList) {
                            Map nItem = new HashMap();
                            for (Object _k : _m.keySet()) {
                                String key = String.valueOf(_k);
                                if (key == "SHAPE") continue;
                                nItem.put(_k, _m.get(_k));
                            }
                            nList.add(nItem);
                        }
                        mapData.put(String.valueOf(entry.getKey()), nList);
                    }
                } else
                    mapData.put(String.valueOf(entry.getKey()), value.get("result"));

                summaryData.put(String.valueOf(entry.getKey()), value.get("summary"));
            }
            double ogArea = gisManager.getGeoService().getGeoArea(gisManager.getGeoService().readUnTypeGeoJSON(geometry),null);
            /** 判断是否是外部链接方式展示结果 **/
            if(!isNull(link)){
                model.addAttribute("linkData",JSON.toJSONString(mapData));
                model.addAttribute("link",link);
                model.addAttribute("ogArea",ogArea);
                model.addAttribute("ogGeo",geometry);
                model.addAttribute("tpl",tpl);
                Map geoMap= new HashMap();
                for (Map.Entry entry:analysisResult.entrySet()){
                    Map map = (Map) entry.getValue();
                    geoMap.put(MapUtils.getString(map,"alias"),MapUtils.getString(map,"geojson"));
                }
                model.addAttribute("geo",JSON.toJSONString(JSON.toJSONString(geoMap)));
                return "analysis/result-multi-link";
            }
            /** 根据level级别的不同 显示不同展示结果 **/
            switch (EnumUtils.MULTI_ANALYZE_LEVEL.valueOf(level)) {
                case wuzhong:
                    if (!mapData.isEmpty()) {
                        model.addAttribute("xlsData", JSON.toJSONString(mapData));
                    }
                    break;
                case mas:
                    if (!mapData.isEmpty()) {
                        model.addAttribute("xlsData", JSON.toJSONString(mapData));
                    }
                    if (!summaryData.isEmpty()) {
                        Map tmp = new HashMap();
                        tmp.put("result", summaryData);
                        tmp.put("ogArea",ogArea);
                        model.addAttribute("summaryData", JSON.toJSONString(tmp));
                    }
                    model.addAttribute("ogArea",gisManager.getGeoService().getGeoArea(gisManager.getGeoService().readUnTypeGeoJSON(geometry),null));
                    model.addAttribute("tpl",tpl);
                    break;
            }
            return "analysis/result-multi-".concat(level);

        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 返回相交分析的结果页面
     * 返回相交分析的一般性页面(只展示组织好的数据)
     * 结果是分组后的带面积的结果
     * @param data
     * @param general
     * @param model
     * @return
     */
    @RequestMapping(value = "/analysis/result/common", method = RequestMethod.POST)
    public String groupedAnalysisResult(@RequestParam(value = "data") String data,
                                        @RequestParam(value="general",required = false)String general,
                                        Model model) {
        if (StringUtils.isNotBlank(data)) {
            if(StringUtils.isNotBlank(general))
                model.addAttribute("general",JSON.parseObject(general,Map.class));
            List result = JSON.parseObject(data, List.class);
            model.addAttribute("info", result);
            model.addAttribute("excelData", JSON.toJSONString(gisManager.getGISService().analysisExcelData(result)));
            model.addAttribute("excelList",JSON.toJSONString(gisManager.getGISService().analysisExcelList(result)));
        }
        return "analysis/result-grouped";
    }

    /**
     *  返回管制区用地分析的结果展示页面
     * @param data
     * @param model
     * @return
     */
    @RequestMapping(value = "/analysis/result/gzqyd", method = RequestMethod.POST)
    public String commonAnalysisResult(@RequestParam(value = "data") String data, Model model) {
        if (StringUtils.isNotBlank(data)) {
            List result = JSON.parseObject(data, List.class);
            model.addAttribute("result", result);
            model.addAttribute("excelData", JSON.toJSONString(gisManager.getGISService().gzqydExcelData(result)));
        }
        return "analysis/result-gzqyd";
    }

    /***
     * 返回规划审查修改分析结果页面(省厅定制功能)
     * @param data
     * @param showFields
     * @param model
     * @since v2.1.2 last modified by yingxiufeng
     * @return
     */
    @RequestMapping(value = "/analysis/result/ghscxg",method = {RequestMethod.POST,RequestMethod.GET})
    public String ghModifyAnalysisResult(@RequestParam("data") String data,
                                         @RequestParam(value = "showFields", defaultValue = "*") String showFields,
                                         Model model) {

        try {
            logger.debug("[ghscxg analysis data]:"+data);
            List<Map> results = JSON.parseObject(data,List.class);
            Map map = gisManager.getGISService().ghscxgData(results,JSON.parseObject(showFields,List.class));
            model.addAttribute("showFields",JSON.parseObject(showFields,List.class));
            model.addAttribute("results",map.get("result"));
            model.addAttribute("excelData",JSON.toJSONString(map.get("excel")));
            return "analysis/result-ghscxg";

        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw new JSONMessageException(e.getLocalizedMessage());
        }

    }

    /**
     * 获取分析时分组字段相对应的字典项配置,字典名称:前缀_分组字段名(如,analysis_dlbm)
     *
     * @param tpl
     * @param groupName
     * @return
     */
    @RequestMapping(value = "/analysis/dict/fetch")
    @ResponseBody
    public List getAnalysisDict(@RequestParam(value = "tpl", required = true) String tpl,
                                @RequestParam(value = "groupName", required = true) String groupName) {
        return dictService.getAnalysisDict(tpl, groupName);
    }

    /**
     * 导出数据至excel
     * @param data      组织好的序列化数据
     * @param fileName  excel文件模板名称
     * @param response
     */
    @RequestMapping(value = "/export/excel")
    public void export2Excel(@RequestParam("data") String data,
                             @RequestParam(value = "fileName", required = false) String fileName,
                             HttpServletResponse response) {
        try {
            Document document = null;
            if (isNull(fileName)) {
                document = documentService.writeExcel(JSON.parseObject(data, List.class));
                sendFile(new ByteArrayInputStream(document.getContent()), response, document.getFileName());
            } else {
                document = documentService.writeExcel(JSON.parseObject(data, Map.class), fileName);
                sendFile(new ByteArrayInputStream(document.getContent()), response, document.getFileName());
            }
        } catch (Exception e) {
            throw new JSONMessageException(getMessage("doc.excel.export.error", e.getLocalizedMessage()));
        }
    }
    /**
     * 导出分析结果至excel文件(xml模板方式导出)
     * @param data      导出数据
     * @param fileName  xml模板名称
     * @param response
     */
    @RequestMapping(value = "/export/analysis")
    public void exportDocument(@RequestParam("data") String data,
                               @RequestParam(value = "fileName", required = true) String fileName,
                               @RequestParam(value = "fileType",defaultValue = "xls")String fileType,
                               HttpServletResponse response) {
        try {
            sendDocument(response, documentService.renderAnalysisExcel(JSON.parseObject(data, Map.class), fileName, Document.Type.valueOf(fileType)));
        } catch (Exception e) {
            throw new JSONMessageException(getMessage("doc.excel.export.error", e.getLocalizedMessage()));
        }
    }
    /**
     * 导出shapefile的zip包
     * @param geometry
     * @param response
     */
    @RequestMapping(value = "/export/shp")
    public void exportShp(@RequestParam(value = "geometry", required = true) String geometry,
                          @RequestParam(value = "sr", required = false) String sr,
                          HttpServletResponse response) {
        try {
            File shpFile;
            if (!isNull(sr))
                shpFile = gisManager.getGeoService().exportToShp(geometry,gisManager.getGeoService().parseUndefineSR(sr));
            else
                shpFile = gisManager.getGeoService().exportToShp(geometry);
            if (shpFile.exists())
                sendStream(new FileInputStream(shpFile),response,shpFile.getName());
            else
                throw new RuntimeException(getMessage("shp.export.error", "file not found"));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * 利用ArcGIS的GP服务 生成导出dwg文件的在线下载url地址
     * @param shpUrl  shp的zip包在线地址
     * @param gpUrl    ArcGIS的GP服务地址
     * @return
     */
    @RequestMapping(value = "/rest/export/dwg")
    @ResponseBody
    public Map exportDwg(@RequestParam(value = "shpUrl",required = true)String shpUrl,
                            @RequestParam(value = "gpUrl",required = true)String gpUrl){

        try {
            return result(gisManager.getGeoService().convertShpToDwg(shpUrl,gpUrl));
        } catch (Exception e) {
            throw new RuntimeException(getMessage("dwg.export.error",e.getMessage()));
        }
    }
    /**
     * 生成shp的zip包 并上传到fileStore 以供在线调用处理
     * @param geometry
     * @param sr
     * @return   返回上传成功的zip文件的filestore的id
     */
    @RequestMapping(value = "/rest/export/shp")
    @ResponseBody
    public String exportShpRest(@RequestParam(value = "geometry", required = true) String geometry,
                                @RequestParam(value = "sr", required = false) String sr) {
        try {
            File shpFile;
            if (!isNull(sr))
                shpFile = gisManager.getGeoService().exportToShp(geometry, gisManager.getGeoService().parseUndefineSR(sr));
            else
                shpFile = gisManager.getGeoService().exportToShp(geometry);
            if (shpFile.exists())
            {
                FileStore fileStore = fileStoreService.save3(shpFile, UUIDHexGenerator.generate());
                return fileStore.getId();
            }
            else
                throw new RuntimeException(getMessage("shp.export.error", "file not found"));
        }catch (Exception e){
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }
    /**
     * 管制区用地分析结果导出excel
     * @param data
     * @param fileName
     * @param response
     */
    @RequestMapping(value = "/export/excel/gzqyd")
    public void export2ExcelGZQ(@RequestParam("data") String data,
                                @RequestParam(value = "fileName", required = true) String fileName,
                                HttpServletResponse response) {
        try {
            Document document = null;
            document = documentService.writeExcelGZQFX(JSON.parseObject(data, List.class), fileName);
            sendFile(new ByteArrayInputStream(document.getContent()), response, document.getFileName());
        } catch (Exception e) {
            throw new JSONMessageException(getMessage("doc.excel.export.error", e.getLocalizedMessage()));
        }
    }

    /**
     * excel 文件导入
     * @param file
     * @param response
     */
    @RequestMapping(value = "/excel/upload")
    @ResponseBody
    public void xlsUpload(@RequestParam(value = "file") MultipartFile file, HttpServletResponse response) {
        try {
            result(gisManager.getGeoService().getExcelCoordinates(file.getInputStream()), response);
        } catch (Exception e) {
            error(getMessage("excel.upload.error", file.getOriginalFilename(), e.getLocalizedMessage()), response);
        }
    }

    /***
     * zip包上传解析(电子报件 shape的zip包 统一处理)
     * @param file
     * @param response
     */
    @RequestMapping(value = "/zip/upload")
    @ResponseBody
    public void zipUpload(@RequestParam(value = "file") MultipartFile file, HttpServletResponse response) {
        try {
            result(gisManager.getGeoService().getZipCoordinates(file.getInputStream()), response);
        } catch (Exception e) {
            error(getMessage("zip.coord.get.error", e.getLocalizedMessage()), response);
        }
    }
    /**
     * 电子报件导入
     * @return
     * v2.1.4后弃用  <br/>改用 @see zipUpload
     */
    @RequestMapping(value = "/bj/upload")
    @ResponseBody
    public void bjUpload(@RequestParam(value = "file") MultipartFile file, HttpServletResponse response) {
        try {
            result(gisManager.getGeoService().getBJCoordinates(file.getInputStream()), response);
        } catch (Exception e) {
            error(getMessage("bj.upload.error", file.getOriginalFilename(), e.getLocalizedMessage()), response);
        }
    }

    /**
     * shapefile上传解析
     * @param file  shape zip
     * @param response
     * v2.1.4后弃用  <br/>改用 zipUpload
     */
    @RequestMapping(value = "/shp/upload")
    @ResponseBody
    public void shpUpload(@RequestParam(value = "file",required = false) MultipartFile file,
                         HttpServletResponse response) {
        try {
            result(gisManager.getGeoService().getShpCoordinates(file.getInputStream()), response);
        } catch (Exception e) {
           error(getMessage("shp.upload.error",file.getOriginalFilename(),e.getLocalizedMessage()),response);
        }
    }

    /**
     * shapefile导入2
     * @param filePath  zip file 路径
     * @param layerName 插入的空间图层名称
     * @param check
     * @param dataSource
     * @return
     */
    @RequestMapping(value = "/shp/file/upload")
    @ResponseBody
    public Map shpZipUpload(@RequestParam(value = "file", required = true) String filePath,
                             @RequestParam(value = "properties", required = false) String properties,
                             @RequestParam("layerName") String layerName,
                             @RequestParam(value = "check", required = false, defaultValue = "true") Boolean check,
                             @RequestParam(value = "dataSource", defaultValue = "") String dataSource) {
        try {
            File zipFile = new File(filePath);
            if(zipFile.exists())
            {
                String geometry = gisManager.getGeoService().getShpCoordinates(zipFile,properties);
                return result(gisManager.getGISService().insert2(layerName, geometry, check, dataSource));
            }else
                throw new JSONMessageException(getMessage("shp.zipfile.exist.error",filePath));
        } catch (Exception e) {
            logger.error(e.getLocalizedMessage());
            throw new JSONMessageException(getMessage("shp.upload.error",filePath,e.getLocalizedMessage()));
        }
    }

    /***
     * 获取geojson的面积
     * @param geometry
     * @return
     * @since  v2.1.4
     */
    @RequestMapping(value = "/geo/area")
    @ResponseBody
    public double geoArea(@RequestParam(value = "geometry") String geometry,
                          @RequestParam(value = "crs", required = false, defaultValue = "4610") String crs) {
        try {
            return geometryService.getGeoArea(geometryService.readUnTypeGeoJSON(geometry), geometryService.parseUndefineSR(crs));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }


    /**
     * format json
     *
     * @param json
     * @param pretty
     * @return
     */
    @RequestMapping(value = "/json/format")
    @ResponseBody
    public String formatJSON(@RequestParam(value = "json") String json,
                             @RequestParam(value = "pretty", defaultValue = "true") boolean pretty) {
        try {
            return JSON.toJSONString(JSON.parseObject(json), pretty);
        } catch (Exception e) {
            throw new JSONMessageException(getMessage("json.format.error", e.getLocalizedMessage()));
        }
    }

    /**
     * crs坐标系转为wkt描述
     * @param sr
     * @return
     */
    @RequestMapping(value = "/crs/wkt")
    @ResponseBody
    public Map crsToWkt(@RequestParam(value = "sr", required = true) String sr) {
        try {
            CoordinateReferenceSystem in = gisManager.getGeoService().parseUndefineSR(sr);
            return result(in.toWKT());
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

    /**
     * json editor
     */
    @Deprecated
    @RequestMapping(value = "/jsoneditor")
    public String jsonEditor() {
        return "jsoneditor";
    }

    /**
     * 出图
     * @param where
     * @param scale
     * @param width
     * @param height
     * @param type
     * @param response
     */
    @Deprecated
    @RequestMapping(value = "/rest/export/map")
    @ResponseBody
    public void exportMap(@RequestParam(value = "where",required = true)String where,
                          @RequestParam(value = "scale",required = false)String scale,
                          @RequestParam(value = "width",defaultValue = "800")int width,
                          @RequestParam(value = "height",defaultValue = "600")int height,
                          @RequestParam(value = "type",required = true)String type,
                          HttpServletResponse response){

        try {
            InputStream inputStream = gisManager.getGISService().exportMap(where, scale, width, height, type);
            sendFile(inputStream,response, String.valueOf(System.currentTimeMillis()));
        } catch (IOException e) {
            throw new JSONMessageException(getMessage("map.export.error", e.getLocalizedMessage()));
        }
    }
    /**
     * 界址点导入与关联
     *
     * @return
     */
    @RequestMapping(value = "/bmark")
    public String boundaryMark(@RequestParam(value = "id", required = false) String id,
                               @RequestParam(value = "type", required = false) String type,
                               @RequestParam(value = "btype", defaultValue = "bmi") String btype,
                               Model model) {
        model.addAttribute("id", id);
        model.addAttribute("btype", btype);
        model.addAttribute("type", type);
        return "geo/bmark";
    }
    /**
     * 界址点导入与关联
     *
     * @return
     */
    @RequestMapping(value = "/bmark/coords")
    @ResponseBody
    public List bMarkUpload(@RequestParam(value = "fileData", required = true) MultipartFile file,
                            @RequestParam(value = "fileType", defaultValue = "xls") String fileType) {
        try {
            return bMarkService.getCoordsByFile(file.getInputStream(), BMarkService.Type.getType(fileType));
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }
    /**
     *
     * @param id
     * @param coords
     * @param type
     * @return
     */
    @RequestMapping(value = "/bmark/gen", method = RequestMethod.POST)
    @ResponseBody
    public boolean generateShape(@RequestParam(value = "id", required = false) String id,
                                 @RequestParam(value = "coords", required = false) String coords,
                                 @RequestParam(value = "type", required = false) String type) {
        try {
            return bMarkService.insert(id, JSON.parseObject(coords, List.class), type);
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }
    /**
     * 界址点导入与关联
     *
     * @return
     */
    @RequestMapping(value = "/bmark/dks")
    @ResponseBody
    public List getBMarkDKById(@RequestParam(value = "key", required = false) String key,
                               @RequestParam(value = "type", required = true) String type) {
        try {
            return bMarkService.query(key, type);
        } catch (Exception e) {
            throw new JSONMessageException(e.getLocalizedMessage());
        }
    }

}
