package cn.gtmap.gtc.resource.utils;

import cn.gtmap.gtc.resource.domain.resource.metadata.Dict;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;

import java.io.UnsupportedEncodingException;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static cn.gtmap.gtc.resource.Constant.*;


/**
 * Created by shanhuashuiqing11@163.com at date 2018/4/17 16:18.
 *
 * @author zhangsongshan
 */
public class Utils {

    private Utils() {

    }

    private static final Map<String, String> wellKnownSimplePrefixes = new HashMap<>(4);

    private static final Map<String, String> wellKnownSimpleRegexs = new HashMap<>(4);

    static final Logger logger = LoggerFactory.getLogger(Utils.class.getName());

    static String regExOfchinese = "[\\u4e00-\\u9fa5]";
    static Pattern pattern = Pattern.compile(regExOfchinese);

    static {
        wellKnownSimplePrefixes.put("}", "{");
        wellKnownSimplePrefixes.put("]", "[");
        wellKnownSimplePrefixes.put(")", "(");

        wellKnownSimpleRegexs.put("${", "\\$\\{+|\\}");
        wellKnownSimpleRegexs.put("{", "\\{+|\\}");
    }

    /**
     * 处理double类型数据,返回处理后的新数据
     *
     * @param value
     * @param precision
     * @return
     */
    public static double doubleRound(Double value, int precision) {
        switch (precision) {
            case 1:
                return Math.round(value * 100) / 100.0;
            case 2:
                return Math.round(value * 100) / 100.00;
            case 3:
                return Math.round(value * 100) / 100.000;
            default:
                return Math.round(value * 100) / 100.00;

        }
    }

    /**
     * 提供精确的小数位四舍五入处理。
     *
     * @param v     需要四舍五入的数字
     * @param scale 小数点后保留几位
     * @return 四舍五入后的结果
     */
    public static double round(double v, int scale) {

        if (scale < 0) {
            throw new IllegalArgumentException(
                    "The scale must be a positive integer or zero");
        }
        BigDecimal b = new BigDecimal(Double.toString(v));
        BigDecimal one = new BigDecimal("1");
        return b.divide(one, scale, BigDecimal.ROUND_HALF_UP).doubleValue();
    }

    /**
     * 替换占位符${}
     *
     * @param data
     * @param params
     * @return
     * @throws Exception
     */
    public static String replaceHolder(String data, Map<String, Object> params) {
        String result = data;
        String[] values = data.split("\\$\\{+|\\}");
        for (int i = 0; i < values.length; i++) {
            if (params.containsKey(values[i])) {
                result = result.replaceAll("\\$\\{" + values[i] + "\\}", String.valueOf(params.get(values[i])));
            }
        }
        return result;
    }

    /***
     * 替换占位符${0}..
     * @param data
     * @param params  不定参数 对应 字符串中的　0,1,2..
     * @return
     * @throws Exception
     */
    public static String replaceHolder(String data, Object... params) {
        String result = data;
        String[] values = data.split("\\$\\{+|\\}");
        for (int i = 0; i < values.length; i++) {
            if (params[i] != null) {
                result = result.replaceAll("\\$\\{" + values[i] + "\\}", String.valueOf(params[i]));
            }
        }
        return result;
    }

    /**
     * replace placeholder with custom prefix /suffix
     *
     * @param placeholderPrefix
     * @param placeholderSuffix
     * @param data
     * @param params
     * @return
     * @throws Exception
     */
    public static String replaceHolder(String placeholderPrefix, String placeholderSuffix, String data, Map<String, Object> params) {
        String result = data;
        String simplePrefixForSuffix = wellKnownSimplePrefixes.get(placeholderSuffix);
        if (simplePrefixForSuffix != null && placeholderPrefix.endsWith(simplePrefixForSuffix)) {
            placeholderPrefix = simplePrefixForSuffix;
        }
        String[] values = data.split(wellKnownSimpleRegexs.get(placeholderPrefix) == null ? "\\$\\{+|\\}" : wellKnownSimpleRegexs.get(placeholderPrefix));
        for (int i = 0; i < values.length; i++) {
            if (params.containsKey(values[i])) {
                result = result.replaceAll("\\{" + values[i] + "\\}", String.valueOf(params.get(values[i])));
            }
        }
        return result;
    }


    /***
     * 替换url占位符${} 并对替换的参数值进行编码
     * @param data
     * @param params
     * @return
     * @throws Exception
     */
    public static String urlReplaceHolder(String data, Map<String, Object> params) throws Exception {
        String result = data;
        String[] values = data.split("\\$\\{+|\\}");
        for (int i = 0; i < values.length; i++) {
            if (params.containsKey(values[i])) {
                result = result.replaceAll("\\$\\{" + values[i] + "\\}", URLEncoder.encode(MapUtils.getString(params, values[i]), CHARSET_UTF8));
            }
        }
        return result;
    }

    /***
     * update/create hash map
     * @param key
     * @param value
     * @param map
     * @return
     */
    public static Map updateMap(Object key, Object value, Map map) {
        Map result = new HashMap();
        result.put(key, value);
        if (map != null && !map.isEmpty()) {
            result.putAll(map);
        }
        return result;
    }

    /***
     * contain chinese char
     * @param value
     * @return
     */
    public static boolean isContainChinese(String value) {
        Matcher matcher = pattern.matcher(value);
        boolean flg = false;
        if (matcher.find()) {
            flg = true;
        }
        return flg;
    }

    /***
     *
     * @param orig
     * @param dest
     * @return
     * @throws Exception
     */
    public static Object copyPropertyForSave(Object orig, Object dest) throws Exception {
        Field[] fieldArr = orig.getClass().getDeclaredFields();
        for (int i = 0; i < fieldArr.length; i++) {
            Field field = fieldArr[i];
            String type = fieldArr[i].getGenericType().toString(); // 获取属性的类型
            String fieldName = field.getName();
            String getMethodName = "get" + fieldName.substring(0, 1).toUpperCase() + fieldName.substring(1);
            String setMethodName = "set" + fieldName.substring(0, 1).toUpperCase() + fieldName.substring(1);
            if (type.equals("class java.lang.String")) { // 如果type是类类型，则前面包含"class "，后面跟类名
                Method method = orig.getClass().getMethod(getMethodName);
                String value = (String) method.invoke(orig); // 调用getter方法获取属性值
                if (StringUtils.isNotBlank(value)) {
                    method = dest.getClass().getMethod(setMethodName, String.class);
                    method.invoke(dest, value);
                }
            }

            if (type.equals("class java.math.BigDecimal")) { // 如果type是类类型，则前面包含"class "，后面跟类名
                Method method = orig.getClass().getMethod(getMethodName);
                BigDecimal value = (BigDecimal) method.invoke(orig); // 调用getter方法获取属性值
                if (value != null) {
                    method = dest.getClass().getMethod(setMethodName, BigDecimal.class);
                    method.invoke(dest, value);
                }
            }

        }
        return dest;
    }

    /***
     * url encode
     * @param url
     * @return
     * @throws UnsupportedEncodingException
     */
    public static String urlEncode(String url) {
        try {
            return URLEncoder.encode(url, CHARSET_UTF8);
        } catch (UnsupportedEncodingException e) {
            logger.error(e.toString());
        }
        return url;
    }

    /**
     * @param value
     * @return
     */
    public static boolean isNull(Object value) {
        if (value == null) {
            return true;
        }
        if (value instanceof String) {
            return StringUtils.isBlank((String) value);
        }
        return false;
    }

    /***
     * is not null
     * @param value
     * @return
     */
    public static boolean isNotNull(Object value) {
        return !isNull(value);
    }

    /**
     * 指定字符集和字节数限制获取子串
     *
     * @param originString
     * @param charsetName
     * @param byteLen
     * @return
     * @throws UnsupportedEncodingException
     */
    public static String substr(String originString, String charsetName, int byteLen)
            throws UnsupportedEncodingException {
        if (originString == null || originString.isEmpty() || byteLen <= 0) {
            return "";
        }
        char[] chars = originString.toCharArray();
        int length = 0;
        int index = chars.length;
        for (int i = 0; i < chars.length; ++i) {
            final int len = String.valueOf(chars[i]).getBytes(charsetName).length + length;
            if (len <= byteLen) {
                length = len;
            } else {
                index = i;
                break;
            }
        }
        return String.valueOf(chars, 0, index);
    }

    /**
     * url 转换
     *
     * @param mappingUrl
     * @param dictList
     * @return
     */
    public static String mappingUrl(String mappingUrl, List<Dict> dictList) {
        List<String> urlList = new ArrayList<>();
        if (!org.springframework.util.CollectionUtils.isEmpty(dictList)) {
            for (Dict dict : dictList) {
                urlList.add(dict.getValue());
            }
        }
        return Utils.urlMapping(mappingUrl, urlList);
    }

    /**
     * url 替换
     *
     * @param repUrl  11.0.22.27
     * @param urlList list<String>  11.0.22.27-172.19.196.142
     * @return
     */
    public static String urlMapping(String repUrl, List<String> urlList) {
        if (CollectionUtils.isNotEmpty(urlList)) {
            for (String url : urlList) {
                String[] urlArr = url.split("-");
                if (repUrl.indexOf(urlArr[0]) >= 0) {
                    repUrl = repUrl.replaceAll(urlArr[0], urlArr[1]);
                    break;
                }
            }
        }
        return repUrl;
    }

    /**
     * 行政区代码判断省、市、区、县
     */
    public static boolean judgeRegionLevel(Integer index, String regionCode) {
        boolean res = false;
        if (Integer.parseInt(regionCode.substring(index, regionCode.length())) == 0) {
            res = true;
        }
        return res;
    }

    public static Map resMap(boolean result, String msg) {
        Map<String, Object> map = new HashMap<String, Object>();
        map.put(MSG, msg);
        map.put(RESULT, result);
        return map;
    }

    /**
     * 判断ie浏览器
     * @param agent
     * @return
     */
    public static boolean isIEBrowser(String agent) {
        agent = agent.toLowerCase();
        String[] IEBrowserSignals = {"nsie", "trident", "edge"};
        for (String signal : IEBrowserSignals) {
            if (agent.contains(signal)) {
                return true;
            }
        }
        return false;
    }

    public static Map parseRes(boolean res,String msg){
        Map map = new HashMap();
        map.put("res",res);
        map.put("msg",msg);
        return map;
    }
    
    public static Pageable layuiPage(Pageable pageable) {
        int page = pageable.getPageNumber();
        if (page > 0) {
            pageable = new PageRequest(page - 1, pageable.getPageSize(), pageable.getSort());
        }
        return pageable;
    }
}
