package cn.gtmap.gtc.clients;

import cn.gtmap.gtc.sso.domain.dto.*;
import org.springframework.cloud.netflix.feign.FeignClient;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * @author <a href="mailto:yangyang@gtmap.cn">yangyang</a>
 * @version 1.0, 2018/7/4
 * @description
 */

@FeignClient(name = "account")
@RequestMapping(value = "/account/rest/v1/users")
public interface UserManagerClient {

    /**
     * @param username 用户名/账户
     * @param alias 姓名
     * @param enabled 1：正常 0：禁用
     * @return
     * @description 分页显示用户信息，模糊查询
     */
    @GetMapping("/list")
    Page<UserDto> listUsers(@RequestParam(name = "username", required = false) String username,
                            @RequestParam(name = "alias", required = false) String alias,
                            @RequestParam(name = "enabled",required = false) Integer enabled,
                            Pageable pageable);

    /**
     * @param inquirer 操作人用户名/账户
     * @param username 用户名/账户
     * @param alias 姓名
     * @param enabled 1：正常 0：禁用 null, 全部
     * @return
     * @description 根据用户归属组织分级查询可查看的用户信息，以操作人所在组织做默认分级搜索
     */
    @GetMapping("/grading-users")
    Page<UserDto> listGradeUsers(Pageable pageable,
                                 @RequestParam(name = "inquirer") String inquirer,
                                 @RequestParam(name = "username",required = false) String username,
                                 @RequestParam(name = "alias",required = false) String alias,
                                 @RequestParam(name = "enabled",required = false) Integer enabled);

    /**
     * @param inquirer 操作人用户名/账户
     * @param orgId 组织id
     * @param username 用户名/账户
     * @param alias 姓名
     * @param enabled 1：正常 0：禁用 null: 全部
     * @return
     * @description 根据登陆用户可管理的组织，分级查询可查看的用户信息
     */
    @GetMapping("/grading-org-users")
    Page<UserDto> listGradingOrgUsers(Pageable pageable,
                                           @RequestParam(name = "inquirer") String inquirer,
                                           @RequestParam(name = "orgId",required = false) String orgId,
                                           @RequestParam(name = "username",required = false) String username,
                                           @RequestParam(name = "alias",required = false) String alias,
                                           @RequestParam(name = "enabled",required = false) Integer enabled);

    /**
     * @param inquirer 操作人用户名/账户
     * @param orgId 组织id
     * @param username 用户名/账户
     * @param alias 姓名
     * @param enabled 1：正常 0：禁用 null: 全部
     * @return
     * @description 根据登陆用户可管理的组织，分级查询可查看的在线用户实时状态信息
     */
    @GetMapping("/grading-org-user-online")
    Page<UserStatusDto> listGradingOrgUserOnline(Pageable pageable,
                                                        @RequestParam(name = "inquirer") String inquirer,
                                                        @RequestParam(name = "orgId",required = false) String orgId,
                                                        @RequestParam(name = "username",required = false) String username,
                                                        @RequestParam(name = "alias",required = false) String alias,
                                                        @RequestParam(name = "enabled",required = false) Integer enabled);

    /**
     * @param userDetail 用户信息
     * @return
     * @description 创建用户
     */
    @PostMapping
    UserDto createNewUser(@RequestBody UserDto userDetail);

    /**
     * @param userDtos 用户信息
     * @return
     * @description 批量导入用户信息
     */
    @PostMapping("/import")
    BaseResultDto importUsers(@RequestBody List<UserDto> userDtos);

    /**
     * @param  id 用户id
     * @param userDetail 用户信息
     * @return
     * @description 更新用户信息
     */
    @PutMapping("/{id}")
    UserDto updateUser(@PathVariable("id") String id, @RequestBody UserDto userDetail);

    /**
     * @param  id 用户id
     * @param userDetail 用户信息
     * @return
     * @description 更新部分用户信息，属性为空不更新
     */
    @PatchMapping("/{id}")
    UserDto updateUserInfo(@PathVariable("id") String id, @RequestBody UserDto userDetail);

    /**
     * @param  username 账号
     * @param password  密码
     * @return
     * @description 更新用户密码
     */
    @PostMapping("/password")
    UserDto updateUserPassword(@RequestParam("username") String username, @RequestParam("password") String password);

    /**
     * @param  id 用户id
     * @return
     * @description 获取用户信息 (页面配置时调用)
     */
    @GetMapping("/{id}")
    UserDto getUserDetail(@PathVariable("id") String id);


    /**
     * @param  username 用户名/账户
     * @return
     * @description 通过用户名获取用户详情（获取信息更全）
     * @deprecated 中文名账号调用出错 used  UserManagerClient.getUserDetailByUsername()
     */
    @Deprecated
    @GetMapping("/username/{username}")
    UserDto getUserByUsername(@PathVariable("username") String username);


    /**
     * @param  username 用户名/账户
     * @return
     * @description 通过用户名获取用户详情（全）
     */
    @GetMapping("/username")
    UserDto getUserDetailByUsername(@RequestParam("username") String username);

    /**
     * @return
     * @description 获取当前访问的用户信息（全）
     */
    @GetMapping("/current-user")
    UserDto getCurrentUser();

    /**
     * @param  id 用户
     * @return
     * @description 启用用户
     */
    @PostMapping("/{id}/enabled")
    Boolean enabledUser(@PathVariable(name = "id") String id);

    /**
     * @param  id 用户
     * @return
     * @description 禁用用户
     */
    @PostMapping("/{id}/disabled")
    Boolean disabledUser(@PathVariable(name = "id") String id);

    /**
     * @param  id 用户
     * @return
     * @description 删除用户
     */
    @DeleteMapping("/{id}")
    Boolean deleteUser(@PathVariable("id") String id);

    /**
     * @param  id 用户
     * @param  signId 签名文件上传文档中心后生成
     * @return
     * @description 更新用户签名ID
     */
    @PostMapping("/{id}/sign")
    Boolean updateUserSign(@PathVariable(name = "id") String id, @RequestParam(name = "signId") String signId);

    /**
     * @param  username 用户名
     * @return
     * @description 获取用户签名ID
     */
    @GetMapping("/sign")
    String getUserSign(@RequestParam(name = "username") String username);

    /**
     * @param  usernames 用户名列表
     * @return
     * @description 根据账号列表查询用户列表
     */
    @PostMapping("/username-list")
    List<UserDto> listUserByUserNames(@RequestBody List<String> usernames);

    /**
     * @param  ids id列表
     * @return
     * @description 根据id列表查询用户列表
     */
    @PostMapping("/id-list")
    List<UserDto> listUserByIds(@RequestBody List<String> ids);


    /**
     * @param  id 用户
     * @param  roleIds 角色id列表
     * @return
     * @description 更新用户归属角色列表
     */
    @PostMapping("/{id}/roles")
    Boolean updateUserRoles(@PathVariable("id") String id, @RequestBody List<String> roleIds);

    /**
     * @param  id 用户
     * @param  orgIds 组织id列表
     * @return
     * @description 更新用户归属组织列表
     */
    @PostMapping("/{id}/orgs")
    Boolean updateUserOrgs(@PathVariable("id") String id, @RequestBody List<String> orgIds);


    /**
     * @param  id 用户
     * @return
     * @description 查询用户所有角色包继承来的角色
     */
    @GetMapping("/{id}/all-roles")
    List<RoleDto> findAllRoles(@PathVariable("id") String id);

    /**
     * @param  id 用户
     * @return
     * @description 查询用户用户页面配置的角色
     */
    @GetMapping("/{id}/roles")
    List<RoleDto> findRoles(@PathVariable("id") String id);

    /**
     * @param  id 用户
     * @return
     * @description 查询用户所在的组织
     */
    @GetMapping("/{id}/orgs")
    List<OrganizationDto> findOrgs(@PathVariable(name = "id") String id);

    /**
     * @param  username 用户名/账户
     * @return
     * @description 查询用户拥有的顶级组织
     * @deprecated as of 1.2.3, in favor of using {@link UserManagerClient#findUserGradingRootOrgs}
     */
    @Deprecated
    @GetMapping("/root-orgs")
    List<OrganizationDto> findUserRootOrgs(@RequestParam(name = "username") String username);

    /**
     * @param  username 用户名/账户
     * @param  enabled 0:禁用， 1启用， null:全部
     * @return
     * @description 查询用户可管理的顶级组织
     */
    @GetMapping("/grading-root-orgs")
    List<OrganizationDto> findUserGradingRootOrgs(@RequestParam(name = "username") String username,
                                                         @RequestParam(name = "enabled", required = false) Integer enabled);

    /**
     * @param  password 密码
     * @param  id 用户
     * @return
     * @description 检查用户密码是否匹配
     */
    @PostMapping("/check-password")
    Boolean checkUserPassword(@RequestParam(name = "password") String password, @RequestParam(name = "id") String id);



    /**
     * @param  id 用户
     * @return
     * @description 初始化密码 123456
     */
    @PostMapping("/init-password")
    Boolean initUserPassword(@RequestParam(name = "id") String id);

    /**
     * @param  username 用户名/账户
     * @return
     * @description 检查用户是否存在
     */
    @GetMapping("/existed")
    Boolean checkUserExist(@RequestParam(name = "username") String username);

    /**
     * @param  id 用户id
     * @return
     * @description 获取当前用户配置的模块权限
     */
    @GetMapping("/{id}/module-authority")
    AuthorityDto findUserModuleAuthority(@PathVariable(name = "id") String id, @RequestParam(name = "moduleId") String moduleId);

    /**
     * @param  id 用户id
     * @return
     * @description 锁定用户
     */
    @PostMapping("{id}/lock")
    Boolean lockUser(@PathVariable(name = "id") String id);

    /**
     * @param  id 用户id
     * @return
     * @description 解锁用户
     */
    @PostMapping("{id}/unlock")
    Boolean unlockUser(@PathVariable(name = "id") String id);

    /**
     * @param  username 用户名，注册时可不填，其他情况必填
     * @param  phone 注册时必填，其余可为空
     * @param  msgType 消息类型  USER_REGISTER： 用户注册，  USER_MODIFY_PWD： 修改密码
     * @return
     * @description 短信验证码发送
     */
    @PostMapping("/sms")
    BaseResultDto sendSms(@RequestParam(name = "username", required = false) String username,
                                 @RequestParam(name = "phone", required = false) String phone,
                                 @RequestParam(name = "msgType") String msgType);

    /**
     * @param  username 用户名，注册时可不填，其他情况必填
     * @param  phone 注册时必填，其余可为空
     * @param  msgType 消息类型  USER_REGISTER： 用户注册，  USER_MODIFY_PWD： 修改密码
     * @param  code 验证码
     * @return
     * @description 短信验证码验证
     */
    @GetMapping("/sms-verify")
    BaseResultDto smsVerify(@RequestParam(name = "username", required = false) String username,
                                   @RequestParam(name = "phone", required = false) String phone,
                                   @RequestParam(name = "msgType") String msgType,
                                   @RequestParam(name = "code") String code);

    /**
     * @param  roleId 角色的uuid，必填
     * @return
     * @description 分页根据角色id查询未选中的用户（角色用户的反选）
     */
    @GetMapping("/list/unSelect")
    public Page<UserDto> queryUnaddUserList(@RequestParam(name = "roleId")String roleId, Pageable pageable);

}
