package cn.gtmap.gtc.clients;

import cn.gtmap.gtc.sso.domain.dto.OrganizationDto;
import cn.gtmap.gtc.sso.domain.dto.ProvinceCityCountyDto;
import cn.gtmap.gtc.sso.domain.dto.RegionDto;
import org.springframework.cloud.netflix.feign.FeignClient;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * @author <a href="mailto:yangyang@gtmap.cn">yangyang</a>
 * @version 1.0.2018/8/3
 * @description
 */

@FeignClient(name = "account")
@RequestMapping("/account/rest/v1/regions")
public interface RegionManagerClient {

    /**
     * @param regionDto 行政区划详情
     * @return
     * @description 创建行政区
     */
    @PostMapping
    RegionDto createRegion(@RequestBody RegionDto regionDto);

    /**
     * @param regionDtos 行政区划详情
     * @return
     * @description 批量导入行政区信息
     */
    @PostMapping("/import")
    boolean importRegions(@RequestBody List<RegionDto> regionDtos);

    /**
     * @param regionDto 行政区划详情
     * @return
     * @description 更新行政区
     */
    @PutMapping
    RegionDto updateRegion(@RequestBody RegionDto regionDto);


    /**
     * @param  id 行政区划id
     * @return
     * @description 删除行政区
     */
    @DeleteMapping("/{id}")
    boolean deleteRegion(@PathVariable(name = "id") String id);

    /**
     * @param  id 行政区划id
     * @return
     * @description 禁用行政区
     */
    @PostMapping("/{id}/disable")
    boolean disableRegion(@PathVariable(name = "id") String id);

    /**
     * @param  id 行政区划id
     * @return
     * @description 启用行政区
     */
    @PostMapping("/{id}/enable")
    boolean enableRegion(@PathVariable(name = "id") String id);

    /**
     * @param  level 行政区划等级（1：省级；2：市；3：区县）
     * @return
     * @description 查找各级行政区
     */
    @GetMapping("/level")
    List<RegionDto> findRegionsByLevel(@RequestParam(name = "level") int level);

    /**
     * @param  code 行政区划编码
     * @return
     * @description 支持根据编号精确查询行政区化
     */
    @GetMapping("/code")
    RegionDto findRegionByCode(@RequestParam(name = "code") String code);

    /**
     * @param  name 行政区划名称
     * @return
     * @description 支持根据编号精确查询行政区化
     */
    @GetMapping("/name")
    RegionDto findRegionByName(@RequestParam(name = "name") String name);

    /**
     * @param  id 行政区划id
     * @param  code 行政区划编码
     * @return
     * @description 校验Code的唯一性
     */
    @GetMapping("/{code}/code-only")
    boolean validOnlyCode(@RequestParam(name = "id", required = false) String id, @PathVariable(name = "code") String code);

    /**
     * @param  id 行政区划id
     * @return
     * @description 根据ID查找行政区
     */
    @GetMapping("/{id}")
    RegionDto findRegionById(@PathVariable(name = "id") String id);

    /**
     * @param  id 行政区划id
     * @return
     * @description 根据ID查找行政区所在的省市区
     */
    @GetMapping("/{id}/province-city-county")
    ProvinceCityCountyDto getProCityCounty(@PathVariable(name = "id") String id);

    /**
     * @param
     * @return
     * @description 查找最顶级行政区
     */
    @GetMapping("/root")
    List<RegionDto> findRootRegions();

    /**
     * @param
     * @return
     * @description 查找启用的最顶级行政区
     */
    @GetMapping("/root/enabled")
    List<RegionDto> findRootRegionsEnabled();

    /**
     * @param id 行政区划id
     * @return
     * @description 查找直接下级行政区
     */
    @GetMapping("/{id}/junior")
    List<RegionDto> findJuniorRegions(@PathVariable(name = "id") String id);

    /**
     * @param id 行政区划id
     * @return
     * @description 查找启用的直接下级行政区
     */
    @GetMapping("/{id}/junior/enabled")
    List<RegionDto> findJuniorRegionsEnabled(@PathVariable(name = "id") String id);

    /**
     * @param id 行政区划id
     * @return
     * @description 查找行政区下的组织机构
     */
    @GetMapping("/{id}/orgs")
    List<OrganizationDto> findOrgsByRegion(@PathVariable(name = "id") String id);

}
