package cn.gtmap.gtc.clients;

import cn.gtmap.gtc.sso.domain.dto.*;
import org.springframework.cloud.netflix.feign.FeignClient;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.web.bind.annotation.*;

import java.util.Collection;
import java.util.List;

/**
 * @author <a href="mailto:yangyang@gtmap.cn">yangyang</a>
 * @version 1.0, 2018/7/7
 * @description
 */

@FeignClient(name = "account")
@RequestMapping("/account/rest/v1/roles")
public interface RoleManagerClient {

    /**
     * @param roleDto 角色信息
     * @return
     * @description 添加角色
     */
    @PostMapping
    RoleDto createRole(@RequestBody RoleDto roleDto);

    /**
     * @param roleDtos 角色信息
     * @return
     * @description 批量导入角色
     */
    @PostMapping("/import")
    boolean importRoles(@RequestBody List<RoleDto> roleDtos);


    /**
     * @param id 角色
     * @return
     * @description 获取角色信息
     */
    @GetMapping("/{id}")
    RoleDto getRoleDetail(@PathVariable("id") String id);

    /**
     * @param name 角色编码
     * @return
     * @description 检查角色是否存在
     */
    @GetMapping("/existed")
    Boolean checkRoleExist(@RequestParam(name = "name") String name);

    /**
     * @param id 角色id
     * @return
     * @description 启用角色
     */
    @PostMapping("/{id}/enabled")
    Boolean enabledRole(@PathVariable(name = "id") String id);

    /**
     * @param id 角色id
     * @return
     * @description 禁用角色
     */
    @PostMapping("/{id}/disabled")
    Boolean disabledRole(@PathVariable(name = "id") String id);

    /**
     * @param id 角色id
     * @param roleDto 角色信息
     * @return
     * @description 更新角色信息
     */
    @PutMapping("/{id}")
    RoleDto updateRoleDetail(@PathVariable("id") String id, @RequestBody RoleDto roleDto);

    /**
     * @param name 角色编码
     * @return
     * @description 通过角色名获取角色信息
     */
    @GetMapping("/name")
    RoleDto queryRoleByRoleName(@RequestParam("name") String name);

    /**
     * @param names 角色编码列表
     * @return
     * @description 通过角色编码列表获取角色列表
     */
    @PostMapping("/names")
    List<RoleDto> queryRoleByRoleNames(@RequestBody Collection<String> names);

    /**
     * @param ids 角色ID列表
     * @return
     * @description 通过Ids获取角色列表
     */
    @PostMapping("/list")
    List<RoleDto> queryRolesByIds(@RequestBody List<String> ids);


    /**
     * @param name 角色编码
     * @param alias 角色名称
     * @param enabled 1：正常， 0: 禁用
     * @return
     * @description 分页查询角色列表，模糊搜索
     */
    @GetMapping("/page")
    Page<RoleDto> listRoleRecords(Pageable pageable, @RequestParam(name = "name", required = false) String name,
                                  @RequestParam(name = "alias", required = false) String alias,
                                  @RequestParam(name = "enabled", required = false) Integer enabled);

    /**
     * @param inquirer 操作人用户名
     * @param name 角色编码
     * @param alias 角色名称
     * @param enabled 1：正常， 0: 禁用
     * @return
     * @description 分级查询角色列表，模糊搜索，根据操作人可操作的角色做默认分级  更换listGradingRoles()
     * @deprecated {@link RoleManagerClient#listGradingRoles}
     */
    @GetMapping("/grading-roles")
    @Deprecated
    Page<RoleDto> listGradeRoleRecords(Pageable pageable, @RequestParam(name = "name", required = false) String name,
                                       @RequestParam(name = "alias", required = false) String alias,
                                       @RequestParam(name = "enabled", required = false) Integer enabled,
                                       @RequestParam(name = "inquirer", required = false) String inquirer);
    /**
     * @param inquirer 操作人用户名
     * @param name 角色编码
     * @param alias 角色名称
     * @param orgId 组织ID, 想要查询哪个组织下的角色
     * @param enabled 1：正常， 0: 禁用 null 全部
     * @return
     * @description 获取当前登录用户分级可授权的可授权角色
     */

    @GetMapping("/grading-org-roles")
    Page<RoleDto> listGradingRoles(Pageable pageable,
                                   @RequestParam(name = "inquirer") String inquirer,
                                   @RequestParam(name = "orgId", required = false) String orgId,
                                   @RequestParam(name = "name", required = false) String name,
                                   @RequestParam(name = "alias", required = false) String alias,
                                   @RequestParam(name = "enabled", required = false) Integer enabled
    );


    /**
     * @param id 角色id
     * @return
     * @description 查询角色及子级角色所有用户
     */
    @GetMapping("/{id}/all-enabled-users")
    Page<UserDto> listAllUsersByRoleId(Pageable pageable, @PathVariable("id") String id);

    /**
     * @param id 角色id
     * @return
     * @description 查询角色及子级角色所有用户
     */
    @GetMapping("/{id}/list-enabled-users")
    List<UserDto> listEnableUsersByRoleId(@PathVariable("id") String id);

    /**
     * @param name 角色编码
     * @return
     * @description 查询角色编码角色及子级角色所有用户
     */
    @GetMapping("/name/{name}/all-enabled-users")
    Page<UserDto> listEnableUsersByName(Pageable pageable, @PathVariable("name") String name);

    /**
     * @param id 角色Id
     * @param userIds 用户Id列表
     * @return
     * @description 角色添加人员列表
     */
    @PostMapping("/{id}/add-users")
    boolean addRoleUsers(@PathVariable(name = "id") String id, @RequestBody List<String> userIds);

    /**
     * @param id 角色Id
     * @param userIds 用户Id列表， 用英文逗号间隔
     * @return
     * @description 角色删除人员列表
     */
    @DeleteMapping("/{id}/del-users")
    boolean delRoleUsers(@PathVariable(name = "id") String id, @RequestParam(name = "userIds") String userIds);

    /**
     * @param
     * @return
     * @description 查询所有角色列表
     */
    @GetMapping
    List<RoleDto> getAllRoleList();

    /**
     * @param
     * @return
     * @description 查询所有启用角色列表
     */
    @GetMapping("/enabled")
    List<RoleDto> getEnabledRoleList();

    /**
     * @param id 角色id
     * @return
     * @description 删除角色
     */
    @DeleteMapping("/{id}")
    Boolean deleteRole(@PathVariable("id") String id);

    /**
     * @param id 角色id
     * @param roleIds 角色id列表
     * @return
     * @description 添加角色继承关系
     */
    @PostMapping("/{id}/inheriting-roles")
    Boolean updateInheritRoles(@PathVariable("id") String id, @RequestBody List<String> roleIds);

    /**
     * @param id 角色id
     * @return
     * @description 获取继承的角色列表
     */
    @GetMapping("/{id}/inheriting-roles")
    List<RoleDto> listInheritRoles(@PathVariable("id") String id);

    /**
     * @param id 角色id
     * @param roleIds 角色id列表
     * @return
     * @description 更新角色互斥关系
     */
    @PostMapping("/{id}/excluded-roles")
    Boolean updateExcludeRoles(@PathVariable("id") String id, @RequestBody List<String> roleIds);

    /**
     * @param id 角色id
     * @return
     * @description 获取角色互斥关系
     */
    @GetMapping("/{id}/excluded-roles")
    List<RoleDto> listExcludeRoles(@PathVariable("id") String id);

    /**
     * @param id 角色id
     * @return
     * @description 获取单个角色互斥和被互斥角色
     */
    @GetMapping("/{id}/all-excluded-roles")
    List<RoleDto> listAllExcludeRoles(@PathVariable("id") String id);

    /**
     * @param ids 角色id列表
     * @return
     * @description 批量获取互斥和被互斥角色
     */
    @PostMapping("/excluded-roles")
    List<RoleDto> listExcludeRolesByIds(@RequestBody List<String> ids);


    /**
     * @param roleDto 当前角色可管理理哪些角色，可管理哪些模块
     * @return
     * @description 保存分级授权权限  use saveGradingConfig()
     * @deprecated {@link RoleManagerClient#saveGradingConfig}
     */
    @Deprecated
    @PutMapping("/{id}/grading-authority")
    boolean saveGradingAuthority(@PathVariable("id") String id, @RequestBody RoleDto roleDto);

    /**
     * @param configDto 当前角色可管理理哪些角色，可管理哪些模块
     * @return
     * @description 保存分级授权权限  use saveGradingConfig()
     */
    @PutMapping("/{id}/grading-config")
    boolean saveGradingConfig(@PathVariable("id") String id, @RequestBody GradingConfigDto configDto);

    /**
     * @param id 当前角色
     * @return
     * @description 获取当前角色可管理/可授权角色
     * @deprecated
     */
    @GetMapping("/{id}/grading-roles")
    @Deprecated
    List<RoleDto> findGradingRoles(@PathVariable("id") String id);


    /**
     * @param id 当前角色
     * @return
     * @description 获取获取当前模块可管理授权的模块权限
     */
    @GetMapping("/{id}/grading-authority")
    List<GradingAuthorityDto> findGradingAuthority(@PathVariable("id") String id);

    /**
     * @param username 操作人用户名
     * @return
     * @description 获取当前登录用户的可授权角色
     */
    @GetMapping("/user/grading-roles")
    List<RoleDto> findGradingRolesByLoginUser(@RequestParam(name = "username") String username);

    /**
     * @param username 操作人用户名
     * @param pid 模块
     * @param gradedRoleId
     * @param clientId 应用Id
     * @return
     * @description 获取当前登录用户的可授权模块，用于分级授权及授权
     */
    @GetMapping("/user/grading-modules")
    List<ModuleDto> findLoginUserGradingModules(@RequestParam(name = "username") String username,
                                                @RequestParam(name = "pid", required = false) String pid,
                                                @RequestParam(name = "gradedRoleId", required = false) String gradedRoleId,
                                                @RequestParam(name = "clientId", required = false) String clientId);

    /**
     * @param id 角色Id
     * @param enabled 0:禁用 1：正常 null: 全部
     * @return
     * @description 获取角色可管理的组织
     */
    @GetMapping("/{id}/grading-orgs")
    List<OrganizationDto> findGradingOrgs(@PathVariable(name = "id") String id, @RequestParam(name = "enabled", required = false) Integer enabled);

}
