package cn.gtmap.gtc.starter.gcas.util;

import cn.gtmap.gtc.feign.common.exception.ResponseException;
import okhttp3.*;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * @author <a href="mailto:yangyang@gtmap.cn">yangyang</a>
 * @version 1.0.2019/9/19
 * @description
 */
public class HttpUtils {
    private static final Logger logger = LoggerFactory.getLogger(HttpUtils.class);
    private static OkHttpClient okHttpClient;

    static {
        okHttpClient = new OkHttpClient.Builder()
                .retryOnConnectionFailure(false)//是否开启缓存
                .connectionPool(new ConnectionPool(5, 5, TimeUnit.MINUTES))//连接池
                .connectTimeout(10L, TimeUnit.SECONDS)
                .readTimeout(30L, TimeUnit.SECONDS)
                .build();
    }

    /**
     * 根据map获取get请求参数
     * @param queries
     * @return
     */
    public static StringBuffer getQueryString(String url,Map<String,String> queries){
        StringBuffer sb = new StringBuffer(url);
        if (queries != null && queries.keySet().size() > 0) {
            boolean firstFlag = true;
            Iterator iterator = queries.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry entry = (Map.Entry<String, String>) iterator.next();
                if (firstFlag) {
                    sb.append("?" + entry.getKey() + "=" + entry.getValue());
                    firstFlag = false;
                } else {
                    sb.append("&" + entry.getKey() + "=" + entry.getValue());
                }
            }
        }
        return sb;
    }

    /**
     * 调用okhttp的newCall方法
     * @param request
     * @return
     */
    private static String execNewCall(Request request){
        Response response = null;
        try {
            response = okHttpClient.newCall(request).execute();
            if (response.isSuccessful()) {
                return response.body().string();
            } else {
                throw new ResponseException(response.body().string());
            }
        } catch (Exception e) {

            logger.error("okhttp3 put error >> ex = {}", ExceptionUtils.getStackTrace(e));
            throw new ResponseException(e.getMessage());
        } finally {
            if (response != null) {
                response.close();
            }
        }
    }

    /**
     * get
     * @param url     请求的url
     * @param queries 请求的参数，在浏览器？后面的数据，没有可以传null
     * @return
     */
    public static String get(String url, Map<String, String> queries, String authorizationHeader) {
        StringBuffer sb = getQueryString(url,queries);
        Request.Builder builder = new Request.Builder();
        builder.addHeader("Accept", "application/json");
        builder.addHeader("Content-Type", "application/json");
        if (StringUtils.isNotBlank(authorizationHeader)){
            builder.addHeader("Authorization", authorizationHeader);
        }
        Request request = builder.url(sb.toString())
                .build();
        return execNewCall(request);
    }

    public static String delete(String url,String jsonParams, Map<String, String> queries, String authorizationHeader) {
        StringBuffer sb = getQueryString(url,queries);
        RequestBody requestBody = null;
        if (StringUtils.isNotBlank(jsonParams)) {
            requestBody = RequestBody.create(MediaType.parse("application/json; charset=utf-8"), jsonParams);
        }
        Request.Builder builder = new Request.Builder();
        if (StringUtils.isNotBlank(authorizationHeader)){
            builder.addHeader("Authorization", authorizationHeader);
        }
        Request request = builder.url(sb.toString())
                .delete(requestBody)
                .build();
        return execNewCall(request);
    }


    public static String put(String url, String jsonParams, Map<String, String> queries,  String authorizationHeader) {
        StringBuffer sb = getQueryString(url,queries);
        RequestBody requestBody = null;
        if (StringUtils.isNotBlank(jsonParams)) {
            requestBody = RequestBody.create(MediaType.parse("application/json; charset=utf-8"), jsonParams);
        } else {
            requestBody = RequestBody.create(MediaType.parse("application/json; charset=utf-8"), "");
        }

        Request.Builder builder = new Request.Builder();
        if (StringUtils.isNotBlank(authorizationHeader)){
            builder.addHeader("Authorization", authorizationHeader);
        }
        Request request = builder.url(sb.toString())
                .put(requestBody)
                .build();
        return execNewCall(request);
    }
    public static String post(String url, String jsonParams, Map<String, String> queries,  String authorizationHeader) {
        StringBuffer sb = getQueryString(url,queries);
        RequestBody requestBody = null;
        if (StringUtils.isNotBlank(jsonParams)) {
            requestBody = RequestBody.create(MediaType.parse("application/json; charset=utf-8"), jsonParams);
        } else {
            requestBody = RequestBody.create(MediaType.parse("application/json; charset=utf-8"), "");
        }

        Request.Builder builder = new Request.Builder();
        if (StringUtils.isNotBlank(authorizationHeader)){
            builder.addHeader("Authorization", authorizationHeader);
        }
        Request request = builder.url(sb.toString())
                .post(requestBody)
                .build();
        return execNewCall(request);
    }


    /**
     * post
     *
     * @param url    请求的url
     * @param params post form 提交的参数
     * @return
     */
    public static String postFormParams(String url, Map<String, String> params) {
        FormBody.Builder builder = new FormBody.Builder();
        //添加参数
        if (params != null && params.keySet().size() > 0) {
            for (String key : params.keySet()) {
                builder.add(key, params.get(key));
            }
        }
        Request request = new Request.Builder()
                .url(url)
                .post(builder.build())
                .build();
        return execNewCall(request);
    }


    /**
     * Post请求发送JSON数据....{"name":"zhangsan","pwd":"123456"}
     * 参数一：请求Url
     * 参数二：请求的JSON
     * 参数三：请求回调
     */
    public static String postJsonParams(String url, String jsonParams) {
        RequestBody requestBody = RequestBody.create(MediaType.parse("application/json; charset=utf-8"), jsonParams);
        Request request = new Request.Builder()
                .url(url)
                .post(requestBody)
                .build();
        return execNewCall(request);
    }

    /**
     * Post请求发送xml数据....
     * 参数一：请求Url
     * 参数二：请求的xmlString
     * 参数三：请求回调
     */
    public static String postXmlParams(String url, String xml) {
        RequestBody requestBody = RequestBody.create(MediaType.parse("application/xml; charset=utf-8"), xml);
        Request request = new Request.Builder()
                .url(url)
                .post(requestBody)
                .build();
        return execNewCall(request);
    }

}
