/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     Codecs.java
 * Modifier: yangxin
 * Modified: 2014-06-11 10:38
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.egovplat.core.util;

import cn.gtmap.egovplat.core.ex.AppException;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;
import com.google.common.io.BaseEncoding;
import org.apache.commons.lang3.StringUtils;

import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 12-8-17
 */
public final class Codecs {

    private static HashFunction HF = Hashing.murmur3_128("hp".hashCode());
    private static BaseEncoding BE = BaseEncoding.base64Url().omitPadding();

    public static byte[] hash(byte[] bytes) {
        return HF.hashBytes(bytes).asBytes();
    }

    public static String hash(String str) {
        return encode(hash(bytes(str)));
    }

    public static String hash(String str, int len) {
        return StringUtils.left(hash(str), len);
    }

    public static String md5Hex(byte[] bytes) {
        return Hashing.md5().hashBytes(bytes).toString();
    }

    public static String md5Hex(String str) {
        return md5Hex(bytes(str));
    }

    public static String sha1Hex(byte[] bytes) {
        return Hashing.sha1().hashBytes(bytes).toString();
    }

    public static String sha1Hex(String str) {
        return sha1Hex(bytes(str));
    }

    public static String encode(byte[] bytes) {
        return BE.encode(bytes);
    }

    public static String encode(long l) {
        ByteBuffer buf = ByteBuffer.allocate(8);
        buf.putLong(l);
        return encode(buf.array());
    }

    public static String encode(int i) {
        ByteBuffer buf = ByteBuffer.allocate(4);
        buf.putInt(i);
        return encode(buf.array());
    }

    public static byte[] decode(String str) {
        return BE.decode(str);
    }

    public static byte[] xor(byte[] bytes, byte[] keyBytes) {
        for (int i = 0, len = bytes.length, keyLen = keyBytes.length; i < len; i++) {
            bytes[i] = (byte) (bytes[i] ^ keyBytes[i % keyLen]);
        }
        return bytes;
    }

    public static String xorEncode(String str, String key) {
        return encode(xor(bytes(str), bytes(key)));
    }

    public static String xorDecode(String str, String key) {
        return string(xor(decode(str), bytes(key)));
    }

    public static String uuid() {
        return encode(hash(UUID.bytes()));
    }

    public static String uuid(int len) {
        return StringUtils.left(encode(hash(UUID.bytes())), len);
    }

    public static byte[] bytes(String s) {
        if (s == null) {
            return ArrayUtils.EMPTY_BYTE_ARRAY;
        }
        try {
            return s.getBytes(Charsets.UTF8);
        } catch (UnsupportedEncodingException e) {
            throw new AppException(e);
        }
    }

    public static String string(byte[] bytes) {
        try {
            return new String(bytes, Charsets.UTF8);
        } catch (UnsupportedEncodingException e) {
            throw new AppException(e);
        }
    }
}
