/*
 * Project:  gtbroker
 * Module:   gtbroker-server
 * File:     FreemarkerTemplateRenderer.java
 * Modifier: xyang
 * Modified: 2013-10-28 04:41
 *
 * Copyright (c) 2014 Gtmap Ltd. All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent or the
 * registration of a utility model, design or code.
 */

package cn.gtmap.egovplat.core.template;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Maps;
import cn.gtmap.egovplat.core.util.Codecs;
import freemarker.template.Configuration;
import freemarker.template.Template;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.servlet.view.freemarker.FreeMarkerConfig;

import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 12-8-28
 */
public class FreemarkerTemplateRenderer implements TemplateRenderer {

    private static Logger LOG = LoggerFactory.getLogger(FreemarkerTemplateRenderer.class);
    private Configuration configuration;
    private Map<String, Object> variables;
    private Cache<String, Template> templateCache = CacheBuilder.newBuilder().maximumSize(1000).build();

    public void setFreeMarkerConfig(FreeMarkerConfig freeMarkerConfig) {
        setConfiguration(freeMarkerConfig.getConfiguration());
    }

    public void setConfiguration(Configuration configuration) {
        this.configuration = configuration;
    }

    public void setVariables(Map<String, Object> variables) {
        this.variables = variables;
    }

    @Override
    public String render(String templateName, Map<String, Object> params) throws RenderingException, TemplateNotFoundException{
        return renderTpl(getTemplate(templateName), params);
    }

    @Override
    public void render(String templateName, Map<String, Object> params, Writer writer) throws RenderingException, TemplateNotFoundException{
        renderTpl(getTemplate(templateName), params, writer);
    }

    @Override
    public String renderFragment(String fragment, Map<String, Object> params) throws RenderingException {
        return renderTpl(getStringTemplate(fragment), params);
    }

    private void renderTpl(Template tpl, Map<String, Object> params, Writer writer) {
        try {
            tpl.process(createModel(params), writer);
        } catch (Exception e) {
            LOG.error("Render template:[" + tpl + "] error", e);
            throw new RenderingException(e);
        }
    }

    private Map<String, Object> createModel(Map<String, Object> params) {
        if (variables != null) {
            Map<String, Object> model = Maps.newHashMap(variables);
            model.putAll(params);
            return model;
        } else {
            return params;
        }
    }

    private String renderTpl(Template tpl, Map<String, Object> params) {
        StringWriter writer = new StringWriter();
        renderTpl(tpl, params, writer);
        return writer.toString();
    }

    private Template getTemplate(String name) {
        try {
            return configuration.getTemplate(name);
        } catch (IOException e) {
            throw new TemplateNotFoundException("Tpl [" + name + "] not found", e);
        }
    }

    private Template getStringTemplate(final String content) {
        final String key = Codecs.hash(content);
        try {
            return templateCache.get(key, new Callable<Template>() {
                @Override
                public Template call() throws Exception {
                    return new Template(key, content, configuration);
                }
            });
        } catch (ExecutionException ignored) {
        }
        return null;
    }
}
