/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     FsAndMemoryCacheStrategy.java
 * Modifier: yangxin
 * Modified: 2014-06-30 16:23
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.egovplat.core.support.wro;

import com.google.common.base.Optional;
import com.google.common.collect.Maps;
import cn.gtmap.egovplat.core.util.Codecs;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.SerializationUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ro.isdc.wro.cache.CacheStrategy;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.util.Map;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 14-6-30
 */
public class FsAndMemoryCacheStrategy<K, V> implements CacheStrategy<K, V> {
    private static final Logger LOG = LoggerFactory.getLogger(FsAndMemoryCacheStrategy.class);
    private final Map<K, Optional<V>> map = Maps.newConcurrentMap();
    private final File tmpPath;

    public FsAndMemoryCacheStrategy() {
        tmpPath = new File(System.getProperty("java.io.tmpdir"), "wro-" + Codecs.hash(System.getProperty("user.dir"), 6));
        if (!tmpPath.exists()) {
            tmpPath.mkdirs();
        } else {
            if (tmpPath.list().length > 1000) {
                clear();
            }
        }
    }

    @Override
    public void put(K key, V value) {
        map.put(key, Optional.fromNullable(value));
        if (value != null) {
            try {
                FileUtils.writeByteArrayToFile(new File(tmpPath, getStringKey(key)), SerializationUtils.serialize((Serializable) value));
            } catch (IOException e) {
                LOG.warn("Error to store [" + key + "=" + value + "]", e);
            }
        }
    }

    @Override
    public V get(K key) {
        Optional<V> optional = map.get(key);
        if (optional!=null && optional.isPresent()) {
            return optional.get();
        }
        File file = new File(tmpPath, getStringKey(key));
        try {
            if (file.exists()) {
                V value = SerializationUtils.deserialize(FileUtils.readFileToByteArray(file));
                map.put(key, Optional.of(value));
                return value;
            }
        } catch (IOException ignored) {
        }
        return null;
    }

    @Override
    public void clear() {
        try {
            FileUtils.cleanDirectory(tmpPath);
        } catch (IOException ignored) {
        }
    }

    @Override
    public void destroy() {
    }

    private String getStringKey(K key) {
        return Codecs.hash(StringUtils.substringAfter(key.toString(), "["));
    }
}
