/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     DefaultAppFilter.java
 * Modifier: yangxin
 * Modified: 2014-07-02 17:24
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.egovplat.core.support.spring;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import cn.gtmap.egovplat.core.env.Env;
import cn.gtmap.egovplat.core.ex.AppException;
import cn.gtmap.egovplat.core.ex.Source;
import cn.gtmap.egovplat.core.ex.SourceExtractor;
import cn.gtmap.egovplat.core.ex.Sourceable;
import cn.gtmap.egovplat.core.ex.impl.SpringConfSourceExtractor;
import cn.gtmap.egovplat.core.util.Charsets;
import cn.gtmap.egovplat.core.util.ExUtils;
import freemarker.cache.WebappTemplateLoader;
import freemarker.template.Configuration;
import freemarker.template.TemplateException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.context.WebApplicationContext;

import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.util.List;
import java.util.Map;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 14-7-2
 */
public class AppFilter implements Filter {
    private static final Logger LOG = LoggerFactory.getLogger(FriendlyExceptionRenderInterceptor.class);
    public static final String EX_RENDERED = "AppFilter.exRendered";
    private static final String FILTERED = "AppFilter.filtered";

    private ServletContext servletContext;
    private Configuration configuration;
    private List<SourceExtractor> sourceExtractors = Lists.newArrayList();

    @Override
    public void init(FilterConfig config) throws ServletException {
        servletContext = config.getServletContext();
        configuration = new Configuration();
        configuration.setLocalizedLookup(false);
        configuration.setDefaultEncoding(Charsets.UTF8);
        configuration.setTemplateLoader(new WebappTemplateLoader(servletContext, "/WEB-INF/views/common"));
        sourceExtractors.add(new SpringConfSourceExtractor());
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        if (request.getAttribute(FILTERED) != null) {
            chain.doFilter(request, response);
            return;
        }
        try {
            request.setAttribute(FILTERED, Boolean.TRUE);
            request.setCharacterEncoding(Charsets.UTF8);
            response.setCharacterEncoding(Charsets.UTF8);
            Object attr = servletContext.getAttribute(WebApplicationContext.ROOT_WEB_APPLICATION_CONTEXT_ATTRIBUTE);
            if (attr instanceof Throwable) {
                renderException(request, response, (Throwable) attr);
            }
            try {
                chain.doFilter(request, response);
            } catch (Throwable e) {
                if (request.getAttribute(EX_RENDERED) != null) {
                    renderException(request, response, e);
                }
            }
        } finally {
            request.removeAttribute(FILTERED);
        }
    }

    private void renderException(ServletRequest request, ServletResponse response, Throwable ex) throws IOException {
        response.resetBuffer();
        Sourceable sEx = findSourceable(ex);
        if (sEx != null) {
            ex = (Throwable) sEx;
        }
        Map<String, Object> model = Maps.newHashMap();
        AppException appEx = ExUtils.wrap(ex);
        model.put("type", ex.getClass());
        model.put("rawEx", ex);
        model.put("ex", appEx);
        if (LOG.isWarnEnabled()) {
            LOG.warn("Handler request exception [" + ((HttpServletRequest) request).getRequestURI() + "],id=" + appEx.getId(), appEx);
        }
        if (Env.isDev()) {
            Source source = ExUtils.extractSource(sourceExtractors, ex);
            if (source == null) {
                source = new Source();
                if (sEx != null) {
                    source.setTitle(sEx.getTitle());
                    source.setFile(sEx.getFile());
                    source.setLineNumber(sEx.getLineNumber());
                    source.setLines(sEx.getLines());
                } else {
                    source.setTitle("Application startup error");
                }
                source.setDescription(ExUtils.buildNestedMessage(ex));
            }
            model.put("source", source);
        }
        try {
            configuration.getTemplate((Env.isDev() ? "ex" : "error") + ".ftl").process(model, response.getWriter());
        } catch (TemplateException e) {
            ExUtils.throwEx(e);
        }
    }

    private Sourceable findSourceable(Throwable cause) {
        Throwable lastCause = null;
        while (cause != null && cause != lastCause) {
            if (cause instanceof Sourceable) {
                return (Sourceable) cause;
            }
            lastCause = cause;
            cause = cause.getCause();
        }
        return null;
    }

    @Override
    public void destroy() {
    }
}
