/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     SpringConfSourceExtractor.java
 * Modifier: yangxin
 * Modified: 2014-07-02 19:20
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.egovplat.core.ex.impl;

import cn.gtmap.egovplat.core.ex.Source;
import cn.gtmap.egovplat.core.ex.SourceExtractor;
import cn.gtmap.egovplat.core.util.Charsets;
import cn.gtmap.egovplat.core.util.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanCreationException;
import org.springframework.beans.factory.CannotLoadBeanClassException;
import org.springframework.beans.factory.xml.XmlBeanDefinitionStoreException;

import java.io.File;
import java.io.IOException;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 14-7-2
 */
public class SpringConfSourceExtractor implements SourceExtractor<BeansException> {
    private String[] confPaths = new String[]{"src/main/resources"};

    public void setConfPaths(String[] confPaths) {
        this.confPaths = confPaths;
    }

    @Override
    public boolean isSupport(Throwable ex) {
        return ex instanceof BeansException;
    }

    @Override
    public Source extract(BeansException ex) {
        Source source = new Source();
        source.setTitle("Spring context error");
        ExInfo exInfo = getExInfo(ex);
        if (exInfo.ex != null) {
            String beanName = exInfo.beanName;
            if (beanName != null) {
                int pos = beanName.lastIndexOf("#");
                if (pos > 0) {
                    beanName = "class=\"" + beanName.substring(0, pos) + "\"";
                } else {
                    beanName = "id=\"" + beanName + "\"";
                }
            }
            readLines(source, exInfo.path);
            if (exInfo.line > 0) {
                source.setLineNumber(exInfo.line);
            } else {
                for (int i = 0, len = source.getLines().size(); i < len; i++) {
                    String line = source.getLines().get(i);
                    if (line.contains(beanName)) {
                        source.setLineNumber(i + 1);
                    }
                }
            }
            source.setDescription(exInfo.ex.getMessage());
            return source;
        }
        return null;
    }

    private ExInfo getExInfo(Throwable ex) {
        ExInfo exInfo = new ExInfo();
        while (ex != null) {
            if (ex instanceof BeanCreationException) {
                BeanCreationException e = (BeanCreationException) ex;
                String desc = e.getResourceDescription();
                if (StringUtils.contains(desc, ".xml]")) {
                    exInfo.ex = e;
                    exInfo.beanName = e.getBeanName();
                    exInfo.path = getPath(desc);
                }
            } else if (ex instanceof CannotLoadBeanClassException) {
                CannotLoadBeanClassException e = (CannotLoadBeanClassException) ex;
                String desc = e.getResourceDescription();
                if (StringUtils.contains(desc, ".xml]")) {
                    exInfo.ex = e;
                    exInfo.beanName = e.getBeanName();
                    exInfo.path = getPath(desc);
                }
            } else if (ex instanceof XmlBeanDefinitionStoreException) {
                XmlBeanDefinitionStoreException e = (XmlBeanDefinitionStoreException) ex;
                exInfo.ex = e;
                exInfo.path = getPath(e.getResourceDescription());
                exInfo.line = e.getLineNumber();
            }
            ex = ex.getCause();
        }
        return exInfo;
    }

    private static String getPath(String desc) {
        return StringUtils.substringBetween(desc, "[", "]");
    }

    private void readLines(Source source, String path) {
        for (String sp : confPaths) {
            File file = new File(sp, path);
            if (file.exists()) {
                try {
                    source.setFile(file.getPath());
                    source.setLines(FileUtils.readLines(file, Charsets.UTF8));
                    return;
                } catch (IOException ignored) {
                }
            }
        }
    }

    static class ExInfo {
        String beanName;
        String path;
        int line;
        Exception ex;
    }
}
