/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     EnvConfLoader.java
 * Modifier: yangxin
 * Modified: 2014-06-11 17:37
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.egovplat.core.env;

import cn.gtmap.egovplat.core.Constants;
import cn.gtmap.egovplat.core.attr.Attrs;
import cn.gtmap.egovplat.core.util.IO;
import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.core.env.AbstractEnvironment;
import org.springframework.util.StringUtils;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Properties;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 11-5-30
 */
public class EnvConfLoader {

    protected final Log logger = LogFactory.getLog(EnvConfLoader.class);

    public static final String HOME_NAME = "egov.home";
    public static final String CONF_NAME = "egov.conf";
    public static final String DATA_NAME = "egov.data";
    public static final String PROPERTY_FILE_NAME = "egov.properties";
    public static final String DEFAULT_ACTIVE_CONF = "default";
    public static final String EGOV_HOME_FOLDER = "egov-home";

    public static void load(final String... paths) {
        EnvConfLoader cl = new EnvConfLoader();
        cl.loadConfig(paths);
    }

    public void loadConfig(final String... paths) {
        File root = determineRootDir(paths);
        logger.info("Use egov-home dir:[" + root + "]");

        File home;
        if(root!=null){
            logger.info("Use egov-home dir:[" + root + "]");
            File activeFile = new File(root, "active.conf");
            String active = DEFAULT_ACTIVE_CONF;
            try {
                active = FileUtils.readFileToString(activeFile);
            } catch (IOException e) {
                logger.info("Read active profile from [" + activeFile.getAbsolutePath() + "] fail,use default");
            }
            home = new File(root, active);
            if (!home.exists()) {
                home = new File(root, DEFAULT_ACTIVE_CONF);
                if (!home.exists()) {
                    logger.error("Egov active conf [" + home.getAbsoluteFile() + "] not exsit");
                }
            }
        }else{
            URL url = this.getClass().getResource("/META-INF/conf/" + PROPERTY_FILE_NAME);
            if (url != null) {
                try {
                    home = new File(url.toURI()).getParentFile().getParentFile();
                    logger.info("Custom config home not found,Use classpath config home dir [" + home.getAbsolutePath() + "]");
                } catch (URISyntaxException ignored) {
                    return;
                }
            } else {
                logger.error("Load config error,config not found");
                return;
            }
        }

        String homePath = home.getAbsolutePath();
        String confPath = homePath + File.separator + "conf";
        File confFile = new File(confPath, PROPERTY_FILE_NAME);
        Properties props = new Properties();
        setConfigProperty(props, HOME_NAME, home);
        setConfigProperty(props, CONF_NAME, new File(confPath));
        loadProperties(props, confFile);
        Env.configuration = Attrs.compositeAttrable(props, System.getProperties());

        String ps = Env.get(Env.APP_PROFILES);
        if (!confFile.exists()) {
            logger.info("Egov config file [" + confFile.getAbsolutePath() + "] not found,enable install mode");
            if (ps == null) {
                ps = "install";
            } else {
                ps += ",install";
            }
        }
        if (ps == null) {
            ps = "default";
        }
        if (System.getProperty("prod") != null) {
            ps += ",prod";
        }
        Env.setAttr(Env.APP_PROFILES, ps);
        System.setProperty(AbstractEnvironment.ACTIVE_PROFILES_PROPERTY_NAME, ps);
        logger.info("Use App profiles [" + ps + "]");

        Env.profiles = StringUtils.commaDelimitedListToSet(ps);
        if (Env.profiles.contains(Env.Mode.PROD.getTitle())) {
            Env.mode = Env.Mode.PROD;
        } else if (Env.profiles.contains(Env.Mode.TEST.getTitle())) {
            Env.mode = Env.Mode.TEST;
        } else {
            Env.mode = Env.Mode.DEV;
        }
        setConfigProperty(props, "app.model", Env.mode.getTitle());
        logger.info("Use App mode     [" + Env.mode.getTitle() + "]");
    }

    private void loadProperties(Properties p, File file) {
        try {
            if (!file.exists()) {
                return;
            }
            Properties props = IO.readProperties(file);
            for (String name : props.stringPropertyNames()) {
                setConfigProperty(p, name, Constants.PLACEHOLDER_HELPER.replacePlaceholders(props.getProperty(name), props));
            }
        } catch (IOException e) {
            logger.error("error to load props file:[" + file.getAbsolutePath() + "]", e);
        }
    }

    protected void setConfigProperty(Properties props, String key, String value) {
        props.put(key, value);
        logger.debug("Set env property [" + key + "=" + value + "]");
    }

    protected void setConfigProperty(Properties props,String key, File file) {
        setConfigPropertyAndSystemProperty(props, key, file.toURI().toString());
        setConfigPropertyAndSystemProperty(props, key + ".path", file.getAbsolutePath());
    }

    protected void setConfigPropertyAndSystemProperty(Properties props, String key, String value) {
        System.setProperty(key,value);
        setConfigProperty(props,key,value);
    }

    private File determineRootDir(final String... paths) {
        File root;
        if (paths != null) {
            for (String path : paths) {
                root = new File(path);
                if (root.exists())
                    return root;
            }
        }
        for (String path : new String[]{
                System.getProperty("EGOV_HOME"),
                System.getenv("EGOV_HOME")
        }) {
            if (path != null) {
                root = new File(path);
                if (root.exists())
                    return root;
            }
        }

        for (String path : new String[]{
                System.getProperty("catalina.base"),
                System.getProperty("catalina.home"),
                System.getProperty("user.home"),
                "/opt"
        }) {
            if (path != null) {
                root = new File(path, EGOV_HOME_FOLDER);
                if (root.exists())
                    return root;
            }
        }
        return null;
    }
}
