/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     PageRequest.java
 * Modifier: yangxin
 * Modified: 2014-06-11 10:38
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.egovplat.core.data;

import com.alibaba.fastjson.annotation.JSONField;
import com.google.common.collect.Lists;
import cn.gtmap.egovplat.core.util.ArrayUtils;
import com.mysema.query.types.OrderSpecifier;

import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 13-7-13
 */
public class PageRequest implements Pageable, Serializable {
    private static final long serialVersionUID = 4668590265538872363L;
    private final int offset;
    private final int size;
    private final Collection<Order> orders;

    public PageRequest(int offset, int size, Collection<Order> orders) {
        if (offset < 0) {
            throw new IllegalArgumentException("Offset must not be less than zero!");
        }
        this.offset = offset;
        this.size = size;
        this.orders = orders;
    }

    public PageRequest(int offset, int size, Order... orders) {
        this(offset, size, ArrayUtils.asList(orders));
    }

    public PageRequest(int offset, int size) {
        this(offset, size, Collections.<Order>emptyList());
    }

    public PageRequest(int offset) {
        this(offset, SIZE_DEFAULT);
    }

    public PageRequest() {
        this(0);
    }

    @Override
    public int getSize() {
        return size;
    }

    @Override
    @JSONField(serialize = false)
    public int getOffset() {
        return offset;
    }

    @Override
    public int getIndex() {
        return toIndex(offset, size);
    }

    @Override
    public Collection<Order> getOrders() {
        return orders;
    }

    @Override
    public int hashCode() {
        int result = 17;
        result = 31 * result + offset;
        result = 31 * result + size;
        result = 31 * result + orders.hashCode();
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (!(obj instanceof Pageable)) {
            return false;
        }
        Pageable other = (Pageable) obj;
        return offset == other.getOffset() && size == other.getSize() && orders.equals(other.getOrders());
    }

    @Override
    public String toString() {
        return "offset:" + offset + ", size:" + size + (orders.isEmpty() ? "" : ", orders:" + orders.toString());
    }

    @Override
    public Pageable next() {
        return new PageRequest(offset + size, size, orders);
    }

    @Override
    public Pageable previous() {
        return new PageRequest(offset - size, size, orders);
    }

    @Override
    public Pageable order(Order order) {
        List<Order> orders = Lists.newArrayList(this.orders);
        orders.add(order);
        return new PageRequest(offset, size, orders);
    }

    @Override
    public Pageable order(OrderSpecifier order) {
        List<Order> orders = Lists.newArrayList(this.orders);
        orders.add(Order.order(order));
        return new PageRequest(offset, size, orders);
    }

    public static int toOffset(int index, int size) {
        return (index - 1) * size;
    }

    public static int toIndex(int offset, int size) {
        return offset / size + 1;
    }
}
