/*
 * Project:  hydroplat-parent
 * Module:   hydroplat-common
 * File:     Order.java
 * Modifier: yangxin
 * Modified: 2014-06-11 10:38
 *
 * Copyright (c) 2014 Mapjs All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent
 * or the registration of a utility model, design or code.
 */

package cn.gtmap.egovplat.core.data;

import com.alibaba.fastjson.annotation.JSONField;
import com.google.common.collect.Lists;
import com.mysema.query.types.OrderSpecifier;
import com.mysema.query.types.Path;

import java.io.Serializable;
import java.util.*;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:yangxin@gtmap.cn">yangxin</a>
 * @version V1.0, 13-7-12
 */
public final class Order implements Serializable {
    private static final long serialVersionUID = 6285805100711077135L;
    public static final Direction DEFAULT_DIRECTION = Direction.DESC;
    private final String field;
    private final Direction direction;

    public Order(String field, Direction direction) {
        this.field = field;
        this.direction = direction;
    }

    public String getField() {
        return field;
    }

    @JSONField(serialize = false)
    public Direction getDirection() {
        return direction;
    }

    public boolean isAsc() {
        return direction.isAsc();
    }

    public static Order order(String field, Direction direction) {
        return new Order(field, direction);
    }

    public static Order order(String field, boolean asc) {
        return order(field, asc ? Direction.ASC : Direction.DESC);
    }

    public static Order order(String field) {
        return order(field, DEFAULT_DIRECTION);
    }

    public static Order order(OrderSpecifier order) {
        return order(((Path) order.getTarget()).getMetadata().getName(), order.isAscending());
    }

    public static Order asc(String field) {
        return order(field, true);
    }

    public static Order desc(String field) {
        return order(field, false);
    }

    public static List<Order> orders(boolean asc, Collection<String> fields) {
        List<Order> orders = Lists.newArrayListWithCapacity(fields.size());
        for (String field : fields) {
            orders.add(Order.order(field, asc));
        }
        return orders;
    }

    public static List<Order> orders(boolean asc, String... fields) {
        if (fields == null) {
            return Collections.emptyList();
        }
        return orders(asc, Arrays.asList(fields));
    }

    public static List<Order> orders(Collection<String> fields) {
        return orders(DEFAULT_DIRECTION.isAsc(), fields);
    }

    public static List<Order> orders(String... fields) {
        return orders(DEFAULT_DIRECTION.isAsc(), fields);
    }

    public static List<Order> orders(OrderSpecifier... orders) {
        if (orders == null) {
            return Collections.emptyList();
        }
        List<Order> list = Lists.newArrayList();
        for (OrderSpecifier os : orders) {
            list.add(Order.order(os));
        }
        return list;
    }

    @Override
    public int hashCode() {
        return field.hashCode() << 1 | direction.ordinal();
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (!(obj instanceof Order)) {
            return false;
        }
        Order other = (Order) obj;
        return direction == other.direction && field.equals(other.field);
    }

    @Override
    public String toString() {
        return field + " " + direction.getValue();
    }

    public static enum Direction {
        ASC("asc"), DESC("desc");

        private String value;

        private Direction(String value) {
            this.value = value;
        }

        public String getValue() {
            return value;
        }

        public boolean isAsc() {
            return this == ASC;
        }
    }
}
