/*
 * Decompiled with CFR 0.152.
 */
package org.janusgraph.diskstorage.es;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterators;
import com.google.common.collect.LinkedListMultimap;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.commons.lang.StringUtils;
import org.elasticsearch.Version;
import org.elasticsearch.action.admin.indices.create.CreateIndexResponse;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexRequest;
import org.elasticsearch.action.admin.indices.exists.indices.IndicesExistsRequest;
import org.elasticsearch.action.admin.indices.exists.indices.IndicesExistsResponse;
import org.elasticsearch.action.admin.indices.mapping.put.PutMappingResponse;
import org.elasticsearch.action.bulk.BulkItemResponse;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkRequestBuilder;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.search.SearchRequestBuilder;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.update.UpdateRequestBuilder;
import org.elasticsearch.client.Client;
import org.elasticsearch.client.transport.TransportClient;
import org.elasticsearch.common.settings.ImmutableSettings;
import org.elasticsearch.common.transport.InetSocketTransportAddress;
import org.elasticsearch.common.transport.TransportAddress;
import org.elasticsearch.common.unit.DistanceUnit;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.index.query.AndFilterBuilder;
import org.elasticsearch.index.query.FilterBuilder;
import org.elasticsearch.index.query.FilterBuilders;
import org.elasticsearch.index.query.OrFilterBuilder;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.indices.IndexMissingException;
import org.elasticsearch.node.Node;
import org.elasticsearch.node.NodeBuilder;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.script.ScriptService;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.SearchHits;
import org.elasticsearch.search.sort.FieldSortBuilder;
import org.elasticsearch.search.sort.SortBuilder;
import org.elasticsearch.search.sort.SortOrder;
import org.janusgraph.core.Cardinality;
import org.janusgraph.core.JanusGraphException;
import org.janusgraph.core.attribute.Cmp;
import org.janusgraph.core.attribute.Geo;
import org.janusgraph.core.attribute.Geoshape;
import org.janusgraph.core.attribute.Text;
import org.janusgraph.core.schema.Mapping;
import org.janusgraph.diskstorage.BackendException;
import org.janusgraph.diskstorage.BaseTransaction;
import org.janusgraph.diskstorage.BaseTransactionConfig;
import org.janusgraph.diskstorage.BaseTransactionConfigurable;
import org.janusgraph.diskstorage.PermanentBackendException;
import org.janusgraph.diskstorage.TemporaryBackendException;
import org.janusgraph.diskstorage.configuration.ConfigNamespace;
import org.janusgraph.diskstorage.configuration.ConfigOption;
import org.janusgraph.diskstorage.configuration.Configuration;
import org.janusgraph.diskstorage.es.ElasticSearchConstants;
import org.janusgraph.diskstorage.es.ElasticSearchSetup;
import org.janusgraph.diskstorage.indexing.IndexEntry;
import org.janusgraph.diskstorage.indexing.IndexFeatures;
import org.janusgraph.diskstorage.indexing.IndexMutation;
import org.janusgraph.diskstorage.indexing.IndexProvider;
import org.janusgraph.diskstorage.indexing.IndexQuery;
import org.janusgraph.diskstorage.indexing.KeyInformation;
import org.janusgraph.diskstorage.indexing.RawQuery;
import org.janusgraph.diskstorage.util.DefaultTransaction;
import org.janusgraph.graphdb.configuration.GraphDatabaseConfiguration;
import org.janusgraph.graphdb.configuration.PreInitializeConfigOptions;
import org.janusgraph.graphdb.database.serialize.AttributeUtil;
import org.janusgraph.graphdb.internal.Order;
import org.janusgraph.graphdb.query.JanusGraphPredicate;
import org.janusgraph.graphdb.query.condition.And;
import org.janusgraph.graphdb.query.condition.Condition;
import org.janusgraph.graphdb.query.condition.Not;
import org.janusgraph.graphdb.query.condition.Or;
import org.janusgraph.graphdb.query.condition.PredicateCondition;
import org.janusgraph.util.system.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@PreInitializeConfigOptions
public class ElasticSearchIndex
implements IndexProvider {
    private static final Logger log = LoggerFactory.getLogger(ElasticSearchIndex.class);
    private static final String TTL_FIELD = "_ttl";
    private static final String STRING_MAPPING_SUFFIX = "__STRING";
    public static final ImmutableList<String> DATA_SUBDIRS = ImmutableList.of((Object)"data", (Object)"work", (Object)"logs");
    public static final ConfigNamespace ELASTICSEARCH_NS = new ConfigNamespace(GraphDatabaseConfiguration.INDEX_NS, "elasticsearch", "Elasticsearch index configuration");
    public static final ConfigOption<Boolean> CLIENT_ONLY = new ConfigOption(ELASTICSEARCH_NS, "client-only", "The Elasticsearch node.client option is set to this boolean value, and the Elasticsearch node.data option is set to the negation of this value.  True creates a thin client which holds no data.  False creates a regular Elasticsearch cluster node that may store data.", ConfigOption.Type.GLOBAL_OFFLINE, (Object)true);
    public static final ConfigOption<String> CLUSTER_NAME = new ConfigOption(ELASTICSEARCH_NS, "cluster-name", "The name of the Elasticsearch cluster.  This should match the \"cluster.name\" setting in the Elasticsearch nodes' configuration.", ConfigOption.Type.GLOBAL_OFFLINE, (Object)"elasticsearch");
    public static final ConfigOption<Boolean> LOCAL_MODE = new ConfigOption(ELASTICSEARCH_NS, "local-mode", "On the legacy config track, this option chooses between starting a TransportClient (false) or a Node with JVM-local transport and local data (true).  On the interface config track, this option is considered by (but optional for) the Node client and ignored by the TransportClient.  See the manual for more information about ES config tracks.", ConfigOption.Type.GLOBAL_OFFLINE, (Object)false);
    public static final ConfigOption<Boolean> CLIENT_SNIFF = new ConfigOption(ELASTICSEARCH_NS, "sniff", "Whether to enable cluster sniffing.  This option only applies to the TransportClient.  Enabling this option makes the TransportClient attempt to discover other cluster nodes besides those in the initial host list provided at startup.", ConfigOption.Type.MASKABLE, (Object)true);
    public static final ConfigOption<String> INTERFACE = new ConfigOption(ELASTICSEARCH_NS, "interface", "Whether to connect to ES using the Node or Transport client (see the \"Talking to Elasticsearch\" section of the ES manual for discussion of the difference).  Setting this option enables the interface config track (see manual for more information about ES config tracks).", ConfigOption.Type.MASKABLE, String.class, (Object)ElasticSearchSetup.TRANSPORT_CLIENT.toString(), ConfigOption.disallowEmpty(String.class));
    public static final ConfigOption<Boolean> IGNORE_CLUSTER_NAME = new ConfigOption(ELASTICSEARCH_NS, "ignore-cluster-name", "Whether to bypass validation of the cluster name of connected nodes.  This option is only used on the interface configuration track (see manual for information about ES config tracks).", ConfigOption.Type.MASKABLE, (Object)true);
    public static final ConfigOption<String> TTL_INTERVAL = new ConfigOption(ELASTICSEARCH_NS, "ttl-interval", "The period of time between runs of ES's bulit-in expired document deleter.  This string will become the value of ES's indices.ttl.interval setting and should be formatted accordingly, e.g. 5s or 60s.", ConfigOption.Type.MASKABLE, (Object)"5s");
    public static final ConfigOption<String> HEALTH_REQUEST_TIMEOUT = new ConfigOption(ELASTICSEARCH_NS, "health-request-timeout", "When JanusGraph initializes its ES backend, JanusGraph waits up to this duration for the ES cluster health to reach at least yellow status.  This string should be formatted as a natural number followed by the lowercase letter \"s\", e.g. 3s or 60s.", ConfigOption.Type.MASKABLE, (Object)"30s");
    public static final ConfigOption<Boolean> LOAD_DEFAULT_NODE_SETTINGS = new ConfigOption(ELASTICSEARCH_NS, "load-default-node-settings", "Whether ES's Node client will internally attempt to load default configuration settings from system properties/process environment variables.  Only meaningful when using the Node client (has no effect with TransportClient).", ConfigOption.Type.MASKABLE, (Object)true);
    public static final ConfigOption<Boolean> USE_EDEPRECATED_IGNORE_UNMAPPED_OPTION = new ConfigOption(ELASTICSEARCH_NS, "use-deprecated-ignore-unmapped-option", "Elasticsearch versions before 1.4.0 supported the \"ignore_unmapped\" sort option. In 1.4.0, it was deprecated by the new \"unmapped_type\" sort option.  This configurationsetting controls which ES option JanusGraph uses: false for the newer \"unmapped_type\",true for the older \"ignore_unmapped\".", ConfigOption.Type.MASKABLE, (Object)false);
    public static final ConfigNamespace ES_EXTRAS_NS = new ConfigNamespace(ELASTICSEARCH_NS, "ext", "Overrides for arbitrary elasticsearch.yaml settings", true);
    public static final ConfigNamespace ES_CREATE_NS = new ConfigNamespace(ELASTICSEARCH_NS, "create", "Settings related to index creation");
    public static final ConfigOption<Long> CREATE_SLEEP = new ConfigOption(ES_CREATE_NS, "sleep", "How long to sleep, in milliseconds, between the successful completion of a (blocking) index creation request and the first use of that index.  This only applies when creating an index in ES, which typically only happens the first time JanusGraph is started on top of ES. If the index JanusGraph is configured to use already exists, then this setting has no effect.", ConfigOption.Type.MASKABLE, (Object)200L);
    public static final ConfigNamespace ES_CREATE_EXTRAS_NS = new ConfigNamespace(ES_CREATE_NS, "ext", "Overrides for arbitrary settings applied at index creation", true);
    private static final IndexFeatures ES_FEATURES = new IndexFeatures.Builder().supportsDocumentTTL().setDefaultStringMapping(Mapping.TEXT).supportedStringMappings(new Mapping[]{Mapping.TEXT, Mapping.TEXTSTRING, Mapping.STRING}).setWildcardField("_all").supportsCardinality(Cardinality.SINGLE).supportsCardinality(Cardinality.LIST).supportsCardinality(Cardinality.SET).supportsNanoseconds().build();
    public static final int HOST_PORT_DEFAULT = 9300;
    private final Node node;
    private final Client client;
    private final String indexName;
    private final int maxResultsSize;
    private final boolean useDeprecatedIgnoreUnmapped;

    public ElasticSearchIndex(Configuration config) {
        this.indexName = (String)config.get(GraphDatabaseConfiguration.INDEX_NAME, new String[0]);
        this.useDeprecatedIgnoreUnmapped = (Boolean)config.get(USE_EDEPRECATED_IGNORE_UNMAPPED_OPTION, new String[0]);
        this.checkExpectedClientVersion();
        ElasticSearchSetup.Connection c = !config.has(INTERFACE, new String[0]) ? this.legacyConfiguration(config) : this.interfaceConfiguration(config);
        this.node = c.getNode();
        this.client = c.getClient();
        this.maxResultsSize = (Integer)config.get(GraphDatabaseConfiguration.INDEX_MAX_RESULT_SET_SIZE, new String[0]);
        log.debug("Configured ES query result set max size to {}", (Object)this.maxResultsSize);
        this.client.admin().cluster().prepareHealth(new String[0]).setTimeout((String)config.get(HEALTH_REQUEST_TIMEOUT, new String[0])).setWaitForYellowStatus().execute().actionGet();
        this.checkForOrCreateIndex(config);
    }

    private void checkForOrCreateIndex(Configuration config) {
        Preconditions.checkState((null != this.client ? 1 : 0) != 0);
        IndicesExistsResponse response = (IndicesExistsResponse)this.client.admin().indices().exists(new IndicesExistsRequest(new String[]{this.indexName})).actionGet();
        if (!response.isExists()) {
            ImmutableSettings.Builder settings = ImmutableSettings.settingsBuilder();
            ElasticSearchSetup.applySettingsFromJanusGraphConf(settings, config, ES_CREATE_EXTRAS_NS);
            CreateIndexResponse create = (CreateIndexResponse)this.client.admin().indices().prepareCreate(this.indexName).setSettings(settings.build()).execute().actionGet();
            try {
                long sleep = (Long)config.get(CREATE_SLEEP, new String[0]);
                log.debug("Sleeping {} ms after {} index creation returned from actionGet()", (Object)sleep, (Object)this.indexName);
                Thread.sleep(sleep);
            }
            catch (InterruptedException e) {
                throw new JanusGraphException("Interrupted while waiting for index to settle in", (Throwable)e);
            }
            if (!create.isAcknowledged()) {
                throw new IllegalArgumentException("Could not create index: " + this.indexName);
            }
        }
    }

    private ElasticSearchSetup.Connection interfaceConfiguration(Configuration config) {
        ElasticSearchSetup clientMode = (ElasticSearchSetup)ConfigOption.getEnumValue((String)((String)config.get(INTERFACE, new String[0])), ElasticSearchSetup.class);
        try {
            return clientMode.connect(config);
        }
        catch (IOException e) {
            throw new JanusGraphException((Throwable)e);
        }
    }

    private ElasticSearchSetup.Connection legacyConfiguration(Configuration config) {
        TransportClient client;
        Node node;
        block14: {
            block11: {
                NodeBuilder builder;
                boolean local;
                boolean clientOnly;
                block13: {
                    block12: {
                        if (!((Boolean)config.get(LOCAL_MODE, new String[0])).booleanValue()) break block11;
                        log.debug("Configuring ES for JVM local transport");
                        clientOnly = (Boolean)config.get(CLIENT_ONLY, new String[0]);
                        local = (Boolean)config.get(LOCAL_MODE, new String[0]);
                        builder = NodeBuilder.nodeBuilder();
                        Preconditions.checkArgument((config.has(GraphDatabaseConfiguration.INDEX_CONF_FILE, new String[0]) || config.has(GraphDatabaseConfiguration.INDEX_DIRECTORY, new String[0]) ? 1 : 0) != 0, (Object)"Must either configure configuration file or base directory");
                        if (!config.has(GraphDatabaseConfiguration.INDEX_CONF_FILE, new String[0])) break block12;
                        String configFile = (String)config.get(GraphDatabaseConfiguration.INDEX_CONF_FILE, new String[0]);
                        ImmutableSettings.Builder sb = ImmutableSettings.settingsBuilder();
                        log.debug("Configuring ES from YML file [{}]", (Object)configFile);
                        FileInputStream fis = null;
                        try {
                            fis = new FileInputStream(configFile);
                            sb.loadFromStream(configFile, (InputStream)fis);
                            builder.settings(sb.build());
                        }
                        catch (FileNotFoundException e) {
                            try {
                                throw new JanusGraphException((Throwable)e);
                            }
                            catch (Throwable throwable) {
                                IOUtils.closeQuietly(fis);
                                throw throwable;
                            }
                        }
                        IOUtils.closeQuietly((Closeable)fis);
                        break block13;
                    }
                    String dataDirectory = (String)config.get(GraphDatabaseConfiguration.INDEX_DIRECTORY, new String[0]);
                    log.debug("Configuring ES with data directory [{}]", (Object)dataDirectory);
                    File f = new File(dataDirectory);
                    if (!f.exists()) {
                        f.mkdirs();
                    }
                    ImmutableSettings.Builder b = ImmutableSettings.settingsBuilder();
                    for (String sub : DATA_SUBDIRS) {
                        String subdir = dataDirectory + File.separator + sub;
                        f = new File(subdir);
                        if (!f.exists()) {
                            f.mkdirs();
                        }
                        b.put("path." + sub, subdir);
                    }
                    b.put("script.disable_dynamic", false);
                    b.put("indices.ttl.interval", "5s");
                    builder.settings(b.build());
                    String clustername = (String)config.get(CLUSTER_NAME, new String[0]);
                    Preconditions.checkArgument((boolean)StringUtils.isNotBlank((String)clustername), (String)"Invalid cluster name: %s", (Object[])new Object[]{clustername});
                    builder.clusterName(clustername);
                }
                node = builder.client(clientOnly).data(!clientOnly).local(local).node();
                client = node.client();
                break block14;
            }
            log.debug("Configuring ES for network transport");
            ImmutableSettings.Builder settings = ImmutableSettings.settingsBuilder();
            if (config.has(CLUSTER_NAME, new String[0])) {
                String clustername = (String)config.get(CLUSTER_NAME, new String[0]);
                Preconditions.checkArgument((boolean)StringUtils.isNotBlank((String)clustername), (String)"Invalid cluster name: %s", (Object[])new Object[]{clustername});
                settings.put("cluster.name", clustername);
            } else {
                settings.put("client.transport.ignore_cluster_name", true);
            }
            log.debug("Transport sniffing enabled: {}", config.get(CLIENT_SNIFF, new String[0]));
            settings.put("client.transport.sniff", ((Boolean)config.get(CLIENT_SNIFF, new String[0])).booleanValue());
            settings.put("script.disable_dynamic", false);
            TransportClient tc = new TransportClient(settings.build());
            int defaultPort = config.has(GraphDatabaseConfiguration.INDEX_PORT, new String[0]) ? (Integer)config.get(GraphDatabaseConfiguration.INDEX_PORT, new String[0]) : 9300;
            for (String host : (String[])config.get(GraphDatabaseConfiguration.INDEX_HOSTS, new String[0])) {
                String[] hostparts = host.split(":");
                String hostname = hostparts[0];
                int hostport = defaultPort;
                if (hostparts.length == 2) {
                    hostport = Integer.parseInt(hostparts[1]);
                }
                log.info("Configured remote host: {} : {}", (Object)hostname, (Object)hostport);
                tc.addTransportAddress((TransportAddress)new InetSocketTransportAddress(hostname, hostport));
            }
            client = tc;
            node = null;
        }
        return new ElasticSearchSetup.Connection(node, (Client)client);
    }

    private BackendException convert(Exception esException) {
        if (esException instanceof InterruptedException) {
            return new TemporaryBackendException("Interrupted while waiting for response", (Throwable)esException);
        }
        return new PermanentBackendException("Unknown exception while executing index operation", (Throwable)esException);
    }

    private static String getDualMappingName(String key) {
        return key + STRING_MAPPING_SUFFIX;
    }

    public void register(String store, String key, KeyInformation information, BaseTransaction tx) throws BackendException {
        XContentBuilder mapping;
        Class dataType = information.getDataType();
        Mapping map = Mapping.getMapping((KeyInformation)information);
        Preconditions.checkArgument((map == Mapping.DEFAULT || AttributeUtil.isString((Class)dataType) ? 1 : 0) != 0, (String)"Specified illegal mapping [%s] for data type [%s]", (Object[])new Object[]{map, dataType});
        try {
            block32: {
                block31: {
                    mapping = XContentFactory.jsonBuilder().startObject().startObject(store).field(TTL_FIELD, (Map)new HashMap<String, Object>(){
                        {
                            this.put("enabled", true);
                        }
                    }).startObject("properties").startObject(key);
                    if (!AttributeUtil.isString((Class)dataType)) break block31;
                    if (map == Mapping.DEFAULT) {
                        map = Mapping.TEXT;
                    }
                    log.debug("Registering string type for {} with mapping {}", (Object)key, (Object)map);
                    mapping.field("type", "string");
                    switch (map) {
                        case STRING: {
                            mapping.field("index", "not_analyzed");
                            break block32;
                        }
                        case TEXT: {
                            break block32;
                        }
                        case TEXTSTRING: {
                            mapping.endObject();
                            mapping.startObject(ElasticSearchIndex.getDualMappingName(key));
                            mapping.field("type", "string");
                            mapping.field("index", "not_analyzed");
                            break block32;
                        }
                        default: {
                            throw new AssertionError((Object)("Unexpected mapping: " + map));
                        }
                    }
                }
                if (dataType == Float.class) {
                    log.debug("Registering float type for {}", (Object)key);
                    mapping.field("type", "float");
                } else if (dataType == Double.class) {
                    log.debug("Registering double type for {}", (Object)key);
                    mapping.field("type", "double");
                } else if (dataType == Byte.class) {
                    log.debug("Registering byte type for {}", (Object)key);
                    mapping.field("type", "byte");
                } else if (dataType == Short.class) {
                    log.debug("Registering short type for {}", (Object)key);
                    mapping.field("type", "short");
                } else if (dataType == Integer.class) {
                    log.debug("Registering integer type for {}", (Object)key);
                    mapping.field("type", "integer");
                } else if (dataType == Long.class) {
                    log.debug("Registering long type for {}", (Object)key);
                    mapping.field("type", "long");
                } else if (dataType == Boolean.class) {
                    log.debug("Registering boolean type for {}", (Object)key);
                    mapping.field("type", "boolean");
                } else if (dataType == Geoshape.class) {
                    log.debug("Registering geo_point type for {}", (Object)key);
                    mapping.field("type", "geo_point");
                } else if (dataType == Date.class || dataType == Instant.class) {
                    log.debug("Registering date type for {}", (Object)key);
                    mapping.field("type", "date");
                } else if (dataType == Boolean.class) {
                    log.debug("Registering boolean type for {}", (Object)key);
                    mapping.field("type", "boolean");
                } else if (dataType == UUID.class) {
                    log.debug("Registering uuid type for {}", (Object)key);
                    mapping.field("type", "string");
                    mapping.field("index", "not_analyzed");
                }
            }
            mapping.endObject().endObject().endObject().endObject();
        }
        catch (IOException e) {
            throw new PermanentBackendException("Could not render json for put mapping request", (Throwable)e);
        }
        try {
            PutMappingResponse e = (PutMappingResponse)this.client.admin().indices().preparePutMapping(new String[]{this.indexName}).setIgnoreConflicts(false).setType(store).setSource(mapping).execute().actionGet();
        }
        catch (Exception e) {
            throw this.convert(e);
        }
    }

    private static Mapping getStringMapping(KeyInformation information) {
        assert (AttributeUtil.isString((Class)information.getDataType()));
        Mapping map = Mapping.getMapping((KeyInformation)information);
        if (map == Mapping.DEFAULT) {
            map = Mapping.TEXT;
        }
        return map;
    }

    private static boolean hasDualStringMapping(KeyInformation information) {
        return AttributeUtil.isString((Class)information.getDataType()) && ElasticSearchIndex.getStringMapping(information) == Mapping.TEXTSTRING;
    }

    public XContentBuilder getNewDocument(List<IndexEntry> additions, KeyInformation.StoreRetriever informations, int ttl) throws BackendException {
        Preconditions.checkArgument((ttl >= 0 ? 1 : 0) != 0);
        try {
            XContentBuilder builder = XContentFactory.jsonBuilder().startObject();
            LinkedListMultimap uniq = LinkedListMultimap.create();
            for (IndexEntry indexEntry : additions) {
                uniq.put((Object)indexEntry.field, (Object)indexEntry);
            }
            for (Map.Entry entry : uniq.asMap().entrySet()) {
                KeyInformation keyInformation = informations.get((String)entry.getKey());
                Object[] value = null;
                switch (keyInformation.getCardinality()) {
                    case SINGLE: {
                        value = ElasticSearchIndex.convertToEsType(((IndexEntry)Iterators.getLast(((Collection)entry.getValue()).iterator())).value);
                        break;
                    }
                    case SET: 
                    case LIST: {
                        value = ((Collection)entry.getValue()).stream().map(v -> ElasticSearchIndex.convertToEsType(v.value)).collect(Collectors.toList()).toArray();
                    }
                }
                builder.field((String)entry.getKey(), value);
                if (!ElasticSearchIndex.hasDualStringMapping(informations.get((String)entry.getKey())) || keyInformation.getDataType() != String.class) continue;
                builder.field(ElasticSearchIndex.getDualMappingName((String)entry.getKey()), (Object)value);
            }
            if (ttl > 0) {
                builder.field(TTL_FIELD, TimeUnit.MILLISECONDS.convert(ttl, TimeUnit.SECONDS));
            }
            builder.endObject();
            return builder;
        }
        catch (IOException e) {
            throw new PermanentBackendException("Could not write json");
        }
    }

    private static Object convertToEsType(Object value) {
        if (value instanceof Number) {
            if (AttributeUtil.isWholeNumber((Number)((Number)value))) {
                return ((Number)value).longValue();
            }
            return ((Number)value).doubleValue();
        }
        if (AttributeUtil.isString((Object)value)) {
            return value;
        }
        if (value instanceof Geoshape) {
            Geoshape shape = (Geoshape)value;
            if (shape.getType() == Geoshape.Type.POINT) {
                Geoshape.Point p = shape.getPoint();
                return new double[]{p.getLongitude(), p.getLatitude()};
            }
            throw new UnsupportedOperationException("Geo type is not supported: " + shape.getType());
        }
        if (value instanceof Date || value instanceof Instant) {
            return value;
        }
        if (value instanceof Boolean) {
            return value;
        }
        if (value instanceof UUID) {
            return value.toString();
        }
        throw new IllegalArgumentException("Unsupported type: " + value.getClass() + " (value: " + value + ")");
    }

    public void mutate(Map<String, Map<String, IndexMutation>> mutations, KeyInformation.IndexRetriever informations, BaseTransaction tx) throws BackendException {
        BulkRequestBuilder brb = this.client.prepareBulk();
        int bulkrequests = 0;
        try {
            BulkResponse bulkItemResponses;
            for (Map.Entry<String, Map<String, IndexMutation>> stores : mutations.entrySet()) {
                String storename = stores.getKey();
                for (Map.Entry<String, IndexMutation> entry : stores.getValue().entrySet()) {
                    String docid = entry.getKey();
                    IndexMutation mutation = entry.getValue();
                    assert (mutation.isConsolidated());
                    Preconditions.checkArgument((!mutation.isNew() || !mutation.isDeleted() ? 1 : 0) != 0);
                    Preconditions.checkArgument((!mutation.isNew() || !mutation.hasDeletions() ? 1 : 0) != 0);
                    Preconditions.checkArgument((!mutation.isDeleted() || !mutation.hasAdditions() ? 1 : 0) != 0);
                    if (mutation.hasDeletions()) {
                        if (mutation.isDeleted()) {
                            log.trace("Deleting entire document {}", (Object)docid);
                            brb.add(new DeleteRequest(this.indexName, storename, docid));
                        } else {
                            String script = this.getDeletionScript(informations, storename, mutation);
                            brb.add(this.client.prepareUpdate(this.indexName, storename, docid).setScript(script, ScriptService.ScriptType.INLINE));
                            log.trace("Adding script {}", (Object)script);
                        }
                        ++bulkrequests;
                    }
                    if (!mutation.hasAdditions()) continue;
                    int ttl = mutation.determineTTL();
                    if (mutation.isNew()) {
                        log.trace("Adding entire document {}", (Object)docid);
                        brb.add(new IndexRequest(this.indexName, storename, docid).source(this.getNewDocument(mutation.getAdditions(), informations.get(storename), ttl)));
                    } else {
                        Preconditions.checkArgument((ttl == 0 ? 1 : 0) != 0, (String)"Elasticsearch only supports TTL on new documents [%s]", (Object[])new Object[]{docid});
                        boolean needUpsert = !mutation.hasDeletions();
                        String script = this.getAdditionScript(informations, storename, mutation);
                        UpdateRequestBuilder update = this.client.prepareUpdate(this.indexName, storename, docid).setScript(script, ScriptService.ScriptType.INLINE);
                        if (needUpsert) {
                            XContentBuilder doc = this.getNewDocument(mutation.getAdditions(), informations.get(storename), ttl);
                            update.setUpsert(doc);
                        }
                        brb.add(update);
                        log.trace("Adding script {}", (Object)script);
                    }
                    ++bulkrequests;
                }
            }
            if (bulkrequests > 0 && (bulkItemResponses = (BulkResponse)brb.execute().actionGet()).hasFailures()) {
                boolean actualFailure = false;
                for (BulkItemResponse response : bulkItemResponses.getItems()) {
                    if (!response.isFailed() || response.getFailure().getStatus() == RestStatus.NOT_FOUND) continue;
                    log.error("Failed to execute ES query {}", (Object)response.getFailureMessage());
                    actualFailure = true;
                }
                if (actualFailure) {
                    throw new Exception(bulkItemResponses.buildFailureMessage());
                }
            }
        }
        catch (Exception e) {
            log.error("Failed to execute ES query {}", (Object)((BulkRequest)brb.request()).timeout(), (Object)e);
            throw this.convert(e);
        }
    }

    private String getDeletionScript(KeyInformation.IndexRetriever informations, String storename, IndexMutation mutation) throws PermanentBackendException {
        StringBuilder script = new StringBuilder();
        for (IndexEntry deletion : mutation.getDeletions()) {
            KeyInformation keyInformation = informations.get(storename).get(deletion.field);
            switch (keyInformation.getCardinality()) {
                case SINGLE: {
                    script.append("ctx._source.remove(\"" + deletion.field + "\");");
                    if (!ElasticSearchIndex.hasDualStringMapping(informations.get(storename, deletion.field))) break;
                    script.append("ctx._source.remove(\"" + ElasticSearchIndex.getDualMappingName(deletion.field) + "\");");
                    break;
                }
                case SET: 
                case LIST: {
                    String jsValue = ElasticSearchIndex.convertToJsType(deletion.value);
                    script.append("def index = ctx._source[\"" + deletion.field + "\"].indexOf(" + jsValue + "); ctx._source[\"" + deletion.field + "\"].remove(index);");
                    if (!ElasticSearchIndex.hasDualStringMapping(informations.get(storename, deletion.field))) break;
                    script.append("def index = ctx._source[\"" + ElasticSearchIndex.getDualMappingName(deletion.field) + "\"].indexOf(" + jsValue + "); ctx._source[\"" + ElasticSearchIndex.getDualMappingName(deletion.field) + "\"].remove(index);");
                }
            }
        }
        return script.toString();
    }

    private String getAdditionScript(KeyInformation.IndexRetriever informations, String storename, IndexMutation mutation) throws PermanentBackendException {
        StringBuilder script = new StringBuilder();
        for (IndexEntry e : mutation.getAdditions()) {
            KeyInformation keyInformation = informations.get(storename).get(e.field);
            switch (keyInformation.getCardinality()) {
                case SINGLE: {
                    script.append("ctx._source[\"" + e.field + "\"] = " + ElasticSearchIndex.convertToJsType(e.value) + ";");
                    if (!ElasticSearchIndex.hasDualStringMapping(keyInformation)) break;
                    script.append("ctx._source[\"" + ElasticSearchIndex.getDualMappingName(e.field) + "\"] = " + ElasticSearchIndex.convertToJsType(e.value) + ";");
                    break;
                }
                case SET: 
                case LIST: {
                    script.append("if(ctx._source[\"" + e.field + "\"] == null) {ctx._source[\"" + e.field + "\"] = []};");
                    script.append("ctx._source[\"" + e.field + "\"].add(" + ElasticSearchIndex.convertToJsType(e.value) + ");");
                    if (!ElasticSearchIndex.hasDualStringMapping(keyInformation)) break;
                    script.append("if(ctx._source[\"" + ElasticSearchIndex.getDualMappingName(e.field) + "\"] == null) {ctx._source[\"" + e.field + "\"] = []};");
                    script.append("ctx._source[\"" + ElasticSearchIndex.getDualMappingName(e.field) + "\"].add(" + ElasticSearchIndex.convertToJsType(e.value) + ");");
                }
            }
        }
        return script.toString();
    }

    private static String convertToJsType(Object value) throws PermanentBackendException {
        try {
            XContentBuilder builder = XContentFactory.jsonBuilder().startObject();
            builder.field("value", ElasticSearchIndex.convertToEsType(value));
            String s = builder.string();
            int prefixLength = "{\"value\":".length();
            int suffixLength = "}".length();
            String result = s.substring(prefixLength, s.length() - suffixLength);
            result = result.replace("$", "\\$");
            return result;
        }
        catch (IOException e) {
            throw new PermanentBackendException("Could not write json");
        }
    }

    public void restore(Map<String, Map<String, List<IndexEntry>>> documents, KeyInformation.IndexRetriever informations, BaseTransaction tx) throws BackendException {
        BulkRequestBuilder bulk = this.client.prepareBulk();
        int requests = 0;
        try {
            for (Map.Entry<String, Map<String, List<IndexEntry>>> stores : documents.entrySet()) {
                String store = stores.getKey();
                for (Map.Entry<String, List<IndexEntry>> entry : stores.getValue().entrySet()) {
                    String docID = entry.getKey();
                    List<IndexEntry> content = entry.getValue();
                    if (content == null || content.size() == 0) {
                        if (log.isTraceEnabled()) {
                            log.trace("Deleting entire document {}", (Object)docID);
                        }
                        bulk.add(new DeleteRequest(this.indexName, store, docID));
                        ++requests;
                        continue;
                    }
                    if (log.isTraceEnabled()) {
                        log.trace("Adding entire document {}", (Object)docID);
                    }
                    bulk.add(new IndexRequest(this.indexName, store, docID).source(this.getNewDocument(content, informations.get(store), IndexMutation.determineTTL(content))));
                    ++requests;
                }
            }
            if (requests > 0) {
                bulk.execute().actionGet();
            }
        }
        catch (Exception e) {
            throw this.convert(e);
        }
    }

    public FilterBuilder getFilter(Condition<?> condition, KeyInformation.StoreRetriever informations) {
        if (condition instanceof PredicateCondition) {
            PredicateCondition atom = (PredicateCondition)condition;
            Object value = atom.getValue();
            String key = (String)atom.getKey();
            JanusGraphPredicate janusgraphPredicate = atom.getPredicate();
            if (value instanceof Number) {
                Preconditions.checkArgument((boolean)(janusgraphPredicate instanceof Cmp), (Object)("Relation not supported on numeric types: " + janusgraphPredicate));
                Cmp numRel = (Cmp)janusgraphPredicate;
                Preconditions.checkArgument((boolean)(value instanceof Number));
                switch (numRel) {
                    case EQUAL: {
                        return FilterBuilders.inFilter((String)key, (Object[])new Object[]{value});
                    }
                    case NOT_EQUAL: {
                        return FilterBuilders.notFilter((FilterBuilder)FilterBuilders.inFilter((String)key, (Object[])new Object[]{value}));
                    }
                    case LESS_THAN: {
                        return FilterBuilders.rangeFilter((String)key).lt(value);
                    }
                    case LESS_THAN_EQUAL: {
                        return FilterBuilders.rangeFilter((String)key).lte(value);
                    }
                    case GREATER_THAN: {
                        return FilterBuilders.rangeFilter((String)key).gt(value);
                    }
                    case GREATER_THAN_EQUAL: {
                        return FilterBuilders.rangeFilter((String)key).gte(value);
                    }
                }
                throw new IllegalArgumentException("Unexpected relation: " + numRel);
            }
            if (value instanceof String) {
                Mapping map = ElasticSearchIndex.getStringMapping(informations.get(key));
                String fieldName = key;
                if (map == Mapping.TEXT && !janusgraphPredicate.toString().startsWith("CONTAINS")) {
                    throw new IllegalArgumentException("Text mapped string values only support CONTAINS queries and not: " + janusgraphPredicate);
                }
                if (map == Mapping.STRING && janusgraphPredicate.toString().startsWith("CONTAINS")) {
                    throw new IllegalArgumentException("String mapped string values do not support CONTAINS queries: " + janusgraphPredicate);
                }
                if (map == Mapping.TEXTSTRING && !janusgraphPredicate.toString().startsWith("CONTAINS")) {
                    fieldName = ElasticSearchIndex.getDualMappingName(key);
                }
                if (janusgraphPredicate == Text.CONTAINS) {
                    value = ((String)value).toLowerCase();
                    AndFilterBuilder b = FilterBuilders.andFilter((FilterBuilder[])new FilterBuilder[0]);
                    for (String term : Text.tokenize((String)((String)value))) {
                        b.add((FilterBuilder)FilterBuilders.termFilter((String)fieldName, (String)term));
                    }
                    return b;
                }
                if (janusgraphPredicate == Text.CONTAINS_PREFIX) {
                    value = ((String)value).toLowerCase();
                    return FilterBuilders.prefixFilter((String)fieldName, (String)((String)value));
                }
                if (janusgraphPredicate == Text.CONTAINS_REGEX) {
                    value = ((String)value).toLowerCase();
                    return FilterBuilders.regexpFilter((String)fieldName, (String)((String)value));
                }
                if (janusgraphPredicate == Text.PREFIX) {
                    return FilterBuilders.prefixFilter((String)fieldName, (String)((String)value));
                }
                if (janusgraphPredicate == Text.REGEX) {
                    return FilterBuilders.regexpFilter((String)fieldName, (String)((String)value));
                }
                if (janusgraphPredicate == Cmp.EQUAL) {
                    return FilterBuilders.termFilter((String)fieldName, (String)((String)value));
                }
                if (janusgraphPredicate == Cmp.NOT_EQUAL) {
                    return FilterBuilders.notFilter((FilterBuilder)FilterBuilders.termFilter((String)fieldName, (String)((String)value)));
                }
                throw new IllegalArgumentException("Predicate is not supported for string value: " + janusgraphPredicate);
            }
            if (value instanceof Geoshape) {
                Preconditions.checkArgument((janusgraphPredicate == Geo.WITHIN ? 1 : 0) != 0, (Object)("Relation is not supported for geo value: " + janusgraphPredicate));
                Geoshape shape = (Geoshape)value;
                if (shape.getType() == Geoshape.Type.CIRCLE) {
                    Geoshape.Point center = shape.getPoint();
                    return FilterBuilders.geoDistanceFilter((String)key).lat((double)center.getLatitude()).lon((double)center.getLongitude()).distance((double)shape.getRadius(), DistanceUnit.KILOMETERS);
                }
                if (shape.getType() == Geoshape.Type.BOX) {
                    Geoshape.Point southwest = shape.getPoint(0);
                    Geoshape.Point northeast = shape.getPoint(1);
                    return FilterBuilders.geoBoundingBoxFilter((String)key).bottomRight((double)southwest.getLatitude(), (double)northeast.getLongitude()).topLeft((double)northeast.getLatitude(), (double)southwest.getLongitude());
                }
                throw new IllegalArgumentException("Unsupported or invalid search shape type: " + shape.getType());
            }
            if (value instanceof Date || value instanceof Instant) {
                Preconditions.checkArgument((boolean)(janusgraphPredicate instanceof Cmp), (Object)("Relation not supported on date types: " + janusgraphPredicate));
                Cmp numRel = (Cmp)janusgraphPredicate;
                switch (numRel) {
                    case EQUAL: {
                        return FilterBuilders.inFilter((String)key, (Object[])new Object[]{value});
                    }
                    case NOT_EQUAL: {
                        return FilterBuilders.notFilter((FilterBuilder)FilterBuilders.inFilter((String)key, (Object[])new Object[]{value}));
                    }
                    case LESS_THAN: {
                        return FilterBuilders.rangeFilter((String)key).lt(value);
                    }
                    case LESS_THAN_EQUAL: {
                        return FilterBuilders.rangeFilter((String)key).lte(value);
                    }
                    case GREATER_THAN: {
                        return FilterBuilders.rangeFilter((String)key).gt(value);
                    }
                    case GREATER_THAN_EQUAL: {
                        return FilterBuilders.rangeFilter((String)key).gte(value);
                    }
                }
                throw new IllegalArgumentException("Unexpected relation: " + numRel);
            }
            if (value instanceof Boolean) {
                Cmp numRel = (Cmp)janusgraphPredicate;
                switch (numRel) {
                    case EQUAL: {
                        return FilterBuilders.inFilter((String)key, (Object[])new Object[]{value});
                    }
                    case NOT_EQUAL: {
                        return FilterBuilders.notFilter((FilterBuilder)FilterBuilders.inFilter((String)key, (Object[])new Object[]{value}));
                    }
                }
                throw new IllegalArgumentException("Boolean types only support EQUAL or NOT_EQUAL");
            }
            if (value instanceof UUID) {
                if (janusgraphPredicate == Cmp.EQUAL) {
                    return FilterBuilders.termFilter((String)key, (Object)value);
                }
                if (janusgraphPredicate == Cmp.NOT_EQUAL) {
                    return FilterBuilders.notFilter((FilterBuilder)FilterBuilders.termFilter((String)key, (Object)value));
                }
                throw new IllegalArgumentException("Only equal or not equal is supported for UUIDs: " + janusgraphPredicate);
            }
            throw new IllegalArgumentException("Unsupported type: " + value);
        }
        if (condition instanceof Not) {
            return FilterBuilders.notFilter((FilterBuilder)this.getFilter(((Not)condition).getChild(), informations));
        }
        if (condition instanceof And) {
            AndFilterBuilder b = FilterBuilders.andFilter((FilterBuilder[])new FilterBuilder[0]);
            for (Condition c : condition.getChildren()) {
                b.add(this.getFilter(c, informations));
            }
            return b;
        }
        if (condition instanceof Or) {
            OrFilterBuilder b = FilterBuilders.orFilter((FilterBuilder[])new FilterBuilder[0]);
            for (Condition c : condition.getChildren()) {
                b.add(this.getFilter(c, informations));
            }
            return b;
        }
        throw new IllegalArgumentException("Invalid condition: " + condition);
    }

    public List<String> query(IndexQuery query, KeyInformation.IndexRetriever informations, BaseTransaction tx) throws BackendException {
        SearchRequestBuilder srb = this.client.prepareSearch(new String[]{this.indexName});
        srb.setTypes(new String[]{query.getStore()});
        srb.setQuery((QueryBuilder)QueryBuilders.matchAllQuery());
        srb.setPostFilter(this.getFilter(query.getCondition(), informations.get(query.getStore())));
        if (!query.getOrder().isEmpty()) {
            List orders = query.getOrder();
            for (int i = 0; i < orders.size(); ++i) {
                IndexQuery.OrderEntry orderEntry = (IndexQuery.OrderEntry)orders.get(i);
                FieldSortBuilder fsb = new FieldSortBuilder(((IndexQuery.OrderEntry)orders.get(i)).getKey()).order(orderEntry.getOrder() == Order.ASC ? SortOrder.ASC : SortOrder.DESC);
                if (this.useDeprecatedIgnoreUnmapped) {
                    fsb.ignoreUnmapped(true);
                } else {
                    Class datatype = orderEntry.getDatatype();
                    fsb.unmappedType(this.convertToEsDataType(datatype));
                }
                srb.addSort((SortBuilder)fsb);
            }
        }
        srb.setFrom(0);
        if (query.hasLimit()) {
            srb.setSize(query.getLimit());
        } else {
            srb.setSize(this.maxResultsSize);
        }
        srb.setNoFields();
        SearchResponse response = (SearchResponse)srb.execute().actionGet();
        log.debug("Executed query [{}] in {} ms", (Object)query.getCondition(), (Object)response.getTookInMillis());
        SearchHits hits = response.getHits();
        if (!query.hasLimit() && hits.totalHits() >= (long)this.maxResultsSize) {
            log.warn("Query result set truncated to first [{}] elements for query: {}", (Object)this.maxResultsSize, (Object)query);
        }
        ArrayList<String> result = new ArrayList<String>(hits.hits().length);
        for (SearchHit hit : hits) {
            result.add(hit.id());
        }
        return result;
    }

    private String convertToEsDataType(Class<?> datatype) {
        if (String.class.isAssignableFrom(datatype)) {
            return "string";
        }
        if (Integer.class.isAssignableFrom(datatype)) {
            return "integer";
        }
        if (Long.class.isAssignableFrom(datatype)) {
            return "long";
        }
        if (Float.class.isAssignableFrom(datatype)) {
            return "float";
        }
        if (Double.class.isAssignableFrom(datatype)) {
            return "double";
        }
        if (Boolean.class.isAssignableFrom(datatype)) {
            return "boolean";
        }
        if (Date.class.isAssignableFrom(datatype)) {
            return "date";
        }
        if (Instant.class.isAssignableFrom(datatype)) {
            return "date";
        }
        if (Geoshape.class.isAssignableFrom(datatype)) {
            return "geo_point";
        }
        return null;
    }

    public Iterable<RawQuery.Result<String>> query(RawQuery query, KeyInformation.IndexRetriever informations, BaseTransaction tx) throws BackendException {
        SearchRequestBuilder srb = this.client.prepareSearch(new String[]{this.indexName});
        srb.setTypes(new String[]{query.getStore()});
        srb.setQuery((QueryBuilder)QueryBuilders.queryStringQuery((String)query.getQuery()));
        srb.setFrom(query.getOffset());
        if (query.hasLimit()) {
            srb.setSize(query.getLimit());
        } else {
            srb.setSize(this.maxResultsSize);
        }
        srb.setNoFields();
        SearchResponse response = (SearchResponse)srb.execute().actionGet();
        log.debug("Executed query [{}] in {} ms", (Object)query.getQuery(), (Object)response.getTookInMillis());
        SearchHits hits = response.getHits();
        if (!query.hasLimit() && hits.totalHits() >= (long)this.maxResultsSize) {
            log.warn("Query result set truncated to first [{}] elements for query: {}", (Object)this.maxResultsSize, (Object)query);
        }
        ArrayList<RawQuery.Result<String>> result = new ArrayList<RawQuery.Result<String>>(hits.hits().length);
        for (SearchHit hit : hits) {
            result.add((RawQuery.Result<String>)new RawQuery.Result((Object)hit.id(), (double)hit.getScore()));
        }
        return result;
    }

    public boolean supports(KeyInformation information, JanusGraphPredicate janusgraphPredicate) {
        Class dataType = information.getDataType();
        Mapping mapping = Mapping.getMapping((KeyInformation)information);
        if (mapping != Mapping.DEFAULT && !AttributeUtil.isString((Class)dataType)) {
            return false;
        }
        if (Number.class.isAssignableFrom(dataType)) {
            if (janusgraphPredicate instanceof Cmp) {
                return true;
            }
        } else {
            if (dataType == Geoshape.class) {
                return janusgraphPredicate == Geo.WITHIN;
            }
            if (AttributeUtil.isString((Class)dataType)) {
                switch (mapping) {
                    case TEXT: 
                    case DEFAULT: {
                        return janusgraphPredicate == Text.CONTAINS || janusgraphPredicate == Text.CONTAINS_PREFIX || janusgraphPredicate == Text.CONTAINS_REGEX;
                    }
                    case STRING: {
                        return janusgraphPredicate == Cmp.EQUAL || janusgraphPredicate == Cmp.NOT_EQUAL || janusgraphPredicate == Text.REGEX || janusgraphPredicate == Text.PREFIX;
                    }
                    case TEXTSTRING: {
                        return janusgraphPredicate instanceof Text || janusgraphPredicate == Cmp.EQUAL || janusgraphPredicate == Cmp.NOT_EQUAL;
                    }
                }
            } else if (dataType == Date.class || dataType == Instant.class) {
                if (janusgraphPredicate instanceof Cmp) {
                    return true;
                }
            } else {
                if (dataType == Boolean.class) {
                    return janusgraphPredicate == Cmp.EQUAL || janusgraphPredicate == Cmp.NOT_EQUAL;
                }
                if (dataType == UUID.class) {
                    return janusgraphPredicate == Cmp.EQUAL || janusgraphPredicate == Cmp.NOT_EQUAL;
                }
            }
        }
        return false;
    }

    public boolean supports(KeyInformation information) {
        Class dataType = information.getDataType();
        Mapping mapping = Mapping.getMapping((KeyInformation)information);
        return Number.class.isAssignableFrom(dataType) || dataType == Geoshape.class || dataType == Date.class || dataType == Instant.class || dataType == Boolean.class || dataType == UUID.class ? mapping == Mapping.DEFAULT : AttributeUtil.isString((Class)dataType) && (mapping == Mapping.DEFAULT || mapping == Mapping.STRING || mapping == Mapping.TEXT || mapping == Mapping.TEXTSTRING);
    }

    public String mapKey2Field(String key, KeyInformation information) {
        Preconditions.checkArgument((!StringUtils.containsAny((String)key, (char[])new char[]{' '}) ? 1 : 0) != 0, (String)"Invalid key name provided: %s", (Object[])new Object[]{key});
        return key;
    }

    public IndexFeatures getFeatures() {
        return ES_FEATURES;
    }

    public BaseTransactionConfigurable beginTransaction(BaseTransactionConfig config) throws BackendException {
        return new DefaultTransaction(config);
    }

    public void close() throws BackendException {
        if (this.node != null && !this.node.isClosed()) {
            this.node.close();
        }
        this.client.close();
    }

    public void clearStorage() throws BackendException {
        try {
            try {
                this.client.admin().indices().delete(new DeleteIndexRequest(this.indexName)).actionGet();
                Thread.sleep(1000L);
            }
            catch (IndexMissingException indexMissingException) {
                // empty catch block
            }
        }
        catch (Exception e) {
            throw new PermanentBackendException("Could not delete index " + this.indexName, (Throwable)e);
        }
        finally {
            this.close();
        }
    }

    Node getNode() {
        return this.node;
    }

    private void checkExpectedClientVersion() {
        try {
            if (!Version.CURRENT.toString().equals(ElasticSearchConstants.ES_VERSION_EXPECTED)) {
                log.warn("ES client version ({}) does not match the version with which JanusGraph was compiled ({}).  This might cause problems.", (Object)Version.CURRENT, (Object)ElasticSearchConstants.ES_VERSION_EXPECTED);
            } else {
                log.debug("Found ES client version matching JanusGraph's compile-time version: {} (OK)", (Object)Version.CURRENT);
            }
        }
        catch (RuntimeException e) {
            log.warn("Unable to check expected ES client version", (Throwable)e);
        }
    }
}

