/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.BiFunction;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.BlacklistedDirectories;
import org.apache.cassandra.db.lifecycle.LifecycleTransaction;
import org.apache.cassandra.io.FSDiskFullWriteError;
import org.apache.cassandra.io.FSError;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.SnapshotDeletingTask;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.DirectorySizeCalculator;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.Pair;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Directories {
    private static final Logger logger = LoggerFactory.getLogger(Directories.class);
    public static final String BACKUPS_SUBDIR = "backups";
    public static final String SNAPSHOT_SUBDIR = "snapshots";
    public static final String TMP_SUBDIR = "tmp";
    public static final String SECONDARY_INDEX_NAME_SEPARATOR = ".";
    public static final DataDirectory[] dataDirectories;
    private final CFMetaData metadata;
    private final DataDirectory[] paths;
    private final File[] dataPaths;

    public static boolean verifyFullPermissions(File dir, String dataDir) {
        if (!dir.isDirectory()) {
            logger.error("Not a directory {}", (Object)dataDir);
            return false;
        }
        if (!FileAction.hasPrivilege(dir, FileAction.X)) {
            logger.error("Doesn't have execute permissions for {} directory", (Object)dataDir);
            return false;
        }
        if (!FileAction.hasPrivilege(dir, FileAction.R)) {
            logger.error("Doesn't have read permissions for {} directory", (Object)dataDir);
            return false;
        }
        if (dir.exists() && !FileAction.hasPrivilege(dir, FileAction.W)) {
            logger.error("Doesn't have write permissions for {} directory", (Object)dataDir);
            return false;
        }
        return true;
    }

    public Directories(CFMetaData metadata) {
        this(metadata, dataDirectories);
    }

    public Directories(CFMetaData metadata, Collection<DataDirectory> paths) {
        this(metadata, paths.toArray(new DataDirectory[paths.size()]));
    }

    public Directories(final CFMetaData metadata, DataDirectory[] paths) {
        this.metadata = metadata;
        this.paths = paths;
        String cfId = ByteBufferUtil.bytesToHex(ByteBufferUtil.bytes(metadata.cfId));
        int idx = metadata.cfName.indexOf(SECONDARY_INDEX_NAME_SEPARATOR);
        String cfName = idx >= 0 ? metadata.cfName.substring(0, idx) : metadata.cfName;
        String indexNameWithDot = idx >= 0 ? metadata.cfName.substring(idx) : null;
        this.dataPaths = new File[paths.length];
        String oldSSTableRelativePath = Directories.join(metadata.ksName, cfName);
        for (int i = 0; i < paths.length; ++i) {
            this.dataPaths[i] = new File(paths[i].location, oldSSTableRelativePath);
        }
        boolean olderDirectoryExists = Iterables.any(Arrays.asList(this.dataPaths), (Predicate)new Predicate<File>(){

            public boolean apply(File file) {
                return file.exists();
            }
        });
        if (!olderDirectoryExists) {
            String newSSTableRelativePath = Directories.join(metadata.ksName, cfName + '-' + cfId);
            for (int i = 0; i < paths.length; ++i) {
                this.dataPaths[i] = new File(paths[i].location, newSSTableRelativePath);
            }
        }
        if (indexNameWithDot != null) {
            for (int i = 0; i < paths.length; ++i) {
                this.dataPaths[i] = new File(this.dataPaths[i], indexNameWithDot);
            }
        }
        for (File dir : this.dataPaths) {
            try {
                FileUtils.createDirectory(dir);
            }
            catch (FSError e) {
                logger.error("Failed to create {} directory", (Object)dir);
                FileUtils.handleFSError(e);
            }
        }
        if (indexNameWithDot != null) {
            for (File dataPath : this.dataPaths) {
                File[] indexFiles;
                for (File indexFile : indexFiles = dataPath.getParentFile().listFiles(new FileFilter(){

                    @Override
                    public boolean accept(File file) {
                        if (file.isDirectory()) {
                            return false;
                        }
                        Pair<Descriptor, Component> pair = SSTable.tryComponentFromFilename(file.getParentFile(), file.getName());
                        return pair != null && ((Descriptor)pair.left).ksname.equals(metadata.ksName) && ((Descriptor)pair.left).cfname.equals(metadata.cfName);
                    }
                })) {
                    File destFile = new File(dataPath, indexFile.getName());
                    logger.trace("Moving index file {} to {}", (Object)indexFile, (Object)destFile);
                    FileUtils.renameWithConfirm(indexFile, destFile);
                }
            }
        }
    }

    public File getLocationForDisk(DataDirectory dataDirectory) {
        if (dataDirectory != null) {
            for (File dir : this.dataPaths) {
                if (!dir.getAbsolutePath().startsWith(dataDirectory.location.getAbsolutePath())) continue;
                return dir;
            }
        }
        return null;
    }

    public DataDirectory getDataDirectoryForFile(File directory) {
        if (directory != null) {
            for (DataDirectory dataDirectory : this.paths) {
                if (!directory.getAbsolutePath().startsWith(dataDirectory.location.getAbsolutePath())) continue;
                return dataDirectory;
            }
        }
        return null;
    }

    public Descriptor find(String filename) {
        for (File dir : this.dataPaths) {
            if (!new File(dir, filename).exists()) continue;
            return (Descriptor)Descriptor.fromFilename((File)dir, (String)filename).left;
        }
        return null;
    }

    public File getDirectoryForNewSSTables() {
        return this.getWriteableLocationAsFile(-1L);
    }

    public File getWriteableLocationAsFile(long writeSize) {
        File location = this.getLocationForDisk(this.getWriteableLocation(writeSize));
        if (location == null) {
            throw new FSWriteError((Throwable)new IOException("No configured data directory contains enough space to write " + writeSize + " bytes"), "");
        }
        return location;
    }

    public File getTemporaryWriteableDirectoryAsFile(long writeSize) {
        File location = this.getLocationForDisk(this.getWriteableLocation(writeSize));
        if (location == null) {
            return null;
        }
        return new File(location, TMP_SUBDIR);
    }

    public void removeTemporaryDirectories() {
        for (File dataDir : this.dataPaths) {
            File tmpDir = new File(dataDir, TMP_SUBDIR);
            if (!tmpDir.exists()) continue;
            logger.debug("Removing temporary directory {}", (Object)tmpDir);
            FileUtils.deleteRecursive(tmpDir);
        }
    }

    public DataDirectory getWriteableLocation(long writeSize) {
        ArrayList<DataDirectoryCandidate> candidates = new ArrayList<DataDirectoryCandidate>();
        long totalAvailable = 0L;
        boolean tooBig = false;
        for (DataDirectory dataDir : this.paths) {
            if (BlacklistedDirectories.isUnwritable(this.getLocationForDisk(dataDir))) {
                logger.trace("removing blacklisted candidate {}", (Object)dataDir.location);
                continue;
            }
            DataDirectoryCandidate candidate = new DataDirectoryCandidate(dataDir);
            if (candidate.availableSpace < writeSize) {
                logger.trace("removing candidate {}, usable={}, requested={}", new Object[]{candidate.dataDirectory.location, candidate.availableSpace, writeSize});
                tooBig = true;
                continue;
            }
            candidates.add(candidate);
            totalAvailable += candidate.availableSpace;
        }
        if (candidates.isEmpty()) {
            if (tooBig) {
                throw new FSDiskFullWriteError((Throwable)new IOException("Insufficient disk space to write " + writeSize + " bytes"), "");
            }
            throw new FSWriteError((Throwable)new IOException("All configured data directories have been blacklisted as unwritable for erroring out"), "");
        }
        if (candidates.size() == 1) {
            return ((DataDirectoryCandidate)candidates.get((int)0)).dataDirectory;
        }
        Directories.sortWriteableCandidates(candidates, totalAvailable);
        return Directories.pickWriteableDirectory(candidates);
    }

    static DataDirectory pickWriteableDirectory(List<DataDirectoryCandidate> candidates) {
        double rnd = ThreadLocalRandom.current().nextDouble();
        for (DataDirectoryCandidate candidate : candidates) {
            if (!((rnd -= candidate.perc) <= 0.0)) continue;
            return candidate.dataDirectory;
        }
        return candidates.get((int)0).dataDirectory;
    }

    static void sortWriteableCandidates(List<DataDirectoryCandidate> candidates, long totalAvailable) {
        for (DataDirectoryCandidate candidate : candidates) {
            candidate.calcFreePerc(totalAvailable);
        }
        Collections.sort(candidates);
    }

    public boolean hasAvailableDiskSpace(long estimatedSSTables, long expectedTotalWriteSize) {
        long writeSize = expectedTotalWriteSize / estimatedSSTables;
        long totalAvailable = 0L;
        for (DataDirectory dataDir : this.paths) {
            if (BlacklistedDirectories.isUnwritable(this.getLocationForDisk(dataDir))) continue;
            DataDirectoryCandidate candidate = new DataDirectoryCandidate(dataDir);
            if (candidate.availableSpace < writeSize) continue;
            totalAvailable += candidate.availableSpace;
        }
        return totalAvailable > expectedTotalWriteSize;
    }

    public DataDirectory[] getWriteableLocations() {
        ArrayList<DataDirectory> nonBlacklistedDirs = new ArrayList<DataDirectory>();
        for (DataDirectory dir : this.paths) {
            if (BlacklistedDirectories.isUnwritable(dir.location)) continue;
            nonBlacklistedDirs.add(dir);
        }
        Collections.sort(nonBlacklistedDirs, new Comparator<DataDirectory>(){

            @Override
            public int compare(DataDirectory o1, DataDirectory o2) {
                return o1.location.compareTo(o2.location);
            }
        });
        return nonBlacklistedDirs.toArray(new DataDirectory[nonBlacklistedDirs.size()]);
    }

    public static File getSnapshotDirectory(Descriptor desc, String snapshotName) {
        return Directories.getSnapshotDirectory(desc.directory, snapshotName);
    }

    public static File getSnapshotDirectory(File location, String snapshotName) {
        if (location.getName().startsWith(SECONDARY_INDEX_NAME_SEPARATOR)) {
            return Directories.getOrCreate(location.getParentFile(), SNAPSHOT_SUBDIR, snapshotName, location.getName());
        }
        return Directories.getOrCreate(location, SNAPSHOT_SUBDIR, snapshotName);
    }

    public File getSnapshotManifestFile(String snapshotName) {
        File snapshotDir = Directories.getSnapshotDirectory(this.getDirectoryForNewSSTables(), snapshotName);
        return new File(snapshotDir, "manifest.json");
    }

    public File getSnapshotSchemaFile(String snapshotName) {
        File snapshotDir = Directories.getSnapshotDirectory(this.getDirectoryForNewSSTables(), snapshotName);
        return new File(snapshotDir, "schema.cql");
    }

    public File getNewEphemeralSnapshotMarkerFile(String snapshotName) {
        File snapshotDir = new File(this.getWriteableLocationAsFile(1L), Directories.join(SNAPSHOT_SUBDIR, snapshotName));
        return Directories.getEphemeralSnapshotMarkerFile(snapshotDir);
    }

    private static File getEphemeralSnapshotMarkerFile(File snapshotDirectory) {
        return new File(snapshotDirectory, "ephemeral.snapshot");
    }

    public static File getBackupsDirectory(Descriptor desc) {
        return Directories.getBackupsDirectory(desc.directory);
    }

    public static File getBackupsDirectory(File location) {
        if (location.getName().startsWith(SECONDARY_INDEX_NAME_SEPARATOR)) {
            return Directories.getOrCreate(location.getParentFile(), BACKUPS_SUBDIR, location.getName());
        }
        return Directories.getOrCreate(location, BACKUPS_SUBDIR);
    }

    public SSTableLister sstableLister(OnTxnErr onTxnErr) {
        return new SSTableLister(onTxnErr);
    }

    public Map<String, Pair<Long, Long>> getSnapshotDetails() {
        HashMap<String, Pair<Long, Long>> snapshotSpaceMap = new HashMap<String, Pair<Long, Long>>();
        for (File snapshot : this.listSnapshots()) {
            long sizeOnDisk = FileUtils.folderSize(snapshot);
            long trueSize = this.getTrueAllocatedSizeIn(snapshot);
            Pair<Long, Long> spaceUsed = (Pair<Long, Long>)snapshotSpaceMap.get(snapshot.getName());
            spaceUsed = spaceUsed == null ? Pair.create(sizeOnDisk, trueSize) : Pair.create((Long)spaceUsed.left + sizeOnDisk, (Long)spaceUsed.right + trueSize);
            snapshotSpaceMap.put(snapshot.getName(), spaceUsed);
        }
        return snapshotSpaceMap;
    }

    public List<String> listEphemeralSnapshots() {
        LinkedList<String> ephemeralSnapshots = new LinkedList<String>();
        for (File snapshot : this.listSnapshots()) {
            if (!Directories.getEphemeralSnapshotMarkerFile(snapshot).exists()) continue;
            ephemeralSnapshots.add(snapshot.getName());
        }
        return ephemeralSnapshots;
    }

    private List<File> listSnapshots() {
        LinkedList<File> snapshots = new LinkedList<File>();
        for (File dir : this.dataPaths) {
            File[] snapshotDirs;
            File snapshotDir;
            File file = snapshotDir = dir.getName().startsWith(SECONDARY_INDEX_NAME_SEPARATOR) ? new File(dir.getParent(), SNAPSHOT_SUBDIR) : new File(dir, SNAPSHOT_SUBDIR);
            if (!snapshotDir.exists() || !snapshotDir.isDirectory() || (snapshotDirs = snapshotDir.listFiles()) == null) continue;
            for (File snapshot : snapshotDirs) {
                if (!snapshot.isDirectory()) continue;
                snapshots.add(snapshot);
            }
        }
        return snapshots;
    }

    public boolean snapshotExists(String snapshotName) {
        for (File dir : this.dataPaths) {
            File snapshotDir = dir.getName().startsWith(SECONDARY_INDEX_NAME_SEPARATOR) ? new File(dir.getParentFile(), Directories.join(SNAPSHOT_SUBDIR, snapshotName, dir.getName())) : new File(dir, Directories.join(SNAPSHOT_SUBDIR, snapshotName));
            if (!snapshotDir.exists()) continue;
            return true;
        }
        return false;
    }

    public static void clearSnapshot(String snapshotName, List<File> snapshotDirectories) {
        String tag = snapshotName == null ? "" : snapshotName;
        for (File dir : snapshotDirectories) {
            File snapshotDir = new File(dir, Directories.join(SNAPSHOT_SUBDIR, tag));
            if (!snapshotDir.exists()) continue;
            logger.trace("Removing snapshot directory {}", (Object)snapshotDir);
            try {
                FileUtils.deleteRecursive(snapshotDir);
            }
            catch (FSWriteError e) {
                if (FBUtilities.isWindows) {
                    SnapshotDeletingTask.addFailedSnapshot(snapshotDir);
                    continue;
                }
                throw e;
            }
        }
    }

    public long snapshotCreationTime(String snapshotName) {
        for (File dir : this.dataPaths) {
            File snapshotDir = Directories.getSnapshotDirectory(dir, snapshotName);
            if (!snapshotDir.exists()) continue;
            return snapshotDir.lastModified();
        }
        throw new RuntimeException("Snapshot " + snapshotName + " doesn't exist");
    }

    public long trueSnapshotsSize() {
        long result = 0L;
        for (File dir : this.dataPaths) {
            File snapshotDir = dir.getName().startsWith(SECONDARY_INDEX_NAME_SEPARATOR) ? new File(dir.getParent(), SNAPSHOT_SUBDIR) : new File(dir, SNAPSHOT_SUBDIR);
            result += this.getTrueAllocatedSizeIn(snapshotDir);
        }
        return result;
    }

    public long getRawDiretoriesSize() {
        long totalAllocatedSize = 0L;
        for (File path : this.dataPaths) {
            totalAllocatedSize += FileUtils.folderSize(path);
        }
        return totalAllocatedSize;
    }

    public long getTrueAllocatedSizeIn(File input) {
        if (!input.isDirectory()) {
            return 0L;
        }
        SSTableSizeSummer visitor = new SSTableSizeSummer(input, this.sstableLister(OnTxnErr.THROW).listFiles());
        try {
            Files.walkFileTree(input.toPath(), visitor);
        }
        catch (IOException e) {
            logger.error("Could not calculate the size of {}. {}", (Object)input, (Object)e);
        }
        return visitor.getAllocatedSize();
    }

    public static List<File> getKSChildDirectories(String ksName) {
        return Directories.getKSChildDirectories(ksName, dataDirectories);
    }

    public static List<File> getKSChildDirectories(String ksName, DataDirectory[] directories) {
        ArrayList<File> result = new ArrayList<File>();
        for (DataDirectory dataDirectory : directories) {
            File ksDir = new File(dataDirectory.location, ksName);
            File[] cfDirs = ksDir.listFiles();
            if (cfDirs == null) continue;
            for (File cfDir : cfDirs) {
                if (!cfDir.isDirectory()) continue;
                result.add(cfDir);
            }
        }
        return result;
    }

    public List<File> getCFDirectories() {
        ArrayList<File> result = new ArrayList<File>();
        for (File dataDirectory : this.dataPaths) {
            if (!dataDirectory.isDirectory()) continue;
            result.add(dataDirectory);
        }
        return result;
    }

    private static File getOrCreate(File base, String ... subdirs) {
        File dir;
        File file = dir = subdirs == null || subdirs.length == 0 ? base : new File(base, Directories.join(subdirs));
        if (dir.exists()) {
            if (!dir.isDirectory()) {
                throw new AssertionError((Object)String.format("Invalid directory path %s: path exists but is not a directory", dir));
            }
        } else if (!(dir.mkdirs() || dir.exists() && dir.isDirectory())) {
            throw new FSWriteError((Throwable)new IOException("Unable to create directory " + dir), dir);
        }
        return dir;
    }

    private static String join(String ... s) {
        return StringUtils.join((Object[])s, (String)File.separator);
    }

    @VisibleForTesting
    static void overrideDataDirectoriesForTest(String loc) {
        for (int i = 0; i < dataDirectories.length; ++i) {
            Directories.dataDirectories[i] = new DataDirectory(new File(loc));
        }
    }

    @VisibleForTesting
    static void resetDataDirectoriesAfterTest() {
        String[] locations = DatabaseDescriptor.getAllDataFileLocations();
        for (int i = 0; i < locations.length; ++i) {
            Directories.dataDirectories[i] = new DataDirectory(new File(locations[i]));
        }
    }

    static {
        String[] locations = DatabaseDescriptor.getAllDataFileLocations();
        dataDirectories = new DataDirectory[locations.length];
        for (int i = 0; i < locations.length; ++i) {
            Directories.dataDirectories[i] = new DataDirectory(new File(locations[i]));
        }
    }

    private class SSTableSizeSummer
    extends DirectorySizeCalculator {
        private final HashSet<File> toSkip;

        SSTableSizeSummer(File path, List<File> files) {
            super(path);
            this.toSkip = new HashSet<File>(files);
        }

        @Override
        public boolean isAcceptable(Path path) {
            File file = path.toFile();
            Pair<Descriptor, Component> pair = SSTable.tryComponentFromFilename(path.getParent().toFile(), file.getName());
            return pair != null && ((Descriptor)pair.left).ksname.equals(((Directories)Directories.this).metadata.ksName) && ((Descriptor)pair.left).cfname.equals(((Directories)Directories.this).metadata.cfName) && !this.toSkip.contains(file);
        }
    }

    public class SSTableLister {
        private final OnTxnErr onTxnErr;
        private boolean skipTemporary;
        private boolean includeBackups;
        private boolean onlyBackups;
        private int nbFiles;
        private final Map<Descriptor, Set<Component>> components = new HashMap<Descriptor, Set<Component>>();
        private boolean filtered;
        private String snapshotName;

        private SSTableLister(OnTxnErr onTxnErr) {
            this.onTxnErr = onTxnErr;
        }

        public SSTableLister skipTemporary(boolean b) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.skipTemporary = b;
            return this;
        }

        public SSTableLister includeBackups(boolean b) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.includeBackups = b;
            return this;
        }

        public SSTableLister onlyBackups(boolean b) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.onlyBackups = b;
            this.includeBackups = b;
            return this;
        }

        public SSTableLister snapshots(String sn) {
            if (this.filtered) {
                throw new IllegalStateException("list() has already been called");
            }
            this.snapshotName = sn;
            return this;
        }

        public Map<Descriptor, Set<Component>> list() {
            this.filter();
            return ImmutableMap.copyOf(this.components);
        }

        public List<File> listFiles() {
            this.filter();
            ArrayList<File> l = new ArrayList<File>(this.nbFiles);
            for (Map.Entry<Descriptor, Set<Component>> entry : this.components.entrySet()) {
                for (Component c : entry.getValue()) {
                    l.add(new File(entry.getKey().filenameFor(c)));
                }
            }
            return l;
        }

        private void filter() {
            if (this.filtered) {
                return;
            }
            for (File location : Directories.this.dataPaths) {
                if (BlacklistedDirectories.isUnreadable(location)) continue;
                if (this.snapshotName != null) {
                    LifecycleTransaction.getFiles(Directories.getSnapshotDirectory(location, this.snapshotName).toPath(), this.getFilter(), this.onTxnErr);
                    continue;
                }
                if (!this.onlyBackups) {
                    LifecycleTransaction.getFiles(location.toPath(), this.getFilter(), this.onTxnErr);
                }
                if (!this.includeBackups) continue;
                LifecycleTransaction.getFiles(Directories.getBackupsDirectory(location).toPath(), this.getFilter(), this.onTxnErr);
            }
            this.filtered = true;
        }

        private BiFunction<File, FileType, Boolean> getFilter() {
            return (file, type) -> {
                switch (type) {
                    case TXN_LOG: {
                        return false;
                    }
                    case TEMPORARY: {
                        if (this.skipTemporary) {
                            return false;
                        }
                    }
                    case FINAL: {
                        Pair<Descriptor, Component> pair = SSTable.tryComponentFromFilename(file.getParentFile(), file.getName());
                        if (pair == null) {
                            return false;
                        }
                        if (!((Descriptor)pair.left).ksname.equals(((Directories)Directories.this).metadata.ksName) || !((Descriptor)pair.left).cfname.equals(((Directories)Directories.this).metadata.cfName)) {
                            return false;
                        }
                        Set<Component> previous = this.components.get(pair.left);
                        if (previous == null) {
                            previous = new HashSet<Component>();
                            this.components.put((Descriptor)pair.left, previous);
                        } else if (((Component)pair.right).type == Component.Type.DIGEST && pair.right != ((Descriptor)pair.left).digestComponent) {
                            this.components.remove(pair.left);
                            Descriptor updated = ((Descriptor)pair.left).withDigestComponent((Component)pair.right);
                            this.components.put(updated, previous);
                        }
                        previous.add((Component)pair.right);
                        ++this.nbFiles;
                        return false;
                    }
                }
                throw new AssertionError();
            };
        }
    }

    public static enum OnTxnErr {
        THROW,
        IGNORE;

    }

    public static enum FileType {
        FINAL,
        TEMPORARY,
        TXN_LOG;

    }

    static final class DataDirectoryCandidate
    implements Comparable<DataDirectoryCandidate> {
        final DataDirectory dataDirectory;
        final long availableSpace;
        double perc;

        public DataDirectoryCandidate(DataDirectory dataDirectory) {
            this.dataDirectory = dataDirectory;
            this.availableSpace = dataDirectory.getAvailableSpace();
        }

        void calcFreePerc(long totalAvailableSpace) {
            double w = this.availableSpace;
            this.perc = w /= (double)totalAvailableSpace;
        }

        @Override
        public int compareTo(DataDirectoryCandidate o) {
            if (this == o) {
                return 0;
            }
            int r = Double.compare(this.perc, o.perc);
            if (r != 0) {
                return -r;
            }
            return System.identityHashCode(this) - System.identityHashCode(o);
        }
    }

    public static class DataDirectory {
        public final File location;

        public DataDirectory(File location) {
            this.location = location;
        }

        public long getAvailableSpace() {
            long availableSpace = this.location.getUsableSpace() - DatabaseDescriptor.getMinFreeSpacePerDriveInBytes();
            return availableSpace > 0L ? availableSpace : 0L;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            DataDirectory that = (DataDirectory)o;
            return this.location.equals(that.location);
        }

        public int hashCode() {
            return this.location.hashCode();
        }
    }

    public static enum FileAction {
        X,
        W,
        XW,
        R,
        XR,
        RW,
        XRW;


        public static boolean hasPrivilege(File file, FileAction action) {
            boolean privilege = false;
            switch (action) {
                case X: {
                    privilege = file.canExecute();
                    break;
                }
                case W: {
                    privilege = file.canWrite();
                    break;
                }
                case XW: {
                    privilege = file.canExecute() && file.canWrite();
                    break;
                }
                case R: {
                    privilege = file.canRead();
                    break;
                }
                case XR: {
                    privilege = file.canExecute() && file.canRead();
                    break;
                }
                case RW: {
                    privilege = file.canRead() && file.canWrite();
                    break;
                }
                case XRW: {
                    privilege = file.canExecute() && file.canRead() && file.canWrite();
                }
            }
            return privilege;
        }
    }
}

