/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.utils;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.File;
import java.io.RandomAccessFile;
import java.lang.reflect.Constructor;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.LockSupport;
import org.apache.cassandra.io.util.FileUtils;
import org.slf4j.Logger;

public class CoalescingStrategies {
    private static final String DEBUG_COALESCING_PROPERTY = "cassandra.coalescing_debug";
    private static final boolean DEBUG_COALESCING = Boolean.getBoolean("cassandra.coalescing_debug");
    private static final String DEBUG_COALESCING_PATH_PROPERTY = "cassandra.coalescing_debug_path";
    private static final String DEBUG_COALESCING_PATH = System.getProperty("cassandra.coalescing_debug_path", "/tmp/coleascing_debug");
    @VisibleForTesting
    static Clock CLOCK;
    private static final Parker PARKER;

    @VisibleForTesting
    static void parkLoop(long nanos) {
        long now = System.nanoTime();
        long timer = now + nanos;
        do {
            LockSupport.parkNanos(timer - now);
        } while (timer - (now = System.nanoTime()) > nanos / 16L);
    }

    private static boolean maybeSleep(int messages, long averageGap, long maxCoalesceWindow, Parker parker) {
        long sleep = (long)messages * averageGap;
        if (sleep <= 0L || sleep > maxCoalesceWindow) {
            return false;
        }
        while (sleep * 2L < maxCoalesceWindow) {
            sleep *= 2L;
        }
        parker.park(sleep);
        return true;
    }

    @VisibleForTesting
    static CoalescingStrategy newCoalescingStrategy(String strategy, int coalesceWindow, Parker parker, Logger logger, String displayName) {
        String strategyCleaned;
        String classname = null;
        switch (strategyCleaned = strategy.trim().toUpperCase()) {
            case "MOVINGAVERAGE": {
                classname = MovingAverageCoalescingStrategy.class.getName();
                break;
            }
            case "FIXED": {
                classname = FixedCoalescingStrategy.class.getName();
                break;
            }
            case "TIMEHORIZON": {
                classname = TimeHorizonMovingAverageCoalescingStrategy.class.getName();
                break;
            }
            case "DISABLED": {
                classname = DisabledCoalescingStrategy.class.getName();
                break;
            }
            default: {
                classname = strategy;
            }
        }
        try {
            Class<?> clazz = Class.forName(classname);
            if (!CoalescingStrategy.class.isAssignableFrom(clazz)) {
                throw new RuntimeException(classname + " is not an instance of CoalescingStrategy");
            }
            Constructor<?> constructor = clazz.getConstructor(Integer.TYPE, Parker.class, Logger.class, String.class);
            return (CoalescingStrategy)constructor.newInstance(coalesceWindow, parker, logger, displayName);
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public static CoalescingStrategy newCoalescingStrategy(String strategy, int coalesceWindow, Logger logger, String displayName) {
        return CoalescingStrategies.newCoalescingStrategy(strategy, coalesceWindow, PARKER, logger, displayName);
    }

    static {
        if (DEBUG_COALESCING) {
            File directory = new File(DEBUG_COALESCING_PATH);
            if (directory.exists()) {
                FileUtils.deleteRecursive(directory);
            }
            if (!directory.mkdirs()) {
                throw new ExceptionInInitializerError("Couldn't create log dir");
            }
        }
        CLOCK = new Clock(){

            @Override
            public long nanoTime() {
                return System.nanoTime();
            }
        };
        PARKER = new Parker(){

            @Override
            public void park(long nanos) {
                CoalescingStrategies.parkLoop(nanos);
            }
        };
    }

    @VisibleForTesting
    static class DisabledCoalescingStrategy
    extends CoalescingStrategy {
        public DisabledCoalescingStrategy(int coalesceWindowMicros, Parker parker, Logger logger, String displayName) {
            super(parker, logger, displayName);
        }

        @Override
        protected <C extends Coalescable> void coalesceInternal(BlockingQueue<C> input, List<C> out, int maxItems) throws InterruptedException {
            if (input.drainTo(out, maxItems) == 0) {
                out.add(input.take());
                input.drainTo(out, maxItems - 1);
            }
            this.debugTimestamps(out);
        }

        public String toString() {
            return "Disabled";
        }
    }

    @VisibleForTesting
    static class FixedCoalescingStrategy
    extends CoalescingStrategy {
        private final long coalesceWindow;

        public FixedCoalescingStrategy(int coalesceWindowMicros, Parker parker, Logger logger, String displayName) {
            super(parker, logger, displayName);
            this.coalesceWindow = TimeUnit.MICROSECONDS.toNanos(coalesceWindowMicros);
        }

        @Override
        protected <C extends Coalescable> void coalesceInternal(BlockingQueue<C> input, List<C> out, int maxItems) throws InterruptedException {
            if (input.drainTo(out, maxItems) == 0) {
                out.add(input.take());
                this.parker.park(this.coalesceWindow);
                input.drainTo(out, maxItems - 1);
            }
            this.debugTimestamps(out);
        }

        public String toString() {
            return "Fixed";
        }
    }

    @VisibleForTesting
    static class MovingAverageCoalescingStrategy
    extends CoalescingStrategy {
        private final int[] samples = new int[16];
        private long lastSample = 0L;
        private int index = 0;
        private long sum = 0L;
        private final long maxCoalesceWindow;

        public MovingAverageCoalescingStrategy(int maxCoalesceWindow, Parker parker, Logger logger, String displayName) {
            super(parker, logger, displayName);
            this.maxCoalesceWindow = TimeUnit.MICROSECONDS.toNanos(maxCoalesceWindow);
            for (int ii = 0; ii < this.samples.length; ++ii) {
                this.samples[ii] = Integer.MAX_VALUE;
            }
            this.sum = Integer.MAX_VALUE * (long)this.samples.length;
        }

        private long logSample(int value) {
            this.sum -= (long)this.samples[this.index];
            this.sum += (long)value;
            this.samples[this.index] = value;
            ++this.index;
            this.index &= 0xF;
            return this.sum / 16L;
        }

        private long notifyOfSample(long sample) {
            this.debugTimestamp(sample);
            if (sample > this.lastSample) {
                int delta = (int)Math.min(Integer.MAX_VALUE, sample - this.lastSample);
                this.lastSample = sample;
                return this.logSample(delta);
            }
            return this.logSample(1);
        }

        @Override
        protected <C extends Coalescable> void coalesceInternal(BlockingQueue<C> input, List<C> out, int maxItems) throws InterruptedException {
            if (input.drainTo(out, maxItems) == 0) {
                out.add(input.take());
            }
            long average = this.notifyOfSample(((Coalescable)out.get(0)).timestampNanos());
            this.debugGap(average);
            CoalescingStrategies.maybeSleep(out.size(), average, this.maxCoalesceWindow, this.parker);
            input.drainTo(out, maxItems - out.size());
            for (int ii = 1; ii < out.size(); ++ii) {
                this.notifyOfSample(((Coalescable)out.get(ii)).timestampNanos());
            }
        }

        public String toString() {
            return "Moving average";
        }
    }

    @VisibleForTesting
    static class TimeHorizonMovingAverageCoalescingStrategy
    extends CoalescingStrategy {
        private static final int INDEX_SHIFT = 26;
        private static final long BUCKET_INTERVAL = 0x4000000L;
        private static final int BUCKET_COUNT = 16;
        private static final long INTERVAL = 0x40000000L;
        private static final long MEASURED_INTERVAL = 0x3C000000L;
        private long epoch = CLOCK.nanoTime();
        private final int[] samples = new int[16];
        private long sum = 0L;
        private final long maxCoalesceWindow;

        public TimeHorizonMovingAverageCoalescingStrategy(int maxCoalesceWindow, Parker parker, Logger logger, String displayName) {
            super(parker, logger, displayName);
            this.maxCoalesceWindow = TimeUnit.MICROSECONDS.toNanos(maxCoalesceWindow);
            this.sum = 0L;
        }

        private void logSample(long nanos) {
            int ix;
            this.debugTimestamp(nanos);
            long epoch = this.epoch;
            long delta = nanos - epoch;
            if (delta < 0L) {
                return;
            }
            if (delta > 0x40000000L) {
                epoch = this.rollepoch(delta, epoch, nanos);
            }
            int n = ix = this.ix(nanos);
            this.samples[n] = this.samples[n] + 1;
            if (ix != this.ix(epoch - 1L)) {
                ++this.sum;
            }
        }

        private long averageGap() {
            if (this.sum == 0L) {
                return Integer.MAX_VALUE;
            }
            return 0x3C000000L / this.sum;
        }

        private long rollepoch(long delta, long epoch, long nanos) {
            if (delta > 0x80000000L) {
                epoch = this.epoch(nanos);
                this.sum = 0L;
                Arrays.fill(this.samples, 0);
            } else {
                this.sum += (long)this.samples[this.ix(epoch - 1L)];
                while (epoch + 0x40000000L < nanos) {
                    int index = this.ix(epoch);
                    this.sum -= (long)this.samples[index];
                    this.samples[index] = 0;
                    epoch += 0x4000000L;
                }
            }
            this.epoch = epoch;
            return epoch;
        }

        private long epoch(long latestNanos) {
            return latestNanos - 0x3C000000L & 0xFFFFFFFFFC000000L;
        }

        private int ix(long nanos) {
            return (int)(nanos >>> 26 & 0xFL);
        }

        @Override
        protected <C extends Coalescable> void coalesceInternal(BlockingQueue<C> input, List<C> out, int maxItems) throws InterruptedException {
            if (input.drainTo(out, maxItems) == 0) {
                out.add(input.take());
                input.drainTo(out, maxItems - 1);
            }
            for (Coalescable qm : out) {
                this.logSample(qm.timestampNanos());
            }
            long averageGap = this.averageGap();
            this.debugGap(averageGap);
            int count = out.size();
            if (CoalescingStrategies.maybeSleep(count, averageGap, this.maxCoalesceWindow, this.parker)) {
                input.drainTo(out, maxItems - out.size());
                int prevCount = count;
                count = out.size();
                for (int i = prevCount; i < count; ++i) {
                    this.logSample(((Coalescable)out.get(i)).timestampNanos());
                }
            }
        }

        public String toString() {
            return "Time horizon moving average";
        }
    }

    @VisibleForTesting
    static interface Parker {
        public void park(long var1);
    }

    public static abstract class CoalescingStrategy {
        protected final Parker parker;
        protected final Logger logger;
        protected volatile boolean shouldLogAverage = false;
        protected final ByteBuffer logBuffer;
        private RandomAccessFile ras;
        private final String displayName;

        protected CoalescingStrategy(Parker parker, Logger logger, String displayName) {
            this.parker = parker;
            this.logger = logger;
            this.displayName = displayName;
            if (DEBUG_COALESCING) {
                new Thread(displayName + " debug thread"){

                    @Override
                    public void run() {
                        while (true) {
                            try {
                                Thread.sleep(5000L);
                            }
                            catch (InterruptedException e) {
                                throw new AssertionError();
                            }
                            CoalescingStrategy.this.shouldLogAverage = true;
                        }
                    }
                }.start();
            }
            RandomAccessFile rasTemp = null;
            MappedByteBuffer logBufferTemp = null;
            if (DEBUG_COALESCING) {
                try {
                    File outFile = File.createTempFile("coalescing_" + this.displayName + "_", ".log", new File(DEBUG_COALESCING_PATH));
                    rasTemp = new RandomAccessFile(outFile, "rw");
                    logBufferTemp = this.ras.getChannel().map(FileChannel.MapMode.READ_WRITE, 0L, Integer.MAX_VALUE);
                    logBufferTemp.putLong(0L);
                }
                catch (Exception e) {
                    logger.error("Unable to create output file for debugging coalescing", (Throwable)e);
                }
            }
            this.ras = rasTemp;
            this.logBuffer = logBufferTemp;
        }

        protected final void debugGap(long averageGap) {
            if (DEBUG_COALESCING && this.shouldLogAverage) {
                this.shouldLogAverage = false;
                this.logger.info(this.toString() + " gap " + TimeUnit.NANOSECONDS.toMicros(averageGap) + "\u03bcs");
            }
        }

        protected final void debugTimestamp(long timestamp) {
            if (DEBUG_COALESCING && this.logBuffer != null) {
                this.logBuffer.putLong(0, this.logBuffer.getLong(0) + 1L);
                this.logBuffer.putLong(timestamp);
            }
        }

        protected final <C extends Coalescable> void debugTimestamps(Collection<C> coalescables) {
            if (DEBUG_COALESCING) {
                for (Coalescable coalescable : coalescables) {
                    this.debugTimestamp(coalescable.timestampNanos());
                }
            }
        }

        public <C extends Coalescable> void coalesce(BlockingQueue<C> input, List<C> out, int maxItems) throws InterruptedException {
            Preconditions.checkArgument((boolean)out.isEmpty(), (Object)"out list should be empty");
            this.coalesceInternal(input, out, maxItems);
        }

        protected abstract <C extends Coalescable> void coalesceInternal(BlockingQueue<C> var1, List<C> var2, int var3) throws InterruptedException;
    }

    public static interface Coalescable {
        public long timestampNanos();
    }

    @VisibleForTesting
    static interface Clock {
        public long nanoTime();
    }
}

