package com.jpattern.orm.session.datasource;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jpattern.orm.exception.OrmException;
import com.jpattern.orm.exception.sql.OrmSqlException;
import com.jpattern.orm.session.BatchPreparedStatementSetter;
import com.jpattern.orm.session.GeneratedKeyReader;
import com.jpattern.orm.session.PreparedStatementSetter;
import com.jpattern.orm.session.ResultSetReader;
import com.jpattern.orm.session.SqlPerformerStrategy;
import com.jpattern.orm.session.datasource.exception.SpringBasedSQLStateSQLExceptionTranslator;

/**
 * 
 * @author Francesco Cina
 *
 * 02/lug/2011
 * 
 * ISqlExecutor implementation using java.sql.Connection as backend.
 */
public class DataSourceSqlPerformerStrategy implements SqlPerformerStrategy, IConnectionCaller {

	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final DataSourceSessionProvider dataSourceSessionProvider;

	public DataSourceSqlPerformerStrategy(final DataSourceSessionProvider dataSourceSessionProvider) {
		this.dataSourceSessionProvider = dataSourceSessionProvider;
	}

	@Override
	public void execute(final String sql, final int timeout) throws OrmException {
		this.logger.debug("Method called");
		PreparedStatement preparedStatement = null;
		IConnection conn = this.dataSourceSessionProvider.getConnection(false, this);
		try {
			preparedStatement = conn.prepareStatement( sql );
			preparedStatement.setQueryTimeout(timeout);
			preparedStatement.execute();
			conn.commit();
		} catch (Exception e) {
			conn.rollback();
			throw translateException("execute", sql, e);
		} finally {
			try {
				if (preparedStatement!=null) {
					preparedStatement.close();
				}
			} catch (Exception e) {
				throw translateException("execute", sql, e);
			} finally {
				conn.close(this);
			}
		}
	}

	@Override
	public <T> T query(final String sql, final int timeout, final int maxRows, final PreparedStatementSetter pss, final ResultSetReader<T> rse) 	throws OrmException {
		this.logger.debug("Method called");
		ResultSet resultSet = null;
		PreparedStatement preparedStatement = null;
		IConnection conn = this.dataSourceSessionProvider.getConnection(true, this);
		try {
			preparedStatement = conn.prepareStatement( sql );
			pss.set(preparedStatement);
			preparedStatement.setMaxRows(maxRows);
			preparedStatement.setQueryTimeout(timeout);
			resultSet = preparedStatement.executeQuery();
			return rse.read(resultSet);
		} catch (Exception e) {
			throw translateException("query", sql, e);
		} finally {
			try {
				if ((resultSet!=null) && !resultSet.isClosed()) {
					resultSet.close();
				}
				if (preparedStatement!=null) {
					preparedStatement.close();
				}
			} catch (Exception e) {
				throw translateException("query", sql, e);
			} finally {
				conn.close(this);
			}
		}
	}

	@Override
	public int update(final String sql, final int timeout, final PreparedStatementSetter pss) throws OrmException {
		this.logger.debug("Method called");
		DataSourceConnection conn = this.dataSourceSessionProvider.getConnection(false, this);
		PreparedStatement preparedStatement = null;
		try {
			preparedStatement = conn.prepareStatement( sql );
			preparedStatement.setQueryTimeout(timeout);
			pss.set(preparedStatement);
			int result = preparedStatement.executeUpdate();
			conn.commit();
			return result;
		} catch (Exception e) {
			conn.rollback();
			throw translateException("update", sql, e);
		} finally {
			try {
				if (preparedStatement!=null) {
					preparedStatement.close();
				}
			} catch (Exception e) {
				throw translateException("update", sql, e);
			} finally {
				conn.close(this);
			}
		}
	}

	@Override
	public int update(final String sql, final int timeout, final GeneratedKeyReader generatedKeyExtractor, final PreparedStatementSetter pss) throws OrmException {
		this.logger.debug("Method called");
		DataSourceConnection conn = this.dataSourceSessionProvider.getConnection(false, this);
		ResultSet generatedKeyResultSet = null;
		PreparedStatement preparedStatement = null;
		int result = 0;
		try {
			preparedStatement = conn.prepareStatement( sql , generatedKeyExtractor.generatedColumnNames());
			preparedStatement.setQueryTimeout(timeout);
			pss.set(preparedStatement);
			result = preparedStatement.executeUpdate();
			generatedKeyResultSet = preparedStatement.getGeneratedKeys();
			generatedKeyExtractor.read(generatedKeyResultSet);
			conn.commit();
			return result;
		} catch (Exception e) {
			conn.rollback();
			throw translateException("update", sql, e);
		} finally {
			try {
				if (preparedStatement!=null) {
					preparedStatement.close();
				}
				if ((generatedKeyResultSet!=null) && !generatedKeyResultSet.isClosed()) {
					generatedKeyResultSet.close();
				}
			} catch (Exception e) {
				throw translateException("update", sql, e);
			} finally {
				conn.close(this);
			}
		}
	}

	@Override
	public int[] batchUpdate(final List<String> sqls, final int timeout) throws OrmException {
		this.logger.debug("Method called");
		IConnection conn = this.dataSourceSessionProvider.getConnection(false, this);
		IStatement statement = null;
		try {
			statement = conn.createStatement();
			statement.setQueryTimeout(timeout);
			for (String sql : sqls) {
				statement.addBatch(sql);
			}
			int[] result = statement.executeBatch();
			conn.commit();
			return result;
		} catch (Exception e) {
			conn.rollback();
			throw translateException("batchUpdate", "", e);
		} finally {
			try {
				if (statement!=null) {
					statement.close();
				}
			} catch (Exception e) {
				throw translateException("batchUpdate", "", e);
			} finally {
				conn.close(this);
			}
		}
	}

	@Override
	public int[] batchUpdate(final String sql, final List<Object[]> args, final int timeout) throws OrmException {
		this.logger.debug("Method called");
		IConnection conn = this.dataSourceSessionProvider.getConnection(false, this);
		PreparedStatement preparedStatement = null;
		try {
			preparedStatement = conn.prepareStatement( sql );
			preparedStatement.setQueryTimeout(timeout);
			for (Object[] arg : args) {
				int i = 0;
				for (Object value : arg) {
					preparedStatement.setObject(++i, value);
				}
				preparedStatement.addBatch();
			}
			int[] result = preparedStatement.executeBatch();
			conn.commit();
			return result;
		} catch (Exception e) {
			conn.rollback();
			throw translateException("batchUpdate", sql, e);
		} finally {
			try {
				if (preparedStatement!=null) {
					preparedStatement.close();
				}
			} catch (Exception e) {
				throw translateException("batchUpdate", sql, e);
			} finally {
				conn.close(this);
			}
		}
	}

	@Override
	public int[] batchUpdate(final String sql, final BatchPreparedStatementSetter psc, final int timeout) throws OrmException {
		this.logger.debug("Method called");
		IConnection conn = this.dataSourceSessionProvider.getConnection(false, this);
		PreparedStatement preparedStatement = null;
		try {
			preparedStatement = conn.prepareStatement( sql );
			preparedStatement.setQueryTimeout(timeout);
			for (int i=0; i<psc.getBatchSize(); i++) {
				psc.set(preparedStatement, i);
				preparedStatement.addBatch();
			}
			int[] result = preparedStatement.executeBatch();
			conn.commit();
			return result;
		} catch (Exception e) {
			conn.rollback();
			throw translateException("batchUpdate", sql, e);
		} finally {
			try {
				if (preparedStatement!=null) {
					preparedStatement.close();
				}
			} catch (Exception e) {
				throw translateException("batchUpdate", sql, e);
			} finally {
				conn.close(this);
			}
		}
	}

	private OrmException translateException(final String task, final String sql, final Exception ex) {
		if (ex instanceof OrmException) {
			return (OrmException) ex;
		}
		if (ex instanceof SQLException) {
			return SpringBasedSQLStateSQLExceptionTranslator.doTranslate(task, sql, (SQLException) ex);
		}
		return new OrmSqlException(ex);
	}

}
