package com.jpattern.orm.session;

import java.util.Collection;
import java.util.List;

import com.jpattern.orm.exception.OrmException;
import com.jpattern.orm.exception.OrmNotUniqueResultException;
import com.jpattern.orm.query.delete.DeleteQuery;
import com.jpattern.orm.query.find.CustomFindQuery;
import com.jpattern.orm.query.find.FindQuery;
import com.jpattern.orm.query.sql.SqlExecutor;
import com.jpattern.orm.query.update.UpdateQuery;
import com.jpattern.orm.script.ScriptExecutor;
import com.jpattern.orm.transaction.Transaction;
import com.jpattern.orm.transaction.TransactionDefinition;

/**
 * 
 * @author Francesco Cina
 *
 * 21/mag/2011
 * 
 */
public interface Session {

	/**
	 * Begin a transaction or participate to an existing one using the default ITransactionDefinition
	 * @return
	 * @throws OrmException
	 */
	Transaction transaction() throws OrmException;

	/**
	 * Begin a or participate to an existing one depending on the specific transactionDefinition
	 * @return
	 * @throws OrmException
	 */
	Transaction transaction(TransactionDefinition transactionDefinition) throws OrmException;

	/**
	 * Execute a block of code inside a Transaction or participate to an existing one
	 * @param transactionCallback
	 * @return
	 */
	<T> T doInTransaction(TransactionCallback<T> transactionCallback);

	/**
	 * Execute a block of code inside a Transaction or participate to an existing one
	 * @param transactionCallback
	 * @return
	 */
	<T> T doInTransaction(TransactionDefinition transactionDefinition, TransactionCallback<T> transactionCallback);

	/**
	 * Return whether a Bean exist in the DB using the Object type and id(s).
	 * @param <T>
	 * @param object
	 * @return
	 * @throws OrmException
	 */
	<T> boolean exist(T object) throws OrmException;

	/**
	 * Return whether a Bean with a specific id exists in the DB.
	 * @param <T>
	 * @param clazz The Class of the object to load
	 * @param idValue the value of the identifying column of the object
	 * @return
	 * @throws OrmException
	 */
	<T> boolean exist(Class<T> clazz, Object idValue) throws OrmException;

	/**
	 * Return whether a Bean with the specific ids exists in the DB.
	 * @param <T>
	 * @param clazz The Class of the object to load
	 * @param idValues an ordered array with the values of the identifying columns of the object
	 * @return
	 * @throws OrmException
	 */
	<T> boolean exist(Class<T> clazz, Object[] idValues) throws OrmException;

	/**
	 * Find a bean using is ID.
	 * @param <T>
	 * @param clazz The Class of the object to load
	 * @param idValue the value of the identifying column of the object
	 * @return
	 * @throws OrmException
	 */
	<T> T find(Class<T> clazz, Object idValue) throws OrmException;

	/**
	 * Find a bean using is IDs.
	 * @param <T>
	 * @param clazz The Class of the object to load
	 * @param idValues an ordered array with the values of the identifying columns of the object
	 * @return
	 * @throws OrmException
	 */
	<T> T find(Class<T> clazz, Object[] idValues) throws OrmException;

	/**
	 * Find a bean using is ID. An OrmNotUniqueResultException is thrown if nothing is found.
	 * @param <T>
	 * @param clazz The Class of the object to load
	 * @param idValue the value of the identifying column of the object
	 * @return
	 * @throws OrmException
	 */
	<T> T findUnique(Class<T> clazz, Object idValue) throws OrmException, OrmNotUniqueResultException;

	/**
	 * Find a bean using is IDs. An OrmNotUniqueResultException is thrown if nothing is found.
	 * @param <T>
	 * @param clazz The Class of the object to load
	 * @param idValues an ordered array with the values of the identifying columns of the object
	 * @return
	 * @throws OrmException
	 */
	<T> T findUnique(Class<T> clazz, Object[] idValues) throws OrmException, OrmNotUniqueResultException;

	/**
	 * Persist the new object in the database
	 * @param <T>
	 * @param object
	 * @throws OrmException
	 * @return the bean representing the saved object (with updated generated fields)
	 */
	<T> T save(T object) throws OrmException;

	/**
	 * Update the object if it exists, otherwise save it
	 * @param <T>
	 * @param object
	 * @throws OrmException
	 * @return the bean representing the saved or updated object (with updated generated fields))
	 */
	<T> T saveOrUpdate(T object) throws OrmException;

	/**
	 * Update the values of an existing object in the database
	 * @param <T>
	 * @param object
	 * @throws OrmException
	 * @return the bean representing the saved object (with updated generated fields)
	 */
	<T> T update(T object) throws OrmException;

	/**
	 * Persist the new objects in the database
	 * @param <T>
	 * @param objects the objects to persist
	 * @throws OrmException
	 * @return a new collection that contains the beans representing the saved objects
	 */
	<T> List<T> save(Collection<T> objects) throws OrmException;

	/**
	 * For each object in the list, update the object if it exists,
	 * otherwise save it
	 * @param <T>
	 * @param objects the objects to persist
	 * @throws OrmException
	 * @return a new collection that contains the beans representing the saved or updated objects
	 */
	<T> List<T> saveOrUpdate(Collection<T> objects) throws OrmException;

	/**
	 * Update the values of the existing objects in the database
	 * @param <T>
	 * @param objects the objects to update
	 * @throws OrmException
	 * @return a new collection that contains the beans representing the updated objects
	 */
	<T> List<T> update(Collection<T> objects) throws OrmException;

	/**
	 * Update the objects of a specific TABLE
	 * @param clazz the TABLE related Class
	 * @throws OrmException
	 */
	UpdateQuery updateQuery(Class<?> clazz) throws OrmException;

	/**
	 * Update the objects of a specific TABLE
	 * @param clazz the TABLE related Class
	 * @param alias The alias of the class in the query.
	 * @throws OrmException
	 */
	UpdateQuery updateQuery(Class<?> clazz, String alias) throws OrmException;


	/**
	 * Delete one object from the database
	 * @param <T>
	 * @param object
	 * @throws OrmException
	 * @return the number of deleted objects
	 */
	<T> int delete(T object) throws OrmException;

	/**
	 * Delete the objects from the database
	 * @param <T>
	 * @param objects the objects to delete
	 * @throws OrmException
	 * @return the number of deleted objects
	 */
	<T> int delete(List<T> object) throws OrmException;

	/**
	 * Delete the objects of a specific table
	 * @param clazz the TABLE related Class
	 * @throws OrmException
	 */
	DeleteQuery deleteQuery(Class<?> clazz) throws OrmException;

	/**
	 * Delete the objects of a specific table
	 * @param clazz the TABLE related Class
	 * @param alias The alias of the class in the query.
	 * @throws OrmException
	 */
	DeleteQuery deleteQuery(Class<?> clazz, String alias) throws OrmException;

	/**
	 * Create a new query
	 * @param <T>
	 * @param clazz The class of the object that will be retrieved by the query execution. The simple class name will be used as alias for the class
	 * @return
	 * @throws OrmException
	 */
	<T> FindQuery<T> findQuery(Class<T> clazz ) throws OrmException;

	/**
	 * Create a new query
	 * @param <T>
	 * @param clazz The class of the object that will be retrieved by the query execution.
	 * @param alias The alias of the class in the query.
	 * @return
	 * @throws OrmException
	 */
	<T> FindQuery<T> findQuery(Class<T> clazz, String alias ) throws OrmException;

	/**
	 * Create a new query
	 * @param selectClause the select clause of the query (do not use the "select" keyword)
	 * @param clazz The class of the object that will be retrieved by the query execution.
	 * @param alias The alias of the class in the query.
	 * @return
	 * @throws OrmException
	 */
	CustomFindQuery findQuery(String selectClause, Class<?> clazz, String alias ) throws OrmException;

	/**
	 * An executor to perform any kind of plain SQL statements.
	 * @return
	 */
	SqlExecutor sqlExecutor();

	/**
	 * A script executor useful to execute multiple sql statement from files.
	 * @return
	 * @throws OrmException
	 */
	ScriptExecutor scriptExecutor() throws OrmException;
}
