package com.jpattern.orm.session;

import java.math.BigDecimal;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jpattern.orm.exception.OrmException;
import com.jpattern.orm.exception.OrmNotUniqueResultException;
import com.jpattern.orm.session.reader.ArrayResultSetReader;
import com.jpattern.orm.session.reader.BigDecimalResultSetReader;
import com.jpattern.orm.session.reader.BooleanResultSetReader;
import com.jpattern.orm.session.reader.DoubleResultSetReader;
import com.jpattern.orm.session.reader.FloatResultSetReader;
import com.jpattern.orm.session.reader.IntegerResultSetReader;
import com.jpattern.orm.session.reader.ListResultSetReader;
import com.jpattern.orm.session.reader.LongResultSetReader;
import com.jpattern.orm.session.reader.ResultSetRowReaderToResultSetReader;
import com.jpattern.orm.session.reader.ResultSetRowReaderToResultSetReaderUnique;
import com.jpattern.orm.session.reader.StringResultSetReader;

/**
 * 
 * @author Francesco Cina
 *
 * 02/lug/2011
 */
public class PlainSqlPerformer implements SqlPerformer {

	private final Logger logger = LoggerFactory.getLogger(this.getClass());
	private final SqlPerformerStrategy sqlPerformerStrategy;
	private int queryTimeout = 0;
	private int maxRows = 0;

	public PlainSqlPerformer(final SqlPerformerStrategy sqlPerformerStrategy) {
		this.sqlPerformerStrategy = sqlPerformerStrategy;
	}

	@Override
	public final void setMaxRows(final int maxRows) {
		if (this.logger.isDebugEnabled()) {
			this.logger.debug("max rows sets to " + maxRows);
		}
		this.maxRows = maxRows;
	}

	@Override
	public final int getMaxRows() {
		return this.maxRows;
	}

	@Override
	public final void setQueryTimeout(final int queryTimeout) {
		if (this.logger.isDebugEnabled()) {
			this.logger.debug("queryTimeout sets to " + queryTimeout);
		}
		this.queryTimeout = queryTimeout;
	}

	@Override
	public final int getQueryTimeout() {
		return this.queryTimeout;
	}

	@Override
	public final Integer queryForInt(final String sql, final Object... values) throws OrmException, OrmNotUniqueResultException {
		return this.query(sql, new IntegerResultSetReader(), values);
	}

	@Override
	public final Integer queryForInt(final String sql, final Collection<?> values) throws OrmException, OrmNotUniqueResultException {
		return this.query(sql, new IntegerResultSetReader(), values);
	}

	@Override
	public final Long queryForLong(final String sql, final Object... values) throws OrmException, OrmNotUniqueResultException {
		return this.query(sql, new LongResultSetReader(), values);
	}

	@Override
	public final Long queryForLong(final String sql, final Collection<?> values) throws OrmException, OrmNotUniqueResultException {
		return this.query(sql, new LongResultSetReader(), values);
	}

	@Override
	public final Double queryForDouble(final String sql, final Object... values) throws OrmException, OrmNotUniqueResultException {
		return this.query(sql, new DoubleResultSetReader(), values);
	}

	@Override
	public final Double queryForDouble(final String sql, final Collection<?> values) throws OrmException, OrmNotUniqueResultException {
		return this.query(sql, new DoubleResultSetReader(), values);
	}

	@Override
	public final Float queryForFloat(final String sql, final Object... values) throws OrmException, OrmNotUniqueResultException {
		return this.query(sql, new FloatResultSetReader(), values);
	}

	@Override
	public final Float queryForFloat(final String sql, final Collection<?> values) throws OrmException, OrmNotUniqueResultException {
		return this.query(sql, new FloatResultSetReader(), values);
	}

	@Override
	public final String queryForString(final String sql, final Object... values) throws OrmException, OrmNotUniqueResultException {
		return this.query(sql, new StringResultSetReader(), values);
	}

	@Override
	public final String queryForString(final String sql, final Collection<?> values) throws OrmException, OrmNotUniqueResultException {
		return this.query(sql, new StringResultSetReader(), values);
	}

	@Override
	public final Boolean queryForBoolean(final String sql, final Object... values) throws OrmException, OrmNotUniqueResultException {
		return this.query(sql, new BooleanResultSetReader(), values);
	}

	@Override
	public final Boolean queryForBoolean(final String sql, final Collection<?> values) throws OrmException, OrmNotUniqueResultException {
		return this.query(sql, new BooleanResultSetReader(), values);
	}

	@Override
	public final BigDecimal queryForBigDecimal(final String sql, final Object... values) throws OrmException, OrmNotUniqueResultException {
		return this.query(sql, new BigDecimalResultSetReader(), values);
	}

	@Override
	public final BigDecimal queryForBigDecimal(final String sql, final Collection<?> values) throws OrmException, OrmNotUniqueResultException {
		return this.query(sql, new BigDecimalResultSetReader(), values);
	}

	@Override
	public final Object[] queryForArray(final String sql, final Object... values) throws OrmException, OrmNotUniqueResultException {
		return this.query(sql, new ArrayResultSetReader(), values);
	}

	@Override
	public final Object[] queryForArray(final String sql, final Collection<?> values) throws OrmException, OrmNotUniqueResultException {
		return this.query(sql, new ArrayResultSetReader(), values);
	}

	@Override
	public final List<Object[]> queryForList(final String sql, final Object... values) throws OrmException {
		return this.query(sql, new ListResultSetReader(), values);
	}

	@Override
	public final List<Object[]> queryForList(final String sql, final Collection<?> values) throws OrmException {
		return this.query(sql, new ListResultSetReader(), values);
	}

	@Override
	public void execute(final String sql) throws OrmException {
		if (this.logger.isDebugEnabled()) {
			this.logger.debug("Execute query: [" + sql + "]");
		}
		this.sqlPerformerStrategy.execute(sql, getQueryTimeout());
	}

	@Override
	public <T> T query(final String sql, final ResultSetReader<T> rse, final Object... args) throws OrmException {
		if (this.logger.isDebugEnabled()) {
			this.logger.debug("Execute query: [" + sql + "]");
			if (args!=null) {
				this.logger.debug("Query params: " + Arrays.toString(args) );
			}
		}
		PreparedStatementSetter pss = new PreparedStatementSetter() {
			@Override
			public void set(final PreparedStatement ps) throws SQLException {
				int index = 0;
				for (Object arg : args) {
					ps.setObject(++index, arg);
				}
			}
		};
		return this.sqlPerformerStrategy.query(sql, getQueryTimeout(), getMaxRows(), pss, rse);
	}

	@Override
	public <T> T query(final String sql, final ResultSetReader<T> rse, final Collection<?> args) throws OrmException {
		if (this.logger.isDebugEnabled()) {
			this.logger.debug("Execute query: [" + sql + "]");
			if (args!=null) {
				this.logger.debug("Query params: " + args );
			}
		}
		PreparedStatementSetter pss = new PreparedStatementSetter() {
			@Override
			public void set(final PreparedStatement ps) throws SQLException {
				int index = 0;
				for (Object arg : args) {
					ps.setObject(++index, arg);
				}
			}
		};
		return this.sqlPerformerStrategy.query(sql, getQueryTimeout(), getMaxRows(), pss, rse);
	}

	@Override
	public <T> List<T> query(final String sql, final ResultSetRowReader<T> rsrr, final Object... args) throws OrmException {
		return query(sql, new ResultSetRowReaderToResultSetReader<T>(rsrr), args);
	}

	@Override
	public <T> List<T> query(final String sql, final ResultSetRowReader<T> rsrr, final Collection<?> args) throws OrmException {
		return query(sql, new ResultSetRowReaderToResultSetReader<T>(rsrr), args);
	}

	@Override
	public <T> T queryForUnique(final String sql, final ResultSetRowReader<T> rsrr, final Object... args) throws OrmException, OrmNotUniqueResultException {
		return query(sql, new ResultSetRowReaderToResultSetReaderUnique<T>(rsrr), args);
	}

	@Override
	public <T> T queryForUnique(final String sql, final ResultSetRowReader<T> rsrr, final Collection<?> args) throws OrmException {
		return query(sql, new ResultSetRowReaderToResultSetReaderUnique<T>(rsrr), args);
	}

	@Override
	public int update(final String sql, final Object... args) throws OrmException {
		PreparedStatementSetter pss = new PreparedStatementSetter() {
			@Override
			public void set(final PreparedStatement ps) throws SQLException {
				if (PlainSqlPerformer.this.logger.isDebugEnabled()) {
					PlainSqlPerformer.this.logger.debug("Query params: " + Arrays.toString(args));
				}
				for (int i=0; i<args.length ; i++) {
					ps.setObject(i+1, args[i]);
				}
			}
		};
		return this.update(sql, pss);
	}

	@Override
	public int update(final String sql, final Collection<?> args) throws OrmException {
		PreparedStatementSetter pss = new PreparedStatementSetter() {
			@Override
			public void set(final PreparedStatement ps) throws SQLException {
				if (PlainSqlPerformer.this.logger.isDebugEnabled()) {
					PlainSqlPerformer.this.logger.debug("Query params: " + args);
				}
				int index = 0;
				for (Object object : args) {
					ps.setObject(++index, object);
				}
			}
		};
		return this.update(sql, pss);
	}

	@Override
	public int update(final String sql, final PreparedStatementSetter psc) throws OrmException {
		if (this.logger.isDebugEnabled()) {
			this.logger.debug("Execute query: [" + sql + "]");
		}
		return this.sqlPerformerStrategy.update(sql, getQueryTimeout(), psc);
	}

	@Override
	public int update(final String sql, final GeneratedKeyReader generatedKeyReader, final Object... args) throws OrmException {
		PreparedStatementSetter pss = new PreparedStatementSetter() {
			@Override
			public void set(final PreparedStatement ps) throws SQLException {
				if (PlainSqlPerformer.this.logger.isDebugEnabled()) {
					PlainSqlPerformer.this.logger.debug("Query params: " + Arrays.toString(args));
				}
				for (int i=0; i<args.length ; i++) {
					ps.setObject(i+1, args[i]);
				}
			}
		};
		return this.update(sql, generatedKeyReader, pss);
	}

	@Override
	public int update(final String sql, final GeneratedKeyReader generatedKeyReader, final Collection<?> args) throws OrmException {
		PreparedStatementSetter pss = new PreparedStatementSetter() {
			@Override
			public void set(final PreparedStatement ps) throws SQLException {
				if (PlainSqlPerformer.this.logger.isDebugEnabled()) {
					PlainSqlPerformer.this.logger.debug("Query params: " + args);
				}
				int index = 0;
				for (Object object : args) {
					ps.setObject(++index, object);
				}
			}
		};
		return this.update(sql, generatedKeyReader, pss);
	}

	@Override
	public int update(final String sql, final GeneratedKeyReader generatedKeyReader,
			final PreparedStatementSetter psc) throws OrmException {
		if (this.logger.isDebugEnabled()) {
			this.logger.debug("Execute query: [" + sql + "]");
		}
		return this.sqlPerformerStrategy.update(sql, getQueryTimeout(), generatedKeyReader, psc);
	}

	@Override
	public int[] batchUpdate(final List<String> sqls) throws OrmException {
		return this.sqlPerformerStrategy.batchUpdate(sqls, getQueryTimeout());
	}

	@Override
	public int[] batchUpdate(final String sql, final List<Object[]> args) throws OrmException {
		if (this.logger.isDebugEnabled()) {
			this.logger.debug("Execute query: [" + sql + "]");
		}
		return this.sqlPerformerStrategy.batchUpdate(sql, args, getQueryTimeout());
	}

	@Override
	public int[] batchUpdate(final String sql, final BatchPreparedStatementSetter psc) throws OrmException {
		if (this.logger.isDebugEnabled()) {
			this.logger.debug("Execute query: [" + sql + "]");
		}
		return this.sqlPerformerStrategy.batchUpdate(sql, psc, getQueryTimeout());
	}

}
