package com.jpattern.orm.query.find;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import com.jpattern.orm.exception.OrmException;
import com.jpattern.orm.exception.OrmNotUniqueResultException;
import com.jpattern.orm.exception.OrmNotUniqueResultManyResultsException;
import com.jpattern.orm.exception.OrmNotUniqueResultNoResultException;
import com.jpattern.orm.mapper.IOrmClassTool;
import com.jpattern.orm.mapper.IOrmClassToolMap;
import com.jpattern.orm.persistor.type.TypeFactory;
import com.jpattern.orm.persistor.type.ext.WrapperTypeList;
import com.jpattern.orm.query.NameSolver;
import com.jpattern.orm.query.NameSolverConsumer;
import com.jpattern.orm.query.LockMode;
import com.jpattern.orm.query.NullNameSolver;
import com.jpattern.orm.query.OrmRowMapper;
import com.jpattern.orm.query.SmartRenderableSqlQuery;
import com.jpattern.orm.session.ResultSetReader;
import com.jpattern.orm.session.SessionSqlPerformer;
import com.jpattern.orm.session.SqlPerformer;
import com.jpattern.orm.util.GenericWrapper;

/**
 * 
 * @author Francesco Cina
 *
 * 20/giu/2011
 */
public class FindQueryOrm<BEAN> extends SmartRenderableSqlQuery implements FindQuery<BEAN>, NameSolverConsumer {

	private NameSolver nameSolver = new NullNameSolver();
	private final IOrmClassToolMap ormClassToolMap;
	private final Class<BEAN> clazz;
	private boolean distinct;
	private final SessionSqlPerformer session;
	private int queryTimeout = 0;
	private int maxRows = 0;
	private LockMode lockMode = LockMode.NO_LOCK;
	private final FindWhereImpl<BEAN> where = new FindWhereImpl<BEAN>(this);
	private final FindOrderByImpl<BEAN> orderBy = new FindOrderByImpl<BEAN>(this);
	private final FindFromImpl<BEAN> from;
	private final Integer nameSolverClassId;
	private final TypeFactory typeFactory;
	private int versionStatus = 0;

	public FindQueryOrm(final IOrmClassToolMap ormClassToolMap, final SessionSqlPerformer session, final Class<BEAN> clazz, final Integer nameSolverClassId, final TypeFactory typeFactory) {
		this.ormClassToolMap = ormClassToolMap;
		this.session = session;
		this.clazz = clazz;
		this.nameSolverClassId = nameSolverClassId;
		this.typeFactory = typeFactory;
		this.from = new FindFromImpl<BEAN>(this, ormClassToolMap, clazz, nameSolverClassId);
	}

	@Override
	public final FindWhere<BEAN> where() throws OrmException {
		return this.where;
	}

	@Override
	public final FindOrderBy<BEAN> orderBy() throws OrmException {
		return this.orderBy;
	}

	@Override
	public final int getMaxRows() throws OrmException {
		return this.maxRows;
	}

	@Override
	public final FindQuery<BEAN> setQueryTimeout(final int queryTimeout) {
		this.queryTimeout = queryTimeout;
		return this;
	}

	@Override
	public final int getQueryTimeout() {
		return this.queryTimeout;
	}

	@Override
	public final FindQuery<BEAN> setMaxRows(final int maxRows) throws OrmException {
		this.maxRows = maxRows;
		return this;
	}

	@Override
	public void setNameSolver(final NameSolver nameSolver) {
		this.nameSolver = nameSolver;
		this.where.setNameSolver(nameSolver);
		this.orderBy.setNameSolver(nameSolver);
		this.from.setNameSolver(nameSolver);
	}

	@Override
	public void find(final OrmRowMapper<BEAN> srr) throws OrmException {
		final List<Object> values = new WrapperTypeList(this.typeFactory);
		this.where.appendElementValues(values);
		final IOrmClassTool<BEAN> ormClassTool = this.ormClassToolMap.getOrmClassTool(this.clazz);

		final ResultSetReader<Object> resultSetReader = new ResultSetReader<Object>() {
			@Override
			public Object read(final ResultSet resultSet) throws SQLException {
				int rowCount = 0;
				while ( resultSet.next() ) {
					srr.read( ormClassTool.getOrmPersistor().mapRow("", resultSet, rowCount) , rowCount );
					rowCount++;
				}
				return null;
			}
		};
		final SqlPerformer sqlExec = this.session.sqlPerformer();
		sqlExec.setMaxRows(this.getMaxRows());
		sqlExec.setQueryTimeout(this.getQueryTimeout());
		sqlExec.query(this.renderSql(), resultSetReader, values);
	}

	@Override
	public BEAN findUnique() throws OrmNotUniqueResultException {
		final List<Object> values = new WrapperTypeList(this.typeFactory);
		this.where.appendElementValues(values);
		final IOrmClassTool<BEAN> ormClassTool = this.ormClassToolMap.getOrmClassTool(this.clazz);
		final GenericWrapper<Integer> resultsCount = new GenericWrapper<Integer>(null);
		final GenericWrapper<BEAN> wrapper = new GenericWrapper<BEAN>(null);
		final ResultSetReader<Object> resultSetReader = new ResultSetReader<Object>() {
			@Override
			public Object read(final ResultSet resultSet) throws SQLException {
				if ( resultSet.next() ) {
					resultsCount.setValue(1);
					wrapper.setValue( ormClassTool.getOrmPersistor().mapRow("", resultSet, 0) );
					if ( resultSet.next() ) {
						resultsCount.setValue(2);
					}
				} else {
					resultsCount.setValue(0);
				}
				return null;
			}
		};
		final SqlPerformer sqlExec = this.session.sqlPerformer();
		sqlExec.setMaxRows(2);
		sqlExec.setQueryTimeout(this.getQueryTimeout());
		sqlExec.query(this.renderSql(), resultSetReader, values);
		if ( resultsCount.getValue() != 1) {
			if ( resultsCount.getValue() == 0 ) {
				throw new OrmNotUniqueResultNoResultException("The query execution returned a number of rows different than one: no results found");
			}
			throw new OrmNotUniqueResultManyResultsException("The query execution returned a number of rows different than one: more than one result found");
		}
		return wrapper.getValue();
	}

	@Override
	public int findRowCount() {
		final List<Object> values = new WrapperTypeList(this.typeFactory);
		this.where.appendElementValues(values);
		final SqlPerformer sqlExec = this.session.sqlPerformer();
		sqlExec.setMaxRows(this.getMaxRows());
		sqlExec.setQueryTimeout(this.getQueryTimeout());
		return sqlExec.queryForInt(this.getGeneratedRowCountSql(), values);
	}

	@Override
	public FindQuery<BEAN> setDistinct(final boolean distinct) {
		this.distinct = distinct;
		this.versionStatus++;
		return this;
	}

	@Override
	public boolean isDistinct() throws OrmException {
		return this.distinct;
	}

	@Override
	public final int getStatusVersion() {
		return this.versionStatus + this.from.getElementStatusVersion() + this.where.getElementStatusVersion() + this.orderBy.getElementStatusVersion();

	}

	@Override
	public String getGeneratedRowCountSql() {
		final StringBuilder stringBuilder = new StringBuilder();
		stringBuilder.append("SELECT COUNT(*) ");
		this.from.renderSqlElement(stringBuilder);
		this.where.renderSqlElement(stringBuilder);
		return stringBuilder.toString();
	}

	@Override
	public final void doRender(final StringBuilder stringBuilder) {
		renderSelect(stringBuilder);
		this.from.renderSqlElement(stringBuilder);
		this.where.renderSqlElement(stringBuilder);
		this.orderBy.renderSqlElement(stringBuilder);
		stringBuilder.append(this.lockMode.getMode());
	}

	private void renderSelect(final StringBuilder stringBuilder) {
		final String alias = this.nameSolver.alias(this.nameSolverClassId);
		stringBuilder.append("SELECT ");
		if (this.distinct) {
			stringBuilder.append("DISTINCT ");
		}
		stringBuilder.append(this.ormClassToolMap.getOrmClassTool(this.clazz).getOrmCRUDQuery().getBaseSelectClause(alias + ".") );
		stringBuilder.append(" ");
	}

	@Override
	public final void appendValues(final List<Object> values) {
		this.where.appendElementValues(values);
	}

	@Override
	public FindQuery<BEAN> setLockMode(final LockMode lockMode) {
		this.lockMode = lockMode;
		this.versionStatus++;
		return this;
	}

	@Override
	public LockMode getLockMode() {
		return this.lockMode;
	}

	@Override
	public FindQuery<BEAN> join(final Class<?> joinClass) {
		return this.from.join(joinClass);
	}

	@Override
	public FindQuery<BEAN> join(final Class<?> joinClass, final String joinClassAlias) {
		return this.from.join(joinClass, joinClassAlias);
	}

	@Override
	public FindQuery<BEAN> naturalJoin(final Class<?> joinClass) {
		return this.from.naturalJoin(joinClass);
	}

	@Override
	public FindQuery<BEAN> naturalJoin(final Class<?> joinClass, final String joinClassAlias) {
		return this.from.naturalJoin(joinClass, joinClassAlias);
	}

	@Override
	public FindQuery<BEAN> innerJoin(final Class<?> joinClass) {
		return this.from.innerJoin(joinClass);
	}

	@Override
	public FindQuery<BEAN> innerJoin(final Class<?> joinClass, final String joinClassAlias) {
		return this.from.innerJoin(joinClass, joinClassAlias);
	}

	@Override
	public FindQuery<BEAN> innerJoin(final Class<?> joinClass, final String onLeftProperty,
			final String onRigthProperty) {
		return this.from.innerJoin(joinClass, onLeftProperty, onRigthProperty);
	}

	@Override
	public FindQuery<BEAN> innerJoin(final Class<?> joinClass, final String joinClassAlias,
			final String onLeftProperty, final String onRigthProperty) {
		return this.from.innerJoin(joinClass, joinClassAlias, onLeftProperty,
				onRigthProperty);
	}

	@Override
	public FindQuery<BEAN> leftOuterJoin(final Class<?> joinClass) {
		return this.from.leftOuterJoin(joinClass);
	}

	@Override
	public FindQuery<BEAN> leftOuterJoin(final Class<?> joinClass,
			final String joinClassAlias) {
		return this.from.leftOuterJoin(joinClass, joinClassAlias);
	}

	@Override
	public FindQuery<BEAN> leftOuterJoin(final Class<?> joinClass,
			final String onLeftProperty, final String onRigthProperty) {
		return this.from.leftOuterJoin(joinClass, onLeftProperty, onRigthProperty);
	}

	@Override
	public FindQuery<BEAN> leftOuterJoin(final Class<?> joinClass,
			final String joinClassAlias, final String onLeftProperty, final String onRigthProperty) {
		return this.from.leftOuterJoin(joinClass, joinClassAlias, onLeftProperty,
				onRigthProperty);
	}

	@Override
	public FindQuery<BEAN> rightOuterJoin(final Class<?> joinClass) {
		return this.from.rightOuterJoin(joinClass);
	}

	@Override
	public FindQuery<BEAN> rightOuterJoin(final Class<?> joinClass,
			final String joinClassAlias) {
		return this.from.rightOuterJoin(joinClass, joinClassAlias);
	}

	@Override
	public FindQuery<BEAN> rightOuterJoin(final Class<?> joinClass,
			final String onLeftProperty, final String onRigthProperty) {
		return this.from.rightOuterJoin(joinClass, onLeftProperty, onRigthProperty);
	}

	@Override
	public FindQuery<BEAN> rightOuterJoin(final Class<?> joinClass,
			final String joinClassAlias, final String onLeftProperty, final String onRigthProperty) {
		return this.from.rightOuterJoin(joinClass, joinClassAlias, onLeftProperty,
				onRigthProperty);
	}

	@Override
	public FindQuery<BEAN> fullOuterJoin(final Class<?> joinClass) {
		return this.from.fullOuterJoin(joinClass);
	}

	@Override
	public FindQuery<BEAN> fullOuterJoin(final Class<?> joinClass,
			final String joinClassAlias) {
		return this.from.fullOuterJoin(joinClass, joinClassAlias);
	}

	@Override
	public FindQuery<BEAN> fullOuterJoin(final Class<?> joinClass,
			final String onLeftProperty, final String onRigthProperty) {
		return this.from.fullOuterJoin(joinClass, onLeftProperty, onRigthProperty);
	}

	@Override
	public FindQuery<BEAN> fullOuterJoin(final Class<?> joinClass,
			final String joinClassAlias, final String onLeftProperty, final String onRigthProperty) {
		return this.from.fullOuterJoin(joinClass, joinClassAlias, onLeftProperty,
				onRigthProperty);
	}

	@Override
	public List<BEAN> findList() {
		return findList(this.getMaxRows());
	}

	private List<BEAN> findList(final int maxRows) {
		final List<Object> values = new WrapperTypeList(this.typeFactory);
		this.where.appendElementValues(values);
		final IOrmClassTool<BEAN> ormClassTool = this.ormClassToolMap.getOrmClassTool(this.clazz);

		final ResultSetReader<List<BEAN>> resultSetReader = new ResultSetReader<List<BEAN>>() {

			@Override
			public List<BEAN> read(final ResultSet resultSet) throws SQLException {
				final List<BEAN> resultList = new java.util.ArrayList<BEAN>();
				int rowCount = 0;
				while ( resultSet.next() ) {
					resultList.add( ormClassTool.getOrmPersistor().mapRow("", resultSet, rowCount++) );
				}
				return resultList;
			}
		};
		final SqlPerformer sqlExec = this.session.sqlPerformer();
		sqlExec.setMaxRows(this.getMaxRows());
		sqlExec.setQueryTimeout(this.getQueryTimeout());
		return sqlExec.query(this.renderSql(), resultSetReader, values);
	}

	@Override
	public BEAN find() throws OrmException, OrmNotUniqueResultException {
		List<BEAN> results = findList(1);
		if (!results.isEmpty()) {
			return results.get(0);
		}
		return null;
	}

}
