package com.jpattern.orm.query.find;

import java.math.BigDecimal;
import java.util.List;

import com.jpattern.orm.exception.OrmException;
import com.jpattern.orm.exception.OrmNotUniqueResultException;
import com.jpattern.orm.mapper.IOrmClassToolMap;
import com.jpattern.orm.persistor.type.TypeFactory;
import com.jpattern.orm.persistor.type.ext.WrapperTypeList;
import com.jpattern.orm.query.NameSolver;
import com.jpattern.orm.query.NameSolverConsumer;
import com.jpattern.orm.query.LockMode;
import com.jpattern.orm.query.SmartRenderableSqlQuery;
import com.jpattern.orm.session.ResultSetReader;
import com.jpattern.orm.session.ResultSetRowReader;
import com.jpattern.orm.session.SessionSqlPerformer;
import com.jpattern.orm.session.SqlPerformer;

/**
 * 
 * @author Francesco Cina
 *
 * 20/giu/2011
 */
public class CustomFindQueryOrm extends SmartRenderableSqlQuery implements CustomFindQuery, NameSolverConsumer {

	private final CustomFindSelectImpl select;
	private final SessionSqlPerformer session;
	private int queryTimeout = 0;
	private int maxRows = 0;
	private LockMode lockMode = LockMode.NO_LOCK;
	private final CustomFindWhereImpl where = new CustomFindWhereImpl(this);
	private final CustomFindOrderByImpl orderBy = new CustomFindOrderByImpl(this);
	private final CustomFindFromImpl from;
	private final TypeFactory typeFactory;
	private int versionStatus = 0;

	public CustomFindQueryOrm(final String selectClause, final IOrmClassToolMap ormClassToolMap, final SessionSqlPerformer session, final Class<?> clazz, final Integer nameSolverClassId, final TypeFactory typeFactory) {
		this.session = session;
		this.typeFactory = typeFactory;
		this.from = new CustomFindFromImpl(this, ormClassToolMap, clazz, nameSolverClassId);
		this.select = new CustomFindSelectImpl(selectClause);
	}

	@Override
	public void setNameSolver(final NameSolver nameSolver) {
		this.where.setNameSolver(nameSolver);
		this.orderBy.setNameSolver(nameSolver);
		this.from.setNameSolver(nameSolver);
		this.select.setNameSolver(nameSolver);
	}

	@Override
	public List<Object[]> findList() {
		final List<Object> values = new WrapperTypeList(this.typeFactory);
		this.where.appendElementValues(values);
		final SqlPerformer sqlExec = this.session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForList(renderSql(), values);
	}


	@Override
	public Object[] findUnique() throws OrmNotUniqueResultException {
		final List<Object> values = new WrapperTypeList(this.typeFactory);
		this.where.appendElementValues(values);
		final SqlPerformer sqlExec = this.session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForArray(renderSql(), values);
	}

	@Override
	public final int getStatusVersion() {
		return this.versionStatus + this.select.getElementStatusVersion() + this.from.getElementStatusVersion() + this.where.getElementStatusVersion() + this.orderBy.getElementStatusVersion();
	}

	@Override
	public final void doRender(final StringBuilder stringBuilder) {
		this.select.renderSqlElement(stringBuilder);
		this.from.renderSqlElement(stringBuilder);
		this.where.renderSqlElement(stringBuilder);
		this.orderBy.renderSqlElement(stringBuilder);
		stringBuilder.append(this.lockMode.getMode());
	}

	@Override
	public <T> T find(final ResultSetReader<T> rse) throws OrmException {
		final List<Object> values = new WrapperTypeList(this.typeFactory);
		this.where.appendElementValues(values);
		final SqlPerformer sqlExec = this.session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.query(renderSql(), rse, values);
	}

	@Override
	public <T> List<T> find(final ResultSetRowReader<T> rsrr) throws OrmException {
		final List<Object> values = new WrapperTypeList(this.typeFactory);
		this.where.appendElementValues(values);
		final SqlPerformer sqlExec = this.session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.query(renderSql(), rsrr, values);
	}

	@Override
	public <T> T findUnique(final ResultSetRowReader<T> rsrr) throws OrmException, OrmNotUniqueResultException {
		final List<Object> values = new WrapperTypeList(this.typeFactory);
		this.where.appendElementValues(values);
		final SqlPerformer sqlExec = this.session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForUnique(renderSql(), rsrr, values);
	}

	@Override
	public int findInt() throws OrmException {
		final List<Object> values = new WrapperTypeList(this.typeFactory);
		this.where.appendElementValues(values);
		final SqlPerformer sqlExec = this.session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForInt(renderSql(), values);
	}

	@Override
	public long findLong() throws OrmException {
		final List<Object> values = new WrapperTypeList(this.typeFactory);
		this.where.appendElementValues(values);
		final SqlPerformer sqlExec = this.session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForLong(renderSql(), values);
	}

	@Override
	public double findDouble() throws OrmException {
		final List<Object> values = new WrapperTypeList(this.typeFactory);
		this.where.appendElementValues(values);
		final SqlPerformer sqlExec = this.session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForDouble(renderSql(), values);
	}

	@Override
	public float findFloat() throws OrmException {
		final List<Object> values = new WrapperTypeList(this.typeFactory);
		this.where.appendElementValues(values);
		final SqlPerformer sqlExec = this.session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForFloat(renderSql(), values);
	}

	@Override
	public String findString() throws OrmException {
		final List<Object> values = new WrapperTypeList(this.typeFactory);
		this.where.appendElementValues(values);
		final SqlPerformer sqlExec = this.session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForString(renderSql(), values);
	}

	@Override
	public boolean findBoolean() throws OrmException {
		final List<Object> values = new WrapperTypeList(this.typeFactory);
		this.where.appendElementValues(values);
		final SqlPerformer sqlExec = this.session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForBoolean(renderSql(), values);
	}

	@Override
	public BigDecimal findBigDecimal() throws OrmException {
		final List<Object> values = new WrapperTypeList(this.typeFactory);
		this.where.appendElementValues(values);
		final SqlPerformer sqlExec = this.session.sqlPerformer();
		sqlExec.setMaxRows(getMaxRows());
		sqlExec.setQueryTimeout(getQueryTimeout());
		return sqlExec.queryForBigDecimal(renderSql(), values);
	}

	@Override
	public final int getMaxRows() throws OrmException {
		return this.maxRows;
	}

	@Override
	public final CustomFindQuery setQueryTimeout(final int queryTimeout) {
		this.queryTimeout = queryTimeout;
		return this;
	}

	@Override
	public final int getQueryTimeout() {
		return this.queryTimeout;
	}

	@Override
	public final CustomFindQuery setMaxRows(final int maxRows) throws OrmException {
		this.maxRows = maxRows;
		return this;
	}

	@Override
	public CustomFindQuery setLockMode(final LockMode lockMode) {
		this.lockMode = lockMode;
		this.versionStatus++;
		return this;
	}

	@Override
	public LockMode getLockMode() {
		return this.lockMode;
	}

	@Override
	public final CustomFindWhere where() throws OrmException {
		return this.where;
	}

	@Override
	public final CustomFindOrderBy orderBy() throws OrmException {
		return this.orderBy;
	}

	@Override
	public final void appendValues(final List<Object> values) {
		this.where.appendElementValues(values);
	}

	public final String renderSqlElement() {
		return this.from.renderSqlElement();
	}

	public final void renderSqlElement(final StringBuilder stringBuilder) {
		this.from.renderSqlElement(stringBuilder);
	}

	public final boolean isElementStatusChanged() {
		return this.from.isElementStatusChanged();
	}

	@Override
	public CustomFindQuery join(final Class<?> joinClass) {
		return this.from.join(joinClass);
	}

	@Override
	public CustomFindQuery join(final Class<?> joinClass, final String joinClassAlias) {
		return this.from.join(joinClass, joinClassAlias);
	}

	@Override
	public CustomFindQuery naturalJoin(final Class<?> joinClass) {
		return this.from.naturalJoin(joinClass);
	}

	@Override
	public CustomFindQuery naturalJoin(final Class<?> joinClass, final String joinClassAlias) {
		return this.from.naturalJoin(joinClass, joinClassAlias);
	}

	@Override
	public CustomFindQuery innerJoin(final Class<?> joinClass) {
		return this.from.innerJoin(joinClass);
	}

	@Override
	public CustomFindQuery innerJoin(final Class<?> joinClass, final String joinClassAlias) {
		return this.from.innerJoin(joinClass, joinClassAlias);
	}

	@Override
	public CustomFindQuery innerJoin(final Class<?> joinClass, final String onLeftProperty,
			final String onRigthProperty) {
		return this.from.innerJoin(joinClass, onLeftProperty, onRigthProperty);
	}

	@Override
	public CustomFindQuery innerJoin(final Class<?> joinClass, final String joinClassAlias,
			final String onLeftProperty, final String onRigthProperty) {
		return this.from.innerJoin(joinClass, joinClassAlias, onLeftProperty,
				onRigthProperty);
	}

	@Override
	public CustomFindQuery leftOuterJoin(final Class<?> joinClass) {
		return this.from.leftOuterJoin(joinClass);
	}

	@Override
	public CustomFindQuery leftOuterJoin(final Class<?> joinClass,
			final String joinClassAlias) {
		return this.from.leftOuterJoin(joinClass, joinClassAlias);
	}

	@Override
	public CustomFindQuery leftOuterJoin(final Class<?> joinClass,
			final String onLeftProperty, final String onRigthProperty) {
		return this.from.leftOuterJoin(joinClass, onLeftProperty, onRigthProperty);
	}

	@Override
	public CustomFindQuery leftOuterJoin(final Class<?> joinClass,
			final String joinClassAlias, final String onLeftProperty, final String onRigthProperty) {
		return this.from.leftOuterJoin(joinClass, joinClassAlias, onLeftProperty,
				onRigthProperty);
	}

	@Override
	public CustomFindQuery rightOuterJoin(final Class<?> joinClass) {
		return this.from.rightOuterJoin(joinClass);
	}

	@Override
	public CustomFindQuery rightOuterJoin(final Class<?> joinClass,
			final String joinClassAlias) {
		return this.from.rightOuterJoin(joinClass, joinClassAlias);
	}

	@Override
	public CustomFindQuery rightOuterJoin(final Class<?> joinClass,
			final String onLeftProperty, final String onRigthProperty) {
		return this.from.rightOuterJoin(joinClass, onLeftProperty, onRigthProperty);
	}

	@Override
	public CustomFindQuery rightOuterJoin(final Class<?> joinClass,
			final String joinClassAlias, final String onLeftProperty, final String onRigthProperty) {
		return this.from.rightOuterJoin(joinClass, joinClassAlias, onLeftProperty,
				onRigthProperty);
	}

	@Override
	public CustomFindQuery fullOuterJoin(final Class<?> joinClass) {
		return this.from.fullOuterJoin(joinClass);
	}

	@Override
	public CustomFindQuery fullOuterJoin(final Class<?> joinClass,
			final String joinClassAlias) {
		return this.from.fullOuterJoin(joinClass, joinClassAlias);
	}

	@Override
	public CustomFindQuery fullOuterJoin(final Class<?> joinClass,
			final String onLeftProperty, final String onRigthProperty) {
		return this.from.fullOuterJoin(joinClass, onLeftProperty, onRigthProperty);
	}

	@Override
	public CustomFindQuery fullOuterJoin(final Class<?> joinClass,
			final String joinClassAlias, final String onLeftProperty, final String onRigthProperty) {
		return this.from.fullOuterJoin(joinClass, joinClassAlias, onLeftProperty,
				onRigthProperty);
	}

	public final int getElementStatusVersion() {
		return this.from.getElementStatusVersion();
	}

	public final void doElementRender(final StringBuilder stringBuilder) {
		this.from.doElementRender(stringBuilder);
	}

	public final void appendElementValues(final List<Object> values) {
		this.from.appendElementValues(values);
	}

	@Override
	public String toString() {
		return this.from.toString();
	}

	@Override
	public CustomFindQuery setDistinct(final boolean distinct) {
		this.select.setDistinct(distinct);
		return this;
	}

	@Override
	public boolean isDistinct() throws OrmException {
		return this.select.isDistinct();
	}

}
