package com.jpattern.orm.query.clause.where;

import java.util.Collection;
import java.util.List;

import com.jpattern.orm.query.BaseFindQuery;

/**
 * A factory helper to build {@link ExpressionElement}s
 * @author ufo
 *
 */
public class Expression {

	private Expression() {
	}

	/**
	 * Express the "Equals to" relation between an object's property
	 * and a fixed value.
	 * 
	 * @param property
	 * @param value
	 * @return
	 */
	public static  EqExpressionElement eq(final String property, final Object value) {
		return new EqExpressionElement(property, value);
	}

	/**
	 * Express the "Equals to" relation between objects properties
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	public static  EqPropertiesExpressionElement eqProperties(final String firstProperty, final String secondProperty) {
		return new EqPropertiesExpressionElement(firstProperty, secondProperty);
	}

	/**
	 * Express the "Lesser or equals to" relation between an object's property
	 * and a fixed value.
	 * 
	 * @param property
	 * @param value
	 * @return
	 */
	public static  LeExpressionElement le(final String property, final Object value) {
		return new LeExpressionElement(property, value);
	}

	/**
	 * Express the "Lesser or equals to" relation between objects properties
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	public static  LePropertiesExpressionElement leProperties(final String firstProperty, final String secondProperty) {
		return new LePropertiesExpressionElement(firstProperty, secondProperty);
	}

	/**
	 * Express the "Greater or equals to" relation between an object's property
	 * and a fixed value.
	 * 
	 * @param property
	 * @param value
	 * @return
	 */
	public static  GeExpressionElement ge(final String property, final Object value) {
		return new GeExpressionElement(property, value);
	}

	/**
	 * Express the "Greater or equals to" relation between objects properties
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	public static  GePropertiesExpressionElement geProperties(final String firstProperty, final String secondProperty) {
		return new GePropertiesExpressionElement(firstProperty, secondProperty);
	}

	/**
	 * 
	 * Express the "Lesser than" relation between an object's property
	 * and a fixed value.
	 * 
	 * @param property
	 * @param value
	 * @return
	 */
	public static  LtExpressionElement lt(final String property, final Object value) {
		return new LtExpressionElement(property, value);
	}

	/**
	 * Express the "Lesser than" relation between objects properties
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	public static  LtPropertiesExpressionElement ltProperties(final String firstProperty, final String secondProperty) {
		return new LtPropertiesExpressionElement(firstProperty, secondProperty);
	}

	/**
	 * Express the "Greater than" relation between an object's property
	 * and a fixed value.
	 * 
	 * @param property
	 * @param value
	 * @return
	 */
	public static  GtExpressionElement gt(final String property, final Object value) {
		return new GtExpressionElement(property, value);
	}

	/**
	 * Express the "Greater than" relation between objects properties
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	public static  GtPropertiesExpressionElement gtProperties(final String firstProperty, final String secondProperty) {
		return new GtPropertiesExpressionElement(firstProperty, secondProperty);
	}

	/**
	 * Express the "Insensitive Equal To" between an object's property
	 * and a fixed value (it uses a lower() function to make both case insensitive).
	 * 
	 * @param propertyName
	 * @param value
	 * @return
	 */
	public static  IEqExpressionElement ieq(final String property, final Object value) {
		return new IEqExpressionElement(property, value);
	}

	/**
	 * Express the "Insensitive Equal To" bbetween objects properties
	 * (it uses a lower() function to make both case insensitive).
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	public static  IEqPropertiesExpressionElement ieqProperties(final String firstProperty, final String secondProperty) {
		return new IEqPropertiesExpressionElement(firstProperty, secondProperty);
	}

	/**
	 * Express the "Not Equals to" relation between objects properties.
	 * 
	 * @param property
	 * @param value
	 * @return
	 */
	public static  NeExpressionElement ne(final String property, final Object value) {
		return new NeExpressionElement(property, value);
	}

	/**
	 * Express the "Not Equals to" relation between an object's property
	 * and a fixed value.
	 * 
	 * @param firstProperty
	 * @param secondProperty
	 * @return
	 */
	public static  NePropertiesExpressionElement neProperties(final String firstProperty, final String secondProperty) {
		return new NePropertiesExpressionElement(firstProperty, secondProperty);
	}

	/**
	 * Case insensitive Like - property like value where the value contains the
	 * SQL wild card characters % (percentage) and _ (underscore).
	 * 
	 * @param propertyName
	 * @param value
	 * @return
	 */
	public static  ILikeExpressionElement ilike(final String property, final Object value) {
		return new ILikeExpressionElement(property, value);
	}

	/**
	 * Not In - property has a value in the collection of values.
	 * 
	 * @param propertyName
	 * @param values
	 * @return
	 */
	public static  NInExpressionElement nin(final String property, final Collection<?> values) {
		return new NInExpressionElement(property, values);
	}

	/**
	 * Not In - property has a value in the array of values.
	 * 
	 * @param propertyName
	 * @param values
	 * @return
	 */
	public static  NInExpressionElement nin(final String property, final Object[] values) {
		return new NInExpressionElement(property, values);
	}

	/**
	 * Not In - using a subQuery.
	 * 
	 * @param propertyName
	 * @param subQuery
	 * @return
	 */
	public static  NInSubQueryExpressionElement nin(final String property, final BaseFindQuery subQuery) {
		return new NInSubQueryExpressionElement(property, subQuery);
	}

	/**
	 * In - property has a value in the collection of values.
	 * 
	 * @param propertyName
	 * @param values
	 * @return
	 */
	public static  InExpressionElement in(final String property, final Collection<?> values) {
		return new InExpressionElement(property, values);
	}

	/**
	 * In - property has a value in the array of values.
	 * 
	 * @param propertyName
	 * @param values
	 * @return
	 */
	public static  InExpressionElement in(final String property, final Object[] values) {
		return new InExpressionElement(property, values);
	}

	/**
	 * In - using a subQuery.
	 * 
	 * @param propertyName
	 * @param subQuery
	 * @return
	 */
	public static  InSubQueryExpressionElement in(final String property, final BaseFindQuery subQuery) {
		return new InSubQueryExpressionElement(property, subQuery);
	}

	/**
	 * Is Not Null - property is not null.
	 * 
	 * @param propertyName
	 * @return
	 */
	public static  IsNotNullExpressionElement isNotNull(final String property) {
		return new IsNotNullExpressionElement(property);
	}

	/**
	 * Is Null - property is null.
	 * 
	 * @param propertyName
	 * @return
	 */
	public static  IsNullExpressionElement isNull(final String property) {
		return new IsNullExpressionElement(property);
	}

	/**
	 * Like - property like value where the value contains the SQL wild card
	 * characters % (percentage) and _ (underscore).
	 * 
	 * @param propertyName
	 * @param value
	 */
	public static  LikeExpressionElement like(final String property, final Object value) {
		return new LikeExpressionElement(property, value);
	}

	/**
	 * Not Like - property like value where the value contains the SQL wild card
	 * characters % (percentage) and _ (underscore).
	 * 
	 * @param propertyName
	 * @param value
	 */
	public static  NLikeExpressionElement nlike(final String property, final Object value) {
		return new NLikeExpressionElement(property, value);
	}

	/**
	 * Negate the expression (prefix it with NOT).
	 * 
	 * @param exp
	 * @return
	 */
	public static  NotExpressionElement not(final ExpressionElement expression) {
		return new NotExpressionElement(expression);
	}

	/**
	 * Or - join more expressions with a logical or.
	 * 
	 * @param expressionElements
	 * @return
	 */
	public static OrExpressionElement or(final ExpressionElement... expressionElements) {
		return new OrExpressionElement(expressionElements);
	}

	/**
	 * Or - join more expressions with a logical or.
	 * 
	 * @param expressionElements
	 * @return
	 */
	public static OrExpressionElement or(final List<ExpressionElement> expressionElements) {
		return new OrExpressionElement(expressionElements);
	}

	/**
	 * And - join more expressions with a logical and.
	 * 
	 * @param expressionElements
	 * @return
	 */
	public static AndExpressionElement and(final List<ExpressionElement> expressionElements) {
		return new AndExpressionElement(expressionElements);
	}

	/**
	 * And - join more expressions with a logical and.
	 * 
	 * @param expressionElements
	 * @return
	 */
	public static AndExpressionElement and(final ExpressionElement... expressionElements) {
		return new AndExpressionElement(expressionElements);
	}
}
